

package DESLib

annotation(preferedView="info",
  Documentation(info="<HTML>
<p>
<b>Parallel DEVS and Process-Oriented Modeling in Modelica</b>
</p>

<p>

This library contains four packages that can be used to model discrete-event systems:
<ul>
<li> <a href=\"Modelica://DESLib.RandomLib\">RandomLib</a> can be used to generate random numbers and variates, following continuous and discrete probability distributions.
<li> <a href=\"Modelica://DESLib.DEVSLib\">DEVSLib</a> can be used to model discrete-event systems following the Parallel DEVS formalism.
<li> <a href=\"Modelica://DESLib.SIMANLib\">SIMANLib</a> can be used to model discrete-event systems following the process-oriented approach. Its functionalities are equivalent to some of the functionalities of the SIMAN modeling language.
<li> <a href=\"Modelica://DESLib.ARENALib\">ARENALib</a> can be used to model discrete-event systems following the process-oriented approach. Its functionalities are equivalent to some of the functionalities of BasicProcess Panel in the Arena simulation environment.
</ul>
</p>
<p>
The DEVSLib, SIMANLib and ARENALib libraries include interfaces to communicate with the rest of Modelica libraries, allowing to construct hybrid models.
</p>

<p>
<dl>
<dt><b>Author:</b></dt>
<dd><a href=\"http://www.euclides.dia.uned.es/vsanz\"> Victorino Sanz </a><br>
    Dpto. Informtica y Automtica, UNED<br>
    Juan del Rosal, 16<br>
    28040, Madrid<br>
    Spain<br>
    Email: <A HREF=\"mailto:vsanz@dia.uned.es\">vsanz@dia.uned.es</A><br>
    Website: <A HREF=\"http://www.euclides.dia.uned.es/vsanz\">www.euclides.dia.uned.es/vsanz</A><br>
    Library website: <A HREF=\"http://www.euclides.dia.uned.es/DESLib\">www.euclides.dia.uned.es/DESLib</A></dd>
</dl>
</p>


<p>
<dt><b>Copying:</b></dt>
<p>
<p>
Licensed by Victorino Sanz under the Modelica License 2 <br>
Copyright 2009, Victorino Sanz.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your ouwn risk;</u> it can be redistributed and/or modified under the terms of the Modelica license 2,
see the license conditions (including the disclaimer of warranty) <a href=\"Modelica://DESLib.ModelicaLicense2\">here</a> or at <a href=\"http://www.modelica.org/licenses/ModelicaLicense2\">http://www.modelica.org/licenses/ModelicaLicense2</a>
</p><br>
</HTML>
"), uses(                           DEVSLib(version="2.2"),
      DESLib(version="1"),
      DESLib73(version="1"),
      Modelica(version="2.2.1"),
      Interactive(version="2")),
    version="1",
    conversion(noneFromVersion=""));

class ModelicaLicense2 "Modelica License 2"

  annotation (DocumentationClass=true,Documentation(info="<html>

<P STYLE=\"text-indent: 0pt; font-weight: medium\">
This page contains the &ldquo;Modelica License 2&rdquo; which was
released by the Modelica Association on Nov. 19, 2008. It is used for
all material from the Modelica Association provided to the public
after this date. It is recommended that other providers of free
Modelica packages license their library also under &ldquo;Modelica
License 2&rdquo;. Additionally, this document contains a description
how to apply the license and has a &ldquo;Frequently Asked Questions&rdquo;
section.</P>

<P STYLE=\"text-indent: 0pt; line-height: 150%\">
<A HREF=\"#1. The Modelica License 2|outline\">             The Modelica License 2</A>
   (in other formats:
   <A HREF=\"http://www.modelica.org/licenses/ModelicaLicense2.html\">standalone html</A>,
   <A HREF=\"http://www.modelica.org/licenses/ModelicaLicense2.pdf\">pdf</A>,
   <A HREF=\"http://www.modelica.org/licenses/ModelicaLicense2.odt\">odt</A>,
   <A HREF=\"http://www.modelica.org/licenses/ModelicaLicense2.doc\">doc</A>)<br>
   <A HREF=\"#2. How to Apply the Modelica License 2|outline\">How to Apply the Modelica License 2</A><br>
   <A HREF=\"#3. Frequently Asked Questions|outline\">         Frequently Asked Questions</A><br>&nbsp;</P>

<hr>
<H4><A NAME=\"_Ref208223390\"></A><A NAME=\"1. The Modelica License 2|outline\"></A>
The Modelica License 2</H4>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>Preamble. </B>The goal of this license is that Modelica related
model libraries, software, images, documents, data files etc. can be
used freely in the original or a modified form, in open source and in
commercial environments (as long as the license conditions below are
fulfilled, in particular sections 2c) and 2d). The Original Work is
provided free of charge and the use is completely at your own risk.
Developers of free Modelica packages are encouraged to utilize this
license for their work.</P>
<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
The Modelica License applies to any Original Work that contains the
following licensing notice adjacent to the copyright notice(s) for
this Original Work:</P>
<P ALIGN=CENTER STYLE=\"text-indent: 0pt\"><B>Licensed
by &lt;name of Licensor&gt; under the Modelica License 2</B></P>
<P STYLE=\"text-indent: 0pt; widows: 2; orphans: 2\"><B>1.
Definitions.</B></P>
<OL style=\"list-style-type:lower-latin\">
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        &ldquo;License&rdquo; is this Modelica License.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        &ldquo;Original Work&rdquo; is any work of authorship, including
        software, images, documents, data files, that contains the above
        licensing notice or that is packed together with a licensing notice
        referencing it.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        &ldquo;Licensor&rdquo; is the provider of the Original Work who has
        placed this licensing notice adjacent to the copyright notice(s) for
        the Original Work. The Original Work is either directly provided by
        the owner of the Original Work, or by a licensee of the owner.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        &ldquo;Derivative Work&rdquo; is any modification of the Original
        Work which represents, as a whole, an original work of authorship.
        For the matter of clarity and as examples:
        </P>
        <OL style=\"list-style-type:upper-latin\">
                <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
                Derivative Work shall not include work that remains separable from
                the Original Work, as well as merely extracting a part of the
                Original Work without modifying it.</P>
                <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
                Derivative Work shall not include (a) fixing of errors and/or (b)
                adding vendor specific Modelica annotations and/or (c) using a
                subset of the classes of a Modelica package, and/or (d) using a
                different representation, e.g., a binary representation.</P>
                <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
                Derivative Work shall include classes that are copied from the
                Original Work where declarations, equations or the documentation
                are modified.
                </P>
                <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
                Derivative Work shall include executables to simulate the models
                that are generated by a Modelica translator based on the Original
                Work (of a Modelica package).</P>
        </OL>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        &ldquo;Modified Work&rdquo; is any modification of the Original Work
        with the following exceptions: (a) fixing of errors and/or (b)
        adding vendor specific Modelica annotations and/or (c) using a
        subset of the classes of a Modelica package, and/or (d) using a
        different representation, e.g., a binary representation.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        &quot;Source Code&quot; means the preferred form of the Original
        Work for making modifications to it and all available documentation
        describing how to modify the Original Work.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        &ldquo;You&rdquo; means an individual or a legal entity exercising
        rights under, and complying with all of the terms of, this License.
        </P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        &ldquo;Modelica package&rdquo; means any Modelica library that is
        defined with the<BR>&ldquo;<FONT FACE=\"Courier New, monospace\"><FONT SIZE=2 STYLE=\"font-size: 9pt\"><B>package</B></FONT></FONT><FONT FACE=\"Courier New, monospace\"><FONT SIZE=2 STYLE=\"font-size: 9pt\">
        &lt;Name&gt; ... </FONT></FONT><FONT FACE=\"Courier New, monospace\"><FONT SIZE=2 STYLE=\"font-size: 9pt\"><B>end</B></FONT></FONT><FONT FACE=\"Courier New, monospace\"><FONT SIZE=2 STYLE=\"font-size: 9pt\">
        &lt;Name&gt;;</FONT></FONT>&ldquo; Modelica language element.</P>
</OL>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>2. </B><B>Grant of Copyright License. </B>Licensor grants You a
worldwide, royalty-free, non-exclusive, sublicensable license, for
the duration of the copyright, to do the following:</P>
<OL style=\"list-style-type:lower-latin\">
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        To reproduce the Original Work in copies, either alone or as part of
        a collection.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        To create Derivative Works according to Section 1d) of this License.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        To distribute or communicate to the public copies of the <U>Original
        Work</U> or a <U>Derivative Work</U> under <U>this License</U>. No
        fee, neither as a copyright-license fee, nor as a selling fee for
        the copy as such may be charged under this License. Furthermore, a
        verbatim copy of this License must be included in any copy of the
        Original Work or a Derivative Work under this License.<BR>      For
        the matter of clarity, it is permitted A) to distribute or
        communicate such copies as part of a (possible commercial)
        collection where other parts are provided under different licenses
        and a license fee is charged for the other parts only and B) to
        charge for mere printing and shipping costs.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        To distribute or communicate to the public copies of a <U>Derivative
        Work</U>, alternatively to Section 2c), under <U>any other license</U>
        of your choice, especially also under a license for
        commercial/proprietary software, as long as You comply with Sections
        3, 4 and 8 below. <BR>      For the matter of clarity, no
        restrictions regarding fees, either as to a copyright-license fee or
        as to a selling fee for the copy as such apply.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        To perform the Original Work publicly.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
        To display the Original Work publicly.</P>
</OL>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>3. </B><B>Acceptance. </B>Any use of the Original Work or a
Derivative Work, or any action according to either Section 2a) to 2f)
above constitutes Your acceptance of this License.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>4. </B><B>Designation of Derivative Works and of Modified Works.
</B>The identifying designation of Derivative Work and of Modified
Work must be different to the corresponding identifying designation
of the Original Work. This means especially that the (root-level)
name of a Modelica package under this license must be changed if the
package is modified (besides fixing of errors, adding vendor specific
Modelica annotations, using a subset of the classes of a Modelica
package, or using another representation, e.g. a binary
representation).</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>5. </B><B>Grant of Patent License.</B>
Licensor grants You a worldwide, royalty-free, non-exclusive, sublicensable license,
under patent claims owned by the Licensor or licensed to the Licensor by
the owners of the Original Work that are embodied in the Original Work
as furnished by the Licensor, for the duration of the patents,
to make, use, sell, offer for sale, have made, and import the Original Work
and Derivative Works under the conditions as given in Section 2.
For the matter of clarity, the license regarding Derivative Works covers
patent claims to the extent as they are embodied in the Original Work only.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>6. Provision of </B><B>Source Code. </B>Licensor agrees to provide
You with a copy of the Source Code of the Original Work but reserves
the right to decide freely on the manner of how the Original Work is
provided.<br>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;For the matter of clarity, Licensor might provide only a binary
representation of the Original Work. In that case, You may (a) either
reproduce the Source Code from the binary representation if this is
possible (e.g., by performing a copy of an encrypted Modelica
package, if encryption allows the copy operation) or (b) request the
Source Code from the Licensor who will provide it to You.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>7. </B><B>Exclusions from License Grant. </B>Neither the names of
Licensor, nor the names of any contributors to the Original Work, nor
any of their trademarks or service marks, may be used to endorse or
promote products derived from this Original Work without express
prior permission of the Licensor. Except as otherwise expressly
stated in this License and in particular in Sections 2 and 5, nothing
in this License grants any license to Licensor&rsquo;s trademarks,
copyrights, patents, trade secrets or any other intellectual
property, and no patent license is granted to make, use, sell, offer
for sale, have made, or import embodiments of any patent claims.<br>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;No license is granted to the trademarks of
Licensor even if such trademarks are included in the Original Work,
except as expressly stated in this License. Nothing in this License
shall be interpreted to prohibit Licensor from licensing under terms
different from this License any Original Work that Licensor otherwise
would have a right to license.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>8. </B><B>Attribution Rights. </B>You must retain in the Source
Code of the Original Work and of any Derivative Works that You
create, all author, copyright, patent, or trademark notices, as well
as any descriptive text identified therein as an &quot;Attribution
Notice&quot;. The same applies to the licensing notice of this
License in the Original Work. For the matter of clarity, &ldquo;author
notice&rdquo; means the notice that identifies the original
author(s). <br>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;You must cause the Source Code for any Derivative
Works that You create to carry a prominent Attribution Notice
reasonably calculated to inform recipients that You have modified the
Original Work. <br>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;In case the Original Work or Derivative Work is not provided in
Source Code, the Attribution Notices shall be appropriately
displayed, e.g., in the documentation of the Derivative Work.</P>

<P STYLE=\"text-indent: 0pt\"><B>9. </B><B>Disclaimer
of Warranty. <BR></B><U><B>The Original Work is provided under this
License on an &quot;as is&quot; basis and without warranty, either
express or implied, including, without limitation, the warranties of
non-infringement, merchantability or fitness for a particular
purpose. The entire risk as to the quality of the Original Work is
with You.</B></U><B> </B>This disclaimer of warranty constitutes an
essential part of this License. No license to the Original Work is
granted by this License except under this disclaimer.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>10. </B><B>Limitation of Liability. </B>Under no circumstances and
under no legal theory, whether in tort (including negligence),
contract, or otherwise, shall the Licensor, the owner or a licensee
of the Original Work be liable to anyone for any direct, indirect,
general, special, incidental, or consequential damages of any
character arising as a result of this License or the use of the
Original Work including, without limitation, damages for loss of
goodwill, work stoppage, computer failure or malfunction, or any and
all other commercial damages or losses. This limitation of liability
shall not apply to the extent applicable law prohibits such
limitation.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>11. </B><B>Termination. </B>This License conditions your rights to
undertake the activities listed in Section 2 and 5, including your
right to create Derivative Works based upon the Original Work, and
doing so without observing these terms and conditions is prohibited
by copyright law and international treaty. Nothing in this License is
intended to affect copyright exceptions and limitations. This License
shall terminate immediately and You may no longer exercise any of the
rights granted to You by this License upon your failure to observe
the conditions of this license.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>12. </B><B>Termination for Patent Action. </B>This License shall
terminate automatically and You may no longer exercise any of the
rights granted to You by this License as of the date You commence an
action, including a cross-claim or counterclaim, against Licensor,
any owners of the Original Work or any licensee alleging that the
Original Work infringes a patent. This termination provision shall
not apply for an action alleging patent infringement through
combinations of the Original Work under combination with other
software or hardware.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>13. </B><B>Jurisdiction. </B>Any action or suit relating to this
License may be brought only in the courts of a jurisdiction wherein
the Licensor resides and under the laws of that jurisdiction
excluding its conflict-of-law provisions. The application of the
United Nations Convention on Contracts for the International Sale of
Goods is expressly excluded. Any use of the Original Work outside the
scope of this License or after its termination shall be subject to
the requirements and penalties of copyright or patent law in the
appropriate jurisdiction. This section shall survive the termination
of this License.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>14. </B><B>Attorneys&rsquo; Fees. </B>In any action to enforce the
terms of this License or seeking damages relating thereto, the
prevailing party shall be entitled to recover its costs and expenses,
including, without limitation, reasonable attorneys' fees and costs
incurred in connection with such action, including any appeal of such
action. This section shall survive the termination of this License.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; widows: 2; orphans: 2\">
<B>15. </B><B>Miscellaneous. </B>
</P>
<OL style=\"list-style-type:lower-latin\">
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">If any
        provision of this License is held to be unenforceable, such
        provision shall be reformed only to the extent necessary to make it
        enforceable.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">No verbal
        ancillary agreements have been made. Changes and additions to this
        License must appear in writing to be valid. This also applies to
        changing the clause pertaining to written form.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">You may use the
        Original Work in all ways not otherwise restricted or conditioned by
        this License or by law, and Licensor promises not to interfere with
        or be responsible for such uses by You.</P>
</OL>

<hr>

<H4><A NAME=\"2. How to Apply the Modelica License 2|outline\"></A>
How to Apply the Modelica License 2</H4>

<P STYLE=\"text-indent: 0pt; font-weight: medium\">
At the top level of your Modelica package and at every important
subpackage, add the following notices in the info layer of the
package:</P>

<table border=\"0\"><tr><td>&nbsp;&nbsp;&nbsp;</td><td>
Licensed by &lt;Licensor&gt; under the Modelica License 2<br>
Copyright &copy; &lt;year1&gt;-&lt;year2&gt;, &lt;name of copyright
holder(s)&gt;.
</td></tr></table>

<table border=\"0\"><tr><td>&nbsp;&nbsp;&nbsp;</td><td>
<I>This Modelica package is <U>free</U> software and
the use is completely at <U>your own risk</U>;
it can be redistributed and/or modified under the terms of the
Modelica license 2, see the license conditions (including the
disclaimer of warranty)
<A HREF=\"modelica://Library.UsersGuide.ModelicaLicense2\">here</A></U>
or at
<A HREF=\"http://www.modelica.org/licenses/ModelicaLicense2\">
http://www.modelica.org/licenses/ModelicaLicense2</A>.
</td></tr></table>

<P STYLE=\"text-indent: 0pt; font-weight: medium\">Include
a copy of the Modelica License 2 under
<B>&lt;library&gt;.UsersGuide.ModelicaLicense2</B>
(use <A HREF=\"http://www.modelica.org/licenses/ModelicaLicense2.mo\">
http://www.modelica.org/licenses/ModelicaLicense2.mo</A>). Furthermore, add
the list of authors and contributors under
<B>&lt;library&gt;.UsersGuide.Contributors</B> or
<B>&lt;library&gt;.UsersGuide.Contact</B>.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium\">For
example, sublibrary Modelica.Blocks of the Modelica Standard Library
may have the following notices:</P>

<table border=\"0\"><tr><td>&nbsp;&nbsp;&nbsp;</td><td>
Licensed by Modelica Association under the Modelica License 2<br>
Copyright &copy; 1998-2008, Modelica Association.
</td></tr></table>

<table border=\"0\"><tr><td>&nbsp;&nbsp;&nbsp;</td><td>
<I>This Modelica package is <U>free</U> software and
the use is completely at <U>your own risk</U>;
it can be redistributed and/or modified under the terms of the
Modelica license 2, see the license conditions (including the
disclaimer of warranty)
<A HREF=\"modelica://Modelica.UsersGuide.ModelicaLicense2\">here</A></U>
or at
<A HREF=\"http://www.modelica.org/licenses/ModelicaLicense2\">
http://www.modelica.org/licenses/ModelicaLicense2</A>.</I>
</td></tr></table>


<P STYLE=\"text-indent: 0pt; font-weight: medium\">For
C-source code and documents, add similar notices in the corresponding
file.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium\">For
images, add a &ldquo;readme.txt&rdquo; file to the directories where
the images are stored and include a similar notice in this file.</P>


<P STYLE=\"text-indent: 0pt; font-weight: medium\">In
these cases, save a copy of the Modelica License 2 in one directory
of the distribution, e.g.,
<A HREF=\"http://www.modelica.org/licenses/ModelicaLicense2.html\">
http://www.modelica.org/licenses/ModelicaLicense2.html</A>
in directory <B>&lt;library&gt;/help/documentation/ModelicaLicense2.html</B>.</P>


<hr>
<H4><A NAME=\"_Ref202857474\"></A><A NAME=\"3. Frequently Asked Questions|outline\"></A>
Frequently Asked Questions</H4>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">
This section contains questions/answer to users and/or distributors of
Modelica packages and/or documents under Modelica License 2. Note,
the answers to the questions below are not a legal interpretation of
the Modelica License 2. In case of a conflict, the language of the
license shall prevail.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium\"><BR>
</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; color: green; font-size:140%\">
<b>Using or Distributing a Modelica <U>Package</U> under the Modelica License 2</b></P>

<P STYLE=\"text-indent: 0pt\"><B>What are the main
differences to the previous version of the Modelica License?</B></P>
<OL>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">
        Modelica License 1 is unclear whether the licensed Modelica package
        can be distributed under a different license. Version 2 explicitly
        allows that &ldquo;Derivative Work&rdquo; can be distributed under
        any license of Your choice, see examples in Section 1d) as to what
        qualifies as Derivative Work (so, version 2 is clearer).</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">
        If You modify a Modelica package under Modelica License 2 (besides
        fixing of errors, adding vendor specific Modelica annotations, using
        a subset of the classes of a Modelica package, or using another
        representation, e.g., a binary representation), you must rename the
        root-level name of the package for your distribution. In version 1
        you could keep the name (so, version 2 is more restrictive). The
        reason of this restriction is to reduce the risk that Modelica
        packages are available that have identical names, but different
        functionality.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">
        Modelica License 1 states that &ldquo;It is not allowed to charge a
        fee for the original version or a modified version of the software,
        besides a reasonable fee for distribution and support<SPAN LANG=\"en-GB\">&ldquo;.
        Version 2 has a </SPAN>similar intention for all Original Work under
        <U>Modelica License 2</U> (to remain free of charge and open source)
        but states this more clearly as &ldquo;No fee, neither as a
        copyright-license fee, nor as a selling fee for the copy as such may
        be charged&rdquo;. Contrary to version 1, Modelica License 2 has no
        restrictions on fees for Derivative Work that is provided under a
        different license (so, version 2 is clearer and has fewer
        restrictions).</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">
        Modelica License 2 introduces several useful provisions for the
        licensee (articles 5, 6, 12), and for the licensor (articles 7, 12,
        13, 14) that have no counter part in version 1.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">
        Modelica License 2 can be applied to all type of work, including
        documents, images and data files, contrary to version 1 that was
        dedicated for software only (so, version 2 is more general).</P>
</OL>

<P STYLE=\"text-indent: 0pt\"><B>Can I distribute a
Modelica package (under Modelica License 2) as part of my commercial
Modelica modeling and simulation environment?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">Yes,
according to Section 2c). However, you are not allowed to charge a
fee for this part of your environment. Of course, you can charge for
your part of the environment.
</P>

<P STYLE=\"text-indent: 0pt\"><B>Can I distribute a
Modelica package (under Modelica License 2) under a different
license?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">No.
The license of an unmodified Modelica package cannot be changed
according to Sections 2c) and 2d). This means that you cannot <U>sell</U>
copies of it, any distribution has to be free of charge.</P>

<P STYLE=\"text-indent: 0pt\"><B>Can I distribute a
Modelica package (under Modelica License 2) under a different license
when I first encrypt the package?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">No.
Merely encrypting a package does not qualify for Derivative Work and
therefore the encrypted package has to stay under Modelica License 2.</P>


<P STYLE=\"text-indent: 0pt\"><B>Can I distribute a
Modelica package (under Modelica License 2) under a different license
when I first add classes to the package?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">No.
The package itself remains unmodified, i.e., it is Original Work, and
therefore the license for this part must remain under Modelica
License 2. The newly added classes can be, however, under a different
license.
</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium\"><B>Can
I copy a class out of a Modelica package (under Modelica License 2)
and include it </B><U><B>unmodified</B></U><B> in a Modelica package
under a </B><U><B>commercial/proprietary license</B></U><B>?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">No,
according to article 2c). However, you can include model, block,
function, package, record and connector classes in your Modelica
package under <U>Modelica License 2</U>. This means that your
Modelica package could be under a commercial/proprietary license, but
one or more classes of it are under Modelica License 2.<BR>Note, a
&ldquo;type&rdquo; class (e.g., type Angle = Real(unit=&rdquo;rad&rdquo;))
can be copied and included unmodified under a commercial/proprietary
license (for details, see the next question).</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium\"><B>Can
I copy a type class or </B><U><B>part</B></U><B> of a model, block,
function, record, connector class, out of a Modelica package (under
Modelica License 2) and include it modified or unmodified in a
Modelica package under a </B><U><B>commercial/proprietary</B></U><B>
license</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">Yes,
according to article 2d), since this will in the end usually qualify
as Derivative Work. The reasoning is the following: A type class or
part of another class (e.g., an equation, a declaration, part of a
class description) cannot be utilized &ldquo;by its own&rdquo;. In
order to make this &ldquo;usable&rdquo;, you have to add additional
code in order that the class can be utilized. This is therefore
usually Derivative Work and Derivative Work can be provided under a
different license. Note, this only holds, if the additional code
introduced is sufficient to qualify for Derivative Work. Merely, just
copying a class and changing, say, one character in the documentation
of this class would be no Derivative Work and therefore the copied
code would have to stay under Modelica License 2.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium\"><B>Can
I copy a class out of a Modelica package (under Modelica License 2)
and include it in </B><U><B>modified </B></U><B>form in a
</B><U><B>commercial/proprietary</B></U><B> Modelica package?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">Yes.
If the modification can be seen as a &ldquo;Derivative Work&rdquo;,
you can place it under your commercial/proprietary license. If the
modification does not qualify as &ldquo;Derivative Work&rdquo; (e.g.,
bug fixes, vendor specific annotations), it must remain under
Modelica License 2. This means that your Modelica package could be
under a commercial/proprietary license, but one or more parts of it
are under Modelica License 2.</P>

<P STYLE=\"text-indent: 0pt\"><B>Can I distribute a
&ldquo;save total model&rdquo; under my commercial/proprietary
license, even if classes under Modelica License 2 are included?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">Your
classes of the &ldquo;save total model&rdquo; can be distributed
under your commercial/proprietary license, but the classes under
Modelica License 2 must remain under Modelica License 2. This means
you can distribute a &ldquo;save total model&rdquo;, but some parts
might be under Modelica License 2.</P>

<P STYLE=\"text-indent: 0pt\"><B>Can I distribute a
Modelica package (under Modelica License 2) in encrypted form?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">Yes.
Note, if the encryption does not allow &ldquo;copying&rdquo; of
classes (in to unencrypted Modelica source code), you have to send
the Modelica source code of this package to your customer, if he/she
wishes it, according to article&nbsp;6.</P>

<P STYLE=\"text-indent: 0pt\"><B>Can I distribute an
executable under my commercial/proprietary license, if the model from
which the executable is generated uses models from a Modelica package
under Modelica License 2?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">Yes,
according to article 2d), since this is seen as Derivative Work. The
reasoning is the following: An executable allows the simulation of a
concrete model, whereas models from a Modelica package (without
pre-processing, translation, tool run-time library) are not able to
be simulated without tool support. By the processing of the tool and
by its run-time libraries, significant new functionality is added (a
model can be simulated whereas previously it could not be simulated)
and functionality available in the package is removed (e.g., to build
up a new model by dragging components of the package is no longer
poss" + "ible with the executable).</P>

<P STYLE=\"text-indent: 0pt\"><B>Is my modification to
a Modelica package (under Modelica License 2) a Derivative Work?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">It
is not possible to give a general answer to it. To be regarded as &quot;an
original work of authorship&quot;, a derivative work must be
different enough from the original or must contain a substantial
amount of new material. Making minor changes or additions of little
substance to a preexisting work will not qualify the work as a new
version for such purposes.
</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium\"><BR>
</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium; color: green; font-size:140%\">
<b>Using or Distributing a Modelica <U>Document</U> under the Modelica License 2</b></P>

<P STYLE=\"text-indent: 0pt; font-weight: medium\">This
section is devoted especially for the following applications:</P>
<OL>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">
        A Modelica tool extracts information out of a Modelica package and
        presents the result in form of a &ldquo;manual&rdquo; for this
        package in, e.g., html, doc, or pdf format.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">
        The Modelica language specification is a document defining the
        Modelica language. It will be licensed under Modelica License 2.</P>
        <LI><P STYLE=\"text-indent: 0pt; font-weight: medium\">
        Someone writes a book about the Modelica language and/or Modelica
        packages and uses information which is available in the Modelica
        language specification and/or the corresponding Modelica package.</P>
</OL>

<P STYLE=\"text-indent: 0pt\"><B>Can I sell a manual
that was basically derived by extracting information automatically
from a Modelica package under Modelica License 2 (e.g., a &ldquo;reference
guide&rdquo; of the Modelica Standard Library):</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">Yes.
Extracting information from a Modelica package, and providing it in a
human readable, suitable format, like html, doc or pdf format, where
the content is significantly modified (e.g. tables with interface
information are constructed from the declarations of the public
variables) qualifies as Derivative Work and there are no restrictions
to charge a fee for Derivative Work under alternative 2d).</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium\"><B>Can
I copy a text passage out of a Modelica document (under Modelica
License 2) and use it </B><U><B>unmodified</B></U><B> in my document
(e.g. the Modelica syntax description in the Modelica Specification)?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">Yes.
In case you distribute your document, the copied parts are still
under Modelica License 2 and you are not allowed to charge a license
fee for this part. You can, of course, charge a fee for the rest of
your document.</P>

<P STYLE=\"text-indent: 0pt; font-weight: medium\"><B>Can
I copy a text passage out of a Modelica document (under Modelica
License 2) and use it in </B><U><B>modified</B></U><B> form in my
document?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">Yes,
the creation of Derivative Works is allowed. In case the content is
significantly modified this qualifies as Derivative Work and there
are no restrictions to charge a fee for Derivative Work under
alternative 2d).</P>

<P STYLE=\"text-indent: 0pt\"><B>Can I sell a printed
version of a Modelica document (under Modelica License 2), e.g., the
Modelica Language Specification?</B></P>
<P STYLE=\"text-indent: 0pt; font-weight: medium\">No,
if you are not the copyright-holder, since article 2c) does not allow
a selling fee for a (in this case physical) copy. However, mere
printing and shipping costs may be recovered.</P>
</html>
"));

end ModelicaLicense2;

  package RandomLib "Library for generating random numbers and random variates"

  annotation(preferedView="info",
    Documentation(info="<HTML>
<p>
<b>RandomLib</b> is a Modelica package for uniform random number and random variates generation.
</p>

<p>
The structure of the package is:
<ul>
<li> The <a href=\"Modelica://DESLib.RandomLib.Variates\">Variates</a> package, for generating random variates.
<li> The <a href=\"Modelica://DESLib.RandomLib.CMRG\">CMRG</a> package, contains a uniform random number generator.
<li> The <a href=\"Modelica://DESLib.RandomLib.Examples\">Examples</a> package, with some examples and case studies.
</ul>

<dl>
<dt><b>Author:</b></dt>
<dd><a href=\"http://www.euclides.dia.uned.es/vsanz\"> Victorino Sanz </a><br>
    Dpto. Informtica y Automtica, UNED<br>
    Juan del Rosal, 16<br>
    28040, Madrid<br>
    Spain<br>
    Email: <A HREF=\"mailto:vsanz@dia.uned.es\">vsanz@dia.uned.es</A><br>
    Website: <A HREF=\"http://www.euclides.dia.uned.es/vsanz\">www.euclides.dia.uned.es/vsanz</A></dd>
</dl>
</p>


<p>
<dt><b>Copying:</b></dt>
<p>
<p>
Licensed by Victorino Sanz under the Modelica License 2 <br>
Copyright 2009, Victorino Sanz.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your ouwn risk;</u> it can be redistributed and/or modified under the terms of the Modelica license 2,
see the license conditions (including the disclaimer of warranty) <a href=\"Modelica://DESLib.ModelicaLicense2\">here</a> or at <a href=\"http://www.modelica.org/licenses/ModelicaLicense2\">http://www.modelica.org/licenses/ModelicaLicense2</a>
</p><br>
</HTML>
"),   uses(Modelica(version="2.2.1")));

    package Variates "Random variates generation package"

      annotation(preferedView="info",
      Documentation(info="<HTML>
<p>
<b>Variates</b> Modelica package contains the implementation of random variate generation functions for the following probability distributions.<br>
<br>
Discrete Probability Distributions:
<ul>
<li> Empirical Discrete
<li> Bernoulli
<li> Discrete Uniform
<li> Binomial
<li> Geometric
<li> Negative Binomial
<li> Poisson
</ul>

Continuous Probability Distributions:
<ul>
<li> Empirical Continuous
<li> Uniform
<li> Exponential
<li> Erlang
<li> Gamma
<li> Weibull
<li> Normal
<li> LogNormal
<li> Beta
<li> Johnson (bounded and unbounded)
<li> Triangular
</ul>
</p>

<p> A description of the package structure and its use is given in the <a href=\"Modelica://DESLib.RandomLib.Variates.UsersGuide\">Users Guide</a>

<p>
<dl>
<dt><b>Author:</b></dt>
<dd><a href=\"http://www.euclides.dia.uned.es/vsanz\"> Victorino Sanz </a><br>
    Dpto. Informtica y Automtica, UNED<br>
    Juan del Rosal, 16<br>
    28040, Madrid<br>
    Spain<br>
    email: <A HREF=\"mailto:vsanz@dia.uned.es\">vsanz@dia.uned.es</A><br></dd>
</dl>
</p>

<p>
<dt><b>Copying:</b></dt>
<p>
<p>
Licensed by Victorino Sanz under the Modelica License 2 <br>
Copyright 2009, Victorino Sanz.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your ouwn risk;</u> it can be redistributed and/or modified under the terms of the Modelica license 2,
see the license conditions (including the disclaimer of warranty) <a href=\"Modelica://DESLib.ModelicaLicense2\">here</a> or at <a href=\"http://www.modelica.org/licenses/ModelicaLicense2\">http://www.modelica.org/licenses/ModelicaLicense2</a>
</p><br>
</HTML>
"),     uses(CMRG(version="2.2"), Modelica(version="2.2")));
      model UsersGuide "Users Guide"

          annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Users Guide of the Variates Library</font></h3>

<p>
<b>Variates</b> package contains a set of functions for generating  random observations from the following probability distributions.
</p>
Discrete Probability Distributions:
<ul>
<li> Empirical Discrete
<li> Bernoulli
<li> Discrete Uniform
<li> Binomial
<li> Geometric
<li> Negative Binomial
<li> Poisson
</ul>

Continuous Probability Distributions:
<ul>
<li> Empirical Continuous
<li> Uniform
<li> Exponential
<li> Erlang
<li> Gamma
<li> Weibull
<li> Normal
<li> LogNormal
<li> Beta
<li> Johnson (bounded and unbounded)
<li> Triangular
</ul>

<p>
Most of the algorithms used for generating the random variates are detailed in \"Simulation Modeling and Analysis\" (Averill M. Law, McGraw Hill, 2007), or the documentation of the Arena Simulation environment.<br>
</p>


<h3><font color=\"#008000\">Library Structure</font></h3>
<p>
The structure of the Variates package is divided in three components: 1) general type and functions; 2) Discrete package; and 3) Continuous package.<br><br>
The general type and functions mainly contain the <em>Generator</em> record, which represent the source for uniform random numbers that will be used by
the <em>U01</em> function, which is the function that generates uniform random numbers.<br>
Each <em>Generator</em> must be initialized using the function <em>initGenerator</em>.<br>
<em>Var</em> function is a prototype for the rest of variate generation functions.<br>
<em>GenerateVariate</em> is a function that accepts an integer parameter that indicates the probability distribution for the random variate.<br>
The <em>Constant</em> function generates a constant number, indicated as a parameter.<br>
<br>

Package <em>Discrete</em> contains the functions for generating random variates following discrete probability distributions.<br><br>

Package <em>Continuous</em>, analogously to the previous, contains the functions for generating random variates following continuous probability distributions.<br><br>

Each function of these packages contains a description of the random variates that generates.
</p>

<p>

</p>

<h3><font color=\"#008000\">Random Variate Generation</font></h3>
<p>
In order to generate uniform random observations using Variates package the user has to:
<ul>
<li> Declare a Generator that represents the source of uniform random numbers.
<li> Initialize it with the initGenerator function.
<li> Call the function of the desired probability distribution with the required parameters to generate a random variate. This function will return the generated random variate and the updated state of the Generator.
</ul>

Notice that the updated state of the Generator has to be used to generate the next random number.<br>
</p>
<p>
The following example represents the steps indicated above.

<pre>
 model usingVariates
  // declaration of the generator
  Variates.Generator g1;
  Real u1[6];
algorithm
  when initial() then
    // generator initialization
    g1 := Variates.initGenerator();
  end when;
  when time &lt;= 0 then
    for i in 1:6 loop
      // generation of random variates from g1 with Exponential(8) distribution.
      (u1[i],g1):= Variates.Continuous.Exponential(g1,8);
    end for;
  end when;
end usingVariates;

//results: u1 = {1.20419,12.282,0.713961,5.28482,7.26866,41.0291}

</pre>
</p>

<p>
Several independent sources of random numbers can be created by declaring several Generators (RngStreams). The initialization of each declared Generator will give it a different initial state.<br>
The package seed is automatically managed, so it is not required to set an initial seed.

</p>

<h3><font color=\"#008000\">Use Another Source of Uniform Random Numbers</font></h3>
<p>
CMRG is the default uniform random number generator for the Variates package. However, Variates package can be used with any other Modelica library for uniform random generation.
<br>
To use other uniform random number generator with the Variates package the following requirements have to me meet:
<ul>
<li> The random number generator has to be described by a Modelica record datatype. For example, for a generator that reads numbers from a file, this record will store the filename and the last line read.
<li> The record that describe the random number generator has to be initialized using a function. This function can not have input parameters.
<li> A function to generate random numbers  has to be declared following this prototype:
<pre>
  function name
    input generator g;
    output Real u;
    output generator gout;
  end name;
</pre>
Where generator is the record mentioned in the previous requirement.
</ul><br>
Once the new random number generator meets these requirements, the Variates Generator, initGenerator and U01 values has to be set to the ones in the new random number generator.
</html>"));
      equation

      end UsersGuide;

      record Generator = CMRG.RngStream
        "Uniform random number source (generator)";
      function initGenerator = CMRG.CreateStream
        "Uniform random source initialization function"
      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
initGenerator
</font></h3>
<p>
Initializes the random number generator
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>initGenerator</b>()</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Reads the value of the package seed, initializes the Generator and updates the package seed.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
g := Variates.initGenerator();
</pre>

</HTML>
"));
      function initGeneratorWrap
        "initializes a RNG generator, returning its single components"
      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
initGeneratorWrap
</font></h3>
<p>
Initializes the random number generator (returning its components separately)
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>initGeneratorWrap</b>()</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Reads the value of the package seed, initializes the Generator and updates the package seed. <br>
This function can be used if the components of the CMRG generator are declared using independent variables, instead of a record.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
(g.Cg1,g.Cg2,g.Cg3,g.Cg4,g.Cg5,g.Cg6,<br>
g.Bg1,g.Bg2,g.Bg3,g.Bg4,g.Bg5,g.Bg6<br>
g.Ig1,g.Ig2,g.Ig3,g.Ig4,g.Ig5,g.Ig6<br>
g.Anti,g.IncPrec) := Variates.initGeneratorWrap();
</pre>

</HTML>
"));

        output Real Cg1;
        output Real Cg2;
        output Real Cg3;
        output Real Cg4;
        output Real Cg5;
        output Real Cg6;
        output Real Bg1;
        output Real Bg2;
        output Real Bg3;
        output Real Bg4;
        output Real Bg5;
        output Real Bg6;
        output Real Ig1;
        output Real Ig2;
        output Real Ig3;
        output Real Ig4;
        output Real Ig5;
        output Real Ig6;
        output Integer Anti;
        output Integer IncPrec;
      protected
        Generator g;
      algorithm
        g := initGenerator();
        Cg1 := g.Cg[1];
        Cg2 := g.Cg[2];
        Cg3 := g.Cg[3];
        Cg4 := g.Cg[4];
        Cg5 := g.Cg[5];
        Cg6 := g.Cg[6];
        Bg1 := g.Bg[1];
        Bg2 := g.Bg[2];
        Bg3 := g.Bg[3];
        Bg4 := g.Bg[4];
        Bg5 := g.Bg[5];
        Bg6 := g.Bg[6];
        Ig1 := g.Ig[1];
        Ig2 := g.Ig[2];
        Ig3 := g.Ig[3];
        Ig4 := g.Ig[4];
        Ig5 := g.Ig[5];
        Ig6 := g.Ig[6];
        Anti := g.Anti;
        IncPrec := g.IncPrec;
      end initGeneratorWrap;

      function U01 = CMRG.RandU01 "Uniform random number generation function"
      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
U01
</font></h3>
<p>
Uniform random number generation function
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>U01</b>(g)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Returns a uniform random number in the interval (0,1), and updates the state of the Generator g by one step
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
model Example
  CMRG.RngStream g;
  Real u[5];
algorithm
  when initial() then
    g := Variates.initGenerator();
  end when;
  when time &lt;= 0 then
    for i in 1:5 loop
      (u[i],g) := Variates.U01(g);
    end for;
  end when;
end Example;

// u = {0.988831,0.760616,0.855857,0.546418,0.868702}

</pre>

</HTML>
"));

      partial function Var "prototype for variate generation functions"
        input Generator g "Random number generator";
        input Real a = 0 "First parameter of the prob. distribution";
        input Real b = 0 "Second parameter of the prob. distribution";
        input Real c = 0
          "Third parameter of the prob. distribution (only for the Johnson and Triangular distributions)";
        input Real d = 0
          "Fourth parameter of the prob. distribution (only for the Johnson distribution)";
        output Real x "Generated random variate";
        output Generator gout "Updated random number generator";
      protected
        Real u;
      end Var;

      function GenerateVariate "Generates Variates"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
GenerateVariate
</font></h3>
<p>

</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>GenerateVariate</b>(distribution,g,p1,p2,p3,p4)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
This function generates a random variate from the distribution indicated with the integer pararameter \"distribution\", using the following value correspondence:
<ul>
<li> 1 = Constant
<li> 2 = Bernoulli
<li> 3 = Discrete Uniform
<li> 4 = Binomial
<li> 5 = Geometric
<li> 6 = Negative Binomial
<li> 7 = Poisson
<li> 8 = Uniform
<li> 9 = Exponential
<li> 10 = Erlang
<li> 11 = Gamma
<li> 12 = Weibull
<li> 13 = Normal
<li> 14 = LogNormal
<li> 15 = Beta
<li> 16 = Johnson (bounded and unbounded)
<li> 17 = Triangular
<li> other value = initalization of the generator (call to initGenerator() function).
</ul>

Using this function, the generation of random variates in models can be easily parametrized.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
 (u,g) := GenerateVariate(2,g,0.4);
 // Generates a discrete variate (u) with Bernoulli(0.4) probability distribution.

 (u,g) := GenerateVariate(11,g,0.4,0.6);
 // Generates a continuous variate (u) with Gamma(0.4,0.6) probability distribution.

The example <a href=\"Modelica://DESLib.RandomLib.Examples.VariatesSimple2\">VariatesSimple2</a> uses of this function.



</pre>

</HTML>
"));
        input Integer distribution;
        input Generator g;
        input Real p1 = 0;
        input Real p2 = 0;
        input Real p3 = 0;
        input Real p4 = 0;
        output Real u;
        output Generator gout;
      algorithm
        if distribution == 1 then
          // constant
          (u,gout) := Constant(
                g,
                p1);
        elseif distribution == 2 then
          // bernoulli
          (u,gout) := Discrete.Bernoulli(
                g,
                p1);
        elseif distribution == 3 then
          // DU
          (u,gout) := Discrete.DiscreteUniform(
                g,
                p1,
                p2);
        elseif distribution == 4 then
          // Bin
          (u,gout) := Discrete.Binomial(
                g,
                p1,
                p2);
        elseif distribution == 5 then
          // Geom
          (u,gout) := Discrete.Geometric(
                g,
                p1);
        elseif distribution == 6 then
          // NegBin
          (u,gout) := Discrete.NegativeBinomial(
                g,
                p1,
                p2);
        elseif distribution == 7 then
          // Poisson
          (u,gout) := Discrete.Poisson(
                g,
                p1);
        elseif distribution == 8 then
          // U
          (u,gout) := Continuous.Uniform(
                g,
                p1,
                p2);
        elseif distribution == 9 then
          // Exp
          (u,gout) := Continuous.Exponential(
                g,
                p1);
        elseif distribution == 10 then
          // Erlang
          (u,gout) := Continuous.Erlang(
                g,
                p1,
                p2);
        elseif distribution == 11 then
          // Gamma
          (u,gout) := Continuous.Gamma(
                g,
                p1,
                p2);
        elseif distribution == 12 then
          // Weibull
          (u,gout) := Continuous.Weibull(
                g,
                p1,
                p2);
        elseif distribution == 13 then
          // N
          (u,gout) := Continuous.Normal(
                g,
                p1,
                p2);
        elseif distribution == 14 then
          // LN
          (u,gout) := Continuous.LogNormal(
                g,
                p1,
                p2);
        elseif distribution == 15 then
          // Beta
          (u,gout) := Continuous.Beta(
                g,
                p1,
                p2);
        elseif distribution == 16 then
          // Johnson
          (u,gout) := Continuous.Johnson(
                g,
                p1,
                p2,
                p3,p4);
        elseif distribution == 17 then
          // Triang
          (u,gout) := Continuous.Triangular(
                g,
                p1,
                p2,
                p3);
        else
          u := 0;
          gout := initGenerator();
        end if;
      end GenerateVariate;

      function GenerateVariateWrap "Generates Variates"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
GenerateVariate
</font></h3>
<p>

</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>GenerateVariate</b>(Cg1,Cg2,Cg3,Cg4,Cg5,Cg6,Bg1,Bg2,Bg3,Bg4,Bg5,Bg6,Ig1,Ig2,Ig3,Ig4,Ig5,Ig6,Anti,IncPrec,distribution,p1,p2,p3,p4)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
This function generates a random variate from the distribution indicated in with the integer pararameter \"distribution\", using the following value correspondence:
<ul>
<li> 1 = Constant
<li> 2 = Bernoulli
<li> 3 = Discrete Uniform
<li> 4 = Binomial
<li> 5 = Geometric
<li> 6 = Negative Binomial
<li> 7 = Poisson
<li> 8 = Uniform
<li> 9 = Exponential
<li> 10 = Erlang
<li> 11 = Gamma
<li> 12 = Weibull
<li> 13 = Normal
<li> 14 = LogNormal
<li> 15 = Beta
<li> 16 = Johnson (bounded and unbounded)
<li> 17 = Triangular
<li> other value = initalization of the generator (call to initGenerator() function).
</ul>

Using this function, the generation of random variates in models can be easily parametrized.
</P>
<p>
This function receives as parameters the individual components of a CMRG generator, instead of the record like in the GenerateVariate function.<br>
It returns the updated state of the generator also with its individual components.
</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
 (u,g.Cg1,g.Cg2,g.Cg3,g.Cg4,g.Cg5,g.Cg6,g.Bg1,g.Bg2,g.Bg3,g.Bg4,g.Bg5,g.Bg6,g.Ig1,g.Ig2,g.Ig3,g.Ig4,g.Ig5,g.Ig6,g.Anti,g.IncPrec) := <br> GenerateVariate(g.Cg1,g.Cg2,g.Cg3,g.Cg4,g.Cg5,g.Cg6,g.Bg1,g.Bg2,g.Bg3,g.Bg4,g.Bg5,g.Bg6,g.Ig1,g.Ig2,g.Ig3,g.Ig4,g.Ig5,g.Ig6,g.Anti,g.IncPrec,2,0.4);
 // Generates a discrete variate (u) with Bernoulli(0.4) probability distribution.

 (u,g.Cg1,g.Cg2,g.Cg3,g.Cg4,g.Cg5,g.Cg6,g.Bg1,g.Bg2,g.Bg3,g.Bg4,g.Bg5,g.Bg6,g.Ig1,g.Ig2,g.Ig3,g.Ig4,g.Ig5,g.Ig6,g.Anti,g.IncPrec,) := <br> GenerateVariate(g.Cg1,g.Cg2,g.Cg3,g.Cg4,g.Cg5,g.Cg6,g.Bg1,g.Bg2,g.Bg3,g.Bg4,g.Bg5,g.Bg6,g.Ig1,g.Ig2,g.Ig3,g.Ig4,g.Ig5,g.Ig6,g.Anti,g.IncPrec,11,0.4,0.6);
 // Generates a discrete variate (u) with Gamma(0.4,0.6) probability distribution.

The example <a href=\"Modelica://DESLib.RandomLib.Examples.Simple.VariatesSimple2\">VariatesSimple2</a> uses of this function.



</pre>

</HTML>
"));

        input Real inCg1;
        input Real inCg2;
        input Real inCg3;
        input Real inCg4;
        input Real inCg5;
        input Real inCg6;
        input Real inBg1;
        input Real inBg2;
        input Real inBg3;
        input Real inBg4;
        input Real inBg5;
        input Real inBg6;
        input Real inIg1;
        input Real inIg2;
        input Real inIg3;
        input Real inIg4;
        input Real inIg5;
        input Real inIg6;
        input Integer inAnti;
        input Integer inIncPrec;
        input Integer distribution;
        input Real p1 = 0;
        input Real p2 = 0;
        input Real p3 = 0;
        input Real p4 = 0;
        output Real u;
        output Real Cg1;
        output Real Cg2;
        output Real Cg3;
        output Real Cg4;
        output Real Cg5;
        output Real Cg6;
        output Real Bg1;
        output Real Bg2;
        output Real Bg3;
        output Real Bg4;
        output Real Bg5;
        output Real Bg6;
        output Real Ig1;
        output Real Ig2;
        output Real Ig3;
        output Real Ig4;
        output Real Ig5;
        output Real Ig6;
        output Integer Anti;
        output Integer IncPrec;
      protected
        Generator g;
        Generator gout;
      algorithm
        g.Cg[1] := inCg1;
        g.Cg[2] := inCg2;
        g.Cg[3] := inCg3;
        g.Cg[4] := inCg4;
        g.Cg[5] := inCg5;
        g.Cg[6] := inCg6;
        g.Bg[1] := inBg1;
        g.Bg[2] := inBg2;
        g.Bg[3] := inBg3;
        g.Bg[4] := inBg4;
        g.Bg[5] := inBg5;
        g.Bg[6] := inBg6;
        g.Ig[1] := inIg1;
        g.Ig[2] := inIg2;
        g.Ig[3] := inIg3;
        g.Ig[4] := inIg4;
        g.Ig[5] := inIg5;
        g.Ig[6] := inIg6;
        g.Anti := inAnti;
        g.IncPrec := inIncPrec;
        if distribution == 1 then
          // constant
          (u,gout) := Constant(
                g,
                p1);
        elseif distribution == 2 then
          // bernoulli
          (u,gout) := Discrete.Bernoulli(
                g,
                p1);
        elseif distribution == 3 then
          // DU
          (u,gout) := Discrete.DiscreteUniform(
                g,
                p1,
                p2);
        elseif distribution == 4 then
          // Bin
          (u,gout) := Discrete.Binomial(
                g,
                p1,
                p2);
        elseif distribution == 5 then
          // Geom
          (u,gout) := Discrete.Geometric(
                g,
                p1);
        elseif distribution == 6 then
          // NegBin
          (u,gout) := Discrete.NegativeBinomial(
                g,
                p1,
                p2);
        elseif distribution == 7 then
          // Poisson
          (u,gout) := Discrete.Poisson(
                g,
                p1);
        elseif distribution == 8 then
          // U
          (u,gout) := Continuous.Uniform(
                g,
                p1,
                p2);
        elseif distribution == 9 then
          // Exp
          (u,gout) := Continuous.Exponential(
                g,
                p1);
        elseif distribution == 10 then
          // Erlang
          (u,gout) := Continuous.Erlang(
                g,
                p1,
                p2);
        elseif distribution == 11 then
          // Gamma
          (u,gout) := Continuous.Gamma(
                g,
                p1,
                p2);
        elseif distribution == 12 then
          // Weibull
          (u,gout) := Continuous.Weibull(
                g,
                p1,
                p2);
        elseif distribution == 13 then
          // N
          (u,gout) := Continuous.Normal(
                g,
                p1,
                p2);
        elseif distribution == 14 then
          // LN
          (u,gout) := Continuous.LogNormal(
                g,
                p1,
                p2);
        elseif distribution == 15 then
          // Beta
          (u,gout) := Continuous.Beta(
                g,
                p1,
                p2);
        elseif distribution == 16 then
          // Johnson
          (u,gout) := Continuous.Johnson(
                g,
                p1,
                p2,
                p3,p4);
        elseif distribution == 17 then
          // Triang
          (u,gout) := Continuous.Triangular(
                g,
                p1,
                p2,
                p3);
        else
          u := 0;
          gout := initGenerator();
        end if;
        Cg1 := gout.Cg[1];
        Cg2 := gout.Cg[2];
        Cg3 := gout.Cg[3];
        Cg4 := gout.Cg[4];
        Cg5 := gout.Cg[5];
        Cg6 := gout.Cg[6];
        Bg1 := gout.Bg[1];
        Bg2 := gout.Bg[2];
        Bg3 := gout.Bg[3];
        Bg4 := gout.Bg[4];
        Bg5 := gout.Bg[5];
        Bg6 := gout.Bg[6];
        Ig1 := gout.Ig[1];
        Ig2 := gout.Ig[2];
        Ig3 := gout.Ig[3];
        Ig4 := gout.Ig[4];
        Ig5 := gout.Ig[5];
        Ig6 := gout.Ig[6];
        Anti := gout.Anti;
        IncPrec := gout.IncPrec;
      end GenerateVariateWrap;

      function Constant "Constant value generation"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Constant
</font></h3>
<p>
Generates a constant number.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Constant</b>(g,a)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Always generates the same number, equal to the parameter a.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>

Constant(g,1) =
(1,g)

for i in 1:5 loop
  (u[i],g) := Constant(g,3);
end for;
// u = {3,3,3,3,3};

</pre>

</HTML>
"));
        extends Var;
      algorithm
        x := a;
        gout := g;
      end Constant;

      package Discrete "Discrete probability distribution functions"
      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
The <b>Discrete</b> package contains functions for generating random variates following several discrete probability distributions.
</p>

<p>
<dl>
<dt><b>Author:</b></dt>
<dd><a href=\"http://www.euclides.dia.uned.es/vsanz\"> Victorino Sanz </a><br>
    Dpto. Informtica y Automtica, UNED<br>
    Juan del Rosal, 16<br>
    28040, Madrid<br>
    Spain<br>
    email: <A HREF=\"mailto:vsanz@dia.uned.es\">vsanz@dia.uned.es</A><br></dd>
</dl>
</p>

<p>
<dt><b>Copying:</b></dt>
<p>
<p>
Licensed by Victorino Sanz under the Modelica License 2 <br>
Copyright 2009, Victorino Sanz.
</p>
<p>
<i>This Modelica packages is <u>free</u> software and the use is completely at <u>your ouwn risk;</u> it can be redistributed and/or modified under the terms of the Modelica license 2,
see the license conditions (including the disclaimer of warranty) <a href=\"Modelica://DESLib.ModelicaLicense2\">here</a> or at <a href=\"http://www.modelica.org/licenses/ModelicaLicense2\">http://www.modelica.org/licenses/ModelicaLicense2</a>
</p><br>
</HTML>
"),       uses(CMRG(version="2.2"), Modelica(version="2.2")));

        function Discrete "Discrete"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Discrete
</font></h3>
<p>
Empirical discrete probability distribution.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Discrete</b>(g,p[:],v[:])</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
This function generates random variates following the Empirical Discrete distribution function described by the parameters p and v.<br>
Parameter p represents the list of cummulative probabilities related with the values in v.<br>
Length of p and v must be the same.
<br><br>
The following figure graphically shows the empirical discrete distribution.
</p>
<p>
<IMG SRC=\"../Figs/discrete.png\" ALT=\"discrete\">
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
 Discrete(g,{0.2,0.4,0.6,0.8,1},{1,2,3,4,5});
</pre>

</HTML>
"));
          input CMRG.RngStream g "Random number generator";
          input Real p[:] "Cummulative probabilities";
          input Real v[:] "Values related with the probabilities";
          output Real value "Generated random variate";
          output CMRG.RngStream gout "Updated random number generator";
        protected
          Real u;
          Integer s = size(p,1);
          Boolean out;
          Integer i;
        algorithm
          (u,gout) := U01(g);
          out := false;
          i := 1;
          while not out loop
            if (u < p[i]) then
              value := v[i];
              out := true;
            end if;
            i := i +1;
          end while;
        /*  for i in 1:s loop
    if (u < p[i]) then
      value := v[i];
      u := 10;
    end if;
  end for;*/
        end Discrete;

        function Bernoulli "Bernoulli"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Bernoulli
</font></h3>
<p>
Bernoulli probability distribution.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Bernoulli</b>(g,p)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Bernoulli probability distribution with probability p.
</P>

<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate u as U(0,1).<br>
2. If u &le p, return x = 1. Otherwise, return x = 0.
</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
(u,g) := Bernoulli(g,0.4);

</pre>

</HTML>
"));
          extends Var;
        algorithm
          (u,gout) := U01(g);
          if (u <= a) then
            x := 1;
          else
            x := 0;
          end if;
        end Bernoulli;

        function DiscreteUniform "Discrete Uniform"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
DiscreteUniform
</font></h3>
<p>
Discrete Uniform probability distribution.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>DiscreteUniform</b>(g,i,j)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Discrete Uniform probability distribution in the interval [i,j].
</P>

<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate u as U(0,1).<br>
2. Return x = i + floor((j - i + 1)*u)
</p>



<h3><font color=\"#008000\">Examples</font></h3>
<pre>

(u,g) := DiscreteUniform(g,2,4);
</pre>

</HTML>
"));
          extends Var;
        algorithm
          (u,gout) := U01(g);
          x := a + floor((b - a + 1)*u);
        end DiscreteUniform;

        function Binomial "Binomial"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Binomial
</font></h3>
<p>
Binomial probability distribution.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Binomial</b>(g,t,b)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following a Binomial(t,p) probability distribution. This distribution is the sum of t random variables from the Bernoulli(b) distribution.
</P>
<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate y[1],...,y[t] as IID Bernoulli(p).<br>
2. Return x = y[1]+...+y[t]
</p>



<h3><font color=\"#008000\">Examples</font></h3>
<pre>
(u,g) := Binomial(g,4,0.3);

</pre>

</HTML>
"));
          extends Var;
        algorithm
          x := 0;
          gout := g;
          for i in 1:a loop
            (u,gout) := Bernoulli(
                    gout,
                    b,
                    0,
                    0);
            x := x + u;
          end for;
        end Binomial;

        function Geometric "Geometric"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Geometric
</font></h3>
<p>
Geometric probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Geometric</b>(g,b)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Geometric probability distribution with probabitity b.
</P>
<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Set i = 0.<br>
2. Generate u as U(0,1).<br>
3. If u &le b, return i. Otherwise i=i+1 and go back to 2.
</p>



<h3><font color=\"#008000\">Examples</font></h3>
<pre>

(u,g) := Geometric(g,0.5);
</pre>

</HTML>
"));
          extends Var;
        protected
          Real e;
          Real i;
        algorithm
          i := 0;
          e := 0;
          gout := g;
          while (e == 0) loop
            (u,gout) := U01(gout);
            if (u <= a) then
              x := i;
              e := 1;
            else
              i := i + 1;
            end if;
          end while;
        end Geometric;

        function NegativeBinomial "Negative Binomial"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
NegativeBinomial
</font></h3>
<p>
Negative Binomial probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>NegativeBinomial</b>(g,t,b)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Negative Binomial probability distribution function. This variate is equivalent to the sum of t random variables following the Geometric(b) distribution.
</P>

<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate y[1],...,y[t] as IID Geometric(p).<br>
2. Return x = y[1]+...+y[t]
</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>

(u,g) := Negativebinomial(g,6,0.2);

</pre>

</HTML>
"));
          extends Var;
        algorithm
          x := 0;
          gout := g;
          for i in 1:a loop
            (u,gout) := Geometric(gout,b);
            x := x + u;
          end for;
        end NegativeBinomial;

        function Poisson "Poisson"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Poisson
</font></h3>
<p>
Poisson probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Poisson</b>(g,lambda)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Poisson probability distribution function with mean alpha.
</P>

<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Let a = e(-lambda), b = 1, and i = 0.<br>
2. Generate u as U(0,1), and replace b = b*u. If b &lt a, return x = i. Otherwise, go to step 3.<br>
3. Replace i by i+1 and go to step 2.
</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
(u,g) := Poisson(4.5);
</pre>

</HTML>
"));
          extends Var;
        protected
          Real aux;
          Real bux;
        algorithm
          aux := Modelica.Math.exp(-a);
          bux := 1;
          x := 0;
          gout := g;
          (u,gout) := U01(gout);
          bux := bux*u;
          while (bux >= aux) loop
            (u,gout) := U01(gout);
            bux := bux*u;
            x := x + 1;
          end while;
        end Poisson;

      end Discrete;

      package Continuous "Continuous probability distribution functions"
      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
The <b>Continuous</b> package contains functions to generate random variates following some continuous probability distributions.
</p>

<p>
<dl>
<dt><b>Author:</b></dt>
<dd><a href=\"http://www.euclides.dia.uned.es/vsanz\"> Victorino Sanz </a><br>
    Dpto. Informtica y Automtica, UNED<br>
    Juan del Rosal, 16<br>
    28040, Madrid<br>
    Spain<br>
    email: <A HREF=\"mailto:vsanz@dia.uned.es\">vsanz@dia.uned.es</A><br></dd>
</dl>
</p>

<p>
<dt><b>Copying:</b></dt>
<p>
<p>
Licensed by Victorino Sanz under the Modelica License 2 <br>
Copyright 2009, Victorino Sanz.
</p>
<p>
<i>This Modelica packages is <u>free</u> software and the use is completely at <u>your ouwn risk;</u> it can be redistributed and/or modified under the terms of the Modelica license 2,
see the license conditions (including the disclaimer of warranty) <a href=\"Modelica://DESLib.ModelicaLicense2\">here</a> or at <a href=\"http://www.modelica.org/licenses/ModelicaLicense2\">http://www.modelica.org/licenses/ModelicaLicense2</a>
</p><br>
</HTML>
"),       uses(CMRG(version="2.2"), Modelica(version="2.2")));
        function Continuous "Empirical Continuous"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Continuous
</font></h3>
<p>
Empirical Continuous probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Continuous</b>(g,p[:],v[:])</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
This function generates random variates following the Empirical Continuous distribution function described by the parameters p and v.<br>
Parameter p represents the list of cummulative probabilities related with the values in v.<br>
Length of p and v must be the same.
</P>
The following figure graphically shows the empirical discrete distribution.
</p>
<p>
<IMG SRC=\"../Figs/continuous.png\" ALT=\"continuous\">
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
 Continuous(g,{0.2,0.4,0.6,0.8,1},{1,2,3,4,5});


</pre>

</HTML>
"));
          input CMRG.RngStream g;
          input Real p[:];
          input Real v[:];
          output Real value;
          output CMRG.RngStream gout;
        protected
          Real u;
          Integer s = size(p,1);
        algorithm
          (u,gout) := U01(g);
          for i in 1:s loop
            if (u <= p[i]) then
              if i == 1 then
                value := v[1];
              else
                value := v[i-1] + (((u-p[i-1])*(v[i]-v[i-1]))/(p[i]-p[i-1]));
              end if;
              u := 10;
            end if;
          end for;
        end Continuous;

        function Uniform "Uniform"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Uniform
</font></h3>
<p>
Uniform probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Uniform</b>(g,a,b)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Uniform probability distribution function in the interval [a,b].
</P>

<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate u as U(0,1)<br>
2. Return x = a + (b-a)*u
</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
(u,g) := Uniform(g,2,4);
</pre>

</HTML>
"));
          extends Var;
        protected
          Generator gaux;
        algorithm
          (u,gout) := U01(g);
          x := a + (b - a)*u;
        end Uniform;

        function Exponential "Exponential"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Exponential
</font></h3>
<p>
Exponential probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Exponential</b>(g,beta)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Exponential probability distribution of mean beta.
</P>


<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate u as U(0,1)<br>
2. Return x = -beta * ln(1-u)
</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>

(u,g) := Exponential(g,0.4);
</pre>

</HTML>
"));
          extends Var;
        algorithm
          (u,gout) := U01(g);
          x := (-a)*Modelica.Math.log(1-u);
        end Exponential;

        function Erlang "Erlang"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Erlang
</font></h3>
<p>
Erlang probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Erlang</b>(g,beta,m)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Erlang probability distribution function, with parameters beta and m. The generated variate corresponds to the sum of m random variates following the Exponential distribution of mean beta/m.
</P>

<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate y[1],...,y[m] as IID U(0,1)<br>
2. Return x = (-beta/m)*ln(product(y))
</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
  (u,g) := Erlang(g,0.5,4);
</pre>

</HTML>
"));
          extends Var;
        protected
          Integer k =  integer(b);
          Real y[k];
        algorithm
          gout := g;
          for i in 1:k loop
            (y[i],gout) := U01(gout);
          end for;
          x := (-a/b)*Modelica.Math.log(product(y));
        end Erlang;

        function Gamma "Gamma"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Gamma
</font></h3>
<p>
Gamma probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Gamma</b>(g,alpha,beta)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Gamma probability distribution function, with parameters alpha and beta.
</P>


<p>
The pseudo-code algorithm used for this distribution is:<br><br>


I. If alpha == 1, return y = exponential(1)<br>
II. If alpha == 0.5, return y = normal(0,1)<br>
III. If a &lt 1<br>
&nbsp &nbsp &nbsp &nbsp  b = (e + alpha)/e<br>
&nbsp &nbsp &nbsp &nbsp 1. Generate u1 as U(0,1). Let P = beta*u1. If P > 1, go to step 3. Otherwise, go to step 2.<br>
&nbsp &nbsp &nbsp &nbsp 2. Let Y = P^(1/alpha), and generate u2 as U(0,1). If u2 &le e^(-Y), return y = Y. Otherwise, go to step 1.<br>
&nbsp &nbsp &nbsp &nbsp 3. Let Y = -ln[(b - P)/alpha] and generate u2 as U(0,1). If u2 &le Y^(alpha-1), return y = Y. Otherwise, go to step 1.<br>
IV. If a &gt 1
&nbsp &nbsp &nbsp &nbsp a = 1/sqrt(2*alpha -1)<br>
&nbsp &nbsp &nbsp &nbsp b = alpha -ln(4)<br>
&nbsp &nbsp &nbsp &nbsp q = alpha + 1/a<br>
&nbsp &nbsp &nbsp &nbsp theta = 4.5<br>
&nbsp &nbsp &nbsp &nbsp d = 1 + ln(theta)<br>
&nbsp &nbsp &nbsp &nbsp 1. Generate u1 and u2 as IID U(0,1).<br>
&nbsp &nbsp &nbsp &nbsp 2. Let V = a*ln[u1/(1-u1)], Y = alpha*e^V, Z = u1^2*u2, and W = b + q*V - Y.<br>
&nbsp &nbsp &nbsp &nbsp 3. If W + d - theta*Z &ge 0, return y = Y. Otherwise, go to step 4.<br>
&nbsp &nbsp &nbsp &nbsp 3. If W &ge ln(Z), return y = Y. Otherwise, go to step 1.<br>
V. x = beta*y;

</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
  (u,g) := Gamma(g,0.4,0.7);
</pre>

</HTML>
"));
          extends Var;
        protected
          Real Y;
          Real y;
          Real y1;
          Real b1;
          Real p;
          Real u1;
          Real u2;
          Boolean step1;
          Boolean step2;
          Boolean step3;
          Boolean step4;
          Real ca;
          Real cb;
          Real q;
          Real theta;
          Real cd;
          Real V;
          Real Z;
          Real W;
        algorithm
          gout := g;
          if a == 1 then //<= 1 and a >= 1 then
            (y,gout) := Exponential(gout,a);
          elseif a == 0.5 then //<= 0.5 and a>= 0.5 then
            (Y,gout) := Normal(gout,0,1);
            y := Y^2;
          elseif a < 1 then
            b1 := (Modelica.Constants.e + a)/Modelica.Constants.e;
            step1 := true;
            while step1 loop
              (u1,gout) := U01(gout);
              p := b1*u1;
              if p > 1 then
                step3 := true;
                step2 := false;
              else
                step3 := false;
                step2 := true;
              end if;
              if step2 then
                Y := p^(1/a);
                (u2,gout) := U01(gout);
                if u2 <= Modelica.Constants.e^(-y1) then
                  step1 := false;
                  y := Y;
                end if;
              end if;
              if step3 then
                Y := -Modelica.Math.log((b1-p)/a);
                (u2,gout) := U01(gout);
                if u2 <= Y^(a-1) then
                  step1 := false;
                  y := Y;
                end if;
              end if;
            end while;
          else
            ca := 1/(sqrt(2*a -1));
            cb := a - Modelica.Math.log(4);
            q := a+1;
            theta := 4.5;
            cd := 1 + Modelica.Math.log(theta);
            step1 := true;
            while step1 loop
              (u1,gout) := U01(gout);
              (u2,gout) := U01(gout);
              V := ca*Modelica.Math.log(u1/(1-u1));
              Y := a*(Modelica.Constants.e^V);
              Z := (u1^2)*u2;
              W := cb + q*V - Y;
              step4 := true;
              if (W+cd-(theta*Z) >= 0) then
                step1 := false;
                step4 := false;
                y := Y;
              end if;
              if step4 then
                if W >= Modelica.Math.log(Z) then
                  step1 := false;
                  y := Y;
                end if;
              end if;
            end while;
          end if;
          x := b*y; // Gamma(a,b) = b * Gamma(a,1)
        end Gamma;

        function Weibull "Weibull"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Weibull
</font></h3>
<p>
Weibull probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Weibull</b>(g,alpha,beta)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Weibull probability distribution, with parameters alpha and beta.
</P>

<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate u as U(0,1)<br>
2. Return x = alpha*((-ln(1-u))^(1/beta))
</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>

(u,g) := Weibull(g,0.4,0.8);

</pre>

</HTML>
"));
          extends Var;
        protected
          Real u;
        algorithm
          (u,gout) := U01(g);
          x := a*((-Modelica.Math.log(1-u))^(1/b));
        end Weibull;

        function Normal "Normal"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Normal
</font></h3>
<p>
Normal probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Normal</b>(g,mu,sigma2)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Normal probability distribution, with mean mu and variance sigma2.
</P>

<p>
The algorithm used for this distribution is the same that the one implemented in SIMAN, and is detailed in:<br><br>
J.D. Beasley and S.G. Springer (1977), The Percentage Points of the Normal Distribution, <i>Applied Statistics</i>, vol. 26, pp: 118-121.

</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>

(u,g) := Normal(g,0,1);

</pre>

</HTML>
"));
          extends Var;
        protected
          Real p;
          constant Real a0 = 2.50662823884;
          constant Real a1 = -18.61500062529;
          constant Real a2 = 41.39119773534;
          constant Real a3 = -25.44106049637;
          constant Real b1 = -8.47351093090;
          constant Real b2 = 23.08336743743;
          constant Real b3 = -21.06224101826;
          constant Real b4 = 3.13082909833;
          constant Real c0 = -2.78718931138;
          constant Real c1 = -2.29796479134;
          constant Real c2 = 4.85014127135;
          constant Real c3 = 2.32121276858;
          constant Real d1 = 3.54388924762;
          constant Real d2 = 1.63706781897;
          Real r;
          constant Real split = 0.42;
          Real value;
        algorithm
          gout:= g;
          (p,gout) := U01(gout);
          if (abs(p-0.5) <= split) then
            r := (p-0.5) *(p-0.5);
            value := (p-0.5)*(((a3*r+a2)*r+a1)*r+a0)/((((b4*r+b3)*r+b2)*r+b1)*r+1);
          elseif (0 < p and p < 1) then
            if (0.5 < p) then
              r := sqrt(-Modelica.Math.log(1-p));
            else
              r := sqrt(-Modelica.Math.log(p));
            end if;
            value :=(((c3*r + c2)*r + c1)*r + c0)/((d2*r + d1)*r + 1);
            if (p < 0.5) then
              value :=-value;
            end if;
          else
            value := 0;
          end if;
          x := (value*b)+a;
        end Normal;

        function LogNormal "LogNormal"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
LogNormal
</font></h3>
<p>
LogNormal probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>LogNormal</b>(g,mu,sigma2)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the LogNormal probability distribution function, with mean mu and variance sigma2.
</P>
<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate Y as N(mu,sigma2).<br>
2. Return x = e^Y
</p>

<h3><font color=\"#008000\">Examples</font></h3>
<pre>
(u,g) := LogNormal(g,0,1);
</pre>

</HTML>
"));
          extends Var;
        protected
          Real y;
        algorithm
          //gout := g;
          (y,gout) := Normal(g,a,b);
          x := Modelica.Math.exp(y);
        end LogNormal;

        function Beta "Beta"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Beta
</font></h3>
<p>
Beta probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Beta</b>(g,alpha1,alpha2)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Beta probability distribution, with parameters alpha1 and alpha2.
</P>
<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate Y1 as gamma(alpha1,1), and Y2 as gamma(alpha2,1) independent from Y1.<br>
2. Return x = Y1/(Y1+Y2)
</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
(u,g) := Beta(g,0.6,0.4);
</pre>

</HTML>
"));
          extends Var;
        protected
          Real y1;
          Real y2;
        algorithm
          (y1,gout) := Gamma(g,a,1);
          (y2,gout) := Gamma(gout,b,1);
          x := y1/(y1+y2);
        end Beta;

        function Johnson "Johnson"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Johnson
</font></h3>
<p>
Johnson probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Johnson</b>(g,alpha1,alpha2,a,b) if bounded (alpha2 > 0) </pre></blockquote>
<blockquote><pre><b>Johnson</b>(g,alpha1,alpha2,gamma,beta) if unbounded (alpha2 &lt; 0)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Johnson probability distribution, with parameters alpha1 and alpha2. If alpha2 > 0, for the Johnson bounded distribution, the variate is generated in the interval [a,b]. Otherwise, gamma and beta are the shape parameters.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
 (u,g) := Johnson(g,0.1,0.5,1,4); // bounded
 (u,g) := Johnson(g,0.1,-0.5,1,1); // unbounded
</pre>

</HTML>
"));
        extends Var;
        protected
          Real z;
          Real y;
        algorithm
          if (b > 0) then // bounded
            (z,gout) := Normal(g,0,1);
            y := Modelica.Math.exp((z -a)/b);
            x := (c + d*y)/(y+1);
          else // unbounded
            (z,gout) := Normal(g,0,1);
            y := Modelica.Math.exp((z-a)/b);
            x := c + (d/2)*(y-1/y);
          end if;
        end Johnson;

        function Triangular "Triangular"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
Triangular
</font></h3>
<p>
Triangular probability distribution function.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>Triangular</b>(g,min,mode,max)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Generates a random variate following the Triangular probability distribution function. The variate is generated in the interval [min,max] with the desired mode.
</P>
<p>
The pseudo-code algorithm used for this distribution is:<br><br>

1. Generate u as U(0,1)<br>
2. If u &le ((mode - min)/(max - min)), return x = min + sqrt(u*(max - min)*(mode - min)). Otherwise, x = max - sqrt((1-u)*(max - min)*(max - mode))
</p>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
  (u,g) := Triangular(g,1,3,6);
</pre>

</HTML>
"));
          extends Var;
        protected
          Real cc;
        algorithm
          (u,gout) := U01(g);
          cc := (b-a)/(c-a);
          if u < cc then
            x := sqrt(cc*u);
          else
            x := 1-sqrt((1-cc)*(1-u));
          end if;
          x := a + (c-a)*x;

        /*  if (u < ((b - a)/(c - a))) then
    x := a + sqrt(u*(c - a)*(b - a));
    //x := a + sqrt(u*((b - a)/(c - a)));
  else
    x := c - sqrt((1 - u)*(c - a)*(c - b));
  end if;*/
        end Triangular;

      end Continuous;

      /*******************/

      /*********************/
    end Variates;

    package Sources
      model Discrete
        parameter Integer distribution = 1 "Random Distribution"
      annotation(choices(choice = 1 "Constant",
            choice=2 "Bernoulli",
            choice=3 "Discrete Uniform",
            choice=4 "Binomial",
            choice=5 "Geometric",
            choice=6 "Negative Binomial",
            choice=7 "Poisson"));
        parameter Real p1 = 1 "first parameter";
        parameter Real p2 = 1 "second parameter";

          Variates.Generator g "RNG";
        Modelica.Blocks.Interfaces.IntegerOutput y annotation (extent=[80,-10; 100,10]);
      protected
        Real out;
      algorithm
        // initilization of the RngStream
        when initial() then
          g := Variates.initGenerator();
        end when;
        (out,g) := Variates.GenerateVariate(distribution,g,p1,p2);
        y :=integer(out);
        annotation (Icon, Diagram);
      end Discrete;

      model Continuous
        Variates.Generator g "RNG";
        parameter Integer distribution = 1 "Random distribution"
      annotation(choices(choice = 1 "Constant",
            choice=8 "Uniform",
            choice=9 "Exponential",
            choice=10 "Erlang",
            choice=11 "Gamma",
            choice=12 "Weibull",
            choice=13 "Normal",
            choice=14 "LogNormal",
            choice=15 "Beta",
            choice=16 "Johnson (bounded and unbounded)",
            choice=17 "Triangular"));
        parameter Real p1 = 1;
        parameter Real p2 = 0.1;
        parameter Real p3 = 1;
        parameter Real p4 = 1;

        Modelica.Blocks.Interfaces.RealOutput y annotation (extent=[80,-10; 100,10]);
      algorithm
        // initilization of the RngStream
        when initial() then
          g := Variates.initGenerator();
        end when;
        (y,g) := Variates.GenerateVariate(distribution,g,p1,p2,p3,p4);
        annotation (Diagram);
      end Continuous;
    end Sources;

    package CMRG
      "Combined Multiple Recursive Generator for uniform random numbers generation"

      annotation(preferedView="info",
      Documentation(info="<HTML>
<p>
<b>CMRG</b> package is the implementation in Modelica of the RNG algorithm proposed by L'Ecuyer et al.
(<a href=\"http://www.iro.umontreal.ca/~lecuyer/myftp/papers/streams00.pdf\">pdf</a>)<br>
</p>

<p> A description of the package structure and its use is given in the <a href=\"Modelica://DESLib.RandomLib.CMRG.UsersGuide\">Users Guide</a>

<p>
<dl>
<dt><b>Author:</b></dt>
<dd><a href=\"http://www.euclides.dia.uned.es/vsanz\"> Victorino Sanz </a><br>
    Dpto. Informtica y Automtica, UNED<br>
    Juan del Rosal, 16<br>
    28040, Madrid<br>
    Spain<br>
    email: <A HREF=\"mailto:vsanz@dia.uned.es\">vsanz@dia.uned.es</A><br></dd>
</dl>
</p>

<p>
<dt><b>Copying:</b></dt>
<p>
<p>
Licensed by Victorino Sanz under the Modelica License 2 <br>
Copyright 2009, Victorino Sanz.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your ouwn risk;</u> it can be redistributed and/or modified under the terms of the Modelica license 2,
see the license conditions (including the disclaimer of warranty) <a href=\"Modelica://DESLib.ModelicaLicense2\">here</a> or at <a href=\"http://www.modelica.org/licenses/ModelicaLicense2\">http://www.modelica.org/licenses/ModelicaLicense2</a>
</p>
<br>
</HTML>
"));

      model UsersGuide "Users Guide"

          annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Users Guide of the CMRG Package</font></h3>

<p>
<b>CMRG</b> package is the implementation in Modelica of the RNG algorithm proposed by L'Ecuyer et al.
(<a href=\"http://www.iro.umontreal.ca/~lecuyer/myftp/papers/streams00.pdf\">pdf</a>)<br>
This RNG algorithm belongs to the Combined Multiple Recursive Generators family and combines two recursive generators of order 3.<br>
</p>

<p>
The RNG implemented in CMRG package has a period close to 2^191.<br>
This generator gives the possibility of creating multiple streams and sub-streams than can be considered as independent RNGs.<br>
The whole period of the generator can be divided into disjoint streams, each of length 2^127.
At the same time, each stream can be divided into 2^51 adjacent sub-streams, each of length 2^76.
</p>

<p>
The main class in the CMRG package is the <em>RngStream</em>.
It represents a single stream of sequential random numbers.<br>
As mentioned above, an RngStream is divided into adjacent sub-streams.
The state of the RngStream indicates the start, current and next position in the current stream and sub-stream.
</p>

<h3><font color=\"#008000\">Package Structure</font></h3>
<p>
 The CMRG package is composed of: 1) the developer package, called Src; and 2) the set of public functions.<br>
The developer package contains the private part of the package. This package includes contants and functions to perform the required calculations to manage the state of the RngStreams. They have no utility for an standard user.<br>
The public functions describe the operations that a user can perform with an RngStream, such as generating random numbers, updating the state of the stream or changing the package seed among others.<br>
The operation of each function is described in its documentation.
</p>

<h3><font color=\"#008000\">Random Number Generation</font></h3>
<p>
In order to generate uniform random numbers using CMRG package the user has to:
<ul>
<li> Declare an RngStream that represents the sequential stream of random numbers.
<li> Initialize the RngStream using the CreateStream function.
<li> Call the RandU01 function to generate a random number. This function will return the generated random number and the updated state of the RngStream.
</ul>

Notice that the updated state of the RngStream has to be used to generate the next random number.<br>
</p>
<p>
The following example represents the steps indicated above.

<pre>
model usingCMRG
  // declaration of the streams
  CMRG.RngStream g1;
  Real u1[6];
algorithm
  when initial() then
    // stream initialization
    g1 := CMRG.RngStream_CreateStream();
  end when;
  when time &lt;= 0 then
    for i in 1:6 loop
      // generation of random numbers from g1
      (u1[i],g1):= CMRG.RngStream_RandU01(g1);
    end for;
  end when;
end usingCMRG;
</pre>
</p>

<p>
Several independent sources of random numbers can be created by declaring several RngStreams. The initialization of each declared RngStream will give it a different initial state.<br>
The package seed is automatically managed, so it is not required to set an initial seed.

</p>
</html>"));
      equation

      end UsersGuide;

      package Src "Internal data and functions"

      type Seed = Real[6] "RNG seed data type";

      constant Real norm =  2.328306549295727688e-10;
      constant Real m1 =    4294967087.0;
      constant Real m2 =    4294944443.0;
      constant Real a12 =     1403580.0;
      constant Real a13n =     810728.0;
      constant Real a21 =      527612.0;
      constant Real a23n =    1370589.0;

      constant Real two17 =   131072.0;
      constant Real two53 =   9007199254740992.0;
      constant Real fact =  5.9604644775390625e-8; /* 1 / 2^24 */

      /* Default initial seed of the package. Will be updated to become
     the seed of the next created stream. */

        //constant Real[6] nextSeed = {12345, 12345, 12345, 12345, 12345, 12345};

      /* The following are the transition matrices of the two MRG components */
      /* (in matrix form), raised to the powers -1, 1, 2^76, and 2^127, resp.*/

      /* Inverse of A1p0 */
      constant Real[3,3] InvA1 = {
              { 184888585.0,   0.0,  1945170933.0},
              {         1.0,   0.0,           0.0},
              {         0.0,   1.0,           0.0}};

      /* Inverse of A2p0 */
      constant Real[3,3] InvA2 =  {
              {      0.0,  360363334.0,  4225571728.0},
              {      1.0,          0.0,           0.0},
              {      0.0,          1.0,           0.0}};

      constant Real[3,3] A1p0 = {
              {       0.0,        1.0,       0.0},
              {       0.0,        0.0,       1.0},
              { -810728.0,  1403580.0,       0.0}};

      constant Real[3,3] A2p0 =  {
              {        0.0,        1.0,       0.0},
              {        0.0,        0.0,       1.0},
              { -1370589.0,        0.0,  527612.0}};

      constant Real[3,3] A1p76= {
              {      82758667.0, 1871391091.0, 4127413238.0},
              {    3672831523.0,   69195019.0, 1871391091.0},
              {    3672091415.0, 3528743235.0,   69195019.0}};

      constant Real[3,3] A2p76= {
              {    1511326704.0, 3759209742.0, 1610795712.0},
              {    4292754251.0, 1511326704.0, 3889917532.0},
              {    3859662829.0, 4292754251.0, 3708466080.0}};

      constant Real[3,3] A1p127 = {
              {    2427906178.0, 3580155704.0,  949770784.0},
              {     226153695.0, 1230515664.0, 3580155704.0},
              {    1988835001.0,  986791581.0, 1230515664.0}};

      constant Real[3,3] A2p127 = {
              {    1464411153.0,  277697599.0, 1610723613.0},
              {      32183930.0, 1464411153.0, 1022607788.0},
              {    2824425944.0,   32183930.0, 2093834863.0}};

      /************** from headers file *******************/

      /****************************************************/

      /*-------------------------------------------------------------------------*/

      function MultModM
          "Compute (a*s + c) % m. m must be < 2^35.  Works also for s, c < 0"
        input Real a;
        input Real s;
        input Real c;
        input Real m;
        output Real v;
        protected
        Real a1;
        Real aux;
      algorithm
        aux := a;
        v := aux * s + c;
        if ((v >= two53) or (v <= -two53)) then
          a1 := floor( aux / two17);
          aux := aux - (a1 * two17);
          v := a1 * s;
          a1 := floor(v / m);
          v := v - (a1 * m);
          v := v * two17 + aux * s + c;
        end if;
        a1 := floor( v / m);
        v := v - (a1*m);
        if (v < 0.0) then
          v := v + m;
        end if;
      end MultModM;

      /*-------------------------------------------------------------------------*/

      function MatVecModM
          "MatVecModM returns v = A*s % m.  Assumes that -m < s[i] < m. Works even if v = s."
        input Real[3,3] A;
        input Real[3] s;
        output Real[3] v;
        input Real m;
        protected
        Real[3] x;
      algorithm
        for i in 1:3 loop
          x[i] := MultModM( A[i,1], s[1], 0.0, m);
          x[i] := MultModM( A[i,2], s[2], x[i], m);
          x[i] := MultModM( A[i,3], s[3], x[i], m);
        end for;

        for i in 1:3 loop
          v[i] := x[i];
        end for;
      end MatVecModM;

      /*-------------------------------------------------------------------------*/

      function MatMatModM
          "MatMAtModM Returns C = A*B % m. Work even if A = C or B = C or A = B = C."
        input Real[3,3] A;
        input Real[3,3] B;
        input Real m;
        output Real[3,3] C;
        protected
        Real[3] V;
        Real[3,3] W;
      algorithm
        for i in 1:3 loop
          for j in 1:3 loop
            V[j] := B[j,i];
          end for;
          V := MatVecModM(A, V, m);
          for j in 1:3 loop
            W[j,i] := V[j];
          end for;
        end for;
        for i in 1:3 loop
          for j in 1:3 loop
            C[i,j] := W[i,j];
          end for;
        end for;
      end MatMatModM;

      /*-------------------------------------------------------------------------*/

      function MatTwoPowModM
          "MatTwoPowModM compute matrix B = (A^(2^e) % m);  works even if A = B"
        input Real[3,3] A;
        output Real[3,3] B;
        input Real m;
        input Real e;
        protected
        Integer i;
        Integer j;
      algorithm
        /* initialize: B = A */
        //if (A[1,1] <> B[1,1]) or (A[1,2] <> B[1,2]) or (A[1,3] <> B[1,3]) or (A[2,1] <> B[2,1]) or (A[2,2] <> B[2,2]) or (A[2,3] <> B[2,3]) or (A[3,1] <> B[3,1]) or (A[3,2] <> B[3,2]) or (A[3,3] <> B[3,3]) then
          for i in 1:3 loop
            for j in 1:3 loop
               B[i,j] := A[i,j];
            end for;
          end for;
        //end if;

        /* Compute B = A^{2^e} */
        for i in 1:e loop
          B := MatMatModM( B, B, m);
        end for;
      end MatTwoPowModM;

      /*-------------------------------------------------------------------------*/

      function MatPowModM
          "MatPowModM compute matrix B = A^n % m ;  works even if A = B"
        input Real[3,3] A;
        output Real[3,3] B;
        input Real m;
        input Real n;
        protected
        Real[3,3] W;
        Real naux;
        Real mo;
      algorithm
        /* initialize: W = A; B = I */
        for i in 1:3 loop
          for j in 1:3 loop
            W[i,j] := A[i,j];
            B[i,j] := 0.0;
          end for;
        end for;

        for j in 1:3 loop
          B[j,j] := 1.0;
        end for;

        naux := n;
        /* Compute B = A^n % m using the binary decomposition of n */
        while (naux > 0) loop
          mo := floor(mod(naux,2));
          if (mo <> 0) then
            B := MatMatModM( W, B, m);
          end if;
          W := MatMatModM( W, W, m);
          naux := naux/2;
        end while;
      end MatPowModM;

      /*-------------------------------------------------------------------------*/

      function U01 "Uniform random number generation function"
        input RngStream g "RngStream";
        output Real out "Uniform random number";
        output RngStream gout "Updated RngStream";
        protected
        Real k;
        Real p1;
        Real p2;
        Real u;
      algorithm
        gout := g;
        /* Component 1 */
        p1 := (a12 * gout.Cg[2]) - (a13n * gout.Cg[1]);
        k := floor(p1 / m1);
        p1 := p1 - ( k * m1);
        if (p1 < 0.0) then
          p1 := p1 + m1;
        end if;
        gout.Cg[1] := gout.Cg[2];
        gout.Cg[2] := gout.Cg[3];
        gout.Cg[3] := p1;

        /* Component 2 */
        p2 := (a21 * gout.Cg[6]) - (a23n * gout.Cg[4]);
        k := floor(p2 / m2);
        p2 := p2 - (k * m2);
        if (p2 < 0.0) then
          p2 := p2 + m2;
        end if;
        gout.Cg[4] := gout.Cg[5];
        gout.Cg[5] := gout.Cg[6];
        gout.Cg[6] := p2;

        /* Combination */
        if (p1 > p2) then
          u := (p1 -p2) * norm;
        else
          u := (p1 - p2 + m1) * norm;
        end if;
        if (gout.Anti <> 0) then
          out := 1-u;
        else
          out := u;
        end if;
      end U01;

      /*-------------------------------------------------------------------------*/

      function U01d
          "Uniform random number generation function, with increased precission"
        input RngStream g "RngStream";
        output Real out "Uniform random number";
        output RngStream gout "Updated RngStream";
        protected
        Real u;
        Real uaux;
      algorithm
        (u,gout) := U01(g);
        if (g.Anti == 0) then
          (uaux,gout) := U01(gout);
          u := u +( uaux * fact);
          if (u < 1.0) then
            out := u;
          else
            out := u - 1.0;
          end if;
        else
          /* Don't forget that U01() returns 1 - u in the antithetic case */
          (uaux,gout) := U01(gout);
          u :=u + ((uaux - 1.0)*fact);
          if (u < 0.0) then
            out := u + 1.0;
          else
            out := u;
          end if;
        end if;
      end U01d;

      /*-------------------------------------------------------------------------*/

      function CheckSeed "Checks correctness of a seed."

        input Seed seed "Seed to be checked";
        output Integer out "Results of the check";
        protected
        Integer i;
      algorithm
        /* Check that the seeds are legitimate values. Returns 0 if legal seeds,
       -1 otherwise */
        out := 0;
        for i in 1:3 loop
          if (seed[i] >= m1) then
            out
         := -1;
          end if;
        end for;
        for i in 4:6 loop
          if (seed[i] >= m2) then
            out
         := -1;
          end if;
        end for;
        if (seed[1] == 0) and (seed[2] == 0) and (seed[3] == 0) then
          out := -1;
        end if;
        if (seed[4] == 0) and (seed[5] == 0) and (seed[6] == 0) then
          out := -1;
        end if;
      end CheckSeed;

        function SaveSeed "saves seed to file CMRGSeed.txt"

          input Seed seed;
          output Integer out;
        //protected
        //  Integer lon;
        //  Seed aux;
        algorithm
        /*  (aux,lon) := ReadVar("seed", 6);
  if (lon == 0) then
    CreateVar("seed",seed,6);
  else
    UpdateVar("seed",seed,6);
  end if;
  out :=0;*/
          if Modelica.Utilities.Files.exist("CMRGSeed.txt") then
            Modelica.Utilities.Files.remove("CMRGSeed.txt");
            Modelica.Utilities.Streams.print(String(seed[1],significantDigits=14)+"\t"+String(seed[2],significantDigits=14)+"\t"+String(seed[3],significantDigits=14)+"\t"+String(seed[4],significantDigits=14)+"\t"+String(seed[5],significantDigits=14)+"\t"+String(seed[6],significantDigits=14),"CMRGSeed.txt");
          else
            Modelica.Utilities.Streams.print(String(seed[1],significantDigits=14)+"\t"+String(seed[2],significantDigits=14)+"\t"+String(seed[3],significantDigits=14)+"\t"+String(seed[4],significantDigits=14)+"\t"+String(seed[5],significantDigits=14)+"\t"+String(seed[6],significantDigits=14),"CMRGSeed.txt");
          end if;
          out := 1;
          Modelica.Utilities.Streams.close("CMRGSeed.txt");
        end SaveSeed;

        function ReadSeed
          "reads the current seed from file CMRGSeed.txt. If the file does not exists, saves the default value {12345,12345,12345,12345,12345,12345}"

          output Seed out;
        protected
          Integer found;
          String line = "";
          Integer index = 1;
        algorithm
          if Modelica.Utilities.Files.exist("CMRGSeed.txt") then
            line := Modelica.Utilities.Streams.readLine("CMRGSeed.txt",1);
            for i in 1:6 loop
              (out[i],index) := Modelica.Utilities.Strings.scanReal(line,index,false,"");
            end for;
          else
            for i in 1:6 loop
              out[i] := 12345;
            end for;
            Modelica.Utilities.Streams.print(String(12345)+"\t"+String(12345)+"\t"+String(12345)+"\t"+String(12345)+"\t"+String(12345)+"\t"+String(12345),"CMRGSeed.txt");
          end if;
          Modelica.Utilities.Streams.close("CMRGSeed.txt");
        end ReadSeed;

      end Src;

      /*---------------------------------------------------------------------*/
      /* Public part.                                                        */
      /*---------------------------------------------------------------------*/

    record RngStream "Data structure for an RNG stream"
      Real[6] Cg;
      Real[6] Bg;
      Real[6] Ig;
      Integer Anti;
      Integer IncPrec;
    end RngStream;

      function CreateStream "CreateStream"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
CreateStream
</font></h3>
<p>
Initializes an RngStream g
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>CreateStream</b>()</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Reads the value of the package seed, initializes the RngStream and updates the package seed.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
g := CMRG.CreateStream();
</pre>

</HTML>
"));
        output RngStream g "Created RngStream";
      protected
        Src.Seed seed;
        Src.Seed outseed;
        Real[3] A;
      algorithm
        seed :=  Src.ReadSeed();
        g.Anti := 0;
        g.IncPrec := 0;

        for i in 1:6 loop
        //Modelica.Utilities.Streams.print(String(seed[i]));
          g.Bg[i] := seed[i];
          g.Cg[i] := seed[i];
          g.Ig[i] := seed[i];
        end for;

        for i in 1:3 loop
          A[i] := seed[i];
        end for;
        A := Src.MatVecModM( Src.A1p127, A, Src.m1);
        for i in 1:3 loop
          outseed[i] := A[i];
        end for;

        for i in 4:6 loop
          A[i-3] := seed[i];
        end for;
        A := Src.MatVecModM( Src.A2p127,A,  Src.m2);
        for i in 4:6 loop
          outseed[i] := A[i-3];
        end for;
        Src.SaveSeed(outseed);
      end CreateStream;

      function ResetStartStream "ResetStartStream"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
ResetStartStream
</font></h3>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>ResetStartStream</b>(g)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Reinitializes the state of the RngStream g to the initial state of the stream.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
    for i in 1:3 loop
     (u[i],g) := CMRG.RandU01(g);
    end for;
    g := CMRG.ResetNextSubstream(g);
    for i in 4:6 loop
     (u[i],g) := CMRG.RandU01(g);
    end for;
    g := CMRG.ResetStartStream(g);
    for i in 7:9 loop
     (u[i],g) := CMRG.RandU01(g);
    end for;

  // u = {0.463941,0.733693,0.00275546,
          0.196378,0.218283,0.34092,
          0.463941,0.733693,0.00275546}
</pre>

</HTML>
"));
        input RngStream g "RngStream";
        output RngStream gout "Updated RngStream";
      protected
        Integer i;
      algorithm
        gout := g;
        for i in 1:6 loop
          gout.Cg[i] := gout.Ig[i];
          gout.Bg[i] := gout.Ig[i];
        end for;
      end ResetStartStream;

      /*-------------------------------------------------------------------------*/

      function ResetNextSubstream "ResetNextSubstream"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
ResetNextSubstream
</font></h3>


<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>ResetNextSubstream</b>(g)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
 Reinitializes the state of RngStream g to the beginning of the next sub-stream.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
    g2 := g1;
    g2 := CMRG.ResetNextSubstream(g2);
    for i in 1:3 loop
      (u1[i],g1) := CMRG.RandU01(g1);
      (u2[i],g2) := CMRG.RandU01(g2);
    end for;
    g1 := CMRG.ResetNextSubstream(g1);
    for i in 4:6 loop
      (u1[i],g1) := CMRG.RandU01(g1);
      (u2[i],g2) := CMRG.RandU01(g2);
    end for;

  // u1 = {0.89111  ,0.941477,0.194141,
  //       0.0368941,0.606502,0.597065}
  // u2 = {0.0368941,0.606502,0.597065,
  //       0.768019 ,0.628601,0.443735}

 </pre>

</HTML>
"));
        input RngStream g "RngStream";
        output RngStream gout "Updated RngStream";
      protected
        Integer i;
        Real[3] A;
      algorithm
        gout := g;

        for i in 1:3 loop
          A[i] := gout.Bg[i];
        end for;
        A := Src.MatVecModM( Src.A1p76, A, Src.m1);
        for i in 1:3 loop
          gout.Bg[i] := A[i];
        end for;

        for i in 4:6 loop
          A[i-3] := gout.Bg[i];
        end for;
        A := Src.MatVecModM( Src.A2p76,A,  Src.m2);
        for i in 4:6 loop
          gout.Bg[i] := A[i-3];
        end for;
        for i in 1:6 loop
          gout.Cg[i] := gout.Bg[i];
        end for;
      end ResetNextSubstream;

      /*-------------------------------------------------------------------------*/

      function ResetStartSubstream "ResetStartSubstream"

      annotation(preferedView="info",
        Documentation(info="<HTML>

<h3><font color=\"#008000\" size=5>
ResetStartSubstream
</font></h3>


<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>ResetStartSubstream</b>(g)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
 Reinitializes the state of RngStream g to the beginning of the current sub-stream.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
    for i in 1:3 loop
      (u[i],g) := CMRG.RandU01(g);
    end for;
    g := CMRG.ResetStartSubstream(g);
    for i in 4:6 loop
      (u[i],g) := CMRG.RandU01(g);
    end for;

  // u = {0.584045,0.18949,0.698993,
  //      0.584045,0.18949,0.698993}

</pre>


</HTML>
"));
        input RngStream g "RngStream";
        output RngStream gout "Updated RngStream";
      protected
        Integer i;
      algorithm
        gout := g;
        for i in 1:6 loop
          gout.Cg[i] := gout.Bg[i];
        end for;
      end ResetStartSubstream;

      /*-------------------------------------------------------------------------*/

      function SetPackageSeed "SetPackageSeed"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
SetPackageSeed
</font></h3>


<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>SetPackageSeed</b>(seed)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
 Changes the current seed stored in the file \"CMRGSeed.txt\".
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
  SetPackageSeed({54321,54321,54321,54321,54321,54321});
</pre>

</HTML>
"));
        input Src.Seed seed "New package seed";
        output Integer out "result (0 == success)";
      protected
        Integer i;
      algorithm
        if (Src.CheckSeed( seed) <> 0) then
          out := -1;                    /* FAILURE */
        else
          Src.SaveSeed(seed);
          out := 0;
        end if;                       /* SUCCESS */
      end SetPackageSeed;

      /*-------------------------------------------------------------------------*/

      function SetSeed "SetSeed"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
SetSeed
</font></h3>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>SetSeed</b>(g,seed)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
 Changes the initial seed of the RngStream g. This modificatioin also changes the current state of the RngStream.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
  g2 := g1;
  for i in 1:3 loop
    (u1[i],g1) := CMRG.RandU01(g1);
    (u2[i],g2) := CMRG.RandU01(g2);
  end for;
  g2 := CMRG.SetSeed(g2,{2345,2435,2345,2345,2345,2345});
  for i in 4:6 loop
    (u1[i],g1) := CMRG.RandU01(g1);
    (u2[i],g2) := CMRG.RandU01(g2);
  end for;

  // u1 = {0.586819,0.848015,0.494309,0.609745,0.698027,0.812434}
  // u2 = {0.586819,0.848015,0.494309,0.813362,0.517802,0.947542}
</pre>

</HTML>
"));
        input RngStream g "RngStream";
        input Src.Seed seed "New seed";
        output RngStream gout "Updated RngStream";
        output Integer out "result (0 == success)";
      protected
        Integer i;
      algorithm
        gout := g;
        if (Src.CheckSeed( seed) <> 0) then
          out := -1;  /* FAILURE */
        else
          for i in 1:6 loop
            gout.Cg[i] := seed[i];
            gout.Bg[i] := seed[i];
            gout.Ig[i] := seed[i];

         end for;
          out := 0;      /* SUCCESS */
        end if;
      end SetSeed;

      /*-------------------------------------------------------------------------*/

      function AdvanceState
        "Advances the state of the RngStream g by k values. If e>0 , then k=2^e+c; if e<0, then k=-2^-e+c; and if e=0,then k=c"

        input RngStream g;
        input Real e;
        input Real c;
        output RngStream gout;
      protected
        Real[3,3] B1;
        Real[3,3] C1;
        Real[3,3] B2;
        Real[3,3] C2;
        Real[3] A;
      algorithm
        gout := g;
        if (e > 0) then
          B1 := Src.MatTwoPowModM( Src.A1p0, Src.m1, e);
          B2 := Src.MatTwoPowModM( Src.A2p0, Src.m2, e);
        else
          if (e < 0) then
            B1  := Src.MatTwoPowModM( Src.InvA1, Src.m1, -e);
            B2  := Src.MatTwoPowModM( Src.InvA2, Src.m2, -e);
          end if;
        end if;

        if (c >= 0) then
          C1 := Src.MatPowModM( Src.A1p0, Src.m1, c);
          C2 := Src.MatPowModM( Src.A2p0, Src.m2, c);
        else
          C1 := Src.MatPowModM( Src.InvA1, Src.m1, -c);
          C2 := Src.MatPowModM( Src.InvA2, Src.m2, -c);
        end if;

        if (e <> 0) then
          C1 := Src.MatMatModM( B1, C1, Src.m1);
          C2 := Src.MatMatModM( B2, C2, Src.m2);
        end if;

       for i in 1:3 loop
          A[i] := gout.Cg[i];
        end for;
        A := Src.MatVecModM( C1, A, Src.m1);
        for i in 1:3 loop
          gout.Cg[i] := A[i];
        end for;

        for i in 4:6 loop
          A[i-3] := gout.Cg[i];
        end for;
        A := Src.MatVecModM( C2,A,  Src.m2);
        for i in 4:6 loop
          gout.Cg[i] := A[i-3];
        end for;

      end AdvanceState;

      /*-------------------------------------------------------------------------*/

      function GetState "GetState"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
GetState
</font></h3>


<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>GetState</b>(g)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
 Returns the current state of RngStream g.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
  CMRG.GetState(g) =
  {2.9888e+009,1.4434e+009,3.39191e+009,6.80411e+007,1.94413e+009,1.91104e+008}
</pre>

</HTML>
"));
        input RngStream g "RngStream";
        output Src.Seed seed "Current state of RngStream";
      protected
        Integer i;
      algorithm
        for i in 1:6 loop
          seed[i] := g.Cg[i];
        end for;
      end GetState;

      /*-------------------------------------------------------------------------*/

      function IncreasedPrecis "IncreasedPrecis"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
IncreasedPrecis
</font></h3>
<p>
Increases the precision of the numbers generated from RngStream g
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>IncreasedPrecis</b>(g,incp)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
When called with incp > 0, switches the RngStream to increased precision mode and the generated number will be in increased precision. <br>
To switch the increased precision mode back it is necessary to call this function with incp == 0.<br><br>
Since Modelica only have one precision for Real numbers, the effect of this function is only to generate alternate random numbers from the ones in the stream.
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
    g2 := CMRG.IncreasedPrecis(g2,1);
    for i in 1:6 loop
      (u1[i],g1) := CMRG.RandU01(g1);
      //(u2[i],g2) := CMRG.RandU01(g2);
    end for;
    for i in 1:3 loop
       (u2[i],g2) := CMRG.RandU01(g2);
    end for;

// u1 = {0.736348,0.970465,0.776025,0.860987,0.803047,0.574409}
// u2 = {0.736348,         0.776025,         0.803047}
</pre>

</HTML>
"));
        input RngStream g "RngStream";
        input Integer incp "Increased precission switch";
        output RngStream gout "Updated RngStream";
      algorithm
        gout := g;
        gout.IncPrec := incp;
      end IncreasedPrecis;

      /*-------------------------------------------------------------------------*/

      function SetAntithetic "SetAntithetic"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
SetAntithetic
</font></h3>
<p>
Change to antithetic generation mode.
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>SetAntithetic</b>(g,a)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
With a &lt;> 0, the RngStream g will generate antithetic numbers (U-1 instead of U)
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
    g2 := g1;
    g2 := CMRG.SetAntithetic(g2,1);
    u1 := CMRG.RandU01(g1);
    u2 := CMRG.RandU01(g2);

// u1 = 0.37564
// u2 = 0.62436

</pre>

</HTML>
"));
        input RngStream g "RngStream";
        input Integer a "Antithetic switch";
        output RngStream gout "Updated RngStream";
      algorithm
        gout := g;
        gout.Anti := a;
      end SetAntithetic;

      /*-------------------------------------------------------------------------*/

      function RandU01 "RandU01"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
RandU01
</font></h3>
<p>
Uniform random number generation function
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>RandU01</b>(g)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Returns a uniform random number in the interval (0,1), and updates the state of RngStream g by one step
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
model Example
  CMRG.RngStream g;
  Real u[5];
algorithm
  when initial() then
    g := CMRG.CreateStream();
  end when;
  when time &lt;= 0 then
    for i in 1:5 loop
      (u[i],g) := CMRG.RandU01(g);
    end for;
  end when;
end Example;

// u = {0.988831,0.760616,0.855857,0.546418,0.868702}

</pre>

</HTML>
"));
        input RngStream g "RngStream";
        output Real out "Generated uniform random number";
        output RngStream gout "Updated RngStream";
      algorithm
        if (g.IncPrec <> 0) then
          (out,gout)  := Src.U01d(g);
        else
          (out,gout)  := Src.U01(g);
        end if;
      end RandU01;

      /*-------------------------------------------------------------------------*/

      function RandInt "RandInt"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<h3><font color=\"#008000\" size=5>
RandInt
</font></h3>
<p>
Discrete distribution random number generation function
</p>

<h3><font color=\"#008000\">Syntax</font></h3>
<blockquote><pre><b>RandInt</b>(g,i,j)</pre></blockquote>


<h3><font color=\"#008000\">Description</font></h3>
<P>
Returns a rundom number from the discrete uniform distribution over the integer (i,i+1,...,j), using RngStream g
</P>


<h3><font color=\"#008000\">Examples</font></h3>
<pre>
model Example
  CMRG.RngStream g;
  Real u[5];
algorithm
  when initial() then
    g := CMRG.CreateStream();
  end when;
  when time &lt;= 0 then
    for i in 1:5 loop
      (u[i],g) := CMRG.RandInt(g);
    end for;
  end when;
end Example;

// u = {8,8,7,9,2}

</pre>

</HTML>
"));
        input RngStream g;
        input Integer i;
        input Integer j;
        output Integer out;
        output RngStream gout;
      protected
        Real u;
      algorithm
        (u,gout) := RandU01(g);
        out := i+ integer((j-i +1.0) * u);
      end RandInt;

      annotation (uses(Modelica(version="2.2.1")));
    end CMRG;
    annotation (uses(Modelica(version="2.2.1")));

    package Examples
      "Examples of random uniform numbers and random variates generation"

    annotation(preferedView="info",
      Documentation(info="<HTML>
<p>
This package contains some examples that show the use of the RandomLib library and its components.
</p>


<p>
<em>CMRGSimple</em>, <em>Mean</em>, <em>VariatesSimple</em> and <em>VariatesSimple2</em> are very simple examples of uniform random number and random variates generation.
</P>
<p>
<em>Test</em> is a larger test case for uniform random number generation. This example is based in the test code provided by L'Ecuyer with his <a href=\"http://www.iro.umontreal.ca/~lecuyer/myftp/streams00/c/\">RngStream C implementation</a>.<br>
This example has been used for validating the implementation of the CMRG library in comparison with the results obtained with the mentioned C implementation.
</p>


<dl>
<dt><b>Author:</b></dt>
<dd><a href=\"http://www.euclides.dia.uned.es/vsanz\"> Victorino Sanz </a><br>
    Dpto. Informtica y Automtica, UNED<br>
    Juan del Rosal, 16<br>
    28040, Madrid<br>
    Spain<br>
    email: <A HREF=\"mailto:vsanz@dia.uned.es\">vsanz@dia.uned.es</A><br></dd>
</dl>
</p>

<p>
<dt><b>Copying:</b></dt>
<p>
<p>
Licensed by Victorino Sanz under the Modelica License 2 <br>
Copyright 2009, Victorino Sanz.
</p>
<p>
<i>This Modelica packages is <u>free</u> software and the use is completely at <u>your ouwn risk;</u> it can be redistributed and/or modified under the terms of the Modelica license 2,
see the license conditions (including the disclaimer of warranty) <a href=\"Modelica://DESLib.ModelicaLicense2\">here</a> or at <a href=\"http://www.modelica.org/licenses/ModelicaLicense2\">http://www.modelica.org/licenses/ModelicaLicense2</a>
</p>
<br>
</HTML>
"));

        model CMRGSimple "generates 10 random numbers from an RngStream"
          CMRG.RngStream g "RngStream";
          Real u[10] "vector of random numbers";
          Boolean init( start = false);
        algorithm
        // initilization of the RngStream
        when initial() and not init then
          g := CMRG.CreateStream();
          init := true;
        end when;
        when time <= 10 then
          for i in 1:10 loop
            // generation of uniform random numbers.
            (u[i],g) := CMRG.RandU01(g);
            Modelica.Utilities.Streams.print(String(u[i]));
          end for;
        end when;
        annotation (experiment, experimentSetupOutput);
        end CMRGSimple;

        model Mean
        "generates random variates (one per second) and calculates the mean of the generated samples"
          CMRG.RngStream g "RngStream";
          Real u "random number";
          Boolean init( start = false);
          Integer Obs(start = 0) "number of observations";
          Real Mean( start = 0) "mean";
        algorithm
        // initilization of the RngStream
        when initial() and not init then
          g := CMRG.CreateStream();
          init := true;
        end when;
        when sample(0,1) then
          (u,g) := Variates.GenerateVariate(13,g,5,1); // generate Normal variates with mean 5 and std 1
          Obs := Obs +1;
          Mean := ((Obs/(Obs+1))*Mean)+(u/(Obs+1));
        end when;
          annotation (experiment(StopTime=10000), experimentSetupOutput);
        end Mean;

      model VariatesSimple
        "generates 5 random variates with Exponential(5) distribution"
        Variates.Generator g "RNG";
        Real u[5] "vector of random variates";
      algorithm
        // initilization of the RngStream
        when initial() then
          g := Variates.initGenerator();
        end when;
        when time <= 0 then
          for i in 1:5 loop
            // generation of uniform random numbers.
            (u[i],g) := Variates.Continuous.Exponential(g,5);
            Modelica.Utilities.Streams.print(String(u[i]));
          end for;
        end when;
      end VariatesSimple;

      model VariatesSimple2
        "generates a discrete variate of each distribution at each integration step"
        Variates.Generator g "RNG";
        Real ConstantVariate;
        Real Bernoulli;
        Real DiscreteUniform;
        Real Binomial;
        Real Geometric;
        Real NegativeBinomial;
        Real Poisson;
        Real Uniform;
        Real Exponential;
        Real Erlang;
        Real Gamma;
        Real Weibull;
        Real Normal;
        Real LogNormal;
        Real Beta;
        Real Johnson;
        Real Triangular;
      algorithm
        // initilization of the RngStream
        when initial() then
          g := Variates.initGenerator();
        end when;
        (ConstantVariate,g) := Variates.GenerateVariate(1,g,0.4);
        (Bernoulli,g) := Variates.GenerateVariate(2,g,0.4);
        (DiscreteUniform,g) := Variates.GenerateVariate(3,g,2,4);
        (Binomial,g) := Variates.GenerateVariate(4,g,4,0.3);
        (Geometric,g) := Variates.GenerateVariate(5,g,0.5);
        (NegativeBinomial,g) := Variates.GenerateVariate(6,g,6,0.2);
        (Poisson,g) := Variates.GenerateVariate(7,g,4.5);
        (Uniform,g) := Variates.GenerateVariate(8,g,2,4);
        (Exponential,g) := Variates.GenerateVariate(9,g,0.4);
        (Erlang,g) := Variates.GenerateVariate(10,g,0.5,4);
        (Gamma,g) := Variates.GenerateVariate(11,g,0.4,0.7);
        (Weibull,g) := Variates.GenerateVariate(12,g,0.4,0.8);
        (Normal,g) := Variates.GenerateVariate(13,g,0,1);
        (LogNormal,g) := Variates.GenerateVariate(14,g,0,1);
        (Beta,g) := Variates.GenerateVariate(15,g,0.6,0.4);
        (Johnson,g) := Variates.GenerateVariate(16,g,0.1,0.5,1,4);
        (Triangular,g) := Variates.GenerateVariate(17,g,1,3,6);

        annotation (experiment, experimentSetupOutput);
      end VariatesSimple2;

    model Test "Pierre L'Ecuyer test program for RngStreams"
        import DESLib.RandomLib.CMRG.*;
      RngStream g1;
      RngStream g2;
      RngStream g3;
      RngStream gar1;
      RngStream gar2;
      RngStream gar3;
      RngStream gar4;
      Real germe[6] = { 1, 1, 1, 1, 1, 1};
      Real sum;
      Real sum3;
      Integer sumi;
      Real u1;
      Real u2;
      Real u3;
      Integer i1;

      function WriteState
        input CMRG.RngStream g;
      algorithm
        Modelica.Utilities.Streams.print("The current state of the Rngstream");
        Modelica.Utilities.Streams.print("Cg = {"+String(g.Cg[1],significantDigits=14)+","+String(g.Cg[2],significantDigits=14)+","+String(g.Cg[3],significantDigits=14)+","+String(g.Cg[4],significantDigits=14)+","+String(g.Cg[5],significantDigits=14)+","+String(g.Cg[6],significantDigits=14)+"}\n");
      end WriteState;
    algorithm
      when time <= 0 then
        g1 := CreateStream();
        g2 := CreateStream();
        g3 := CreateStream();
        Modelica.Utilities.Streams.print("Initial states of g1, g2, and g3:\n");
        WriteState(g1);
        WriteState(g2);
        WriteState(g3);
        (u2,g2) := RandU01(g2);
        (u3,g3) := RandU01(g3);
        sum :=u2 + u3;

        for i in 1:12345 loop
          (u2,g2) := RandU01(g2);
        end for;

        g1 := AdvanceState(g1, 5, 3);
        Modelica.Utilities.Streams.print("State of g1 after advancing by 2^5 + 3 = 35 steps:");
        WriteState(g1);
        (u1,g1) := RandU01(g1);
        Modelica.Utilities.Streams.print("RandU01 (g1) = "+String(u1)+"\n");

        g1 :=  ResetStartStream(g1);
        for i in 1:35 loop
          g1 := AdvanceState(g1, 0,1);
        end for;
        Modelica.Utilities.Streams.print("State of g1 after reset and advancing 35 times by 1:");
        WriteState( g1);
        (u1,g1) := RandU01(g1);
        Modelica.Utilities.Streams.print("RandU01 (g1) = "+String(u1)+"\n");

        g1 := ResetStartStream(g1);
        for i in 1:35 loop
          (i1,g1) := RandInt(g1,1,10);
          sumi := sumi + i1;
        end for;
        Modelica.Utilities.Streams.print("State of g1 after reset and 35 calls to RandInt (1, 10):");
        WriteState(g1);
        Modelica.Utilities.Streams.print("   sum of 35 integers in [1, 10] = "+String(sumi)+"\n");
        sum := sum + (sumi / 100.0);

        (u1,g1) := RandU01(g1);
        Modelica.Utilities.Streams.print( "RandU01 (g1) = "+String(u1)+"\n");

        sum3 :=0.0;
        g1 := ResetStartStream( g1);
        g1 := IncreasedPrecis( g1, 1);
        sumi :=0;
        for i in 1:17 loop
          (i1,g1) := RandInt(g1,1,10);
          sumi := sumi + i1;
        end for;
        Modelica.Utilities.Streams.print("State of g1 after reset, IncreasedPrecis (1) and 17 calls");
        Modelica.Utilities.Streams.print(" to RandInt (1, 10):");
        WriteState( g1);
        g1 := IncreasedPrecis( g1, 0);
        (i1,g1) := RandInt(g1,1,10);
        Modelica.Utilities.Streams.print( "State of g1 after IncreasedPrecis (0) and 1 call to RandInt");
        WriteState( g1);
        sum3 :=sumi/10.0;

        g1 := ResetStartStream( g1);
        g1 := IncreasedPrecis( g1, 1);
        for i in 1:17 loop
          (u1,g1) := RandU01(g1);
          sum3 := sum3 + u1;
        end for;
        Modelica.Utilities.Streams.print("State of g1 after reset, IncreasedPrecis (1) and 17 calls");
        Modelica.Utilities.Streams.print(" to RandU01:");
        WriteState( g1);
        g1 := IncreasedPrecis( g1, 0);
        (u1,g1) := RandU01(g1);
        Modelica.Utilities.Streams.print("State of g1 after IncreasedPrecis (0) and 1 call to RandU01");
        WriteState( g1);
        sum := sum + sum3 / 10.0;

        sum3 := 0.0;
        Modelica.Utilities.Streams.print( "Sum of first 100 output values from stream g3:");
        for i in 1:100 loop
          (u3,g3) := RandU01(g3);
          sum3 := sum3 + u3;
        end for;
        Modelica.Utilities.Streams.print( "   sum = "+String(sum3)+"\n");
        sum := sum + sum3 / 10.0;

        Modelica.Utilities.Streams.print( "\nReset stream g3 to its initial seed.");
        g3 := ResetStartSubstream(g3);
        Modelica.Utilities.Streams.print( "First 5 output values from stream g3:");
        for i in 1:5 loop
          (u3,g3) := RandU01(g3);
          Modelica.Utilities.Streams.print(String(u3));
        end for;
        (u3,g3) := RandU01(g3);
        sum :=sum + u3;

        Modelica.Utilities.Streams.print("\nReset stream g3 to the next Substream, 4 times.");
        for i in 1:4 loop
          g3 := ResetNextSubstream( g3);
        end for;
        Modelica.Utilities.Streams.print("First 5 output values from stream g3, fourth Substream:");
        for i in 1:5 loop
          (u3,g3) := RandU01(g3);
          Modelica.Utilities.Streams.print(String(u3));
        end for;
        (u3,g3) := RandU01(g3);
        sum := sum + u3;

        Modelica.Utilities.Streams.print("\nReset stream g2 to the beginning of Substream.");
        g2 := ResetStartSubstream( g2);
        Modelica.Utilities.Streams.print(" Sum of 100000 values from stream g2 with double precision:   ");
        g2 := IncreasedPrecis( g2, 1);
        sum3 :=0.0;
        for i in 1:100000 loop
          (u2,g2) := RandU01(g2);
          sum3 := sum3 + u2;
        end for;
        Modelica.Utilities.Streams.print(String(sum3));
        sum := sum + sum3 / 10000.0;

        g2 := IncreasedPrecis( g2, 0);

        g3 := SetAntithetic( g3, 1);
        Modelica.Utilities.Streams.print(" Sum of 100000 antithetic output values from stream g3:   ");
        sum3 :=0.0;
        for i in 1:100000 loop
          (u3,g3) := RandU01(g3);
          sum3 := sum3 + u3;
        end for;
        Modelica.Utilities.Streams.print(String(sum3));
         sum := sum + sum3 / 10000.0;

        Modelica.Utilities.Streams.print("SetPackageSeed to seed = { 1, 1, 1, 1, 1, 1 }\n");
        SetPackageSeed(germe);

        Modelica.Utilities.Streams.print("\nCreate an array of 4 named streams and ");
        Modelica.Utilities.Streams.print("write their full state\n");
        gar1 := CreateStream();
        gar2 := CreateStream();
        gar3 := CreateStream();
        gar4 := CreateStream();

        WriteState(gar1);
        WriteState(gar2);
        WriteState(gar3);
        WriteState(gar4);
        Modelica.Utilities.Streams.print("Jump stream Galois (gar3) by 2^127 steps backward");
        gar3 := AdvanceState( gar3, -127, 0);
        WriteState( gar3);
        gar3 := ResetNextSubstream(gar3);

        (u1,gar1) := RandU01(gar1);
        sum := sum + u1;
        (u1,gar2) := RandU01(gar2);
        sum := sum + u1;
        (u1,gar3) := RandU01(gar3);
        sum := sum + u1;
        (u1,gar4) := RandU01(gar4);
        sum := sum + u1;

        Modelica.Utilities.Streams.print("--------------------------------------");
        Modelica.Utilities.Streams.print("Final Sum = "+String(sum)+"\n");
      end when;
    end Test;

    end Examples;
  end RandomLib;

  package DEVSLib "Parallel DEVS formalism implementation in Modelica"

  annotation(preferedView="info",
    Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica package for modeling systems using the Parallel DEVS formalism.
</p>

<p>
The structure of the package is:
<ul>
<li> The <a href=\"Modelica://DESLib.DEVSLib.atomicDraft\">atomicDraft</a> package, that represents the starting point for developing new Atomic DEVS models.
<li> The <a href=\"Modelica://DESLib.DEVSLib.coupledDraft\">coupledDraft</a> model, contains the basic structure for the development of new Coupled DEVS models.
<li> The <a href=\"Modelica://DESLib.DEVSLib.AuxModels\">AuxModels</a> package, with some useful auxiliary models.
<li> The <a href=\"Modelica://DESLib.DEVSLib.Examples\">Examples</a> package, with models of simple and more complex systems modeled using DEVSLib.
<li> The <a href=\"Modelica://DESLib.DEVSLib.SRC\">SRC</a> package, contains the source packages and models.

</ul>
<p>
For a detailed description of the package, its contents, and the usage, please read the <a href=\"Modelica://DESLib.DEVSLib.UsersGuide\">Users Guide</a>.
</p>
<p>
For a detailed description of the implementation of the package and its internal components, please read the <a href=\"Modelica://DESLib.DEVSLib.SRC.DevelopersGuide\">Developers Guide</a>.
</p>

<dl>
<dt><b>Author:</b></dt>
<dd><a href=\"http://www.euclides.dia.uned.es/vsanz\"> Victorino Sanz </a><br>
    Dpto. Informtica y Automtica, UNED<br>
    Juan del Rosal, 16<br>
    28040, Madrid<br>
    Spain<br>
    Email: <A HREF=\"mailto:vsanz@dia.uned.es\">vsanz@dia.uned.es</A><br>
    Website: <A HREF=\"http://www.euclides.dia.uned.es/vsanz\">www.euclides.dia.uned.es/vsanz</A></dd>
</dl>
</p>
<p>
<dt><b>Copying:</b></dt>
<p>
Licensed by Victorino Sanz under the Modelica License 2 <br>
Copyright 2009, Victorino Sanz.
</p>
<p>
<i>This Modelica package is <u>free</u> software and the use is completely at <u>your ouwn risk;</u> it can be redistributed and/or modified under the terms of the Modelica license 2,
see the license conditions (including the disclaimer of warranty) <a href=\"Modelica://DESLib.ModelicaLicense2\">here</a> or at <a href=\"http://www.modelica.org/licenses/ModelicaLicense2\">http://www.modelica.org/licenses/ModelicaLicense2</a>
</p><br>

</HTML>
"),   uses(Modelica(version="2.2.1")));

  model UsersGuide "Users Guide"

      annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Users Guide of the DEVSLib Package</font></h3>

<p>
<b>DEVSLib</b> package can be used to develop new models of systems following the Parallel DEVS formalism.
</p>
<p>
DEVS (Discrete Event System Specification) formalism was firstly introduced by <a href=http://www.acims.arizona.edu/MEMBERS/bio/BPZWebBio.htm>Bernard P. Zeigler</a>.
A full description of the formalism can be found in: B.P.Zeigler, H.Praehofer and T.G.Kim. <em>Theory of Modeling and Simulation (2ed)</em>. Academic Press, 2000.
</p>
<p>
In this library the Parallel DEVS formalism has been implemented. This formalism is described in the next section.
</p>

<h3><font color=\"#008000\">Library Structure</font></h3>
<p>
The library has been divided in two parts, the user part and the source/developer part.<br>

The user part contains the required models and packages to develop new atomic and coupled DEVS models. It is composed by:
<ul>
<li>The <b>atomicDraft</b> package, that represents the basic model and functions to develop new atomic DEVS models.
Any user that wants to develop a new atomic model must duplicate this package and edit the atomic model to include the required input and output ports,
and adapt the functions to meet his/her necessities. This procedure is detailed in the <em>New Atomic DEVS models</em> section of this UsersGuide.
<li>The <b>coupledDraft</b> model, consists in the basic structure for a new coupled DEVS model. It can be used to develop new coupled models.
The construction of new coupled models will be described in the <em>New Coupled DEVS models</em> section.
<li>The <b>AuxModels</b> package, contains several auxiliary models that can be useful in combination with the user developed models.
It also contains five models, DICO, DiBO, Quantizer, CrossUP, and CrossDOWN that can be used to communicate continuous and discrete models in a hybrid system.
These models will be detailed in the <em>Hybrid models development</em> section.
<li>The <b>Examples</b> package, includes several examples of simple and complex DEVS systems.
A description of the modeled system is included in the documentation of each system.
</ul>
</p>
<p>
The source/developer part (SRC package) contains the core source packages and models of the library.
It includes the interfaces for event communication, functions, data structures, basic DEVS models and components.
</p>

<h3><font color=\"#008000\">Users Guide Structure</font></h3>
<p>
This guide is organized in four sections.
</p>

<p>
The first section, called <b>Parallel DEVS</b> contains a description of the DEVS formalism implemented in the library.
The information in this section is basic to understand the behaviour of the library models, specially the atomic models.
If the user is not familiar with the formalism, the reading of this section is very encouraged.
</p>

<p>
The other sections describe the models that compose the library and their use to develop new atomic and coupled DEVS models.<br>
Section <b>New Atomic DEVS models</b> details the procedure to develop new atomic models.<br>
Section <b>New Coupled DEVS models</b> is similar to the previous one but about coupled models.<br>
Last section, <b>Hybrid models development</b> describes the use of DICO and CODI models to communicate continuous and discrete models in a hybrid system.
</p>




</html>"));
  class ParallelDEVS "Parallel DEVS"

  annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Parallel DEVS Formalism</font></h3>

<p>
Models in DEVS are constituted by an state (or a set of sequential states), input ports and output ports.
The input and output ports are used to communicate with other models, sending and receiving events.
The state of the model is modified in transitions, that occur at discrete points in time. These transitions are divided into external and internal ones.
External transitions occur when a new event is received in the model.
Internal transitions occur when a predefined time interval passes without receiving any event.
</p>
<p>
The main difference between Parallel DEVS and other DEVS formalisms, like Classic DEVS is that in Parallel DEVS
an external transition and an internal transition can be executed at the same time. In Classic DEVS, only one of them will be executed at a time.
</p>

<p>
As described in [1], the basic formalism of a Parallel DEVS model is:<br>
&nbsp;&nbsp;DEVS = (X,Y,S,&sigma;ext,&sigma;int,&sigma;con,&lambda;,ta) <br>
with<br>
&nbsp;&nbsp; X is the set of input events.<br>
&nbsp;&nbsp; S is the set of sequential states.<br>
&nbsp;&nbsp; Y is the set of output events.<br>
&nbsp;&nbsp; &sigma;int: S -> S is the internal transition function.<br>
&nbsp;&nbsp; &sigma;ext: Q &times; X<sup>b</sup> -> S is the external transition function, where X<sup>b</sup> is a set of bags over elements in X.<br>
&nbsp;&nbsp; &sigma;con: S &times; X<sup>b</sup> -> S S is the confluent transition function, subject to &sigma;con(s,&phi;) = &sigma;int(s).<br>
&nbsp;&nbsp; &lambda;: S -> Y<sup>b</sup> is the output function.<br>
&nbsp;&nbsp; ta: S -> R<sup>+</sup><sub>0Uinf</sub> is the time advance function, where Q = {(s,e)|sS,0&lt;e&lt;ta(s)}. e is the elapsed time since the last transition.
</p>

<h3><font color=\"#008000\">Model behavior</font></h3>

<p>
The behavior of a Parallel DEVS model is the following:
<ul>
<li> When the model receives an external event (or a bag of external events, if more than one event is received at the same time), the state is updated using the external transition function.
After this, the time advance function calculates the new time interval until the next internal transition.
<li> If the time interval, calculated by the time advance function, passes without receiving any event, an internal transition occurs.
First, the model sends an output using the current state and the output function.
After that, the internal transition function updates the state.
At the end, the time advance function recalculates the time interval until the next internal transition.
It has to be noticed that the time interval can be any positive real number, included zero and infinity.
<li> In Parallel DEVS is possible to execute an internal transition and an external transition at the same time.
when this situation happens the confluent transition function is used to update the state, by executing the internal and external transitions in the desired order.
It is also possible to calculate the new state without using the internal and external transition functions.
</ul>
</p>
<img src=\"..\Figs\\DEVS-basic_model.png\">



<h3><font color=\"#008000\">References</font></h3>
<p>
[1] B.P.Zeigler, H.Praehofer and T.G.Kim.<em>Theory of Modeling and Simulation (2ed)</em>. Academic Press,2000.
</p>
</html>"));

  end ParallelDEVS;

  class NewAtomic "New Atomic DEVS models"

  annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Development of new Atomic DEVS models using DEVSLib</font></h3>
<p>
DEVSLib can be used to develop any new Atomic DEVS model. The user only has to define the behavior of this new model and implement it using the atomicDraft package included in the library.<br>
The procedure to create a new atomic model can be divided in the following steps:
</p>
<h3><font color=\"#008000\">1. Create the package for the new model.</font></h3>
<p>
This step has to be performed by just duplicating the atomicDraft package into another package that will store the new atomic model.
</p>
<p>
The best way to duplicate a package is to use the Dymola menu option.
Selecting the atomicDraft package and left clicking in it, the menu is displayed. The duplicate option corresponds to the Edit->Duplicate Class option in the menu.
The following figure shows the menu and the duplicate option.
</p>
 <br>
<img src=\"..\Figs\\newA1.png\">

<p>
The new name for the atomic model has to be indicated, and the place to store the package selected among the Dymola's Packages tree structure.
</p>

<p>
Finally, the atomic model inside the atomicDraft package can be changed to match the new name of the atomic model.
</p>


<h3><font color=\"#008000\">2. Declare the input/output ports</font></h3>
<p>
Next step is to define the input and output ports of the new model. The number of input and output ports depend on the behavior of the new model.
</p>

<p>
New ports can be declared by just copy/paste one of the existing input/output port.
If no ports are needed, the existing ones can also be deleted.
Another way to introduce new ports is to drag and drop them from the DEVSLib.SRC.Interfaces package.
</p>
<p>
After declaring the new ports, some equations must be added and edited in the model.
These equations correspond to the connection between the ports and the iQueue, iEvent, oQueue and oEvent arrays in the model.
</p>
<p>
Each input port in the model must have two equations connecting the event and queue variables of the connector, to the iEvent and iQueue variables in the model.
The index of the iEvent/iQueue arrays assigned to each connector identifies it, and will also be used to identify the received messages.
</p>
<p>
Each output port in the model must have two equations connecting the event and queue variables of the connector, to the oEvent and oQueue variables in the model.
The index of the oEvent/oQueue arrays assigned to each connector identifies it, and will also be used by the output function to identify the port for the output messages.
</p>
<p>
If the model has no input ports, the following equation has to be added <i>iEvent[1] = 0;</i>, since the minimum value for the number of input ports (numIn) is 1.
(see the <a href=\"Modelica://DESLib.DEVSLib.Examples.SimpleModels.Generator.generator\">Generator</a> model for example)<br>
If the model has no output ports, the following equiation has to be added <i>oQueue[1] = 0;</i>, since the minimum value for the number of output ports (numOut) is 1.
(see the <a href=\"Modelica://DESLib.DEVSLib.Examples.SimpleModels.Display.display\">Display</a> model for example)<br>
</p>
<p>
At the end, the numIn and numOut parameters of the model have also to be set, indicating the number of input and output ports in the model.
An example of port definition and equation edition is shown in the next figure.
</p>
<br>
<img src=\"..\Figs\\newA21-2.png\"><img src=\"..\Figs\\newA22-2.png\">




<h3><font color=\"#008000\">3. Define the state of the model</font></h3>
<p>
The state of a DEVS model is composed by two elements.
First, the <em>st</em> record that stores all the variables that are stored in the state.
Second, the <em>initst</em> function used to initialize the state of each model at the beginning of the simulation.
</p>
<p>
The user has to declare in the st record all the variables used to control the model behavior.
By default this record contains the phase variable, normally used to indicate the current <em>phase</em> of the model (active, passive, busy,...),
and the <em>sigma</em> variable, that usually indicates the time for the next internal transition.<br>
Any other variable can be included in the record, using the usual Modelica types and notation.
 Notice that if the model has any parameter, it must be stored in the state of the model in order to use it in the transition and output functions.
So each parameter must have a corresponding variable in the state of the model.
</p>
<p>
The second component of the state is the initialization function. This function is called at the beginning of
the simulation to initialize the state of each model.<br>
The initst function has as inputs the parameters of the model, if any.
The output of the function is the initial state, as desired by the user.
The value set to the sigma variable will be used to generate the first internal transition.
</p>

<h3><font color=\"#008000\">4. Redeclare the external transition function</font></h3>
<p>
As described in the Parallel DEVS section, every time a new message arrives to a model an external transition is executed.
Simultaneous arrivals to the model can also occur when:
<ul>
<li> Several messages arrive simultaneously to the same port.
<li> Several messages arrive at different input ports.
</ul>
</p>
<p>
All the received messages are temporary stored in a <i>bag</i>.
In both cases the external transition function is executed and manages the bag of messages.
</p>
<p>
The actions performed in each external transition have to be implemented in the <em>ext</em> function,
inside the duplicated atomicDraft package.
This function receives as inputs the current state, the elapsed time, the bag of received messages.
It returns the new state after the external transition.
The user has to read the message or messages received in each transition from the bag of messages.
This can be done using the readEvent function. Also, the numEvents function can be used to know the number of messages received.
After reading the messages, the new state has to be calculated depending on the behavior of the model.
</p>

<h3><font color=\"#008000\">5. Redeclare the output function</font></h3>
<p>
Every time an internal transition has to be performed, an output can be sent from the DEVS model.
No output is sent at external transitions.
The DEVS model generates its output using the output fuction and the current state of the model, before executing the internal transition.
</p>
<p>
The generation of outputs can be configured in the DEVS model including the required code in the <em>out</em> function inside the duplicated atomicDraft package.
The inputs of this function are the current state, the arrays of queues that represent the output ports and the number of queues in that array.
Depending on the model state, an event can be generated using the protected variable <em>y</em> and sending it to the desired queue using the function <em>sendEvent</em> (i.e. sendEvent(queue[i],y);).
Several messages can be sent using multiple calls to the sendEvent function.
</p>

<h3><font color=\"#008000\">6. Redeclare the internal transition function</font></h3>
<p>
In an internal transition, the model state has to be updated after generating the output.
The internal transition function is used to update the state.
</p>
<p>
This function simply receives the current state of the model and returns the updated state, depending on the behavior of the model.
</p>
<p>
The implementation of the internal transition function is contained in the <em>int</em> function inside the atomicDraft package.
This function has to be changed to represent the model behavior at internal transitions.
</p>

<h3><font color=\"#008000\">7. Redeclare the confluent transition function</font></h3>
<p>
In Parallel DEVS a confluent transition is executed when an external transition and internal transition occur at the same instant and have to be simultaneously executed.
The confluent transition function is used to update the state of the model in a confluent transition.
This function is usually a combination of the internal and external transition functions, but it is not neccesary to use them to update the state.
</p>
<p>
The confluent transition function is implemented in the <em>con</em> function inside the atomicDraft package.
This function receives as inputs the current state, the elapsed time and the bag of received events.
Its output corresponds to the updated state of the model.<br>
Since the confluent transition involves an internal transition, the output function will be executed before the confluent trasition function.
</p>
<p>
The default behavior for this function is to generate an output and execute the internal transition before executing the external transition.
This behavior can be changed as required by the behavior of the model.
</p>
</html>"));

  end NewAtomic;

  class NewCoupled "New Coupled DEVS models"

  annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Development of new Coupled DEVS models using DEVSLib</font></h3>
<p>
A coupled model is a DEVS model composed by a combination of atomic, coupled or both models.
Like other DEVS models, a coupled model has input and output ports, that have to be connected to the internal components of the coupled model.
Each input port will be connected to an input port of a component.
Each ouptut port will be connected to one or more output ports of one or several components.
</p>
<p>
The following images show examples of DEVS coupled models.
</p>
<br>
<img src=\"..\Figs\\CoupledDEVS.png\"><img src=\"..\Figs\\devs-coupled.png\">


<h3><font color=\"#008000\">Coupled models with DEVSLib</font></h3>
<p>
Coupled DEVS models can be developed easyly using DEVSlib.
The library includes the coupledDraft model that contains the basic structure for a coupled model.
The steps to build a new coupled model are:
<ol>
<li> Duplicate and rename the coupledDraft model to obtain the basic structure for the new model.
<li> Copy/Paste or delete the input and output ports of the model, to define the required amount of input and output ports in the model.
<li> Include the components of the coupled model, using drag and drop from the Package Browser.
<li> Connect the components between them and with the input and output ports as required by the models behavior.
</ol>
</p>
<p>
The following figure shows a simple coupled DEVS model (corresponds to the <a href=\"Modelica://DESLib.DEVSLib.Examples.SimpleModels.NetSwitch\">NetSwitch</a>
model of the library examples).<br>

<img src=\"..\Figs\\coupled-2.png\">
</p>

<h3><font color=\"#008000\">Duplicating Messages</font></h3>
<p>
Due to the design (queues that receive messages) and implementation (external C structures dynamically managed) of the communication
mechanism between models in the DEVSLib package, it is not possible to connect the same output port of a component to several input ports
of other components.
</p>
<p>
However, to provide a solution to this restriction, a DEVS model that provides the duplication of messages has been included in the library.
This model is included in the AuxModels package, and it is called DUP.
The DUP model sends a copy of each received message through its two output ports.
If more that two copies of the same message are needed, several DUP models can be nested in multiple levels.
(to simplify this procedure a DUP3 model with three output ports has been included in the Auxmodels package).
</p>
<p>
An example of DUP usage is provided in the <a href=\"Modelica://DESLib.DEVSLib.Examples.SimpleModels.testDup\">testDup</a> model,
 and is shown in the next figure.
</p>
<br>
<img src=\"..\Figs\\dup-2.png\">

<h3><font color=\"#008000\">Algebraic Loops</font></h3>
<p>
During the construction of coupled models algebraic loops can be created.
This problem can be avoided in two ways:
<ul>
<li> Aggregating the behavior of all the models in the loop into one single atomic model.
<li> Using the BreakLoop model included in the AuxModels package of the library.
</ul>
</p>
<p>
The BreakLoop model breaks the loop by including a \"pre\" statement in the detection of the incomming messages.
</p>
<p>
An example of BreakLoop usage is provided in the <a href=\"Modelica://DESLib.DEVSLib.Examples.SimpleModels.testSwitch2\">testSwitch2</a> model, and is shown in the next figure.
</p>
<br>
<img src=\"..\Figs\\loop.png\">


</html>"));

  end NewCoupled;

  class Hybrid "Hybrid models development"

  annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Hybrid system modeling using DEVSLib</font></h3>
<p>
Modelica is mainly designed for modeling and simulation of continuous-time systems.
There are a lot of available Modelica components for developing continuous-time models in many different domains (electrical, mechanical, thermodynamical,...).
</p>
<p>
Using the DEVSlib implementation of the Parallel DEVS formalism and the Modelica capabilities for modeling continuous-time systems, it is possible to model and simulate hybrid systems.
These kind of hybrid systems usually contain two parts, one continuous and another discrete, that must communicate somehow.
This communication has been implemented in DEVSLib using the following interface models.
</p>

<h3><font color=\"#008000\">1. Continuous to Discrete</font></h3>
<p>
The communication from the continuous part to the discrete part is performed using three models: the Quantifier, the CrossUP and the CrossDOWN.
<ul>
<li> <em>Quantifier</em>. This model behaves as a first order quantifier of the continuous-time signal.
     Every time the value of the signal changes a predefined value (in either positive or negative direction)
     the model sends an output message with value equal to the new value of the signal.
<li> <em>CrossUP</em>. This model is used to detect the signal crossing, in the upwards direction, a predefined value (i.e. zero-cross).
     Every time the value of the signal crosses the defined value, a message is sent with value equal to the signal value.
     <li> <em>CrossDOWN</em>. This model behaves exactly like the CrossUP model, but detects crossings in the downwards direction.
</ul>
</p>

<p>
An example of this kind of communication is shown in the next figure.
</p>
<br>
<img src=\"..\Figs\\codi.png\">



<h3><font color=\"#008000\">1. Discrete to Continuous</font></h3>
<p>
The communication from the discrete part to the continuous part is performed using the DICO model.
This model translates the value of the received messages from the discrete part into a continuous (Real) signal.
The generated continuous signal is a constant piecewise signal, since the changes in the value of the signal are generated by the arrived messages.
</p>
<p>
An example of this kind of communication is shown in the next figure.
</p>
<br>
<img src=\"..\Figs\\dico-2.png\">


</html>"));

  end Hybrid;

  end UsersGuide;

    package atomicDraft "Draft package to construct new atomic DEVS models"

    annotation(preferedView="info",
      Documentation(info="<HTML>
<p>
This package contains the basic structures needed in atomic models.
It can be duplicated, renamed and modified to construct new atomic DEVS models.
</p>
<p>
The user can implement the behavior of a new DEVS model modifying the con, int, ext, out, and ta functions.<br>
The state can be defined using the st record, and initialized using the initst function.
</p>


</HTML>
"));
      import DESLib.DEVSLib.SRC.*;
      model atomic "Draft model to construct new atomic DEVS models"
                 extends AtomicDEVS(numIn=1,numOut=1,
        redeclare record State = st);
        redeclare function Fcon = con;
        redeclare function Fext = ext;
        redeclare function Fint = int;
        redeclare function Fout = out;
        redeclare function Fta = ta;
        redeclare function initState = initst;
        Interfaces.outPort outPort1     annotation (extent=[96,-10;116,10]);
        Interfaces.inPort inPort1     annotation (extent=[-116,-10;-96,10]);
      equation
        // INPUT PORTS
        iEvent[1] = inPort1.event; // = 0; if no input ports
        iQueue[1] = inPort1.queue;
        // OUTPUT PORTS
        oEvent[1] = outPort1.event;
        oQueue[1] = outPort1.queue; // = 0; if no output ports
        annotation (Diagram);
      end atomic;

      function con "Confluent Transtition Function"
        input st s;
        input Real e;
        input Integer bag;
        output st sout;
        output st soutput;
      algorithm
        // first internal
        soutput := s;
        sout := ext(int(s),0,bag);
        //----------------
        // first external
        //soutput := ext(s,e,bag);
        //sout := int(soutput);
      end con;

      function int "Internal Transition Function"
        input st s;
        output st sout;
      algorithm
        sout := s;
        sout.sigma := Modelica.Constants.inf;
      end int;

      function ext "External Transition Function"
        input st s;
        input Real e;
        input Integer bag;
        output st sout;
      protected
        Integer numreceived;
        stdEvent x;
      algorithm
        sout := s;
        numreceived := numEvents(bag);
        for i in 1:numreceived loop
          x := getEvent(bag);
        end for;
        sout.sigma := Modelica.Constants.inf;
      end ext;

      function out "Output Function"
        input st s;
        input Integer queue[nports];
        input Integer nports;
        output Integer port[nports];
      protected
        stdEvent y;
      algorithm
        port[1] := 0;
      end out;

      function ta "Time Advance Function"
        input st s;
        output Real sigma;
      algorithm
        sigma := s.sigma;
      end ta;

      record st "state of the model"
        Integer phase;
        Real sigma;
      end st;

      function initst "state initalization function"
        output st out;
      algorithm
        out.phase := 0;
        out.sigma := Modelica.Constants.inf; // first internal event never happens
      end initst;
    end atomicDraft;

    model coupledDraft "Draft model for constructing new coupled DEVS models"
      import DESLib.DEVSLib.SRC.*;
    annotation(preferedView="diagram",
      Documentation(info="<HTML>
<p>
This package contains the basic structures needed in coupled models.
It can be duplicated, renamed and modified to construct new coupled DEVS models.
</p>
<p>
This model includes one input and one output ports. Any other requiered port can be included.<br>
The components of the new coupled model can be included and interconnected using drag and drop.
</p>


</HTML>
"),     Coordsys(extent=[-100,-100; 100,100], scale=0.1),
        Diagram);

      Interfaces.inPort in1
                     annotation (Placement(transformation(extent=[-116,-10;-96,10], rotation=0)));
      Interfaces.outPort out1
                       annotation (Placement(transformation(extent=[96,-10;116,10], rotation=0)));
      annotation (Diagram);
    end coupledDraft;
    annotation (uses(Modelica(version="2.2.1")));

    package AuxModels "Auxiliary models"

      model DICO = SRC.Hybrid.DiCo "Discrete-To-Continuous translation model"
                            annotation(preferedView="icon",
        Documentation(info="<HTML>
<p>
This model can be used to translate series of messages into a piecewise constant continuous-time signal.
</p>
<p>
The value of the output corresponds to the value of the received messages.
</p>


</HTML>
"),     Icon(Rectangle(extent=[-100,80; 100,-80], style(color=3, rgbcolor={0,0,255})),
                Text(
                extent=[-60,20; 60,-20],
                style(color=3, rgbcolor={0,0,255}),
                string="DICO")));

      model DiBO = SRC.Hybrid.DiBo "Discrete-To-Boolean Translation Model"
        annotation (preferedView="icon",
        Documentation(info="<HTML>
<p>
This model translates messages into a boolean signal.
</p>
<p>
The value of the output signal is true if the value of the received message is greater than 0, or false otherwise.
</p>
</HTML>
"),       Icon(Rectangle(extent=[-100,80; 100,-80], style(color=3, rgbcolor=
                     {0,0,255})), Text(
                extent=[-60,20; 60,-20],
                style(color=3, rgbcolor={0,0,255}),
                string="DIBO")));

      model Quantizer = SRC.Hybrid.Quantizer "First Order Quantization model"
      annotation (preferedView="icon",
        Documentation(info="<HTML>
<p>
This model implements a first order quantization method, translating the real input into series of messages with value equal to the input signal.
</p>
<p>
A message is generated every time the input signal changes its value in a quantum <i>q</i>.
The <i>Threshold</i> parameter is used to define the accuracy in the detection of the signal variation.
The <i>EType</i> parameter defines the value for the type variable of the generated message.
</p>
</HTML>
"));
      model QuantizerUP=SRC.Hybrid.QuantizerUP "First Order Quantization model"
      annotation (preferedView="icon",
        Documentation(info="<HTML>
<p>
This model implements a first order quantization method, translating the real input into series of messages with value equal to the input signal.
</p>
<p>
A message is generated every time the input signal changes its value in a quantum <i>q</i>.
The <i>Threshold</i> parameter is used to define the accuracy in the detection of the signal variation.
The <i>EType</i> parameter defines the value for the type variable of the generated message.
</p>
</HTML>
"));
      model QuantizerDOWN =
                        SRC.Hybrid.QuantizerDOWN
        "First Order Quantization model"
      annotation (preferedView="icon",
        Documentation(info="<HTML>
<p>
This model implements a first order quantization method, translating the real input into series of messages with value equal to the input signal.
</p>
<p>
A message is generated every time the input signal changes its value in a quantum <i>q</i>.
The <i>Threshold</i> parameter is used to define the accuracy in the detection of the signal variation.
The <i>EType</i> parameter defines the value for the type variable of the generated message.
</p>
</HTML>
"));
      model CrossDOWN = SRC.Hybrid.CrossDOWN
        "Downwards cross-condition message generator"
      annotation (preferedView="icon",
        Documentation(info="<HTML>
<p>
This model translates a real signal into messages when the signal crosses a predefined <i>Value</i> downwards.
The generated message has a value equal to the signal and a type equal to the <i>EType</i> parameter.
</p>
</HTML>
"));
      model CrossUP = SRC.Hybrid.CrossUP
        "Upwards cross-condition message generator"
      annotation (preferedView="icon",
        Documentation(info="<HTML>
<p>
This model translates a real signal into messages when the signal crosses a predefined <i>Value</i> upwards.
The generated message has a value equal to the signal and a type equal to the <i>EType</i> parameter.
</p>
</HTML>
"));
      model DUP = SRC.DupEvent.dupevent "Message duplication model (2 outputs)"
      annotation (preferedView="icon",
        Documentation(info="<HTML>
<p>
This model duplicates and forwards each received message.
Each received message is stored in an internal queue and sent through the two output ports of the model.
</p>
<p>
This model is necessary since an output port can not be connected to several input ports,
because the output port can not store the references to the queues for incoming messages in the input ports.
</p>
</HTML>
"));
      model DUP3 "Message duplication model (3 outputs)"
      annotation (preferedView="icon",
        Documentation(info="<HTML>
<p>
This model duplicates and forwards each received message.
Each received message is stored in an internal queue and sent through the three output ports of the model.
</p>
<p>
This model is necessary since an output port can not be connected to several input ports,
because the output port can not store the references to the queues for incoming messages in the input ports.
</p>
</HTML>
"));

        SRC.Interfaces.inPort IN annotation (extent=[-60,-10;-40,10]);
        SRC.Interfaces.outPort out1 annotation (extent=[4,30;24,50]);
        SRC.Interfaces.outPort out2 annotation (extent=[4,-10;24,10]);
        SRC.Interfaces.outPort out3 annotation (extent=[4,-50;24,-30]);
         annotation (Diagram, Icon(
                    Line(points=[-44,0; 8,0; 6,0], style(
                    color=0,
                    rgbcolor={0,0,0},
                    thickness=4)),
                    Line(points=[8,40; -16,40; -16,-40; 8,-40], style(
                    color=0,
                    rgbcolor={0,0,0},
                    thickness=4))));

        DUP dUP annotation (extent=[-40,0;-20,20]);
        DUP dUP1 annotation (extent=[-20,-20;0,0]);
      equation
        connect(dUP.out1, out1) annotation (points=[-29.4,12; -8.5,12; -8.5,40; 14,40], style(color=0));
        connect(dUP1.out1, out2) annotation (points=[-9.4,-8; 2.5,-8; 2.5,0; 14,0], style(color=0));
        connect(dUP1.out2, out3) annotation (points=[-9.4,-12; 3.5,-12;3.5,-40;14,-40], style(color=0));
        connect(IN, dUP.in1) annotation (points=[-50,0; -44,0; -44,10; -32.6,10], style(color=0));
        connect(dUP.out2, dUP1.in1) annotation (points=[-29.4,8; -29.4,-2.5;-12.6,-2.5;-12.6,-10],style( color=0));
      end DUP3;

      model DUP4
        annotation (Diagram, Icon(
              Line(
                points=[-34,0;-6,0;-6,0], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=4)),
              Line(
                points=[18,60;-6,60;-6,-20;18,-20], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=4)),
              Line(
                points=[18,20;-6,20;-6,-60;18,-60], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=4))));
        DUP dUP annotation (extent=[-30,-10;-10,10]);
        DUP dUP1 annotation (extent=[-10,-50;10,-30]);
        SRC.Interfaces.inPort IN annotation (extent=[-50,-10;-30,10]);
        SRC.Interfaces.outPort out1 annotation (extent=[14,50;34,70]);
        SRC.Interfaces.outPort out2 annotation (extent=[14,10;34,30]);
        SRC.Interfaces.outPort out3 annotation (extent=[14,-30;34,-10]);
        DUP dUP2 annotation (extent=[-10,30;10,50]);
        SRC.Interfaces.outPort out4 annotation (extent=[14,-70;34,-50]);
      equation
        connect(IN,dUP. in1) annotation (points=[-40,0;-40,0;-22.6,0], style( color=0));
        connect(dUP.out1, dUP2.in1) annotation (
            points=[-19.4,2;-10,2;-10,40;-2.6,40],style(
            color=0,
            smooth=Smooth.None));
        connect(dUP.out2, dUP1.in1) annotation (
            points=[-19.4,-2;-10,-2;-10,-40;-2.6,-40],style(
            color=0,
            smooth=Smooth.None));
        connect(dUP2.out1, out1) annotation (
            points=[0.6,42; 10.3,42; 10.3,60; 24,60],
                                                   style(
            color=0,
            smooth=Smooth.None));
        connect(dUP2.out2, out2) annotation (
            points=[0.6,38; 10,38; 10,20; 24,20],
                                              style(
            color=0,
            smooth=Smooth.None));
        connect(dUP1.out1, out3) annotation (
            points=[0.6,-38; 12,-38; 12,-20; 24,-20],
                                                  style(
            color=0,
            smooth=Smooth.None));
        connect(dUP1.out2, out4) annotation (
            points=[0.6,-42; 12,-42; 12,-60; 24,-60],
                                                  style(
            color=0,
            smooth=Smooth.None));
      end DUP4;

      model DUP_N
        parameter Integer N = 2 "Number of outputs (>= 2)";
        annotation (Diagram, Icon(
              Line(
                points=[-34,0;-6,0;-6,0], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=4)),
              Line(
                points=[18,60; -6,60; -6,-20; -6,-20],
                                                    style(
                color=0,
                rgbcolor={0,0,0},
                thickness=4)),
              Line(
                points=[18,20;-6,20;-6,-60;18,-60], style(
                color=0,
                rgbcolor={0,0,0},
                thickness=4)),
              Line(
                points=[12,-12; 12,-12; 12,-14; 12,-14],
                                                    style(
                color=0,
                rgbcolor={0,0,0},
                thickness=4)),
              Line(
                points=[12,-22; 12,-24; 12,-24; 12,-24],
                                                    style(
                color=0,
                rgbcolor={0,0,0},
                thickness=4)),
              Line(
                points=[12,-32; 12,-32; 12,-34; 12,-34],
                                                    style(
                color=0,
                rgbcolor={0,0,0},
                thickness=4)),
              Line(
                points=[12,-42; 12,-42; 12,-44; 12,-44],
                                                    style(
                color=0,
                rgbcolor={0,0,0},
                thickness=4))));
        DUP dup[N-1];
        SRC.Interfaces.inPort IN annotation (extent=[-50,-10;-30,10]);
        SRC.Interfaces.outPort OUT[N] annotation (extent=[36,-10; 56,10]);

      equation
        connect(IN,dup[1].in1);
        for i in 1:N-1 loop
          connect(dup[i].out1, OUT[i]);
          if (i == N-1) then
            connect(dup[i].out2, OUT[i+1]);
          else
            connect(dup[i].out2, dup[i+1].in1);
          end if;
        end for;
      end DUP_N;

      model Select = SRC.Select.select "Output selection model"
      annotation (preferedView="icon",
        Documentation(info="<HTML>
<p>
This model forwards each received message through one of its output ports, depending on the value of the condition.
</p>
<p>
If the condition is true, the message is sent through the port out1, and otherwise it is sent through the port out2.
</p>
</HTML>
"));
      model BreakLoop = SRC.breakloop "Model to break algebraic loops"
      annotation (preferedView="icon",
        Documentation(info="<HTML>
<p>
This model can be used to break algebraic loops in coupled models.
</p>
</HTML>
"));

      model Generator = Examples.SimpleModels.Generator.generator
        "Message generator"
      annotation(preferedView="icon",
        Documentation(info="<HTML>
<p>
The <b>Generator</b> model periodically generates messages and send them through its output port.
</p>
<p>
The first message is generated at time = <i>firstGeneration</i>. The value of the generated message is <i>outValue</i>. Another message is generated every elapsed <i>period</i>.<br>
The <i>numGenerations</i> parameter sets the maximum number of messages to be generated.
</p>


</HTML>
"),       Icon( Rectangle(extent=[-100,60; 80,-60], style(color=3, rgbcolor={0,0,255})),
                Line(points=[-80,-20; 60,-20], style(color=3, rgbcolor={0,0,255})),
                Line(points=[-60,-20; -60,40], style(color=3, rgbcolor={0,0,255})),
                Line(points=[-20,-20; -20,40], style(color=3, rgbcolor={0,0,255})),
                Line(points=[20,-20; 20,40], style(color=3, rgbcolor={0,0,255})),
                Polygon(points=[60,-20; 44,-26; 44,-14; 60,-20], style(
                    color=3,
                    rgbcolor={0,0,255},
                    fillColor=3,
                    rgbfillColor={0,0,255})),
                Text(
                  extent=[-100,100; 80,60],
                  style(
                    color=3,
                    rgbcolor={0,0,255},
                    fillColor=3,
                    rgbfillColor={0,0,255},
                    fillPattern=1),
                  string="%name"),
                Text(
                  extent=[-100,-60; 100,-100],
                  style(
                    color=3,
                    rgbcolor={0,0,255},
                    fillColor=3,
                    rgbfillColor={0,0,255},
                    fillPattern=1),
                  string="{%firstGeneration,%period,%numGenerations,%outValue}")),
                Diagram(
                Rectangle(extent=[-100,60; 80,-60], style(color=3, rgbcolor={0,0,255})),
                Line(points=[-80,-20; 60,-20], style(color=3, rgbcolor={0,0,255})),
                Line(points=[-60,-20; -60,40], style(color=3, rgbcolor={0,0,255})),
                Line(points=[-20,-20; -20,40], style(color=3, rgbcolor={0,0,255})),
                Line(points=[20,-20; 20,40], style(color=3, rgbcolor={0,0,255})),
                Text(
                  extent=[-80,-30; -54,-36],
                  style(color=10, rgbcolor={135,135,135}),
                  string="firstGeneration"),
                Line(points=[-80,-26; -60,-26], style(color=10, rgbcolor={135,135,135})),
                Line(points=[-80,-24; -80,-28], style(color=10, rgbcolor={135,135,135})),
                Line(points=[-60,-24; -60,-28], style(color=10, rgbcolor={135,135,135})),
                Line(points=[-20,-26; -20,-30], style(color=10, rgbcolor={135,135,135})),
                Line(points=[20,-26; 20,-30], style(color=10, rgbcolor={135,135,135})),
                Line(points=[-20,-28; 20,-28], style(color=10, rgbcolor={135,135,135})),
                Text(
                  extent=[-6,-30; 6,-34],
                  style(color=10, rgbcolor={135,135,135}),
                  string="period"),
                Text(
                  extent=[-58,18; -40,14],
                  style(color=10, rgbcolor={135,135,135}),
                  string="outValue"),
                Text(
                  extent=[-36,-50; -6,-56],
                  style(color=10, rgbcolor={135,135,135}),
                  string="numGenerations"),
                Line(points=[-60,-40; -60,-46; 20,-46; 20,-40], style(color=10,
                      rgbcolor={135,135,135})),
                Line(points=[-20,-40; -20,-46], style(color=10, rgbcolor={135,135,135}))));

      model Display = Examples.SimpleModels.Display.display "Message display"
      annotation(preferedView="icon",
        Documentation(info="<HTML>
 <p>
The <b>Display</b> model can be used to receive and show messages generated by other models.
</p>


</HTML>
"),       Icon( Rectangle(extent=[-80,78; 100,-82], style(color=3, rgbcolor={0,0,255})),
                Line(points=[-54,-22; 74,-22], style(color=3, rgbcolor={0,0,255})),
                Line(points=[-34,-22; -34,38], style(color=3, rgbcolor={0,0,255})),
                Line(points=[6,-22; 6,38], style(color=3, rgbcolor={0,0,255})),
                Line(points=[46,-22; 46,38], style(color=3, rgbcolor={0,0,255})),
                Polygon(points=[-46,-22; -62,-28; -62,-16; -46,-22], style(
                    color=3,
                    rgbcolor={0,0,255},
                    fillColor=3,
                    rgbfillColor={0,0,255}))));
      model QSS1 = SRC.QSS.IntegratorQSS1.integratorQSS1
        "First order integrator using state quantization"
      annotation(preferedView="icon",
        Documentation(info="<HTML>
 <p>
This model represents a first order Quantized State System integrator.
</p>
<p>
As inputs, it has to receive messages with type == 1 and value the value of the signal to integrate.<br>
It generates an output message with type == 1 and the value of the integrated signal.
</p>
<p>
The <i>quantum</i> parameter represents the threshold to detect changes in the input signal.<br>
The <i>startX</i> parameter represents a guess for the initial value of the input signal.
</p>

</HTML>
"),   Icon(     Rectangle(extent=[-80,80; 80,-80],  style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-62,62; 60,-60], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-60,60; -40,60; -40,40; -20,40; -20,20; 0,20; 0,0; 20,
                      0; 20,-20; 40,-20; 40,-40; 60,-40; 60,-60],     style(color=10,
                          rgbcolor={135,135,135})),
                    Text(
                      extent=[0,14; 6,6],
                      style(color=10, rgbcolor={135,135,135}),
                      string="q"),
                Text(
                  extent=[-26,72; 80,40],
                  style(color=3, rgbcolor={0,0,255}),
                  string="QSS1")));
      model QSS2 = SRC.QSS.IntegratorQSS2.integratorQSS2
        "Second order integrator using state quantization"
      annotation(preferedView="icon",
        Documentation(info="<HTML>
 <p>
This model represents a second order Quantized State System integrator.
</p>
<p>
As inputs, it has to receive two messages: one with type == 1 and as value the value of the signal to integrate  and; one with type == 2 and value equal to the first derivative of the signal.<br>
It generates two output messages: one with type == 1 and the value of the integrated signal and; one with type == 2 and the value of the first derivative of the integrated signal.
</p>
<p>
The <i>quantum</i> parameter represents the threshold to detect changes in the input signal.<br>
The <i>startX</i> parameter represents a guess for the initial value of the input signal.
</p>

</HTML>
"),       Icon( Rectangle(extent=[-80,80; 80,-80],  style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-62,62; 60,-60], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-60,60; -40,60; -40,40; -20,40; -20,20; 0,20; 0,0; 20,
                      0; 20,-20; 40,-20; 40,-40; 60,-40; 60,-60],     style(color=10,
                          rgbcolor={135,135,135})),
                    Text(
                      extent=[0,14; 6,6],
                      style(color=10, rgbcolor={135,135,135}),
                      string="q"),
                Text(
                  extent=[-26,72; 80,40],
                  style(color=3, rgbcolor={0,0,255}),
                  string="QSS2")));
      model QSS3 = SRC.QSS.IntegratorQSS3.integratorQSS3
        "Third order integrator using state quantization"
      annotation(preferedView="icon",
        Documentation(info="<HTML>
 <p>
This model represents a third order Quantized State System integrator.
</p>
<p>
As inputs, it has to receive three messages: one with type == 1 and as value the value of the signal to integrate; one with type == 2 and value equal to the first derivative of the signal and; the last with type == 3 and the value of the second derivative of the signal.<br>
It generates three output messages: one with type == 1 and the value of the integrated signal; one with type == 2 and the value of the first derivative of the integrated signal and; one with type == 3 and the value of the second derivative of the integrated signal..
</p>
<p>
The <i>quantum</i> parameter represents the threshold to detect changes in the input signal.<br>
The <i>startX</i> parameter represents a guess for the initial value of the input signal.
</p>

</HTML>
"),       Icon( Rectangle(extent=[-80,80; 80,-80],  style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-62,62; 60,-60], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-60,60; -40,60; -40,40; -20,40; -20,20; 0,20; 0,0; 20,
                      0; 20,-20; 40,-20; 40,-40; 60,-40; 60,-60],     style(color=10,
                          rgbcolor={135,135,135})),
                    Text(
                      extent=[0,14; 6,6],
                      style(color=10, rgbcolor={135,135,135}),
                      string="q"),
                Text(
                  extent=[-26,72; 80,40],
                  style(color=3, rgbcolor={0,0,255}),
                  string="QSS3")));

    end AuxModels;

    package Examples "Several examples"

    annotation(preferedView="info",
      Documentation(info="<HTML>
<p>
Several examples have been implemented to show the modeling functionalities of the package and to facilitate its use.
</p>


</HTML>
"));
      package SimpleModels "Simple DEVS models"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package contains several simple DEVS models, atomic and coupled.
</p>


</HTML>
"));

          package Generator "Message generator"

          annotation(preferedView="info",
            Documentation(info="<HTML>
<p>
The <b>Generator</b> model periodically generates messages and send them through its output port.
</p>
<p>
The first message is generated at time = <i>firstGeneration</i>. The value of the generated message is <i>outValue</i>. Another message is generated every elapsed <i>period</i>.<br>
The <i>numGenerations</i> parameter sets the maximum number of messages to be generated.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
            model generator

              parameter Real period = 1;
              parameter Real firstGeneration = 1;
              parameter Integer numGenerations = 0 "0 == infinity";
              parameter Real outValue = 1;
              extends AtomicDEVS(redeclare record State = st);
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(p=period,fg=firstGeneration,val=outValue,ng=numGenerations);
              annotation (Diagram(graphics));
              Interfaces.outPort outPort1
                               annotation (extent=[80,-10;100,10]);
            equation
              for i in 1:numIn loop
                iEvent[i] = 0;
              end for;
              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;

            end generator;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              sout.gen := sout.gen +1;
              if (sout.gen < s.ng) or (s.ng == 0) then
                sout.sigma := s.p; // period
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
              sout.p := s.p;
              sout.val := s.val;
              sout.port := 1;
            end int;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent Y;
            algorithm
              Y.Value := s.val;
              Y.Type := integer(s.val);
              sendEvent(queue[1],Y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase; // 1 = passive, 2 = active
              Real sigma;
              Real p;
              Real fg;
              Real val;
              Integer port;
              Integer ng;
              Integer gen;
            end st;

            function initst
              input Real p;
              input Real fg;
              input Real val;
              input Integer ng;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := fg;
              out.p := p;
              out.val := val;
              out.port := 1;
              out.ng := ng;
              out.gen := 0;
            end initst;
          end Generator;

          package GeneratorRamp "Message generator (switched values)"

          annotation(preferedView="info",
            Documentation(info="<HTML>
 <p>
The <b>GeneratorRamp</b> model periodically generates messages and send them through its output port.
This model behaves similarly to the Generator model.
</p>
<p>
The first message is generated at time = 1. Another message is generated every elapsed <i>period</i>.<br>
The value of the generated message switches between <i>outValue</i> and 0, in each generation.<br>
The first generation has a value of 0.
</p>

</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
            model generatorramp

              parameter Real period = 1;
              parameter Real outValue = 1;
                       extends AtomicDEVS(
              redeclare record State = st);
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(p=period,val=outValue);

              Interfaces.outPort outPort1
                               annotation (extent=[80,-10;100,10]);
            equation
              for i in 1:numIn loop
                iEvent[i] = 0;
              end for;
              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;

            end generatorramp;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              if s.phase == 1 then
                sout.phase := 2;
                sout.sigma := s.p; // period
              else
                sout.phase := 1;
                sout.sigma := s.p; // period
              end if;
              sout.p := s.p;
              sout.val := s.val;
              sout.port := 1;
            end int;

            function out "Output Function"
             input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent Y;
            algorithm
              if s.phase == 2 then
                Y.Value := s.val;
                Y.Type := integer(s.val);
                sendEvent(queue[1],Y);
              else
                Y.Value := 0;
                Y.Type := 0;
                sendEvent(queue[1],Y);
              end if;
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase; // 1 = passive, 2 = active
              Real sigma;
              Real p;
              Real val;
              Integer port;
            end st;

            function initst
              input Real p;
              input Real val;
              output st out;
            algorithm
              out.phase := 1; // passive
              out.sigma := 0;
              out.p := p;
              out.val := val;
              out.port := 1;
            end initst;
          end GeneratorRamp;

        package Display "Message display"

        annotation(preferedView="info",
          Documentation(info="<HTML>
 <p>
The <b>Display</b> model can be used to receive and show messages generated by other models.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model display
          parameter Real displayTime = 0.0001;
                   extends AtomicDEVS(
          redeclare record State = st);
          redeclare function Fint = int;
          redeclare function Fext = ext;
          redeclare function Fout = out;
          redeclare function Fta = ta;
          redeclare function initState = initst(dt=displayTime);
            annotation (Diagram,Icon);
            Interfaces.inPort inPort1 annotation (extent=[-100,-10;-80,10]);
          equation
              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;
              oQueue[1] = 0;

          end display;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.phase := 0;
            sout.sigma := Modelica.Constants.inf;
            sout.job := 0;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              //Modelica.Utilities.Streams.print("* EVENT received Type "+String(x.Type)+" ,Value "+String(x.Value));
              sout.value := x.Value;
              sout.received := sout.received +1;
            end for;
            sout.phase := 1;
            sout.sigma := s.dt; // processing_time
            sout.job := numreceived;
            sout.dt := s.dt;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
              sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real job;
            Real value; // value of the received message
            Real dt; // processing time;
            Integer received;
          end st;

          function initst
            input Real dt;
            output st out;
          algorithm
            out.phase := 0; // passive
            out.sigma := Modelica.Constants.inf;  //first generation starts at time == 1
            out.job := 0;
            out.value := 0;
            out.dt := dt;
            out.received := 0;
          end initst;
        end Display;

        package Processor "Processing delay"

        annotation(preferedView="info",
          Documentation(info="<HTML>
 <p>
The <b>Processor</b> model represents a processing delay for the received messages.
</p>
 <p>
Each received message is sent through the output port after a delay equivalent to <i>processTime</i>.
If a new message is received before ending the current process, it is rejected and the process continues.
</p>

</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model processor
          parameter Real processTime = 1;
                   extends AtomicDEVS(
          redeclare record State = st);
          redeclare function Fcon = con;
          redeclare function Fint = int;
          redeclare function Fout = out;
          redeclare function Fext = ext;
          redeclare function Fta = ta;
          redeclare function initState = initst(dt=processTime);
            annotation (Diagram);
              Interfaces.outPort outPort1
                               annotation (extent=[80, -10;100,10]);
              Interfaces.inPort inPort1
                             annotation (extent=[-100,-10;-80,10]);
          equation
              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;
              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
          end processor;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),0,bag);
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.phase := 1;
            sout.sigma := Modelica.Constants.inf; // processing_time;
            sout.job := 0;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            if s.phase == 1 then
              for i in 1:numreceived loop
                x := DEVSLib.SRC.getEvent(bag);
                if i == 1 then
                  sout.job := x.Value;
                  //Modelica.Utilities.Streams.print("* EVENT received Type "+String(x.Type)+" ,Value "+String(x.Value));
                else
                  //Modelica.Utilities.Streams.print("* EVENT BALKED, still processing!");
                end if;
                sout.received := sout.received +1;
              end for;
              sout.phase := 2;
              sout.sigma := s.dt; // processing_time
            else
              sout.sigma := s.sigma -e;
            end if;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            if s.phase == 2 then
              y.Type := 1;
              y.Value := s.job;
              sendEvent(queue[1],y);
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real job;
            Real dt; // processing time;
            Integer received;
          end st;

          function initst
            input Real dt;
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;
            out.job := 0;
            out.dt := dt;
            out.received := 0;
          end initst;
        end Processor;

        package ProcessorTwice "Processing delay"

        annotation(preferedView="info",
          Documentation(info="<HTML>
 <p>
The <b>Processor</b> model represents a processing delay for the received messages.
</p>
 <p>
Each received message is sent through the output port after a delay equivalent to <i>processTime</i>.
If a new message is received before ending the current process, it is rejected and the process continues.
</p>

</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model processortwice
          parameter Real processTime = 1;
                   extends AtomicDEVS(
          redeclare record State = st);
          redeclare function Fcon = con;
          redeclare function Fint = int;
          redeclare function Fout = out;
          redeclare function Fext = ext;
          redeclare function Fta = ta;
          redeclare function initState = initst(dt=processTime);
            annotation (Diagram);
              Interfaces.outPort outPort1
                               annotation (extent=[80,-10;100,10]);
              Interfaces.inPort inPort1
                             annotation (extent=[-100,-10;-80,10]);
          equation
              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;
              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
          end processortwice;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),0,bag);
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            if sout.phase == 2 then
              sout.phase := 3;
              sout.sigma := sout.dt;
              Modelica.Utilities.Streams.print("PRIMERO");
            else
              sout.phase := 1;
              sout.sigma := Modelica.Constants.inf; // processing_time;
              sout.job := 0;
              Modelica.Utilities.Streams.print("SEGUNDO");
            end if;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            if s.phase == 1 then
              for i in 1:numreceived loop
                x := DEVSLib.SRC.getEvent(bag);
                if i == 1 then
                  sout.job := x.Value;
                  //Modelica.Utilities.Streams.print("* EVENT received Type "+String(x.Type)+" ,Value "+String(x.Value));
                else
                  //Modelica.Utilities.Streams.print("* EVENT BALKED, still processing!");
                end if;
                sout.received := sout.received +1;
              end for;
              sout.phase := 2;
              sout.sigma := s.dt; // processing_time
            else
              sout.sigma := s.sigma -e;
            end if;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            if s.phase == 3 then
              y.Type := 1;
              y.Value := s.job;
              sendEvent(queue[1],y);
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real job;
            Real dt; // processing time;
            Integer received;
          end st;

          function initst
            input Real dt;
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;
            out.job := 0;
            out.dt := dt;
            out.received := 0;
          end initst;
        end ProcessorTwice;

        package Switch "Message forwarding though switched ports"

        annotation(preferedView="info",
          Documentation(info="<HTML>
 <p>
The <b>Switch</b> model switches the receiving and sending ports for messages.
</p>
 <p>
A message received in the input port <i>in1</i> is sent through the output port <i>out2</i>, and viceversa.
</p>

</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model switch
                   extends AtomicDEVS(numIn=2,numOut=2,
          redeclare record State = st);
          redeclare function Fcon = con;
          redeclare function Fint = int;
          redeclare function Fext = ext;
          redeclare function Fout = out;
          redeclare function Fta = ta;
          redeclare function initState = initst;
            annotation (Diagram(graphics));
              Interfaces.inPort in1
                         annotation (extent=[-100,30;-80,50]);
              Interfaces.inPort in2
                             annotation (extent=[-100,-50;-80,-30]);
              Interfaces.outPort out1
                           annotation (extent=[80,30;100,50]);
              Interfaces.outPort out2
                           annotation (extent=[80,-50;100,-30]);
          equation
              iEvent[1] = in1.event;
              iQueue[1] = in1.queue;
              iEvent[2] = in2.event;
              iQueue[2] = in2.queue;

              oEvent[1] = out1.event;
              oQueue[1] = out1.queue;
              oEvent[2] = out2.event;
              oQueue[2] = out2.queue;
          end switch;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),0,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.phase := 1;
            sout.sigma := Modelica.Constants.inf; // processing_time;
            sout.er1 := 0;
            sout.er2 := 0;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              if x.Port == 1 then
                sendEvent(sout.q1,x);
                sout.er1 := sout.er1 +1;
              else
                sendEvent(sout.q2,x);
                sout.er2 := sout.er2 +1;
              end if;
              //Modelica.Utilities.Streams.print("* EVENT received Type "+String(x.Type)+" ,Value "+String(x.Value));
            end for;
            sout.received := sout.received + numreceived;
            sout.phase := 2;
            sout.sigma := 0; // processing_time
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            if s.er1 > 0 then
              for i in 1:s.er1 loop
                y := getEvent(s.q1);
                sendEvent(queue[2],y);
              end for;
            end if;
            if s.er2 > 0 then
              for i in 1:s.er2 loop
                y := getEvent(s.q2);
                sendEvent(queue[1],y);
              end for;
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Integer q1;
            Integer q2;
            Integer er1;
            Integer er2;
            Integer received;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;
            out.er1 := 0;
            out.er2 := 0;
            out.q1 := CreateQueue();
            out.q2 := CreateQueue();
            out.received := 0;
          end initst;
        end Switch;

        package Switch1 "Message forwarding through alternated ports"

        annotation(preferedView="info",
          Documentation(info="<HTML>
 <p>
The <b>Switch1</b> model switches the receiving and sending ports for messages.
</p>
 <p>
As opposed to the Switch model, this model only has one input port.<br>
The received messages are alternatively sent through the output ports 1 and 2.(i.e. the first message is sent through port 1, the second through port 2, the third through port 1 again, and so on).
</p>

</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model switch1
          parameter Boolean active =  true;
                   extends AtomicDEVS(numIn=1,numOut=2,
          redeclare record State = st);
          redeclare function Fcon = con;
          redeclare function Fint = int;
          redeclare function Fout = out;
          redeclare function Fext = ext;
          redeclare function Fta = ta;
          redeclare function initState = initst(a=active);
            annotation (Diagram(graphics));
              Interfaces.inPort inPort1
                             annotation (extent=[-100,-10;-80,10]);
              Interfaces.outPort out1
                           annotation (extent=[80,30;100,50]);
              Interfaces.outPort out2
                           annotation (extent=[80,-50;100,-30]);
          equation
            iEvent[1] = inPort1.event;
            iQueue[1] = inPort1.queue;

            oEvent[1] = out1.event;
            oQueue[1] = out1.queue;
            oEvent[2] = out2.event;
            oQueue[2] = out2.queue;
          end switch1;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.phase := 1;
            sout.sigma := Modelica.Constants.inf; // processing_time;
            if s.nextout == 1 then
              sout.nextout := 2;
            else
              sout.nextout := 1;
            end if;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              if i == 1 then
                sout.ev := x.Value;
                sout.et := x.Type;
                Modelica.Utilities.Streams.print("* EVENT received Type "+String(x.Type)+" ,Value "+String(x.Value));
              else
                Modelica.Utilities.Streams.print("* EVENT BALKED, still processing!");
              end if;
              sout.received := sout.received +1;
            end for;
            sout.phase := 2;
            sout.sigma := 0; // processing_time
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            if s.a then
                y.Type := s.et;
                y.Value := s.nextout; //s.ev;
                sendEvent(queue[s.nextout],y);
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Boolean a;
            Integer et;
            Real ev;
            Integer received;
            Integer nextout;
          end st;

          function initst
            input Boolean a;
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;  //first generation starts at time == 1
            out.a := a;
            out.et := 0;
            out.ev := 0;
            out.received := 0;
            out.nextout := 1;
          end initst;
        end Switch1;

          package Ramp "Slope calculator"

          annotation(preferedView="info",
            Documentation(info="<HTML>
 <p>
The <b>Ramp</b> model stores its input and uses it as the value of the slope of its internal state.
</p>

<p>

</p>

</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
            model ramp
              parameter Real stepTime = 0.1;
                       extends AtomicDEVS(
              redeclare record State = st);
              redeclare function Fcon = con;
              redeclare function Fext = ext;
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(s=stepTime);
              annotation (Diagram(graphics));
              Interfaces.inPort inPort1
                             annotation (extent=[-100,-10;-80,10]);
              Interfaces.outPort outPort1
                               annotation (extent=[80,-10;100,10]);
            equation
              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;

            end ramp;

            function con "Confluent Transtition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            algorithm
              sout := ext(int(s),e,bag);
            end con;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              sout.phase := 2;
              sout.sigma := s.step;
              sout.pos := s.pos + (s.sigma*s.inp);
              sout.inp := s.inp;
            end int;

            function ext "External Transition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
          protected
              Integer numreceived;
              stdEvent x;
            algorithm
              numreceived := numEvents(bag);
              for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              end for;
              sout := s;
              sout.phase := 2;
              sout.sigma := s.sigma - e;
              sout.pos := s.pos + (e*x.Value);
              sout.inp := x.Value;
            end ext;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent y;
            algorithm
              y.Value := s.pos +(s.sigma*s.inp);
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase; // 1 = passive, 2 = active
              Real sigma;
              Real inp;
              Real pos;
              Real step;
            end st;

            function initst
              input Real s;
              output st out;
            algorithm
              out.phase := 1; // passive
              out.sigma := s;
              out.inp := 0;
              out.pos := 0;
              out.step := s;
            end initst;
          end Ramp;

          model Pipe "Pipeline of three serial processes"
          annotation(preferedView="info",
            Documentation(info="<HTML>
 <p>
The <b>Pipe</b> model is a coupled model composed by three processes.
</p>
<p>
Received messages are delayed in each process before leaving the Pipe model.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
            parameter Real p1 = 1;
            parameter Real p2 = 1;
            parameter Real p3 = 1;

            annotation (Diagram);
            Processor.processor processor(processTime=p1)
              annotation (extent=[-60,-10; -40,10]);
            Processor.processor processor1(processTime=p2)
              annotation (extent=[-18,-10;2,10]);
            Processor.processor processor2(processTime=p3)
              annotation (extent=[20,-10;40,10]);
            Interfaces.inPort inPort1
                           annotation (extent=[-100,-10;-80,10]);
            Interfaces.outPort outPort1
                             annotation (extent=[80,-10;100,10]);
          equation
          connect(processor.outPort1, processor1.inPort1)
            annotation (points=[-41,0; -17,0],      style(color=0, rgbcolor={0,0,0}));
          connect(processor1.outPort1, processor2.inPort1)
            annotation (points=[1,0;10,0;21,0], style(color=0,rgbcolor={0,0,0}));
          connect(processor2.outPort1, outPort1)
            annotation (points=[39,0;90,0], style(color=0,rgbcolor={0,0,0}));
          connect(inPort1, processor.inPort1)
            annotation (points=[-90,0; -59,0],style(color=0,rgbcolor={0,0,0}));
          end Pipe;

          model NetSwitch "alternated switch and delay"
          annotation(preferedView="info",
            Documentation(info="<HTML>
 <p>
The <b>NetSwitch</b> model is a coupled model composed by a switch1 and two processes.
</p>
<p>
Received messages are alternatively sent to the processes and delayed before leaving the model.
</p>

</HTML>
"),         Diagram(coordinateSystem(preserveAspectRatio=false, extent={{-100,
                    -100},{100,100}}),
                    graphics));
          import DESLib.DEVSLib.SRC.*;
            parameter Real p1 = 1;
            parameter Real p2 = 1;
            annotation (Diagram);
            Processor.processor Process1( processTime=p1, name="p1")
              annotation (extent=[-2,2;24,24]);
            Processor.processor Process2( processTime=p2, name="p2")
              annotation (extent=[-2,-26;24,-4]);
            Interfaces.inPort inPort1
                           annotation (extent=[-104,-12;-84,8]);
            Interfaces.outPort outPort1
                             annotation (extent=[84,-10;104,10]);
            Switch1.switch1 switch1_1(name="switch")
                                      annotation (extent=[-56,-14;-30,10]);
          equation
            connect(Process1.outPort1, outPort1)
                   annotation (points=[22.7,13; 38,13; 38,0; 94,0],
                                                                 style(color=0,rgbcolor={0,0,0}));
            connect(Process2.outPort1, outPort1)
                   annotation (points=[22.7,-15; 38,-15; 38,0; 94,0],
                                                                   style(color=0,rgbcolor={0,0,0}));
            connect(inPort1, switch1_1.inPort1)
                   annotation (points=[-94,-2;-54.7,-2], style(color=0,rgbcolor={0,0,0}));
            connect(switch1_1.out1, Process1.inPort1)
                   annotation (points=[-31.3,2.8; -16,2.8; -16,13; -0.7,13],
                                                                          style(color=0,rgbcolor={0,0,0}));
            connect(switch1_1.out2, Process2.inPort1)
                   annotation (points=[-31.3,-6.8; -16,-6.8; -16,-15; -0.7,-15],
                                                                              style(color=0,rgbcolor={0,0,0}));
          end NetSwitch;

          model testGenerator
            Real Gen = generator.S.gen;
            Real D = display.S.phase;

            Generator.generator generator(name="G", numGenerations=0, firstGeneration=0)
              annotation (extent=[-80,0;-60,20]);
            Display.display display(name="D") annotation (extent=[0,0;20,20]);
            annotation (Diagram, experiment(StopTime=10), experimentSetupOutput);
          equation
          connect(generator.outPort1, display.inPort1) annotation (points=[-61,10;1,10], style(color=0,rgbcolor={0,0,0}));
          end testGenerator;

          model testNetSwitch
          Real D;

            annotation (Diagram,experiment(StopTime=10),experimentSetupOutput);
            Generator.generator generator(name="gen")
              annotation (extent=[-80,-16;-40,20]);
            Display.display display(name="display")
              annotation (extent=[40,-16;80,20]);
            NetSwitch netSwitch
              annotation (extent=[-24,-16;20,20]);
          equation
          D = display.S.value;
            connect(generator.outPort1, netSwitch.inPort1)
                annotation (points=[-42,2; -32.34,2; -32.34,1.64; -22.68,1.64],
                                                                             style(color=0,rgbcolor={0,0,0}));
            connect(netSwitch.outPort1, display.inPort1)
            annotation (points=[18.68,2;42,2], style(color=0,rgbcolor={0,0,0}));
          end testNetSwitch;

          model testRamp
          Real D;

            GeneratorRamp.generatorramp generatorRamp(
            name="gen",
            period=5,
            outValue=1)
              annotation (extent=[-80,20;-60,40]);
            Display.display display(name="display")
              annotation (extent=[0,20;20,40]);
            Ramp.ramp ramp(name="ramp")
               annotation (extent=[-40,20;-20,40]);
            annotation (Diagram, experiment(StopTime=100),experimentSetupOutput);
          equation
          D = display.S.value;
          connect(generatorRamp.outPort1, ramp.inPort1)
            annotation (points=[-61,30;-39,30], style(color=0,rgbcolor={0,0,0}));
          connect(ramp.outPort1, display.inPort1)
            annotation (points=[-21,30;1,30], style(color=0,rgbcolor={0,0,0}));
          end testRamp;

          model testSwitch
            Real Display1;
            Real Display2;

            Generator.generator generator(firstGeneration=1, name="gen1")
              annotation (extent=[-100,20;-80,40]);
            Display.display display1(name="display1")
              annotation (extent=[60,20;80,40]);
              annotation (Diagram,  experiment(StopTime=100),  experimentSetupOutput);
            Switch.switch switch(name="switch")
              annotation (extent=[0,0;20,20]);
            Generator.generator generator2(firstGeneration=1.5, name="gen2")
              annotation (extent=[-100,-18;-80,2]);
            Display.display display2(name="display2")
               annotation (extent=[60,-20;80,0]);
            Pipe pipe  annotation (extent=[-60,20;-40,40]);
          equation
          Display1 = display1.S.phase;
          Display2 = display2.S.phase;
            connect(generator2.outPort1, switch.in2) annotation (points=[-81,-8;-20.5,-8;-20.5,6;1,6], style(color=0,rgbcolor={0,0,255}));
            connect(switch.out1, display1.inPort1) annotation (points=[19,14;39.5,14;39.5,30;61,30],style(color=0,rgbcolor={0,0,255}));
          connect(switch.out2, display2.inPort1)  annotation (points=[19,6;39.5,6;39.5,-10;61,-10], style(color=0,rgbcolor={0,0,255}));
            connect(pipe.outPort1, switch.in1) annotation (points=[-41,30;-19.5,30;-19.5,14;1,14], style(color=0,rgbcolor={0,0,255}));
          connect(generator.outPort1, pipe.inPort1) annotation (points=[-81,30;-59,30], style(color=0,rgbcolor={0,0,0}));
          end testSwitch;

          model testHybrid
          Real S1;
          Real S2;
          Real outS1;
          Real outS2_1;
          Real outS2_2;

            annotation (Diagram,  experiment(StopTime=50),  experimentSetupOutput);
          AuxModels.DICO dICO1
            annotation (extent=[-44,54;-24,74]);
          GeneratorRamp.generatorramp generatorramp1(name="gramp", outValue=2)
            annotation (extent=[-76,54;-56,74]);
            Display.display display4(name="D2")
            annotation (extent=[-30,-20;-10,0]);
          AuxModels.CrossDOWN crossDOWN2( EType=2, Value=0.001)
            annotation (extent=[46,-32;66,-12]);
          AuxModels.CrossUP crossUP2(Value=0.98)
            annotation (extent=[46,-6;66,14]);
            Display.display display2(name="D2")
            annotation (extent=[74,-6;94,14]);
            Display.display display3(name="D2")
            annotation (extent=[74,-32;94,-12]);
          Modelica.Blocks.Sources.Sine sine(freqHz=0.1)
            annotation (extent=[6,-20;26,0]);
          Modelica.Blocks.Sources.Sine sine1(freqHz=0.1, phase=3.3)
            annotation (extent=[-92,-20;-72,0]);
          AuxModels.Quantizer quantizer(q=0.1)
            annotation (extent=[-60,-20;-40,0]);
          equation
          S1 = sine1.y;
          outS1 = display4.S.value;
          S2 = sine.y;
          outS2_1 = display2.S.phase;
          outS2_2 = display3.S.phase;

          connect(generatorramp1.outPort1, dICO1.inport)
            annotation (points=[-57,64;-44.6,64], style(color=0,rgbcolor={0,0,0}));
          connect(crossDOWN2.outport,display3. inPort1)
            annotation (points=[66.6,-22;75,-22], style(color=0,rgbcolor={0,0,0}));
          connect(crossUP2.outport,display2. inPort1)
            annotation (points=[66.6,4;75,4], style(color=0,rgbcolor={0,0,0}));
          connect(sine.y, crossUP2.u)
            annotation (points=[27,-10;36,-10;36,4;45,4], style(color=0,rgbcolor={0,0,127}));
          connect(sine.y, crossDOWN2.u)
            annotation (points=[27,-10;36,-10;36,-22;45,-22], style(color=0,rgbcolor={0,0,127}));
          connect(sine1.y, quantizer.u)
            annotation (points=[-71,-10;-61,-10], style(color=0,rgbcolor={0,0,127}));
          connect(quantizer.outport, display4.inPort1)
            annotation (points=[-39.4,-10;-29,-10], style(color=0,rgbcolor={0,0,0}));
          end testHybrid;

          model testDup
            Real Gen = generator.S.gen;
            Real D1 = display.S.phase;
            Real D2 = display1.S.phase;
            Real D3 = display2.S.phase;

            annotation (Diagram(coordinateSystem(preserveAspectRatio=false,
                  extent={{-100,-100},{100,100}}),
                                graphics),
            experiment(StopTime=10),
            experimentSetupOutput);
            AuxModels.DUP dupevent(name="dup1")
             annotation (extent=[-44,0;-24,20]);
            Generator.generator generator(name="gen")
             annotation (extent=[-64,0;-44,20]);
            Display.display display(name="D1")
             annotation (extent=[4,22;24,42]);
            Display.display display1(name="D2")
             annotation (extent=[4,0;24,20]);
            AuxModels.DUP dupevent1(name="dup2")
             annotation (extent=[-28,-14;-8,6]);
            Display.display display2(name="D3")
             annotation (extent=[4,-22;24,-2]);
          equation
          connect(generator.outPort1, dupevent.in1) annotation (points=[-45,10;-36.6,10], style(color=0,rgbcolor={0,0,0}));
          connect(dupevent.out1, display.inPort1) annotation (points=[-33.4,12;-14,12;-14,32;5,32], style(color=0,rgbcolor={0,0,0}));
          connect(dupevent.out2, dupevent1.in1) annotation (points=[-33.4,8;-26,8;-26,-4;-20.6,-4], style(color=0,rgbcolor={0,0,0}));
          connect(dupevent1.out1, display1.inPort1) annotation (points=[-17.4,-2;-6,-2;-6,10;5,10], style(color=0,rgbcolor={0,0,0}));
          connect(dupevent1.out2, display2.inPort1) annotation (points=[-17.4,-6;-6,-6;-6,-12;5,-12], style(color=0,rgbcolor={0,0,0}));
          end testDup;

          model testConcurrent
            Real Gen = generator1.S.gen;
            Real D1 = Display1.S.phase;
            Real D2 = Display2.S.phase;

            annotation (Diagram, experiment(StopTime=10), experimentSetupOutput);
            Switch.switch switch(name="Switch")
            annotation (extent=[-20,0;0,20]);
            Display.display Display1(name="D1")
            annotation (extent=[20,20;40,40]);
            Display.display Display2(name="D2")
            annotation (extent=[20,-20;40,0]);
          AuxModels.Generator generator1(name="gen1")
            annotation (extent=[-100,0;-80,20]);
          AuxModels.DUP dUP(name="dup")
            annotation (extent=[-68,0;-48,20]);
          equation
            connect(switch.out1,Display1. inPort1) annotation (points=[-1,14;9.5,14;9.5,30;21,30], style(color=0,rgbcolor={0,0,255}));
            connect(switch.out2,Display2. inPort1) annotation (points=[-1,6;10.5,6;10.5,-10;21,-10], style(color=0,rgbcolor={0,0,255}));
          connect(generator1.outPort1, dUP.in1) annotation (points=[-81,10;-74.5,10;-74.5,10;-60.6,10], style(color=0,rgbcolor={0,0,0}));
          connect(dUP.out1, switch.in1) annotation (points=[-57.4,12;-33.5,12;-33.5,14;-19,14], style(color=0,rgbcolor={0,0,0}));
          connect(dUP.out2, switch.in2) annotation (points=[-57.4,8;-33.5,8;-33.5,6;-19,6], style(color=0,rgbcolor={0,0,0}));
          end testConcurrent;

          model testConfluent
            Real Gen = generator.S.gen;
            Real D = display.S.phase;

            Generator.generator generator(name="G", numGenerations=0, firstGeneration=0)
              annotation (extent=[-80,0;-60,20]);
            Processor.processor processor(name="P1")
              annotation (extent=[-40,0;-20,20]);
            Display.display display(name="D")
              annotation (extent=[0,0; 20,20]);
            annotation (Diagram, experiment(StopTime=10), experimentSetupOutput);
          equation
            connect(processor.outPort1, display.inPort1) annotation (points=[-21,10;
                1,10],                                                                     style(color=0,rgbcolor={0,0,255}));
          connect(generator.outPort1, processor.inPort1) annotation (points=[-61,10;-39,10], style(color=0,rgbcolor={0,0,0}));
          end testConfluent;

          model testConfluentTwice
            Real Gen = generator.S.gen;
            Real D = display.S.phase;

            Generator.generator generator(name="G", numGenerations=0, firstGeneration=0,
            period=5)
              annotation (extent=[-80,0;-60,20]);
            Display.display display(name="D")
              annotation (extent=[0,0;20,20]);
            annotation (Diagram,  experiment(StopTime=10),  experimentSetupOutput);
          ProcessorTwice.processortwice processortwice(name="P", processTime=0)
            annotation (extent=[-40,0;-20,20]);
          equation
          connect(generator.outPort1, processortwice.inPort1) annotation (points=[-61,10;-39,10], style(color=0,rgbcolor={0,0,0}));
          connect(processortwice.outPort1, display.inPort1) annotation (points=[-21,10;1,10], style(color=0,rgbcolor={0,0,0}));
          end testConfluentTwice;

          model testSwitch2
            //Real Display1;
            Real Display2;

            annotation (Diagram,  experiment(StopTime=100),  experimentSetupOutput);
            Switch.switch switch(name="switch")
            annotation (extent=[-26,46; -6,66]);
            Generator.generator gen(
              name="gen2",
              firstGeneration=1,
              numGenerations=3,
              period=1)
              annotation (extent=[-78,4; -58,24]);
            Display.display display(name="display2")
               annotation (extent=[28,-4;48,16]);
          Processor.processor proc(name="proc", processTime=0)
            annotation (extent=[-24,-30; -4,-10],
                                               rotation=180, origin={10,32});
          AuxModels.BreakLoop breakLoop annotation (extent=[6,-10; 26,10],
                rotation=90,
                origin=[24,24]);
          equation
          //Display1 = display1.S.phase;
          Display2 = display.S.phase;
            connect(switch.out2, display.inPort1)
            annotation (points=[-7,52; 24,52; 24,6; 29,6],
                                            style(color=0,rgbcolor={0,0,0}));
            connect(gen.outPort1, switch.in2)
            annotation (points=[-59,14; -44,14; -44,52; -25,52],
                                          style(color=0,rgbcolor={0,0,0}));
            connect(proc.outPort1, switch.in1)
            annotation (points=[-23,-20; -34,-20; -34,60; -25,60],
                                                       style(color=0,rgbcolor={0,0,0}));
          connect(switch.out1, breakLoop.IN)
            annotation (points=[-7,60; 16,60; 16,-2.8],
                                                     style(color=0,rgbcolor={0,0,0}));
          connect(breakLoop.OUT, proc.inPort1)
            annotation (points=[16,0.6; 16,-20; -5,-20],
                                                     style(color=0,rgbcolor={0,0,0}));
          end testSwitch2;

          model testSwitch2Twice
            //Real Display1;
            Real Display2;

            annotation (Diagram,  experiment(StopTime=100),  experimentSetupOutput);
            Switch.switch switch(name="switch")
            annotation (extent=[0,0; 20,20]);
            Generator.generator generator2(                     name="gen2",
            firstGeneration=1,
            numGenerations=3,
            period=1)
            annotation (extent=[-86,-4;-66,16]);
            Display.display display2(name="display2")
            annotation (extent=[36,0;56,20]);
          AuxModels.BreakLoop breakLoop
            annotation (origin=[-12,44],extent=[-10,-20; 10,0],rotation=180);
          ProcessorTwice.processortwice processortwice(processTime=0, name=
                "proc")
            annotation (extent=[-34,16;-14,36]);
          equation
          //Display1 = display1.S.phase;
          Display2 = display2.S.phase;
          connect(generator2.outPort1, switch.in2)
            annotation (points=[-67,6; 1,6],style(color=0,rgbcolor={0,0,0}));
          connect(switch.out2, display2.inPort1)
            annotation (points=[19,6; 28,6; 28,10; 37,10],
                                                        style(color=0,rgbcolor={0,0,0}));
          connect(switch.out1, breakLoop.IN)
            annotation (points=[19,14; 22,14; 22,-10; 2.8,-10],
                                                            style(color=0,rgbcolor={0,0,0}));
          connect(breakLoop.OUT, processortwice.inPort1)
            annotation (points=[-0.6,-10; -42,-10; -42,26; -33,26],
                                                               style(color=0,rgbcolor={0,0,0}));
          connect(processortwice.outPort1, switch.in1)
            annotation (points=[-15,26; -6,26; -6,14; 1,14],
                                                         style(color=0,rgbcolor={0,0,0}));
          end testSwitch2Twice;

          model testSwitch3
            //Real Display1;
            Real Display2;

            annotation (Diagram,  experiment(StopTime=100),  experimentSetupOutput);
            Generator.generator generator2(                     name="gen2",
            numGenerations=5,
            firstGeneration=1)
              annotation (extent=[-90,-10; -70,10]);
            Display.display display2(name="display2")
                                    annotation (extent=[70,-10; 90,10]);
          Processor.processor processor(name="proc", processTime=1)
            annotation (extent=[-50,10; -30,30]);
          Switch1.switch1 switch1_1 annotation (extent=[-10,-10; 10,10]);
          AuxModels.BreakLoop breakLoop
            annotation (origin=[-6,50], extent=[0,26; 20,46],   rotation=180);
          equation
          //Display1 = display1.S.phase;
          Display2 = display2.S.phase;
          connect(generator2.outPort1, switch1_1.inPort1) annotation ( points=[-71,0;
                -9,0],                                                                           style(color=0,rgbcolor={0,0,0}));
          connect(processor.outPort1, switch1_1.inPort1) annotation (points=[-31,20;
                -20,20; -20,0; -9,0],                                                                        style(color=0,rgbcolor={0,0,0}));
          connect(switch1_1.out2, display2.inPort1) annotation (points=[9,-4; 40,
                -4; 40,0; 71,0],                                                                      style(color=0,rgbcolor={0,0,0}));
          connect(switch1_1.out1, breakLoop.IN) annotation (points=[9,4; 26,4;
                26,36; 12.8,36],                                                                style(color=0,rgbcolor={0,0,0}));
          connect(breakLoop.OUT, processor.inPort1) annotation (points=[9.4,36;
                -58,36; -58,20; -49,20],                                                               style(color=0,rgbcolor={0,0,0}));
          end testSwitch3;

        package SetValue "Sends a message with the desired value"
          import DESLib.DEVSLib.SRC.*;

        annotation(preferedView="documentation",
          Documentation(info="<HTML>
 <p>
This model generates a message with value and type equal to the <i>Value</i> parameter every time a new input message is received.
</p>

</HTML>
"));
          model setValue
          replaceable Real Value = 1;
                   extends AtomicDEVS(
          redeclare record State = st);
          redeclare function Fcon = con(v=Value);
          redeclare function Fint = int;
          redeclare function Fout = out;
          redeclare function Fext = ext(v=Value);
          redeclare function Fta = ta;
          redeclare function initState = initst;
            annotation (Diagram, Icon(Text(extent=[-44,32;44,-32], style(color=3, rgbcolor={0,0,255}), string="%Value")));
              Interfaces.outPort outPort1
                               annotation (extent=[80,-10;100,10]);
              Interfaces.inPort inPort1
                             annotation (extent=[-100,-10;-80,10]);
          equation
              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
          end setValue;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            input Real v;
            output st sout;
          algorithm
            sout := ext(int(s),0,bag,v=v);
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.phase := 1;
            sout.sigma := Modelica.Constants.inf; // processing_time;
            sout.job := 0;
            sout.n := 0;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            input Real v;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            sout.n := numEvents(bag);
            sout.v := v;
            for i in 1:sout.n loop
              x := DEVSLib.SRC.getEvent(bag);
            end for;
            sout.sigma := 0;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            for i in 1:s.n loop
              y.Type := integer(s.v);
              y.Value := s.v;
              sendEvent(queue[1],y);
            end for;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real job;
            Real v; // processing time;
            Integer received;
            Integer n;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;  //first generation starts at time == 1
            out.job := 0;
            out.v := 0;
            out.received := 0;
            out.n := 0;
          end initst;
        end SetValue;

        package IFType "Conditional message forwarding"

        annotation(preferedView="documentation",
          Documentation(info="<HTML>
 <p>
This model forwards each received message throuhg the port out1 if the type of the message equals the <i> condition </i> parameter and otherwise through port out2.
</p>

</HTML>
"));
          model ifType
            parameter Integer condition = 1;
                     extends SRC.AtomicDEVS(numOut=2,redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst(cond=condition);
            annotation (Diagram, Icon(
                        Rectangle(extent=[-80,80; 80,-80], style(color=3, rgbcolor={0,0,255})),
                        Line(points=[-80,40; 80,40],
                                                  style(color=3, rgbcolor={0,0,255})),
                        Line(points=[20,-50; 80,-50],
                                                   style(color=3, rgbcolor={0,0,255})),
                        Text(
                          extent=[-80,88; -14,34],
                          style(color=3, rgbcolor={0,0,255}),
                          string="IFType"),
                        Text(
                          extent=[40,72; 80,52],
                          style(color=3, rgbcolor={0,0,255}),
                          string="then"),
                        Text(
                          extent=[38,-54; 78,-74],
                          style(color=3, rgbcolor={0,0,255}),
                          string="else"),
                        Line(points=[20,40; 20,-50],
                                                   style(color=3, rgbcolor={0,0,255})),
                        Polygon(points=[80,40; 68,44; 68,36; 80,40], style(color=3, rgbcolor=
                                {0,0,255})),
                        Polygon(points=[80,-50; 68,-46; 68,-54; 80,-50], style(color=3,
                              rgbcolor={0,0,255})),
                        Text(
                          extent=[-80,-20; 0,-80],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%condition"),
                        Text(
                          extent=[-100,100; 100,80],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%name")));

            SRC.Interfaces.outPort out1
                             annotation (extent=[80,30;100,50]);
            SRC.Interfaces.inPort in1
                           annotation (extent=[-100,30;-80,50]);
            SRC.Interfaces.outPort out2
                             annotation (extent=[80,-60;100,-40]);
          equation
            iEvent[1] = in1.event;
            iQueue[1] = in1.queue;

            oEvent[1] = out1.event;
            oQueue[1] = out1.queue;
            oEvent[2] = out2.event;
            oQueue[2] = out2.queue;
          end ifType;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),0,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          protected
            Integer n;
          algorithm
            sout := s;
            sout.sigma := Modelica.Constants.inf;
            sout.nq := 0;
            sout.ne := 0;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            SRC.stdEvent x;
          algorithm
            sout := s;
            numreceived := SRC.numEvents(bag);
            //Modelica.Utilities.Streams.print("RECEIVED");
            for i in 1:numreceived loop
              //Modelica.Utilities.Streams.print("EVENT");
              x := DEVSLib.SRC.getEvent(bag);
              if x.Type == s.cond then
                //Modelica.Utilities.Streams.print("IF THEN");
                SRC.sendEvent(sout.q, x);
                sout.nq := sout.nq +1;
              else
                //Modelica.Utilities.Streams.print("IF ELSE");
                SRC.sendEvent(sout.elseq,x);
                sout.ne := sout.ne +1;
              end if;
            end for;
            sout.sigma := 0;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            SRC.stdEvent y;
          algorithm
            if s.nq > 0 then
              for i in 1:s.nq loop
                //Modelica.Utilities.Streams.print("SENT THEN");
                y := SRC.getEvent(s.q, i);
                SRC.sendEvent(queue[1], y);
              end for;
            elseif s.ne > 0 then
              for i in 1:s.ne loop
                //Modelica.Utilities.Streams.print("SENT ELSE");
                y := SRC.getEvent(s.elseq, i);
                SRC.sendEvent(queue[2], y);
              end for;
            end if;

          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Integer q;
            Integer cond;
            Integer elseq;
            Integer nq;
            Integer ne;
          end st;

          function initst
            input Integer cond;
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;
            out.q := SRC.CreateQueue();
            out.nq := 0;
            out.elseq := SRC.CreateQueue();
            out.ne := 0;
            out.cond := cond;
          end initst;
        end IFType;

          model testIF

            annotation (Diagram);
            Generator.generator generator(                              name=
                "gen1", numGenerations=0)
              annotation (extent=[-60,0;-40,20]);
            Display.display display(name="dis")
                                    annotation (extent=[60,0;80,20]);
            Display.display display1(name="dis1")
                                    annotation (extent=[60,-40;80,-20]);
          SetValue.setValue setValue(Value=3) annotation (extent=[18,2;38,22]);
          SetValue.setValue setValue1(Value=4)
            annotation (extent=[16,-30;36,-10]);
          IFType.ifType ifType(condition=2)
                               annotation (extent=[-26,-2;-6,18]);
          equation
          connect(setValue.outPort1, display.inPort1) annotation (points=[37,12;50,12;50,10;61,10], style(color=0,rgbcolor={0,0,0}));
          connect(setValue1.outPort1, display1.inPort1) annotation (points=[35,-20;48,-20;48,-30;61,-30], style(color=0,rgbcolor={0,0,0}));
          connect(generator.outPort1, ifType.in1) annotation (points=[-41,10;-32,10;-32,12;-25,12], style(color=0,rgbcolor={0,0,0}));
          connect(ifType.out1, setValue.inPort1) annotation (points=[-7,12;19,12], style(color=0,rgbcolor={0,0,0}));
          connect(ifType.out2, setValue1.inPort1) annotation (points=[-7,3;5.5,3;5.5,-20;17,-20], style(color=0,rgbcolor={0,0,0}));
          end testIF;

        package Storage "Stores the value of the last received message"
          import DESLib.DEVSLib.SRC.*;

        annotation(preferedView="documentation",
          Documentation(info="<HTML>
 <p>
This model stores the value of any message received at port STORE.<br>
Also it generates an output message with the stored value if a message is received at port CHECK.
</p>

</HTML>
"));
          model storage
                   extends AtomicDEVS(numIn=2,
          redeclare record State = st);
          redeclare function Fcon = con;
          redeclare function Fint = int;
          redeclare function Fout = out;
          redeclare function Fext = ext;
          redeclare function Fta = ta;
          redeclare function initState = initst;
            annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                              color=3, rgbcolor={0,0,255})), Text(
                          extent=[-80,34; 80,-26],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%name")));
                      Interfaces.outPort outPort1
                                       annotation (extent=[80,-10; 100,10]);
                      Interfaces.inPort CHECK
                                     annotation (extent=[-100,40; -80,60]);
                      Interfaces.inPort STORE
                                     annotation (extent=[-100,-60; -80,-40]);
          equation
              iEvent[1] = CHECK.event;
              iQueue[1] = CHECK.queue;
              iEvent[2] = STORE.event;
              iQueue[2] = STORE.queue;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
          end storage;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),0,bag);
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.sigma := Modelica.Constants.inf; // processing_time;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              if x.Port == 1 then //CHECK
                sout.sigma := 0;
              elseif x.Port == 2 then // STORE
                sout.t := x.Type;
                sout.v := x.Value;
                sout.sigma := Modelica.Constants.inf;
              end if;
            end for;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            y.Type := s.t;
            y.Value := s.v;
            sendEvent(queue[1],y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Integer t;
            Real v;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;  //first generation starts at time == 1
            out.t := 0;
            out.v := 0;
          end initst;
        end Storage;

        package Wait "Non-preemptive delay (with internal queue)"
          import DESLib.DEVSLib.SRC.*;

        annotation(preferedView="documentation",
          Documentation(info="<HTML>
 <p>
This model represents a delay or waiting time.
Each message received at port IN is sent after the processTime is elapsed.<br>
If another message is received at port STOP, the delay is stopped (and thus elongated) until a new message is received at port IN.
</p>

</HTML>
"));
          model wait
          parameter Real processTime = 1;
                   extends AtomicDEVS(numIn=2,
          redeclare record State = st);
          redeclare function Fcon = con;
          redeclare function Fint = int;
          redeclare function Fout = out;
          redeclare function Fext = ext;
          redeclare function Fta = ta;
          redeclare function initState = initst(dt=processTime);
            annotation (Diagram);
              Interfaces.outPort outPort1
                               annotation (extent=[80,-10;100,10]);
              Interfaces.inPort IN
                             annotation (extent=[-100,-10;-80,10]);
              Interfaces.inPort STOP
                             annotation (extent=[-100,-80;-80,-60]);
          equation
              iEvent[1] = IN.event;
              iQueue[1] = IN.queue;
              iEvent[2] = STOP.event;
              iQueue[2] = STOP.queue;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
          end wait;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.phase := 1;
            sout.sigma := Modelica.Constants.inf; // processing_time;
            sout.job := 0;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := DESLib.DEVSLib.SRC.getEvent(bag);
              if x.Port == 1 then // IN
                if i == 1 then
                  if sout.phase == 0 then // continue stopped job
                    if s.wait >= Modelica.Constants.inf then
                      sout.sigma := s.dt;
                    else
                      sout.sigma := s.wait;
                    end if;
                    sout.phase := 1;
                  else
                    sout.job := x.Value; // new job
                    sout.sigma := s.dt;
                    sout.phase := 1;
                  end if;
                else
                  Modelica.Utilities.Streams.print("* EVENT BALKED, still processing!");
                end if;
              elseif x.Port == 2 then // STOP
                sout.wait := sout.sigma - e;
                sout.sigma := Modelica.Constants.inf;
                sout.phase := 0;
              end if;
              sout.received := sout.received +1;
            end for;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
              y.Type := 1;
              y.Value := s.job;
              sendEvent(queue[1],y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real job;
            Real dt; // processing time;
            Real wait;
            Integer received;
          end st;

          function initst
            input Real dt;
            output st out;
          algorithm
            out.phase := 1; // 0 = stop, 1 = active
            out.sigma := Modelica.Constants.inf;  //first generation starts at time == 1
            out.job := 0;
            out.dt := dt;
            out.wait := 0;
            out.received := 0;
          end initst;
        end Wait;

        package BankTeller
          package CustomerArrival
            "Draft package to construct new atomic DEVS models"

          annotation(preferedView="info",
            Documentation(info="<HTML>



</HTML>
"));
            import DESLib.DEVSLib.SRC.*;
            model customers "Draft model to construct new atomic DEVS models"
              replaceable Real interarrival = 1;
                       extends AtomicDEVS(numIn=1,numOut=1,
              redeclare record State = st);
              redeclare function Fint = int(iat=interarrival);
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst;
              Interfaces.outPort outPort1     annotation (extent=[96,-10;116,10]);
            equation
              // INPUT PORTS
              iEvent[1] = 0;
              //iQueue[1] = inPort1.queue;
              // OUTPUT PORTS
              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
              annotation (Diagram(graphics));
            end customers;

            function int "Internal Transition Function"
              input st s;
              input Real iat;
              output st sout;
            algorithm
              sout := s;
              sout.sigma := iat;
            end int;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
            protected
              stdEvent y;
            algorithm
              y.Type := 1;
              y.Value := 1;
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st "state of the model"
              Integer phase;
              Real sigma;
            end st;

            function initst "state initalization function"
              output st out;
            algorithm
              out.phase := 0;
              out.sigma := 1;
            end initst;
          end CustomerArrival;

          package Teller "Draft package to construct new atomic DEVS models"

          annotation(preferedView="info",
            Documentation(info="<HTML>

</HTML>
"));
            import DESLib.DEVSLib.SRC.*;
            model teller "Draft model to construct new atomic DEVS models"
              replaceable Real serviceTime = 1;
                       extends AtomicDEVS(numIn=1,numOut=1,
              redeclare record State = st);
              redeclare function Fcon = con(stm=serviceTime);
              redeclare function Fext = ext(stm=serviceTime);
              redeclare function Fint = int(stm=serviceTime);
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst;
              Interfaces.outPort outPort1     annotation (extent=[96,-10;116,10]);
              Interfaces.inPort inPort1     annotation (extent=[-116,-10;-96,10]);
            equation
              // INPUT PORTS
              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;
              // OUTPUT PORTS
              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
              annotation (Diagram(graphics));
            end teller;

            function con "Confluent Transtition Function"
              input st s;
              input Real e;
              input Integer bag;
              input Real stm;
              output st sout;
            algorithm
              // first internal
              sout := ext(int(s,stm=stm),0,bag,stm=stm);
              //----------------
              // first external
              //soutput := ext(s,e,bag);
              //sout := int(soutput);
            end con;

            function int "Internal Transition Function"
              input st s;
              input Real stm;
              output st sout;
            algorithm
              sout := s;
              if sout.nqueue == 0 then
                sout.sigma := Modelica.Constants.inf; // wait
                sout.phase := 1; // set passive
              else
                sout.nqueue := sout.nqueue - 1;
                sout.sigma := stm;
              end if;
            end int;

            function ext "External Transition Function"
              input st s;
              input Real e;
              input Integer bag;
              input Real stm;
              output st sout;
            protected
              Integer numreceived;
              stdEvent x;
            algorithm
              sout := s;
              numreceived := numEvents(bag);
              for i in 1:numreceived loop
                x := getEvent(bag);
                if sout.phase == 1 then // passive -> serve
                  sout.phase := 2; // set active
                  sout.sigma := stm;
                elseif sout.phase == 2 then // active -> wait in queue
                  sout.nqueue := sout.nqueue + 1;
                  sout.sigma := sout.sigma - e;
                end if;
              end for;

            end ext;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
            protected
              stdEvent y;
            algorithm
              y.Type := 1;
              y.Value := 1;
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st "state of the model"
              Integer phase;
              Real sigma;
              Integer nqueue;
            end st;

            function initst "state initalization function"
              output st out;
            algorithm
              out.phase := 1; //passive
              out.sigma := Modelica.Constants.inf; // first internal event never happens
              out.nqueue := 0; // empty queue
            end initst;
          end Teller;

          model system
            RandomLib.Variates.Generator g "RNG";
            Real a "inter-arrival time variate";
            Real s "service time variate";
            CustomerArrival.customers customers(name="customers", interarrival=
                  a) annotation (extent=[-60,0; -40,20]);
            Teller.teller teller(               name="teller", serviceTime=s)
              annotation (extent=[0,0; 20,20]);
            AuxModels.Display Departures
                                       annotation (extent=[40,0; 60,20]);
            AuxModels.Display Arrivals annotation (extent=[-20,-40; 0,-20]);
            AuxModels.DUP dUP annotation (extent=[-42,0; -22,20]);
          algorithm
            // initilization of the RngStream
            when initial() then
              g := RandomLib.Variates.initGenerator();
            end when;
            (a,g) := RandomLib.Variates.Continuous.Exponential(g,10);
            (s,g) := RandomLib.Variates.Continuous.Exponential(g,8);
          equation

            annotation (Diagram, experiment(StopTime=20),  experimentSetupOutput);
            connect(teller.outPort1, Departures.inPort1)
                                                       annotation (points=[20.6,10;
                  41,10],                                                                   style(color=0,rgbcolor={0,0,0}));
            connect(customers.outPort1, dUP.in1) annotation (points=[-39.4,10;
                  -34.6,10],                                                             style(color=0,rgbcolor={0,0,0}));
            connect(dUP.out2, Arrivals.inPort1) annotation (points=[-31.4,8;
                  -31.4,-5; -19,-5; -19,-30],                                                         style(color=0,rgbcolor={0,0,0}));
            connect(dUP.out1, teller.inPort1) annotation (points=[-31.4,12; -16,
                  12; -16,10; -0.6,10],                                                              style(color=0,rgbcolor={0,0,0}));
          end system;
        end BankTeller;

        package BankTellerAVG
          package CustomerArrival
            "Draft package to construct new atomic DEVS models"

          annotation(preferedView="info",
            Documentation(info="<HTML>



</HTML>
"));
            import DESLib.DEVSLib.SRC.*;
            model customers "Draft model to construct new atomic DEVS models"
              replaceable Real interarrival = 1;
                       extends AtomicDEVS(numIn=1,numOut=1,
              redeclare record State = st);
              redeclare function Fint = int(iat=interarrival);
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst;
              Interfaces.outPort outPort1     annotation (extent=[96,-10;116,10]);
            equation
              // INPUT PORTS
              iEvent[1] = 0;
              //iQueue[1] = inPort1.queue;
              // OUTPUT PORTS
              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
              annotation (Diagram(graphics));
            end customers;

            function int "Internal Transition Function"
              input st s;
              input Real iat;
              output st sout;
            algorithm
              sout := s;
              sout.sigma := iat;
            end int;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
            protected
              stdEvent y;
            algorithm
              y.Type := 1;
              y.Value := 1;
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st "state of the model"
              Integer phase;
              Real sigma;
            end st;

            function initst "state initalization function"
              output st out;
            algorithm
              out.phase := 0;
              out.sigma := 1;
            end initst;
          end CustomerArrival;

          package Teller "Draft package to construct new atomic DEVS models"

          annotation(preferedView="info",
            Documentation(info="<HTML>



</HTML>
"));
            import DESLib.DEVSLib.SRC.*;
            model teller "Draft model to construct new atomic DEVS models"
              replaceable Real serviceTime = 1;
              Real T = time;
                       extends AtomicDEVS(numIn=1,numOut=1,
              redeclare record State = st);
              redeclare function Fcon = con(stm=serviceTime,t=T);
              redeclare function Fext = ext(stm=serviceTime,t=T);
              redeclare function Fint = int(stm=serviceTime,t=T);
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst;
              Interfaces.outPort outPort1     annotation (extent=[96,-10;116,10]);
              Interfaces.inPort inPort1     annotation (extent=[-116,-10;-96,10]);
            equation
              // INPUT PORTS
              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;
              // OUTPUT PORTS
              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
              annotation (Diagram(graphics));
            end teller;

            function con "Confluent Transtition Function"
              input st s;
              input Real e;
              input Integer bag;
              input Real stm;
              input Real t;
              output st sout;
            algorithm
              // first internal
              sout := ext(int(s,stm=stm,t=t),0,bag,stm=stm,t=t);
              //----------------
              // first external
              //soutput := ext(s,e,bag);
              //sout := int(soutput);
            end con;

            function int "Internal Transition Function"
              input st s;
              input Real stm;
              input Real t;
              output st sout;
            algorithm
              sout := s;
              if sout.nqueue == 0 then
                sout.sigma := Modelica.Constants.inf; // wait
                sout.phase := 1; // set passive
              else
                sout.Qsum := sout.Qsum + sout.nqueue*(t-sout.Qlast);
                sout.Qavg := sout.Qsum/t;
                sout.Qlast := t;
                sout.nqueue := sout.nqueue - 1;
                sout.sigma := stm;
              end if;
            end int;

            function ext "External Transition Function"
              input st s;
              input Real e;
              input Integer bag;
              input Real stm;
              input Real t;
              output st sout;
            protected
              Integer numreceived;
              stdEvent x;
            algorithm
              sout := s;
              numreceived := numEvents(bag);
              for i in 1:numreceived loop
                x := getEvent(bag);
                if sout.phase == 1 then // passive -> serve
                  sout.phase := 2; // set active
                  sout.sigma := stm;
                elseif sout.phase == 2 then // active -> wait in queue
                  sout.Qsum := sout.Qsum + sout.nqueue*(t-sout.Qlast);
                  sout.Qavg := sout.Qsum/t;
                  sout.Qlast := t;
                  sout.nqueue := sout.nqueue + 1;
                  sout.sigma := sout.sigma - e;
                end if;
              end for;

            end ext;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
            protected
              stdEvent y;
            algorithm
              y.Type := 1;
              y.Value := 1;
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st "state of the model"
              Integer phase;
              Real sigma;
              Integer nqueue;
              Real Qsum;
              Real Qavg;
              Real Qlast;
            end st;

            function initst "state initalization function"
              output st out;
            algorithm
              out.phase := 1; //passive
              out.sigma := Modelica.Constants.inf; // first internal event never happens
              out.nqueue := 0; // empty queue
              out.Qsum := 0;
              out.Qavg := 0;
              out.Qlast := 0;
            end initst;
          end Teller;

          model system
            RandomLib.Variates.Generator g "RNG";
            Real e "inter-arrival variate";
            Real t "service-time variate";
            Real QAVG;
            CustomerArrival.customers customers(name="customers", interarrival=
                  e) annotation (extent=[-60,0; -40,20]);
            Teller.teller teller(serviceTime=t, name="teller")
              annotation (extent=[0,0; 20,20]);
            AuxModels.Display Departures
                                       annotation (extent=[34,0; 54,20]);
            AuxModels.Display Arrivals annotation (extent=[-28,-28; -8,-8]);
            AuxModels.DUP dUP annotation (extent=[-42,0; -22,20]);
          algorithm
            // initilization of the RngStream
            when initial() then
              g := RandomLib.Variates.initGenerator();
            end when;
            (e,g) := RandomLib.Variates.Continuous.Exponential(g,10);
            (t,g) := RandomLib.Variates.Continuous.Exponential(g,8);
          equation
            QAVG = teller.S.Qavg;

            annotation (Diagram(coordinateSystem(preserveAspectRatio=false,
                    extent={{-100,-100},{100,100}}),
                                graphics),
              experiment(StopTime=1e+006),
              experimentSetupOutput);
            connect(teller.outPort1, Departures.inPort1) annotation (points=[20.6,10;
                  35,10],                                                                     style(color=0,rgbcolor={0,0,0}));
            connect(customers.outPort1, dUP.in1) annotation (points=[-39.4,10;
                  -34.6,10],                                                             style(color=0,rgbcolor={0,0,0}));
            connect(dUP.out2, Arrivals.inPort1) annotation (points=[-31.4,8;
                  -31.4,-5; -27,-5; -27,-18],                                                          style(color=0,rgbcolor={0,0,0}));
            connect(dUP.out1, teller.inPort1) annotation (points=[-31.4,12; -16,
                  12; -16,10; -0.6,10],                                                              style(color=0,rgbcolor={0,0,0}));
          end system;
        end BankTellerAVG;
      end SimpleModels;

      package ATM "Model of an Automatic Teller Machine"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package includes the implementation of an ATM system (Automatic Teller Machine).
A complete description of the system and its implementation using the CD++ environment can be found <a href=http://www.sce.carleton.ca/faculty/wainer/wbgraf/samples/atm.zip>here</a>
</p>
<p>
Basically, the system is composed by three subsystems:
<ul>
<li> CardReader, that receives the inserted card and starts the authorization process.
<li> Authorization, that validates the card, the PIN and the amount of cash to be withdrawn.
<li> CashDispenser, that gives the cash to the user.
</ul>
</p>
<p>
The CardReader and the CashDispenser are modeled as atomic models, and the Authorization as a coupled model.<br>
The Authorization subsystem is composed by the BalanceVerifier, the PINVerifier and the UserInterface.
</p>

<h3><font color=\"#008000\">Simulation</font></h3>
The ATM system is implemented in the testATM model, that can be simulated.<br>
Three variables have been included to facilitate the observation of the results.
<ul>
<li><i>cardIn</i> shows the insertions of cards into the ATM.
<li><i>cash</i> shows the returned cash to the customer.
<li><i>cardOut</i> shows the ejection of the inserted card.
</ul>
<br>
</HTML>
"));
        package BalanceVerifier
          import DESLib.DEVSLib.SRC.*;

          model balanceVerifier
                     extends AtomicDEVS(numOut=2,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst;
            annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                            color=3, rgbcolor={0,0,255})), Text(
                        extent=[-76,44; 74,-36],
                        style(color=3, rgbcolor={0,0,255}),
                        string="%name")));
                  Interfaces.outPort AmntOut
                                   annotation (extent=[80,30; 100,50]);
                  Interfaces.inPort AmntIn
                                 annotation (extent=[-100,-10; -80,10]);
                  Interfaces.outPort GetAmntOut
                                   annotation (extent=[80,-50; 100,-30]);
          equation
            iEvent[1] = AmntIn.event;
            iQueue[1] = AmntIn.queue;

            oEvent[1] = AmntOut.event;
            oQueue[1] = AmntOut.queue;
            oEvent[2] = GetAmntOut.event;
            oQueue[2] = GetAmntOut.queue;
          end balanceVerifier;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.phase := 1; // pasive;
            sout.sigma := Modelica.Constants.inf;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
            Real r;
          algorithm
            x := DEVSLib.SRC.getEvent(bag);
            sout := s;
            if s.phase == 1 then
              sout.amount := integer(x.Value);
              sout.phase := 2; // active
              sout.sigma := RandomUniform(0)*10; // random processing time
              r := RandomUniform(0);
              if (r <= 0.8) then
                sout.balance := true;
              else
                sout.balance := false;
              end if;
            else
              sout.sigma := s.sigma - e;
            end if;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent Y;
          algorithm
            if (s.balance == true) then
              Y.Type := 1;
              Y.Value := s.amount; // output value;
              sendEvent(queue[1],Y);
            else
              Y.Type := 1;
              Y.Value := 1;
              sendEvent(queue[2],Y);
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real amount;
            Boolean balance;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 100;  //first generation starts at time == 100
            out.amount := 0;
            out.balance := false;
          end initst;
        end BalanceVerifier;

        package PINVerifier
          import DESLib.DEVSLib.SRC.*;
          model pinVerifier
                     extends AtomicDEVS(numOut=3,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst;
            annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                            color=3, rgbcolor={0,0,255})), Text(
                        extent=[-76,44; 74,-36],
                        style(color=3, rgbcolor={0,0,255}),
                        string="%name")));
                  Interfaces.outPort GetAmnt
                                   annotation (extent=[80,50; 100,70]);
                  Interfaces.inPort PINIn
                                 annotation (extent=[-100,-10; -80,10]);
                  Interfaces.outPort GetPIN
                                   annotation (extent=[80,-10; 100,10]);
                  Interfaces.outPort outPort3 "Eject"
                                   annotation (extent=[80,-70; 100,-50]);
          equation
            iEvent[1] = PINIn.event;
            iQueue[1] = PINIn.queue;

            oEvent[1] = GetAmnt.event;
            oQueue[1] = GetAmnt.queue;
            oEvent[2] = GetPIN.event;
            oQueue[2] = GetPIN.queue;
            oEvent[3] = outPort3.event;
            oQueue[3] = outPort3.queue;
          end pinVerifier;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.phase := 1;
            sout.sigma := Modelica.Constants.inf;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
            Real r;
          algorithm
            x := DEVSLib.SRC.getEvent(bag);
            sout := s;
            if s.phase == 1 then
              sout.sigma := RandomUniform(0)*10; // random processing time
              sout.phase := 2; // busy
              sout.trials := s.trials +1;
              r := RandomUniform(0);
              if (r <= 0.7) then
                sout.pin_ok := true;
                sout.trials := 0;
              else
                sout.pin_ok := false;
              end if;
            else
              sout.sigma := s.sigma -e; // still processing, event balked
            end if;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent Y;
          algorithm
            if (s.pin_ok == true) then // pin OK
              Y.Value := 1; // output value;
              sendEvent(queue[1],Y);
            else
              if s.trials < 3 then // ask pin again
                Y.Value := 1;
                sendEvent(queue[2],Y);
              else  // eject card
                Y.Value := 1;
                sendEvent(queue[3],Y);
              end if;
            end if;

          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real pin;
            Boolean pin_ok;
            Integer trials;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 100;  //first generation starts at time == 100
            out.pin := 0;
            out.pin_ok := false;
            out.trials := 0;
          end initst;
        end PINVerifier;

        package UserInterface
          import DESLib.DEVSLib.SRC.*;

          model userInterface
                     extends AtomicDEVS(numIn=3,numOut=2,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst;
                  annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                            color=3, rgbcolor={0,0,255})), Text(
                        extent=[-76,44; 74,-36],
                        style(color=3, rgbcolor={0,0,255}),
                        string="%name")));
                  Interfaces.outPort AmntOut
                                   annotation (extent=[80,30; 100,50]);
                  Interfaces.inPort GetPIN
                                 annotation (extent=[-100,50; -80,70]);
                  Interfaces.inPort GetAmnt
                                 annotation (extent=[-100,-10; -80,10]);
                  Interfaces.inPort CardNO
                                 annotation (extent=[-100,-70; -80,-50]);
                  Interfaces.outPort PINOut
                                   annotation (extent=[80,-50; 100,-30]);
          equation
            iEvent[1] = GetPIN.event;
            iQueue[1] = GetPIN.queue;
            iEvent[2] = GetAmnt.event;
            iQueue[2] = GetAmnt.queue;
            iEvent[3] = CardNO.event;
            iQueue[3] = CardNO.queue;

            oEvent[1] = AmntOut.event;
            oQueue[1] = AmntOut.queue;
            oEvent[2] = PINOut.event;
            oQueue[2] = PINOut.queue;
          end userInterface;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
              sout := s;
            sout.phase := 1;
            sout.sigma := Modelica.Constants.inf;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            x := DEVSLib.SRC.getEvent(bag);
            sout := s;
            if s.phase == 1 then
              if x.Port == 1 or x.Port == 3 then // GET PIN
                sout.sigma := RandomUniform(0)*10; // random processing time
                sout.phase := 2; // busy
                sout.pin := integer(RandomUniform(0)*1000);
                sout.pin_entered := true;
                sout.amount_entered := false;
              elseif x.Port == 2 then // GET AMOUNT
                sout.sigma := RandomUniform(0)*10; // random processing time
                sout.phase := 2; // busy
                sout.amount := integer(RandomUniform(0)*1000);
                sout.pin_entered := false;
                sout.amount_entered := true;
              end if;
            else
              sout.sigma := s.sigma -e;
            end if;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent Y;
          algorithm
            if s.pin_entered then // pin entered
              Y.Value := s.pin; // output value;
              sendEvent(queue[1],Y);
            else
              Y.Value := s.amount;
              sendEvent(queue[2],Y);
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Integer pin;
            Integer amount;
            Boolean pin_entered;
            Boolean amount_entered;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 100;  //first generation starts at time == 100
            out.pin := 0;
            out.amount := 0;
            out.pin_entered := false;
            out.amount_entered := false;
          end initst;
        end UserInterface;

        model authorization
          import DESLib.DEVSLib.SRC.*;

         annotation (Icon(Rectangle(extent=[-80,80; 80,-80], style(color=3,
                          rgbcolor={0,0,255})), Text(
                      extent=[-76,44; 74,-36],
                      style(color=3, rgbcolor={0,0,255}),
                      string="%name")),
                                  Diagram);
                UserInterface.userInterface userInterface(name="userInterface")
                  annotation (extent=[-44,-8; -8,30]);
                BalanceVerifier.balanceVerifier balanceVerifier(name="balanceVerifier")
                  annotation (extent=[-2,26; 34,46]);
                PINVerifier.pinVerifier pinVerifier(name="pinVerifier")
                  annotation (extent=[4,-30; 28,-10]);
                Interfaces.inPort CardNO
                              annotation (extent=[-100,-10; -80,10]);
                Interfaces.outPort AmntOut
                                annotation (extent=[80,30; 100,50]);
                Interfaces.outPort Eject
                              annotation (extent=[80,-36; 100,-16]);
                AuxModels.BreakLoop breakLoop
                  annotation (extent=[-18,-62; 2,-42], rotation=180);
                AuxModels.BreakLoop breakLoop1
                  annotation (extent=[-18,-50; 2,-30], rotation=180);
                AuxModels.BreakLoop breakLoop2
                  annotation (extent=[-14,50; 6,70], rotation=180);

        equation
        connect(userInterface.AmntOut, balanceVerifier.AmntIn) annotation (
                      points=[-9.8,18.6; -6,18.6; -6,36; -0.2,36],  style(color=0,
                        rgbcolor={0,0,0}));
                  connect(balanceVerifier.AmntOut, AmntOut) annotation (points=[32.2,40;
                        90,40], style(color=0, rgbcolor={0,0,0}));
                  connect(pinVerifier.outPort3, Eject) annotation (points=[26.8,-26;
                90,-26],      style(color=0, rgbcolor={0,0,0}));
                  connect(userInterface.PINOut, pinVerifier.PINIn) annotation (points=[-9.8,3.4;
                -2,3.4; -2,-20; 5.2,-20],                    style(color=0, rgbcolor={0,0,
                          0}));
                  connect(CardNO, userInterface.CardNO) annotation (points=[-90,0;
                -67.1,0; -67.1,-0.4; -42.2,-0.4],   style(color=0, rgbcolor={0,0,0}));
                  connect(pinVerifier.GetAmnt, breakLoop1.IN) annotation (points=[26.8,-14;
                40,-14; 40,-40; -5.2,-40],         style(color=0, rgbcolor={0,0,0}));
                  connect(breakLoop1.OUT, userInterface.GetAmnt) annotation (points=[
                        -8.6,-40; -50,-40; -50,11; -42.2,11], style(color=0, rgbcolor={
                          0,0,0}));
                  connect(pinVerifier.GetPIN, breakLoop.IN) annotation (points=[26.8,-20;
                44,-20; 44,-52; -5.2,-52],              style(color=0, rgbcolor={0,0,0}));
                  connect(breakLoop.OUT, userInterface.GetPIN) annotation (points=[-8.6,
                        -52; -58,-52; -58,22.4; -42.2,22.4], style(color=0, rgbcolor={0,
                          0,0}));
                  connect(balanceVerifier.GetAmntOut, breakLoop2.IN) annotation (points=[32.2,32;
                38,32; 38,60; -1.2,60],                  style(color=0, rgbcolor={0,0,0}));
                  connect(breakLoop2.OUT, userInterface.GetAmnt) annotation (points=[
                        -4.6,60; -50,60; -50,11; -42.2,11], style(color=0, rgbcolor={0,
                          0,0}));

        end authorization;

        package CardReader
          import DESLib.DEVSLib.SRC.*;
          model cardReader
                     extends AtomicDEVS(numIn=2,numOut=2,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst;

            annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                              color=3, rgbcolor={0,0,255})), Text(
                          extent=[-76,44; 74,-36],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%name")));
                    Interfaces.outPort CardOut
                                     annotation (extent=[80,30; 100,50]);
                    Interfaces.inPort CardIn
                                   annotation (extent=[-100,32; -80,52]);
                    Interfaces.inPort Eject
                                   annotation (extent=[-100,-48; -80,-28]);
                    Interfaces.outPort CardNOOut
                                     annotation (extent=[80,-50; 100,-30]);
          equation
            iEvent[1] = CardIn.event;
            iQueue[1] = CardIn.queue;
            iEvent[2] = Eject.event;
            iQueue[2] = Eject.queue;

            oEvent[1] = CardOut.event;
            oQueue[1] = CardOut.queue;
            oEvent[2] = CardNOOut.event;
            oQueue[2] = CardNOOut.queue;
          end cardReader;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.sigma := Modelica.Constants.inf;
            if s.eject_requested then
              sout.phase := 1;
            end if;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            x := DEVSLib.SRC.getEvent(bag);
            sout := s;
            if x.Port == 1 and not sout.card_entered then // CARD IN
              sout.sigma := RandomUniform(0)*10; // random processing time
              sout.phase := 2; // busy
              sout.card := integer(RandomUniform(0)*1000);
              sout.card_entered := true;
              sout.eject_requested := false;
            elseif x.Port == 2 then // EJECT
              sout.sigma := RandomUniform(0)*10; // random processing time
              sout.card_entered := false;
              sout.eject_requested := true;
            end if;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent Y;
          algorithm
            if s.card_entered then
              Y.Value := s.card;
              sendEvent(queue[2],Y);
            else
              Y.Value := s.card;
              sendEvent(queue[1],Y);
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Integer card;
            Boolean card_entered;
            Boolean eject_requested;
            Boolean empty_reader;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 100;  //first generation starts at time == 100
            out.card := 0;
            out.card_entered := false;
            out.eject_requested := false;
            out.empty_reader := false;
          end initst;
        end CardReader;

        package CashDispenser
          import DESLib.DEVSLib.SRC.*;
          model cashDispenser
                     extends AtomicDEVS(
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst;

            annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                              color=3, rgbcolor={0,0,255})), Text(
                          extent=[-76,44; 74,-36],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%name")));
                    Interfaces.outPort CashOut
                                     annotation (extent=[80,-10; 100,10]);
                    Interfaces.inPort Amnt
                                   annotation (extent=[-100,-10; -80,10]);
          equation
            iEvent[1] = Amnt.event;
            iQueue[1] = Amnt.queue;

            oEvent[1] = CashOut.event;
            oQueue[1] = CashOut.queue;
          end cashDispenser;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
              sout := s;
            sout.sigma := Modelica.Constants.inf;
            sout.phase := 1;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            x := DEVSLib.SRC.getEvent(bag);
            sout := s;
            sout.sigma := RandomUniform(0)*10; // random processing time
            sout.phase := 2; // busy
            sout.amount := integer(x.Value);
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent Y;
          algorithm
            Y.Value := s.amount; // output value;
            sendEvent(queue[1],Y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Integer amount;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 100;  //first generation starts at time == 100
            out.amount := 0;
          end initst;
        end CashDispenser;

        model ATM
          import DESLib.DEVSLib.SRC.*;
          // parameter Real p1 = 1;

          Interfaces.inPort CardIn
                                 annotation (extent=[-100,-6; -80,14]);
                  Interfaces.outPort CardOut
                                   annotation (extent=[80,60; 100,80]);
                  annotation (Diagram, Icon);
                  Interfaces.outPort CashOut
                                   annotation (extent=[80,-80; 100,-60]);
                  authorization Authorization  annotation (extent=[-30,-62; 84,70]);
                  CardReader.cardReader cardReader(name="cardReader")
                                                   annotation (extent=[-72,-6; -42,14]);
                  CashDispenser.cashDispenser cashDispenser
                    annotation (extent=[6,-80; 42,-60]);
                  AuxModels.DUP dUP annotation (extent=[46,-76; 58,-64]);
                  AuxModels.BreakLoop breakLoop
                    annotation (extent=[-36,-92; -16,-72], rotation=180);
                  AuxModels.BreakLoop breakLoop1
                    annotation (extent=[-36,-64; -16,-44], rotation=180);
        equation
          connect(CardIn, cardReader.CardIn) annotation (points=[-90,4; -77,4;
                        -77,8.2; -70.5,8.2], style(color=0, rgbcolor={0,0,0}));
                  connect(cardReader.CardOut, CardOut) annotation (points=[-43.5,8; -32,8;
                        -32,70; 90,70], style(color=0, rgbcolor={0,0,0}));
                  connect(cardReader.CardNOOut, Authorization.CardNO) annotation (points=[-43.5,0;
                -28,0; -28,4; -24.3,4],                   style(color=0, rgbcolor={0,0,0}));
                  connect(Authorization.AmntOut, cashDispenser.Amnt) annotation (points=[78.3,
                30.4; 92,30.4; 92,-58; -4,-58; -4,-70; 7.8,-70],              style(color=
                         0, rgbcolor={0,0,0}));
                  connect(cashDispenser.CashOut, dUP.in1) annotation (points=[40.2,-70; 50.44,
                        -70],      style(color=0, rgbcolor={0,0,0}));
                  connect(dUP.out1, CashOut) annotation (points=[52.36,-68.8; 83.5,-68.8;
                        83.5,-70; 90,-70],
                                      style(color=0, rgbcolor={0,0,0}));
                  connect(dUP.out2, breakLoop.IN) annotation (points=[52.36,-71.2; 66,
                        -71.2; 66,-82; -23.2,-82], style(color=0, rgbcolor={0,0,0}));
                  connect(breakLoop.OUT, cardReader.Eject) annotation (points=[-26.6,
                -82; -78,-82; -78,0.2; -70.5,0.2],         style(color=0, rgbcolor={0,0,
                          0}));
                  connect(Authorization.Eject, breakLoop1.IN) annotation (points=[78.3,
                -13.16; 84,-13.16; 84,-54; -23.2,-54],         style(color=0, rgbcolor=
                          {0,0,0}));
                  connect(breakLoop1.OUT, cardReader.Eject) annotation (points=[-26.6,
                -54; -74,-54; -74,0.2; -70.5,0.2],         style(color=0, rgbcolor={0,0,
                          0}));
        end ATM;

        model testATM
        Real cardIn;
        Real cash;
        Real cardOut;
            AuxModels.Generator generator(period=20)
                    annotation (extent=[-80,0; -60,20]);
                  ATM aTM annotation (extent=[-20,0; 0,20]);
                  annotation (Diagram,
                    experiment(StopTime=50),
                    experimentSetupOutput);
                  AuxModels.Display CardOutDisplay
                                                annotation (extent=[36,18; 56,38]);
                  AuxModels.Display CashOut      annotation (extent=[40,-20; 60,0]);
                  AuxModels.DUP dUP annotation (extent=[-58,0; -38,20]);
                  AuxModels.Display CardInsertion
                                                annotation (extent=[-30,-28; -10,-8]);
        equation
                cardIn = CardInsertion.S.phase;
                cardOut = CardOutDisplay.S.phase;
                cash = CashOut.S.phase;
                  connect(aTM.CardOut, CardOutDisplay.inPort1) annotation (points=[-1,17;
                        20,17; 20,28; 37,28], style(color=0, rgbcolor={0,0,0}));
                  connect(aTM.CashOut, CashOut.inPort1) annotation (points=[-1,3; 20,3;
                        20,-10; 41,-10], style(color=0, rgbcolor={0,0,0}));
                  connect(generator.outPort1, dUP.in1) annotation (points=[-61,10;
                        -50.6,10],
                      style(color=0, rgbcolor={0,0,0}));
                  connect(dUP.out1, aTM.CardIn) annotation (points=[-47.4,12; -29.5,12;
                        -29.5,10.4; -19,10.4], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP.out2, CardInsertion.inPort1) annotation (points=[-47.4,8;
                        -47.4,-5.5; -29,-5.5; -29,-18],
                                                  style(color=0, rgbcolor={0,0,0}));

        end testATM;

      end ATM;

      package Clock2 "Model of a pendulum clock with alarm"
        annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package includes the implementation of a pendulum clock with alarm.
A complete description of the system and its implementation using the CD++ environment can be found <a href=http://www.sce.carleton.ca/faculty/wainer/wbgraf/samples/clock_2.zip>here</a>
</p>

<p>
The clock is guided by the oscillation of a pendulum, and counts the minutes and hours.
It also has an alarm system that is checked and activated when required.<br>
The clock and the alarm can be configured using the input ports.
</p>

<h3><font color=\"#008000\">Simulation</font></h3>
The Clock system is implemented in the testClock model. Three variables have been included to facilitate the observation of the results.
<ul>
<li><i>hour</i> shows the position of the hour hand.
<li><i>min</i> shows the position of the minutes hand.
<li><i>alarm</i> shows the activation of the alarm.
</ul>

The testNoAlarm model includes the Clock system without the alarm subsystem. In this case, two variables have been included to observe the results.
<ul>
<li><i>hour</i> shows the position of the hour hand.
<li><i>min</i> shows the position of the minutes hand.
</ul>

<br>
</HTML>
"));
        package Pendulum
          model pendulum
            import DESLib.DEVSLib.SRC.*;

                      Interfaces.inPort TurnON
                                    annotation (extent=[-100,40; -80,60]);
                      Interfaces.inPort TurnOFF
                                     annotation (extent=[-100,-40; -80,-20]);
                      Interfaces.outPort TIC
                                  annotation (extent=[80,0; 100,20]);
                      AuxModels.CrossUP ZeroCross(Value=6.28)
                                annotation (extent=[40,0; 60,20]);
                      annotation (Diagram);
                    AuxModels.DICO diCo
                              annotation (extent=[-56,0; -36,20]);
                    Sine sine annotation (extent=[-8,0; 12,20]);
          equation
                    connect(ZeroCross.outport, TIC)
                                                 annotation (points=[60.6,10; 90,10],
                                 style(color=3, rgbcolor={0,0,255}));
                    connect(TurnON, diCo.inport) annotation (points=[-90,50; -76,50;
                          -76,10; -56.6,10],
                                       style(color=3, rgbcolor={0,0,255}));
                    connect(TurnOFF, diCo.inport) annotation (points=[-90,-30; -90,10;
                          -56.6,10],       style(color=3, rgbcolor={0,0,255}));
                    connect(sine.y, ZeroCross.u) annotation (points=[13,10; 39,10],
                        style(color=74, rgbcolor={0,0,127}));
                    connect(diCo.y, sine.u) annotation (points=[-35,10; -10,10], style(
                          color=74, rgbcolor={0,0,127}));
          end pendulum;

              block Sine "Generate sine signal"
                parameter Real amplitude=1 "Amplitude of sine wave";
                parameter Modelica.SIunits.Frequency freqHz=1
              "Frequency of sine wave";
                parameter Modelica.SIunits.Angle phase=0 "Phase of sine wave";
                parameter Real offset=0 "Offset of output signal";
                parameter Modelica.SIunits.Time startTime=0
              "Output = offset for time < startTime";
                extends Modelica.Blocks.Interfaces.SISO;
                output Real pos;
          protected
            constant Real pi=Modelica.Constants.pi;

                        annotation (
                          Coordsys(
                            extent=[-100, -100; 100, 100],
                            grid=[1, 1],
                            component=[20, 20]),
                          Window(
                            x=0.23,
                            y=0.08,
                            width=0.66,
                            height=0.68),
                          Icon(
                            Line(points=[-80, 68; -80, -80], style(color=8)),
                            Polygon(points=[-80, 90; -88, 68; -72, 68; -80, 90], style(color=8,
                                   fillColor=8)),
                            Line(points=[-90, 0; 68, 0], style(color=8)),
                            Polygon(points=[90, 0; 68, 8; 68, -8; 90, 0], style(color=8,
                                  fillColor=8)),
                            Line(points=[-80, 0; -68.7, 34.2; -61.5, 53.1; -55.1, 66.4; -49.4,
                                  74.6; -43.8, 79.1; -38.2, 79.8; -32.6, 76.6; -26.9, 69.7; -21.3,
                                   59.4; -14.9, 44.1; -6.83, 21.2; 10.1, -30.8; 17.3, -50.2;
                                  23.7, -64.2; 29.3, -73.1; 35, -78.4; 40.6, -80; 46.2, -77.6;
                                  51.9, -71.5; 57.5, -61.9; 63.9, -47.2; 72, -24.8; 80, 0],
                                style(color=0)),
                            Text(
                              extent=[-147, -152; 153, -112],
                              string="freqHz=%freqHz",
                              style(color=0))),
                          Diagram(
                            Line(points=[-80, -90; -80, 84], style(color=8)),
                            Polygon(points=[-80, 100; -86, 84; -74, 84; -80, 100], style(color=
                                    8, fillColor=8)),
                            Line(points=[-99, -40; 85, -40], style(color=8)),
                            Polygon(points=[101, -40; 85, -34; 85, -46; 101, -40], style(color=
                                    8, fillColor=8)),
                            Line(points=[-40, 0; -31.6, 34.2; -26.1, 53.1; -21.3, 66.4; -17.1,
                                  74.6; -12.9, 79.1; -8.64, 79.8; -4.42, 76.6; -0.201, 69.7;
                                  4.02, 59.4; 8.84, 44.1; 14.9, 21.2; 27.5, -30.8; 33, -50.2;
                                  37.8, -64.2; 42, -73.1; 46.2, -78.4; 50.5, -80; 54.7, -77.6;
                                  58.9, -71.5; 63.1, -61.9; 67.9, -47.2; 74, -24.8; 80, 0],
                                style(color=0, thickness=2)),
                            Line(points=[-41, -2; -80, -2], style(color=0, thickness=2)),
                            Text(
                              extent=[-128, 7; -82, -11],
                              string="offset",
                              style(color=9)),
                            Line(points=[-41, -2; -41, -40], style(color=8, pattern=2)),
                            Text(
                              extent=[-60, -43; -14, -61],
                              string="startTime",
                              style(color=9)),
                            Text(
                              extent=[84, -52; 108, -72],
                              string="time",
                              style(color=9)),
                            Text(
                              extent=[-74, 106; -33, 86],
                              string="y",
                              style(color=9)),
                            Line(points=[-9, 79; 43, 79], style(color=8, pattern=2)),
                            Line(points=[-42, -1; 50, 0], style(color=8, pattern=2)),
                            Polygon(points=[33, 80; 30, 67; 37, 67; 33, 80], style(
                                color=8,
                                fillColor=8,
                                fillPattern=1)),
                            Text(
                              extent=[37, 57; 83, 39],
                              string="amplitude",
                              style(color=9)),
                            Polygon(points=[33, 1; 30, 14; 36, 14; 33, 1; 33, 1], style(
                                color=8,
                                fillColor=8,
                                fillPattern=1)),
                            Line(points=[33, 79; 33, 0], style(
                                color=8,
                                pattern=1,
                                thickness=1,
                                arrow=0))),
              Documentation(info="<html>

</html>"));

              equation
                pos = if u > 0 then  (offset + (if time < startTime then 0 else amplitude*Modelica.Math.sin(2*pi*freqHz*(time - startTime) + phase))) else 0;
                y = der(pos);
                //y = 0;

              end Sine;
        end Pendulum;

        package Control
          import DESLib.DEVSLib.SRC.*;
          model control
                     extends AtomicDEVS(numIn=2,numOut=2,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst;
           annotation (Diagram, Icon);
                    Interfaces.outPort ClockON
                                     annotation (extent=[80,40; 100,60]);
                    Interfaces.inPort TurnON
                                   annotation (extent=[-100,40; -80,60]);
                    Interfaces.inPort TurnOFF
                                   annotation (extent=[-100,-60; -80,-40]);
                    Interfaces.outPort ClockOFF
                                     annotation (extent=[80,-60; 100,-40]);
          equation
            iEvent[1] = TurnON.event;
            iQueue[1] = TurnON.queue;
            iEvent[2] = TurnOFF.event;
            iQueue[2] = TurnOFF.queue;

            oEvent[1] = ClockON.event;
            oQueue[1] = ClockON.queue;
            oEvent[2] = ClockOFF.event;
            oQueue[2] = ClockOFF.queue;

          end control;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout.phase := 1;
            sout.sigma := Modelica.Constants.inf;
            sout.on := s.on;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            x := DEVSLib.SRC.getEvent(bag);
            sout := s;
            if x.Port == 1 then //ON
              if not s.on then
                sout.on := true;
                sout.sigma := 0;
             end if;
            end if;
            if x.Port == 2 then //OFF
              if s.on then
                sout.on := false;
                sout.sigma := 0;
             end if;
            end if;
          end ext;

          function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
            stdEvent Y;
          algorithm
            if s.on then
                Y.Value := 1;
                Y.Type := 1;
                sendEvent(queue[1],Y);
            else
                Y.Value := -1;
                Y.Type := 1;
                sendEvent(queue[2],Y);
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Boolean on;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 100;  //first generation starts at time == 100
            out.on := false;
          end initst;
        end Control;

        package Alarm
          import DESLib.DEVSLib.SRC.*;
          model alarm
                     extends AtomicDEVS(numIn=6,numOut=3,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst;
            annotation (Diagram, Icon);
                    Interfaces.outPort SOUNDAlarm
                                     annotation (extent=[80,40; 100,60]);
                    Interfaces.inPort AlarmHOUR
                                   annotation (extent=[-100,10; -80,30]);
                    Interfaces.inPort AlarmMIN
                                   annotation (extent=[-100,-20; -80,0]);
                    Interfaces.inPort Activate
                                   annotation (extent=[-100,-68; -80,-48]);
                    Interfaces.inPort Deactivate
                                   annotation (extent=[-100,-100; -80,-80]);
                    Interfaces.outPort AlarmON
                                     annotation (extent=[80,-10; 100,10]);
                    Interfaces.outPort AlarmOFF
                                     annotation (extent=[80,-60; 100,-40]);
                    Interfaces.inPort ClockHOUR
                                   annotation (extent=[-100,80; -80,100]);
                    Interfaces.inPort ClockMIN
                                   annotation (extent=[-100,52; -80,72]);
          equation
            iEvent[1] = ClockHOUR.event;
            iQueue[1] = ClockHOUR.queue;
            iEvent[2] = ClockMIN.event;
            iQueue[2] = ClockMIN.queue;
            iEvent[3] = AlarmHOUR.event;
            iQueue[3] = AlarmHOUR.queue;
            iEvent[4] = AlarmMIN.event;
            iQueue[4] = AlarmMIN.queue;
            iEvent[5] = Activate.event;
            iQueue[5] = Activate.queue;
            iEvent[6] = Deactivate.event;
            iQueue[6] = Deactivate.queue;

            oEvent[1] = SOUNDAlarm.event;
            oQueue[1] = SOUNDAlarm.queue;
            oEvent[2] = AlarmON.event;
            oQueue[2] = AlarmON.queue;
            oEvent[3] = AlarmOFF.event;
            oQueue[3] = AlarmOFF.queue;

          end alarm;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.sigma := Modelica.Constants.inf;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            x := DEVSLib.SRC.getEvent(bag);
            sout := s;
            if x.Port == 1 then // clock hour value
              sout.hora_reloj := integer(x.Value);
            end if;
            if x.Port == 2 then // clock min value
              sout.min_reloj := integer(x.Value);
            end if;
            if x.Port == 3 then // alarm hour value
              sout.hora_alarm := integer(x.Value);
            end if;
            if x.Port == 4 then // alarm min value
              sout.min_alarm := integer(x.Value);
            end if;
            if x.Port == 5 then  //activate
              if not s.activo then
                sout.habilitar_salida := true;
                sout.activo := true;
                sout.sigma := 0;
              end if;
            elseif x.Port == 6 then  // deactivate
              if s.activo then
                sout.habilitar_salida := true;
                sout.activo := false;
                sout.sigma := 0;
              end if;
            elseif sout.activo and (sout.hora_reloj == sout.hora_alarm) and (sout.min_reloj == sout.min_alarm) then
              sout.habilitar_salida := false;
              sout.sigma := 0;
            end if;
          end ext;

          function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
            stdEvent Y;
          algorithm
            if s.habilitar_salida then
              if s.activo then
                // alarm on
                  Y.Type := 1;
                  Y.Value := 1;
                  sendEvent(queue[2],Y);
              else
                 // alarm off
                  Y.Type := 1;
                  Y.Value := 1;
                  sendEvent(queue[3],Y);
              end if;
            else
              if s.activo then
                // sound alarm
                  Y.Type := 1;
                  Y.Value := 1;
                  sendEvent(queue[1],Y);
              end if;
            end if;

          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Boolean activo;
            Boolean habilitar_salida;
            Integer hora_reloj;
            Integer min_reloj;
            Integer hora_alarm;
            Integer min_alarm;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 100;  //first generation starts at time == 100
            out.activo := false;
            out.habilitar_salida := false;
            out.hora_reloj := 0;
            out.min_reloj := 0;
            out.hora_alarm := 0;
            out.min_alarm := 0;
          end initst;
        end Alarm;

        package Hours
          import DESLib.DEVSLib.SRC.*;
          model hours
                     extends AtomicDEVS(numIn=2,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst;
            annotation (Diagram, Icon);
                    Interfaces.outPort Value
                                     annotation (extent=[80,0; 100,20]);
                    Interfaces.inPort Increment
                                   annotation (extent=[-100,40; -80,60]);
                    Interfaces.inPort Decrement
                                   annotation (extent=[-100,-40; -80,-20]);
          equation
            iEvent[1] = Increment.event;
            iQueue[1] = Increment.queue;
            iEvent[2] = Decrement.event;
            iQueue[2] = Decrement.queue;

            oEvent[1] = Value.event;
            oQueue[1] = Value.queue;

          end hours;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.sigma := Modelica.Constants.inf;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            x := DEVSLib.SRC.getEvent(bag);
            if x.Port == 1 then // increment
              sout.min := s.min +1;
              if sout.min >= 60 then
                sout.min := 0;
                sout.hora := s.hora +1;
                if sout.hora >= 12 then
                  sout.hora := 1;
                end if;
              end if;
              sout.sigma := 0;
            end if;
            if x.Port == 2 then // decrement
              sout.min := s.min -1;
              if sout.min < 0 then
                sout.min := 59;
                sout.hora := s.hora -1;
                if sout.hora <= 0 then
                  sout.hora := 12;
                end if;
              end if;
              sout.sigma := 0;
            end if;
          end ext;

          function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
            stdEvent Y;
          algorithm
            Y.Type := 1;
            Y.Value := s.hora;
            sendEvent(queue[1],Y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Integer hora;
            Integer min;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 100;  //first generation starts at time == 100
            out.hora := 0;
            out.min := 0;
          end initst;
        end Hours;

        package Minutes
          import DESLib.DEVSLib.SRC.*;
          model minutes
                     extends AtomicDEVS(numIn=3,numOut = 3,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst;
            annotation (Diagram, Icon);
                    Interfaces.outPort OUT_Decrement
                                     annotation (extent=[80,0; 100,20]);
                    Interfaces.inPort IN_Increment
                                   annotation (extent=[-100,60; -80,80]);
                    Interfaces.inPort TIC
                                   annotation (extent=[-100,0; -80,20]);
                    Interfaces.inPort IN_Decrement
                                   annotation (extent=[-100,-60; -80,-40]);
                    Interfaces.outPort Value
                                     annotation (extent=[80,-60; 100,-40]);
                    Interfaces.outPort OUT_Increment
                                     annotation (extent=[80,60; 100,80]);
          equation
            iEvent[1] = IN_Increment.event;
            iQueue[1] = IN_Increment.queue;
            if cardinality(TIC) == 0 then
              iEvent[2] =  0;
              //iQueue[2] =  0;
            else
              iEvent[2] = TIC.event;
              iQueue[2] = TIC.queue;
            end if;
            iEvent[3] = IN_Decrement.event;
            iQueue[3] = IN_Decrement.queue;

            oEvent[1] = OUT_Increment.event;
            oQueue[1] = OUT_Increment.queue;
            oEvent[2] = OUT_Decrement.event;
            oQueue[2] = OUT_Decrement.queue;
            oEvent[3] = Value.event;
            oQueue[3] = Value.queue;

          end minutes;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.evento_saliente := 0;
            sout.sigma := Modelica.Constants.inf;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            x := DEVSLib.SRC.getEvent(bag);
            sout := s;
            if x.Port == 2 then // tic
              sout.tics := s.tics +1;
              if sout.tics >= 10 then
                sout.tics := 0;
                sout.minutos := s.minutos +1;
                sout.evento_saliente := 1;
                if sout.minutos >= 60 then
                  sout.minutos := 0;
                end if;
              end if;
              sout.sigma := 0;
            end if;
            if x.Port == 1 then // increment
              sout.minutos := s.minutos +1;
              sout.evento_saliente := 1;
              if sout.minutos >= 60 then
                sout.minutos := 0;
              end if;
              sout.sigma := 0;
            end if;
            if x.Port == 3 then // decrement
              sout.minutos := s.minutos -1;
              sout.evento_saliente := 2;
              if sout.minutos < 0 then
                sout.minutos := 59;
              end if;
              sout.sigma := 0;
            end if;
          end ext;

          function out "Output Function"
             input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
            stdEvent Y;
          algorithm
            if s.evento_saliente == 1 then
                Y.Type := 1;
                Y.Value := 1;
                sendEvent(queue[1],Y);
            elseif s.evento_saliente == 2 then
                Y.Type := 1;
                Y.Value := 1;
                sendEvent(queue[2],Y);
            end if;
            Y.Type := 1;
            Y.Value := s.minutos;
            sendEvent(queue[3],Y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Integer evento_saliente;
            Integer minutos;
            Integer tics;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 100;  //first generation starts at time == 100
            out.evento_saliente := 0;
            out.minutos := 0;
            out.tics := 0;
          end initst;
        end Minutes;

        model Motor
          import DESLib.DEVSLib.SRC.*;
                  // parameter Real p1 = 1;

                  Interfaces.inPort TurnON
                                 annotation (extent=[-100,40; -80,60]);
                  Interfaces.outPort ClockON
                                   annotation (extent=[80,40; 100,60]);
                  annotation (Diagram);
                  Interfaces.inPort TurnOFF
                                 annotation (extent=[-100,-60; -80,-40]);
                  Interfaces.outPort ClockOFF
                                   annotation (extent=[80,-10; 100,10]);
                  Interfaces.outPort TIC
                                   annotation (extent=[80,-60; 100,-40]);
                  Pendulum.pendulum pendulum annotation (extent=[28,-40; 48,-20]);
                  Control.control control
                    annotation (extent=[-62,12; -28,32]);
                  DupEvent.dupevent ClockONevent       annotation (extent=[-10,34; 10,54]);
                  DupEvent.dupevent ClockOFFevent       annotation (extent=[-10,0; 10,20]);
        equation
                  connect(TurnON, control.TurnON) annotation (points=[-90,50; -68,50;
                        -68,27; -60.3,27], style(color=3, rgbcolor={0,0,255}));
                  connect(TurnOFF, control.TurnOFF) annotation (points=[-90,-50; -70,
                        -50; -70,17; -60.3,17], style(color=3, rgbcolor={0,0,255}));
                  connect(control.ClockON, ClockONevent.in1) annotation (points=[-29.7,27;
                        -13.5,27; -13.5,44; -2.6,44],         style(color=3, rgbcolor={0,0,
                          255}));
                  connect(control.ClockOFF, ClockOFFevent.in1) annotation (points=[-29.7,17;
                        -18.5,17; -18.5,10; -2.6,10],         style(color=3, rgbcolor={0,0,
                          255}));
                  connect(ClockONevent.out1, ClockON) annotation (points=[0.6,46;
                49.5,46; 49.5,50; 90,50],
                                    style(color=3, rgbcolor={0,0,255}));
                  connect(ClockONevent.out2, pendulum.TurnON) annotation (points=[0.6,42;
                16,42; 16,-25; 29,-25],  style(color=3, rgbcolor={0,0,255}));
                  connect(pendulum.TIC, TIC) annotation (points=[47,-29; 65.5,-29; 65.5,-50;
                        90,-50], style(color=3, rgbcolor={0,0,255}));
                  connect(ClockOFFevent.out1, ClockOFF) annotation (points=[0.6,12;
                68,12; 68,0; 90,0],
                               style(color=3, rgbcolor={0,0,255}));
                  connect(ClockOFFevent.out2, pendulum.TurnOFF) annotation (points=[0.6,8;
                12,8; 12,-33; 29,-33],   style(color=3, rgbcolor={0,0,255}));

        end Motor;

        model HandsControl
          import DESLib.DEVSLib.SRC.*;
                  // parameter Real p1 = 1;

                  Interfaces.inPort Increment
                                 annotation (extent=[-100,40; -80,60]);
                  Interfaces.outPort MINValue
                                   annotation (extent=[80,20; 100,40]);
                  annotation (Diagram);
                  Interfaces.inPort Decrement
                                 annotation (extent=[-100,0; -80,20]);
                  Interfaces.inPort TIC
                                 annotation (extent=[-100,-60; -80,-40]);
                  Interfaces.outPort HOURValue
                                   annotation (extent=[80,-40; 100,-20]);
                  Minutes.minutes minutes annotation (extent=[-28,14; 30,70]);
                  Hours.hours hours annotation (extent=[-16,-72; 36,-18]);
        equation
                  connect(Increment, minutes.IN_Increment) annotation (points=[-90,50;
                        -58,50; -58,61.6; -25.1,61.6], style(color=3, rgbcolor={0,0,255}));
                  connect(Decrement, minutes.IN_Decrement) annotation (points=[-90,10;
                        -58,10; -58,28; -25.1,28], style(color=3, rgbcolor={0,0,255}));
                  connect(TIC, minutes.TIC) annotation (points=[-90,-50; -68,-50; -68,-46;
                        -44,-46; -44,44.8; -25.1,44.8], style(color=3, rgbcolor={0,0,255}));
                  connect(minutes.OUT_Increment, hours.Increment) annotation (points=[27.1,
                61.6; 48,61.6; 48,0; -28,0; -28,-31.5; -13.4,-31.5],               style(
                        color=3, rgbcolor={0,0,255}));
                  connect(minutes.OUT_Decrement, hours.Decrement) annotation (points=[27.1,
                44.8; 40,44.8; 40,-14; -24,-14; -24,-53.1; -13.4,-53.1],               style(
                        color=3, rgbcolor={0,0,255}));
                  connect(minutes.Value, MINValue) annotation (points=[27.1,28;
                58.55,28; 58.55,30; 90,30],
                                          style(color=3, rgbcolor={0,0,255}));
                  connect(hours.Value, HOURValue) annotation (points=[33.4,
                -42.3; 46.4,-42.3; 46.4,-30; 90,-30],
                                                  style(color=3, rgbcolor={0,0,255}));

        end HandsControl;

        model AlarmControl
         import DESLib.DEVSLib.SRC.*;
                  // parameter Real p1 = 1;

                  Interfaces.inPort activate
                                 annotation (extent=[-100,80; -80,100]);
                  Interfaces.outPort SoundAlarm
                                   annotation (extent=[80,60; 100,80]);
                  annotation (Diagram);
                  Interfaces.inPort deactivate
                                 annotation (extent=[-100,52; -80,72]);
                  Interfaces.inPort clockHOUR
                                 annotation (extent=[-100,14; -80,34]);
                  Interfaces.inPort clockMIN
                                 annotation (extent=[-100,-18; -80,2]);
                  Interfaces.inPort increment
                                 annotation (extent=[-100,-66; -80,-46]);
                  Interfaces.inPort decrement
                                 annotation (extent=[-100,-100; -80,-80]);
                  Interfaces.outPort AlarmON
                                   annotation (extent=[80,24; 100,44]);
                  Interfaces.outPort AlarmOFF
                                   annotation (extent=[80,-14; 100,6]);
                  Interfaces.outPort AlarmMIN
                                   annotation (extent=[80,-52; 100,-32]);
                  Interfaces.outPort AlarmHOUR
                                   annotation (extent=[80,-90; 100,-70]);
                  HandsControl Hands         annotation (extent=[-52,-96; 8,-34]);
                  Alarm.alarm alarm annotation (extent=[-16,10; 44,68]);
                  AuxModels.DUP HourValue           annotation (extent=[24,-88; 44,-68]);
                  AuxModels.DUP MinValue           annotation (extent=[24,-62; 44,-42]);

        equation
         connect(activate, alarm.Activate) annotation (points=[-90,90; -74,90; -74,
                        22.18; -13,22.18], style(color=3, rgbcolor={0,0,255}));
                  connect(deactivate, alarm.Deactivate) annotation (points=[-90,62;
                -76,62; -76,12.9; -13,12.9], style(color=3, rgbcolor={0,0,255}));
                  connect(clockHOUR, alarm.ClockHOUR) annotation (points=[-90,24; -78,24;
                        -78,65.1; -13,65.1], style(color=3, rgbcolor={0,0,255}));
                  connect(clockMIN, alarm.ClockMIN) annotation (points=[-90,-8;
                -60,-8; -60,56.98; -13,56.98],
                                           style(color=3, rgbcolor={0,0,255}));
                  connect(increment, Hands.Increment)         annotation (points=[-90,-56;
                        -70,-56; -70,-49.5; -49,-49.5], style(color=3, rgbcolor={0,0,255}));
                  connect(decrement, Hands.Decrement)         annotation (points=[-90,-90;
                        -70,-90; -70,-61.9; -49,-61.9], style(color=3, rgbcolor={0,0,255}));
                  connect(Hands.MINValue, MinValue.in1)         annotation (points=[5,-55.7;
                        15.5,-55.7; 15.5,-52; 31.4,-52],
                                                       style(color=3, rgbcolor={0,0,255}));
                  connect(Hands.HOURValue, HourValue.in1)         annotation (points=[5,-74.3;
                        15.5,-74.3; 15.5,-78; 31.4,-78],
                                                       style(color=3, rgbcolor={0,0,255}));
                  connect(MinValue.out1, alarm.AlarmMIN) annotation (points=[34.6,-50;
                        52,-50; 52,-4; -38,-4; -38,36.1; -13,36.1],
                                                            style(color=3, rgbcolor={0,0,255}));
                  connect(MinValue.out2, AlarmMIN) annotation (points=[34.6,-54; 66,-54;
                        66,-42; 90,-42],
                                 style(color=3, rgbcolor={0,0,255}));
                  connect(HourValue.out2, AlarmHOUR) annotation (points=[34.6,-80; 65.5,
                        -80; 65.5,-80; 90,-80],
                                           style(color=3, rgbcolor={0,0,255}));
                  connect(HourValue.out1, alarm.AlarmHOUR) annotation (points=[34.6,-76;
                        48,-76; 48,-10; -32,-10; -32,44.8; -13,44.8],
                                                                   style(color=3, rgbcolor={0,
                          0,255}));
                  connect(alarm.SOUNDAlarm, SoundAlarm) annotation (points=[41,53.5; 64.5,
                        53.5; 64.5,70; 90,70], style(color=3, rgbcolor={0,0,255}));
                  connect(alarm.AlarmON, AlarmON) annotation (points=[41,39; 64.5,39; 64.5,
                        34; 90,34], style(color=3, rgbcolor={0,0,255}));
                  connect(alarm.AlarmOFF, AlarmOFF) annotation (points=[41,24.5; 63.5,24.5;
                        63.5,-4; 90,-4], style(color=3, rgbcolor={0,0,255}));
        end AlarmControl;

        model Clock
          import DESLib.DEVSLib.SRC.*;
                  // parameter Real p1 = 1;

                  Interfaces.inPort TurnON
                                 annotation (extent=[-100,80; -80,100]);
                  Interfaces.outPort clockON
                                   annotation (extent=[80,80; 100,100]);
                  annotation (Diagram);
                  AuxModels.DUP MinValue           annotation (extent=[14,10; 34,30]);
                  AuxModels.DUP HourValue           annotation (extent=[14,-12; 34,8]);
                  Motor motor annotation (extent=[-56,60; -36,80]);
                  HandsControl handsC        annotation (extent=[-38,-12; 0,30]);
                  AlarmControl AlarmC       annotation (extent=[10,-78; 54,-32]);
                  Interfaces.inPort TurnOFF
                                 annotation (extent=[-100,54; -80,74]);
                  Interfaces.inPort Increment
                                 annotation (extent=[-100,20; -80,40]);
                  Interfaces.inPort Decrement
                                 annotation (extent=[-100,-10; -80,10]);
                  Interfaces.inPort Activate
                                 annotation (extent=[-100,-50; -80,-30]);
                  Interfaces.inPort Deactivate
                                 annotation (extent=[-100,-80; -80,-60]);
                  Interfaces.inPort INCAlarm
                                 annotation (extent=[-60,-100; -40,-80]);
                  Interfaces.inPort DECAlarm
                                 annotation (extent=[-28,-100; -8,-80]);
                  Interfaces.outPort clockOFF
                                   annotation (extent=[80,60; 100,80]);
                  Interfaces.outPort clockHOUR
                                   annotation (extent=[80,40; 100,60]);
                  Interfaces.outPort clockMIN
                                   annotation (extent=[80,20; 100,40]);
                  Interfaces.outPort SoundAlarm
                                   annotation (extent=[80,-20; 100,0]);
                  Interfaces.outPort AlarmON
                                   annotation (extent=[80,-40; 100,-20]);
                  Interfaces.outPort AlarmOFF
                                   annotation (extent=[80,-60; 100,-40]);
                  Interfaces.outPort AlarmMIN
                                   annotation (extent=[80,-80; 100,-60]);
                  Interfaces.outPort AlarmHOUR
                                   annotation (extent=[80,-100; 100,-80]);

        equation
         connect(handsC.MINValue, MinValue.in1)        annotation (points=[-1.9,
                15.3; 7.05,15.3; 7.05,20; 21.4,20], style(color=3, rgbcolor={0,0,255}));
                  connect(handsC.HOURValue, HourValue.in1)        annotation (points=[-1.9,2.7;
                7.05,2.7; 7.05,-2; 21.4,-2],            style(color=3, rgbcolor={0,0,255}));
                  connect(TurnON, motor.TurnON) annotation (points=[-90,90; -74,90; -74,75;
                        -55,75], style(color=3, rgbcolor={0,0,255}));
                  connect(TurnOFF, motor.TurnOFF) annotation (points=[-90,64; -72.5,64; -72.5,
                        65; -55,65], style(color=3, rgbcolor={0,0,255}));
                  connect(motor.ClockON, clockON) annotation (points=[-37,75; 23.5,75; 23.5,
                        90; 90,90], style(color=3, rgbcolor={0,0,255}));
                  connect(motor.ClockOFF, clockOFF) annotation (points=[-37,70; 90,70],
                                style(color=3, rgbcolor={0,0,255}));
                  connect(motor.TIC, handsC.TIC)        annotation (points=[-37,65; -30,65;
                        -30,34; -52,34; -52,-1.5; -36.1,-1.5], style(color=3, rgbcolor={0,0,
                          255}));
                  connect(Increment, handsC.Increment)        annotation (points=[-90,30; -62,
                        30; -62,19.5; -36.1,19.5], style(color=3, rgbcolor={0,0,255}));
                  connect(Decrement, handsC.Decrement)        annotation (points=[-90,0; -64,
                        0; -64,11.1; -36.1,11.1], style(color=3, rgbcolor={0,0,255}));
                  connect(HourValue.out1, clockHOUR) annotation (points=[24.6,0; 56,0;
                        56,50; 90,50],
                                style(color=3, rgbcolor={0,0,255}));
                  connect(MinValue.out1, clockMIN) annotation (points=[24.6,22; 63.5,22;
                        63.5,30; 90,30],
                                    style(color=3, rgbcolor={0,0,255}));
                  connect(Activate, AlarmC.activate)       annotation (points=[-90,-40; -40,
                        -40; -40,-34.3; 12.2,-34.3], style(color=3, rgbcolor={0,0,255}));
                  connect(Deactivate, AlarmC.deactivate)       annotation (points=[-90,-70;
                        -38,-70; -38,-40.74; 12.2,-40.74], style(color=3, rgbcolor={0,0,255}));
                  connect(HourValue.out2, AlarmC.clockHOUR)       annotation (points=[24.6,-4;
                        40,-4; 40,-26; 4,-26; 4,-49.48; 12.2,-49.48], style(color=3, rgbcolor=
                         {0,0,255}));
                  connect(MinValue.out2, AlarmC.clockMIN)       annotation (points=[24.6,18;
                        48,18; 48,-20; -4,-20; -4,-56.84; 12.2,-56.84], style(color=3,
                        rgbcolor={0,0,255}));
                  connect(INCAlarm, AlarmC.increment)       annotation (points=[-50,-90; -50,
                        -72; -32,-72; -32,-67.88; 12.2,-67.88], style(color=3, rgbcolor={0,0,
                          255}));
                  connect(DECAlarm, AlarmC.decrement)       annotation (points=[-18,-90; -18,
                        -75.7; 12.2,-75.7], style(color=3, rgbcolor={0,0,255}));
                  connect(AlarmC.SoundAlarm, SoundAlarm)       annotation (points=[51.8,
                        -38.9; 66,-38.9; 66,-10; 90,-10], style(color=3, rgbcolor={0,0,255}));
                  connect(AlarmC.AlarmON, AlarmON)       annotation (points=[51.8,-47.18; 70,
                        -47.18; 70,-30; 90,-30], style(color=3, rgbcolor={0,0,255}));
                  connect(AlarmC.AlarmOFF, AlarmOFF)       annotation (points=[51.8,-55.92;
                        72,-55.92; 72,-50; 90,-50], style(color=3, rgbcolor={0,0,255}));
                  connect(AlarmC.AlarmMIN, AlarmMIN)       annotation (points=[51.8,-64.66;
                        68,-64.66; 68,-70; 90,-70], style(color=3, rgbcolor={0,0,255}));
                  connect(AlarmC.AlarmHOUR, AlarmHOUR)       annotation (points=[51.8,-73.4;
                        62,-73.4; 62,-90; 90,-90], style(color=3, rgbcolor={0,0,255}));

        end Clock;

        model testClock
          Real hour = clockHOUR.S.value;
          Real min = clockMIN.S.value;
          Real alarm = SoundAlarm.S.phase;

         AuxModels.Generator ON(period=1,
                    numGenerations=1,
                    firstGeneration=1)
                    annotation (extent=[-100,80; -80,100]);
                  AuxModels.Generator OFF(outValue=-1, period=100,
                    firstGeneration=1e8)  annotation (extent=[-100,58; -80,78]);
                  AuxModels.Display clockON     annotation (extent=[80,80; 100,100]);
                  annotation (Diagram,
                    experiment(StopTime=100),
                    experimentSetupOutput);
                  Clock clock annotation (extent=[-70,-80; 72,100]);
                  AuxModels.Display clockOFF    annotation (extent=[80,60; 100,80]);
                  AuxModels.Display clockHOUR   annotation (extent=[80,40; 100,60]);
                  AuxModels.Display clockMIN    annotation (extent=[80,20; 100,40]);
                  AuxModels.Display SoundAlarm  annotation (extent=[80,-6; 100,14]);
                  AuxModels.Display AlarmON     annotation (extent=[80,-26; 100,-6]);
                  AuxModels.Display AlarmOFF    annotation (extent=[80,-46; 100,-26]);
                  AuxModels.Display AlarmMIN    annotation (extent=[80,-66; 100,-46]);
                  AuxModels.Display AlarmHOUR   annotation (extent=[80,-86; 100,-66]);
                  AuxModels.Generator INC(period=1, firstGeneration=1e8)
                    annotation (extent=[-100,26; -80,46]);
                  AuxModels.Generator DEC(outValue=-1, period=100,
                    firstGeneration=1e8)  annotation (extent=[-100,0; -80,20]);
                  AuxModels.Generator ACT(period=100,
                    firstGeneration=10,
                    numGenerations=1)
                    annotation (extent=[-100,-36; -80,-16]);
                  AuxModels.Generator DEACT(outValue=-1,
                    period=1000,
                    numGenerations=1,
                    firstGeneration=1e8)  annotation (extent=[-100,-62; -80,-42]);
                  AuxModels.Generator IncAlarm(
                    firstGeneration=1,
                    period=1,
                    numGenerations=1)
                    annotation (extent=[-62,-100; -42,-80]);
                  AuxModels.Generator DecAlarm(outValue=-1, period=100,
                    firstGeneration=1e8)  annotation (extent=[-32,-100; -12,-80]);

        equation
        connect(clock.clockON, clockON.inPort1) annotation (points=[64.9,91; 72.95,
                        91; 72.95,90; 81,90],
                             style(color=3, rgbcolor={0,0,255}));
                  connect(clock.clockOFF, clockOFF.inPort1) annotation (points=[64.9,73;
                        72.45,73; 72.45,70; 81,70], style(color=3, rgbcolor={0,0,255}));
                  connect(clock.clockHOUR, clockHOUR.inPort1) annotation (points=[64.9,55;
                        73.45,55; 73.45,50; 81,50], style(color=3, rgbcolor={0,0,255}));
                  connect(clock.clockMIN, clockMIN.inPort1) annotation (points=[64.9,37;
                        73.45,37; 73.45,30; 81,30], style(color=3, rgbcolor={0,0,255}));
                  connect(clock.SoundAlarm, SoundAlarm.inPort1) annotation (points=[64.9,1;
                        73.45,1; 73.45,4; 81,4],    style(color=3, rgbcolor={0,0,255}));
                  connect(clock.AlarmON, AlarmON.inPort1) annotation (points=[64.9,-17; 73.45,
                        -17; 73.45,-16; 81,-16],       style(color=3, rgbcolor={0,0,255}));
                  connect(clock.AlarmOFF, AlarmOFF.inPort1) annotation (points=[64.9,-35;
                        72.95,-35; 72.95,-36; 81,-36], style(color=3, rgbcolor={0,0,255}));
                  connect(clock.AlarmMIN, AlarmMIN.inPort1) annotation (points=[64.9,-53;
                        73.45,-53; 73.45,-56; 81,-56], style(color=3, rgbcolor={0,0,255}));
                  connect(clock.AlarmHOUR, AlarmHOUR.inPort1) annotation (points=[64.9,-71;
                        73.45,-71; 73.45,-76; 81,-76],      style(color=3, rgbcolor={0,0,
                          255}));
                  connect(IncAlarm.outPort1,clock. INCAlarm) annotation (points=[-43,-90; -43,
                        -82.5; -34.5,-82.5; -34.5,-71],     style(color=3, rgbcolor={0,0,
                          255}));
                  connect(DecAlarm.outPort1,clock. DECAlarm) annotation (points=[-13,-90; -13,
                        -83.5; -11.78,-83.5; -11.78,-71],     style(color=3, rgbcolor={0,
                          0,255}));
                  connect(ON.outPort1,clock. TurnON) annotation (points=[-81,90; -71.95,90;
                        -71.95,91; -62.9,91],
                      style(color=3, rgbcolor={0,0,255}));
                  connect(OFF.outPort1,clock. TurnOFF) annotation (points=[-81,68; -73.5,68;
                        -73.5,67.6; -62.9,67.6],     style(color=3, rgbcolor={0,0,255}));
                  connect(INC.outPort1,clock. Increment) annotation (points=[-81,36; -74.5,36;
                        -74.5,37; -62.9,37],           style(color=3, rgbcolor={0,0,255}));
                  connect(DEC.outPort1,clock. Decrement) annotation (points=[-81,10; -62.9,10],
                                                       style(color=3, rgbcolor={0,0,255}));
                  connect(ACT.outPort1,clock. Activate) annotation (points=[-81,-26;
                        -62.9,-26],                       style(color=3, rgbcolor={0,0,
                          255}));
                  connect(DEACT.outPort1,clock. Deactivate) annotation (points=[-81,-52; -71.5,
                        -52; -71.5,-53; -62.9,-53],       style(color=3, rgbcolor={0,0,
                          255}));
        end testClock;

        model ClockNoAlarm
          import DESLib.DEVSLib.SRC.*;
                  // parameter Real p1 = 1;

                  Interfaces.inPort TurnON
                                 annotation (extent=[-100,32; -80,52]);
                  Interfaces.outPort clockON
                                   annotation (extent=[80,32; 100,52]);
                  annotation (Diagram);
                  Motor motor annotation (extent=[-56,12; -36,32]);
                  HandsControl controlAgujas annotation (extent=[-38,-60; 0,-18]);
                  Interfaces.inPort TurnOFF
                                 annotation (extent=[-100,6; -80,26]);
                  Interfaces.inPort Increment
                                 annotation (extent=[-100,-28; -80,-8]);
                  Interfaces.inPort Decrement
                                 annotation (extent=[-100,-58; -80,-38]);
                  Interfaces.outPort clockOFF
                                   annotation (extent=[80,12; 100,32]);
                  Interfaces.outPort clockHOUR
                                   annotation (extent=[80,-8; 100,12]);
                  Interfaces.outPort clockMIN
                                   annotation (extent=[80,-28; 100,-8]);
        equation
          connect(TurnON, motor.TurnON) annotation (points=[-90,42; -74,42; -74,
                        27; -55,27],
                                 style(color=3, rgbcolor={0,0,255}));
                  connect(TurnOFF, motor.TurnOFF) annotation (points=[-90,16; -72.5,16;
                        -72.5,17; -55,17],
                                     style(color=3, rgbcolor={0,0,255}));
                  connect(motor.ClockON, clockON) annotation (points=[-37,27; 23.5,27;
                        23.5,42; 90,42],
                                    style(color=3, rgbcolor={0,0,255}));
                  connect(motor.ClockOFF, clockOFF) annotation (points=[-37,22; 90,22],
                                style(color=3, rgbcolor={0,0,255}));
                  connect(motor.TIC, controlAgujas.TIC) annotation (points=[-37,17; -30,
                        17; -30,-14; -52,-14; -52,-49.5; -36.1,-49.5],
                                                               style(color=3, rgbcolor={0,0,
                          255}));
                  connect(Increment, controlAgujas.Increment) annotation (points=[-90,-18;
                        -62,-18; -62,-28.5; -36.1,-28.5],
                                                   style(color=3, rgbcolor={0,0,255}));
                  connect(Decrement, controlAgujas.Decrement) annotation (points=[-90,-48;
                        -64,-48; -64,-36.9; -36.1,-36.9],
                                                  style(color=3, rgbcolor={0,0,255}));
                  connect(controlAgujas.HOURValue, clockHOUR) annotation (points=[-1.9,
                -45.3; 70,-45.3; 70,2; 90,2],  style(color=3, rgbcolor={0,0,255}));
                  connect(controlAgujas.MINValue, clockMIN) annotation (points=[-1.9,
                -32.7; 41.05,-32.7; 41.05,-18; 90,-18],
                                                      style(color=3, rgbcolor={0,0,255}));
        end ClockNoAlarm;

        model testNoAlarm
           Real hour = clockHOUR.S.value;
                  Real min = clockMIN.S.value;

                  AuxModels.Generator ON(period=1, numGenerations=1)
                    annotation (extent=[-100,36; -80,56]);
                  AuxModels.Generator OFF(outValue=-1, firstGeneration=1e8)
                                          annotation (extent=[-100,16; -80,36]);
                  AuxModels.Display clockON     annotation (extent=[80,40; 100,60]);
                  annotation (Diagram,
                    experiment(StopTime=100000),
                    experimentSetupOutput);
                  AuxModels.Display clockOFF    annotation (extent=[80,20; 100,40]);
                  AuxModels.Display clockHOUR   annotation (extent=[80,0; 100,20]);
                  AuxModels.Display clockMIN    annotation (extent=[80,-20; 100,0]);
                  AuxModels.Generator INC(firstGeneration=1e8)
                    annotation (extent=[-100,-20; -80,0]);
                  AuxModels.Generator DEC(outValue=-1, firstGeneration=1e8)
                                          annotation (extent=[-100,-40; -80,-20]);
                  ClockNoAlarm clockNoAlarm  annotation (extent=[-72,-76; 68,96]);
        equation
         connect(clockNoAlarm.clockON, clockON.inPort1)  annotation (points=[61,46.12;
                71.5,46.12; 71.5,50; 81,50],              style(color=3, rgbcolor={0,0,
                          255}));
                  connect(clockNoAlarm.clockOFF, clockOFF.inPort1)  annotation (points=[61,28.92;
                71.5,28.92; 71.5,30; 81,30],                 style(color=3, rgbcolor={0,0,
                          255}));
                  connect(clockNoAlarm.clockHOUR, clockHOUR.inPort1)  annotation (points=[61,11.72;
                71.5,11.72; 71.5,10; 81,10],             style(color=3, rgbcolor={0,0,255}));
                  connect(clockNoAlarm.clockMIN, clockMIN.inPort1)  annotation (points=[61,-5.48;
                71.5,-5.48; 71.5,-10; 81,-10],               style(color=3, rgbcolor={0,0,
                          255}));
                  connect(INC.outPort1, clockNoAlarm.Increment)  annotation (points=[-81,-10;
                        -72.5,-10; -72.5,-5.48; -65,-5.48],  style(color=3, rgbcolor={0,0,
                          255}));
                  connect(DEC.outPort1, clockNoAlarm.Decrement)  annotation (points=[-81,-30;
                        -72.5,-30; -72.5,-31.28; -65,-31.28],
                                                         style(color=3, rgbcolor={0,0,255}));
                  connect(ON.outPort1, clockNoAlarm.TurnON)  annotation (points=[-81,46; -72.5,
                        46; -72.5,46.12; -65,46.12],     style(color=3, rgbcolor={0,0,255}));
                  connect(OFF.outPort1, clockNoAlarm.TurnOFF)  annotation (points=[-81,26;
                -72.5,26; -72.5,23.76; -65,23.76],             style(color=3, rgbcolor={0,
                          0,255}));

        end testNoAlarm;
      end Clock2;

      package CarFactory "Model of a car factory"
      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package includes the implementation of a simple car factory.
A complete description of the system and its implementation using the CD++ environment can be found <a href=http://www.sce.carleton.ca/faculty/wainer/wbgraf/samples/auto.zip>here</a>
</p>
<p>
The production of one car is divided into:
<ul>
<li> Engine.
<li> Chassis.
<li> Body.
<li> Transmission.
</ul>
All these pieces are assembled in the FinalAssembly subsystem.
</p>

<h3><font color=\"#008000\">Simulation</font></h3>
The system is implemented in the carfactory model. Two variables have been included to facilitate the observation of the results.
<ul>
<li><i>petitions</i> shows the rate for new car manufacturing petitions.
<li><i>assembled</i> shows the production of new cars.
</ul>

<br>
</HTML>
"));
        package Engine
          package EngineAssembly

          annotation(preferedView="info",
            Window(
              x=0.02,
              y=0.01,
              width=0.2,
              height=0.57,
              library=1,
              autolayout=1),
            version="2.2",
            versionDate="2005-04-15",
            Settings(NewStateSelection=true),
            Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica library for modeling and simulation systems using the DEVS formalism.
</p>


</HTML>
"));
            import DESLib.DEVSLib.SRC.*;
            model engineAssembly
                       extends AtomicDEVS(numIn=3,
              redeclare record State = st);
              parameter Real process_time = 2;
              redeclare function Fcon = con;
              redeclare function Fext = ext;
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(pt=process_time);
              annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80],
                      style(      color=3, rgbcolor={0,0,255})), Text(
                              extent=[-76,44; 74,-36],
                              style(color=3, rgbcolor={0,0,255}),
                              string="%name")));
                        Interfaces.outPort OUT          annotation (extent=[80,-10; 100,10]);
                        Interfaces.inPort INPISTON    annotation (extent=[-100,50; -80,70]);
                        Interfaces.inPort INBODY      annotation (extent=[-100,-10; -80,10]);
                        Interfaces.inPort DONE        annotation (extent=[-100,-68; -80,-48]);
            equation
              iEvent[1] = INPISTON.event;
              iQueue[1] = INPISTON.queue;
              iEvent[2] = INBODY.event;
              iQueue[2] = INBODY.queue;
              iEvent[3] = DONE.event;
              iQueue[3] = DONE.queue;

              oQueue[1] = OUT.queue;
              oEvent[1] = OUT.event;
            end engineAssembly;

            function con "Confluent Transtition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            algorithm
              sout := ext(int(s),e,bag);
            end con;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              sout.elements_piston := sout.elements_piston -4;
              sout.elements_body := sout.elements_body -1;
              sout.phase := 1;
              sout.sigma := Modelica.Constants.inf;
            end int;

            function ext "External Transition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            protected
              Integer numreceived;
              stdEvent x;
            algorithm
              sout := s;
              x := DEVSLib.SRC.getEvent(bag);
              if x.Port == 1 then // new piston
                sout.elements_piston := sout.elements_piston +1;
                if sout.phase == 1 and sout.elements_piston >= 4 and sout.elements_body >= 1 then
                  sout.phase := 2;
                  sout.sigma := sout.pt;
                elseif sout.phase == 2 then
                  sout.sigma := sout.pt - e;
                else
                  sout.sigma := Modelica.Constants.inf;
                end if;
              elseif x.Port == 2 then // new body
                sout.elements_body := sout.elements_body +1;
                if sout.phase == 1 and sout.elements_piston >= 4 and sout.elements_body >= 1 then
                  sout.phase := 2;
                  sout.sigma := sout.pt;
                elseif sout.phase == 2 then
                  sout.sigma := sout.pt - e;
                else
                  sout.sigma := Modelica.Constants.inf;
                end if;
              else // done
                if sout.phase == 1 and sout.elements_piston >= 4 and sout.elements_body >= 1 then
                  sout.phase := 2;
                  sout.sigma := sout.pt;
                else
                  sout.phase := 1;
                  sout.sigma := Modelica.Constants.inf;
                end if;
              end if;
            end ext;

            function out "Output Function"
               input st s;
                input Integer queue[nports];
                input Integer nports;
            protected
              stdEvent y;
            algorithm
              y.Value := 1;
              y.Type := 1;
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase;
              Real sigma;
              Integer elements_piston;
              Integer elements_body;
              Real pt;
            end st;

            function initst
              input Real pt;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := Modelica.Constants.inf; // first internal event never happens
              out.elements_piston :=0;
              out.elements_body :=0;
              out.pt :=pt;
            end initst;
          end EngineAssembly;

          package Piston

          annotation(preferedView="info",
            Window(
              x=0.02,
              y=0.01,
              width=0.2,
              height=0.57,
              library=1,
              autolayout=1),
            version="2.2",
            versionDate="2005-04-15",
            Settings(NewStateSelection=true),
            Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica library for modeling and simulation systems using the DEVS formalism.
</p>


</HTML>
"));
            import DESLib.DEVSLib.SRC.*;
            model piston
                       extends AtomicDEVS(numIn=2,
              redeclare record State = st);
              redeclare function Fcon = con;
              redeclare function Fext = ext;
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(pt = process_time);
             annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                                  color=3, rgbcolor={0,0,255})), Text(
                              extent=[-76,44; 74,-36],
                              style(color=3, rgbcolor={0,0,255}),
                              string="%name")));
                        parameter Real process_time = 1;
                        Interfaces.outPort OUT          annotation (extent=[80,-10; 100,10]);
                        Interfaces.inPort IN          annotation (extent=[-100,30; -80,50]);
                        Interfaces.inPort DONE        annotation (extent=[-100,-50; -80,-30]);

            equation
              iEvent[1] = IN.event;
              iQueue[1] = IN.queue;
              iEvent[2] = DONE.event;
              iQueue[2] = DONE.queue;

              oEvent[1] = OUT.event;
              oQueue[1] = OUT.queue;
            end piston;

            function con "Confluent Transtition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            algorithm
              sout := ext(int(s),e,bag);
            end con;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              sout.elements := sout.elements -1;
              sout.phase := 1;
              sout.sigma :=  Modelica.Constants.inf;

            end int;

            function ext "External Transition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            protected
              Integer numreceived;
              stdEvent x;
            algorithm
              sout := s;
              x := DEVSLib.SRC.getEvent(bag);
              if x.Port == 1 then // IN
                sout.elements := sout.elements +4;
                if sout.phase == 1 then // start producing new elements
                  sout.phase := 2;
                  sout.sigma := sout.pt;
                elseif sout.phase == 2 then // element currently being produced.
                  sout.sigma := sout.pt - e;
                else
                  sout.sigma := Modelica.Constants.inf;
                end if;
              else // DONE
                if sout.elements == 0 then
                  sout.sigma := Modelica.Constants.inf;  // stop producing
                  sout.phase := 1;
                else
                  sout.sigma := sout.pt; // produce next element.
                end if;
              end if;
            end ext;

            function out "Output Function"
               input st s;
                input Integer queue[nports];
                input Integer nports;
            protected
              stdEvent y;
            algorithm
              y.Value := 1;
              y.Type := 1;
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase;
              Real sigma;
              Real pt;
              Integer elements;
            end st;

            function initst
              input Real pt;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := Modelica.Constants.inf; // first internal event never happens
              out.elements := 0;
              out.pt := pt;
            end initst;
          end Piston;

          package EngineBody

          annotation(preferedView="info",
            Window(
              x=0.02,
              y=0.01,
              width=0.2,
              height=0.57,
              library=1,
              autolayout=1),
            version="2.2",
            versionDate="2005-04-15",
            Settings(NewStateSelection=true),
            Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica library for modeling and simulation systems using the DEVS formalism.
</p>


</HTML>
"));
            import DESLib.DEVSLib.SRC.*;
            model enginebody
                       extends AtomicDEVS(numIn=2,
              redeclare record State = st);
              redeclare function Fcon = con;
              redeclare function Fext = ext;
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(pt = process_time);
              annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80],
                      style(      color=3, rgbcolor={0,0,255})), Text(
                              extent=[-76,44; 74,-36],
                              style(color=3, rgbcolor={0,0,255}),
                              string="%name")));
                        parameter Real process_time = 1;
                        Interfaces.outPort OUT          annotation (extent=[80,-10; 100,10]);
                        Interfaces.inPort IN          annotation (extent=[-100,30; -80,50]);
                        Interfaces.inPort DONE        annotation (extent=[-100,-50; -80,-30]);

            equation
              iEvent[1] = IN.event;
              iQueue[1] = IN.queue;
              iEvent[2] = DONE.event;
              iQueue[2] = DONE.queue;

              oEvent[1] = OUT.event;
              oQueue[1] = OUT.queue;
            end enginebody;

            function con "Confluent Transtition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            algorithm
              sout := ext(int(s),e,bag);
            end con;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              sout.elements := sout.elements -1;
              sout.phase := 1;
              sout.sigma :=  Modelica.Constants.inf;

            end int;

            function ext "External Transition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            protected
              Integer numreceived;
              stdEvent x;
            algorithm
              sout := s;
              x := DEVSLib.SRC.getEvent(bag);
              if x.Port == 1 then // IN
                sout.elements := sout.elements +1;
                if sout.phase == 1 then // start producing new element
                  sout.phase := 2;
                  sout.sigma := sout.pt;
                elseif sout.phase == 2 then // element currently being produced.
                  sout.sigma := sout.pt - e;
                else
                  sout.sigma := Modelica.Constants.inf;
                end if;
              else // DONE
                if sout.elements == 0 then
                  sout.sigma := Modelica.Constants.inf;  // stop producing
                  sout.phase := 1;
                else
                  sout.sigma := sout.pt; // produce next element.
                end if;
              end if;
            end ext;

            function out "Output Function"
               input st s;
                input Integer queue[nports];
                input Integer nports;
            protected
              stdEvent y;
            algorithm
              y.Value := 1;
              y.Type := 1;
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase;
              Real sigma;
              Real pt;
              Real elements;
            end st;

            function initst
              input Real pt;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := Modelica.Constants.inf; // first internal event never happens
              out.elements := 0;
              out.pt := pt;
            end initst;
          end EngineBody;

          model engine
            import DESLib.DEVSLib.SRC.*;
                      // parameter Real p1 = 1;

                      Interfaces.inPort inPort1
                                     annotation (extent=[-100,-10; -80,10]);
                      Interfaces.outPort outPort1
                                       annotation (extent=[80,-10; 100,10]);
                      annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                                color=3, rgbcolor={0,0,255})), Text(
                            extent=[-76,44; 74,-36],
                            style(color=3, rgbcolor={0,0,255}),
                            string="%name")));
                      Piston.piston piston annotation (extent=[-46,44; -26,64]);
                      EngineBody.enginebody enginebody
                        annotation (extent=[-48,-40; -28,-20]);
                      EngineAssembly.engineAssembly engineAssembly
                        annotation (extent=[20,0; 40,20]);
                      AuxModels.DUP dUP annotation (extent=[46,0; 66,20]);
                      AuxModels.DUP dUP1 annotation (extent=[-20,44; 0,64]);
                      AuxModels.DUP dUP2 annotation (extent=[-20,-40; 0,-20]);
                      AuxModels.DUP dUP3 annotation (extent=[-76,-10; -56,10]);
                      AuxModels.BreakLoop breakLoop
                        annotation (extent=[-32,28; -12,48], rotation=180);
                      AuxModels.BreakLoop breakLoop1
                        annotation (extent=[-32,-58; -12,-38], rotation=180);
                      AuxModels.BreakLoop breakLoop2
                        annotation (extent=[30,-16; 50,4], rotation=180);

          equation
          connect(piston.OUT, dUP1.in1) annotation (points=[-27,54; -23,54;
                            -23,54; -12.6,54],
                          style(color=3, rgbcolor={0,0,255}));
                      connect(enginebody.OUT, dUP2.in1) annotation (points=[-29,-30; -24,
                            -30; -24,-30; -12.6,-30],
                                  style(color=3, rgbcolor={0,0,255}));
                      connect(engineAssembly.OUT, dUP.in1) annotation (points=[39,10;
                            43.5,10; 43.5,10; 53.4,10],
                                                 style(color=3, rgbcolor={0,0,255}));
                      connect(dUP.out1, outPort1) annotation (points=[56.6,12; 75.5,12;
                            75.5,0; 90,0],
                                        style(color=3, rgbcolor={0,0,255}));
                      connect(dUP1.out1, engineAssembly.INPISTON) annotation (points=[-9.4,56;
                            9.5,56; 9.5,16; 21,16], style(color=3, rgbcolor={0,0,255}));
                      connect(dUP2.out1, engineAssembly.INBODY) annotation (points=[-9.4,-28;
                            10,-28; 10,10; 21,10], style(color=3, rgbcolor={0,0,255}));
                      connect(inPort1, dUP3.in1) annotation (points=[-90,0;
                  -68.6,0],                    style(color=3, rgbcolor={0,0,255}));
                      connect(dUP3.out1, piston.IN) annotation (points=[-65.4,2;
                  -52,2; -52,58; -45,58],style(color=3, rgbcolor={0,0,255}));
                      connect(dUP3.out2, enginebody.IN) annotation (points=[-65.4,-2;
                  -52,-2; -52,-26; -47,-26],   style(color=3, rgbcolor={0,0,255}));
                      connect(dUP1.out2, breakLoop.IN) annotation (points=[-9.4,52; 2,52;
                            2,38; -19.2,38], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop.OUT, piston.DONE) annotation (points=[-22.6,38;
                            -48,38; -48,50; -45,50], style(color=0, rgbcolor={0,0,0}));
                      connect(dUP2.out2, breakLoop1.IN) annotation (points=[-9.4,-32; 10,
                            -32; 10,-48; -19.2,-48], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop1.OUT, enginebody.DONE) annotation (points=[-22.6,
                            -48; -52,-48; -52,-34; -47,-34], style(color=0, rgbcolor={0,0,
                              0}));
                      connect(dUP.out2, breakLoop2.IN) annotation (points=[56.6,8; 64,8;
                            64,-6; 42.8,-6], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop2.OUT, engineAssembly.DONE) annotation (points=[39.4,-6;
                  14,-6; 14,4.2; 21,4.2],                    style(color=0, rgbcolor={0,0,
                              0}));
          end engine;
        end Engine;

        package Chassis

        annotation(preferedView="info",
          Window(
            x=0.02,
            y=0.01,
            width=0.2,
            height=0.57,
            library=1,
            autolayout=1),
          version="2.2",
          versionDate="2005-04-15",
          Settings(NewStateSelection=true),
          Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica library for modeling and simulation systems using the DEVS formalism.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model chassis
                     extends AtomicDEVS(numIn=2,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst(pt = process_time);
            annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                              color=3, rgbcolor={0,0,255})), Text(
                          extent=[-76,44; 74,-36],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%name")));
                    parameter Real process_time = 1;
                    Interfaces.outPort OUT          annotation (extent=[80,-10; 100,10]);
                    Interfaces.inPort IN          annotation (extent=[-100,30; -80,50]);
                    Interfaces.inPort DONE        annotation (extent=[-100,-50; -80,-30]);
          equation
            iEvent[1] = IN.event;
            iQueue[1] = IN.queue;
            iEvent[2] = DONE.event;
            iQueue[2] = DONE.queue;

            oEvent[1] = OUT.event;
            oQueue[1] = OUT.queue;
          end chassis;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.elements := sout.elements -1;
            sout.phase := 1;
            sout.sigma :=  Modelica.Constants.inf;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            x := DEVSLib.SRC.getEvent(bag);
            if x.Port == 1 then // IN
              sout.elements := sout.elements +1;
              if sout.phase == 1 then // start producing new element
                sout.phase := 2;
                sout.sigma := sout.pt;
              elseif sout.phase == 2 then // element currently being produced.
                sout.sigma := sout.pt - e;
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
            else // DONE
              if sout.elements == 0 then
                sout.sigma := Modelica.Constants.inf;  // stop producing
                sout.phase := 1;
              else
                sout.sigma := sout.pt; // produce next element.
              end if;
            end if;
          end ext;

          function out "Output Function"
             input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
            stdEvent y;
          algorithm
            y.Value := 1;
            y.Type := 1;
            sendEvent(queue[1],y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase;
            Real sigma;
            Real pt;
            Real elements;
          end st;

          function initst
            input Real pt;
            output st out;
          algorithm
            out.phase := 1;
            out.sigma := Modelica.Constants.inf; // first internal event never happens
            out.elements := 0;
            out.pt := pt;
          end initst;
        end Chassis;

        package Body

        annotation(preferedView="info",
          Window(
            x=0.02,
            y=0.01,
            width=0.2,
            height=0.57,
            library=1,
            autolayout=1),
          version="2.2",
          versionDate="2005-04-15",
          Settings(NewStateSelection=true),
          Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica library for modeling and simulation systems using the DEVS formalism.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model body
                     extends AtomicDEVS(numIn=2,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst(pt = process_time);
            annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                              color=3, rgbcolor={0,0,255})), Text(
                          extent=[-76,44; 74,-36],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%name")));
                    parameter Real process_time = 1;
                    Interfaces.outPort OUT          annotation (extent=[80,-10; 100,10]);
                    Interfaces.inPort IN          annotation (extent=[-100,30; -80,50]);
                    Interfaces.inPort DONE        annotation (extent=[-100,-50; -80,-30]);
          equation
            iEvent[1] = IN.event;
            iQueue[1] = IN.queue;
            iEvent[2] = DONE.event;
            iQueue[2] = DONE.queue;

            oEvent[1] = OUT.event;
            oQueue[1] = OUT.queue;
          end body;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.elements := sout.elements -1;
            sout.phase := 1;
            sout.sigma :=  Modelica.Constants.inf;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            x := DEVSLib.SRC.getEvent(bag);
            if x.Port == 1 then // IN
              sout.elements := sout.elements +1;
              if sout.phase == 1 then // start producing new element
                sout.phase := 2;
                sout.sigma := sout.pt;
              elseif sout.phase == 2 then // element currently being produced.
                sout.sigma := sout.pt - e;
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
            else // DONE
              if sout.elements == 0 then
                sout.sigma := Modelica.Constants.inf;  // stop producing
                sout.phase := 1;
              else
                sout.sigma := sout.pt; // produce next element.
              end if;
            end if;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            y.Value := 1;
            y.Type := 1;
            sendEvent(queue[1],y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase;
            Real sigma;
            Real pt;
            Real elements;
          end st;

          function initst
            input Real pt;
            output st out;
          algorithm
            out.phase := 1;
            out.sigma := Modelica.Constants.inf; // first internal event never happens
            out.elements := 0;
            out.pt := pt;
          end initst;
        end Body;

        package Transmission

        annotation(preferedView="info",
          Window(
            x=0.02,
            y=0.01,
            width=0.2,
            height=0.57,
            library=1,
            autolayout=1),
          version="2.2",
          versionDate="2005-04-15",
          Settings(NewStateSelection=true),
          Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica library for modeling and simulation systems using the DEVS formalism.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model transmission
                     extends AtomicDEVS(numIn=2,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst(pt = process_time);
            annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                              color=3, rgbcolor={0,0,255})), Text(
                          extent=[-76,44; 74,-36],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%name")));
                    parameter Real process_time = 1;
                    Interfaces.outPort OUT          annotation (extent=[80,-10; 100,10]);
                    Interfaces.inPort IN          annotation (extent=[-100,30; -80,50]);
                    Interfaces.inPort DONE        annotation (extent=[-100,-50; -80,-30]);
          equation
            iEvent[1] = IN.event;
            iQueue[1] = IN.queue;
            iEvent[2] = DONE.event;
            iQueue[2] = DONE.queue;

            oQueue[1] = OUT.queue;
            oEvent[1] = OUT.event;
          end transmission;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.elements := sout.elements -1;
            sout.phase := 1;
            sout.sigma :=  Modelica.Constants.inf;

          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            x := DEVSLib.SRC.getEvent(bag);
            if x.Port == 1 then // IN
              sout.elements := sout.elements +1;
              if sout.phase == 1 then // start producing new element
                sout.phase := 2;
                sout.sigma := sout.pt;
              elseif sout.phase == 2 then // element currently being produced.
                sout.sigma := sout.pt - e;
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
            else // DONE
              if sout.elements == 0 then
                sout.sigma := Modelica.Constants.inf;  // stop producing
                sout.phase := 1;
              else
                sout.sigma := sout.pt; // produce next element.
              end if;
            end if;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            y.Value := 1;
            y.Type := 1;
            sendEvent(queue[1],y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase;
            Real sigma;
            Real pt;
            Real elements;
          end st;

          function initst
            input Real pt;
            output st out;
          algorithm
            out.phase := 1;
            out.sigma := Modelica.Constants.inf; // first internal event never happens
            out.elements := 0;
            out.pt := pt;
          end initst;
        end Transmission;

        package FinalAssembly

        annotation(preferedView="info",
          Window(
            x=0.02,
            y=0.01,
            width=0.2,
            height=0.57,
            library=1,
            autolayout=1),
          version="2.2",
          versionDate="2005-04-15",
          Settings(NewStateSelection=true),
          Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica library for modeling and simulation systems using the DEVS formalism.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model finalAssembly
                     extends AtomicDEVS(numIn=5,
            redeclare record State = st);
            parameter Real process_time = 2;
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst(pt=process_time);
            annotation (Diagram, Icon(Rectangle(extent=[-80,100; 80,-100],
                    style(    color=3, rgbcolor={0,0,255})), Text(
                          extent=[-72,42; 78,-38],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%name")));
                    Interfaces.outPort OUT          annotation (extent=[80,-10; 100,10]);
                    Interfaces.inPort INCHASSIS   annotation (extent=[-100,70; -80,90]);
                    Interfaces.inPort INBODY      annotation (extent=[-100,30; -80,50]);
                    Interfaces.inPort DONE        annotation (extent=[-100,-90; -80,-70]);
                    Interfaces.inPort INTRANS     annotation (extent=[-100,-10; -80,10]);
                    Interfaces.inPort INENGINE    annotation (extent=[-100,-50; -80,-30]);
          equation
            iEvent[1] = INCHASSIS.event;
            iQueue[1] = INCHASSIS.queue;
            iEvent[2] = INBODY.event;
            iQueue[2] = INBODY.queue;
            iEvent[3] = INTRANS.event;
            iQueue[3] = INTRANS.queue;
            iEvent[4] = INENGINE.event;
            iQueue[4] = INENGINE.queue;
            iEvent[5] = DONE.event;
            iQueue[5] = DONE.queue;

            oQueue[1] = OUT.queue;
            oEvent[1] = OUT.event;
          end finalAssembly;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),e,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.elements_chassis := sout.elements_chassis -1;
            sout.elements_body := sout.elements_body -1;
            sout.elements_trans := sout.elements_trans -1;
            sout.elements_engine := sout.elements_engine -1;
            sout.phase := 1;
            sout.sigma := Modelica.Constants.inf;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            x := DEVSLib.SRC.getEvent(bag);
            if x.Port == 1 then // new chassis
              sout.elements_chassis := sout.elements_chassis +1;
              if sout.phase == 1 and sout.elements_chassis >= 1 and sout.elements_body >= 1 and sout.elements_trans >= 1 and sout.elements_engine >= 1 then
                sout.phase := 2;
                sout.sigma := sout.pt;
              elseif sout.phase == 2 then
                sout.sigma := sout.pt - e;
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
            elseif x.Port == 2 then // new body
              sout.elements_body := sout.elements_body +1;
              if sout.phase == 1 and sout.elements_chassis >= 1 and sout.elements_body >= 1 and sout.elements_trans >= 1 and sout.elements_engine >= 1 then
                sout.phase := 2;
                sout.sigma := sout.pt;
              elseif sout.phase == 2 then
                sout.sigma := sout.pt - e;
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
            elseif x.Port == 3 then // new trans
              sout.elements_trans := sout.elements_trans +1;
              if sout.phase == 1 and sout.elements_chassis >= 1 and sout.elements_body >= 1 and sout.elements_trans >= 1 and sout.elements_engine >= 1 then
                sout.phase := 2;
                sout.sigma := sout.pt;
              elseif sout.phase == 2 then
                sout.sigma := sout.pt - e;
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
            elseif x.Port == 4 then // new engine
              sout.elements_engine := sout.elements_engine +1;
              if sout.phase == 1 and sout.elements_chassis >= 1 and sout.elements_body >= 1 and sout.elements_trans >= 1 and sout.elements_engine >= 1 then
                sout.phase := 2;
                sout.sigma := sout.pt;
              elseif sout.phase == 2 then
                sout.sigma := sout.pt - e;
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
            else // done
              if sout.phase == 1 and sout.elements_chassis >= 1 and sout.elements_body >= 1 and sout.elements_trans >= 1 and sout.elements_engine >= 1 then
                sout.phase := 2;
                sout.sigma := sout.pt;
              else
                sout.phase := 1;
                sout.sigma := Modelica.Constants.inf;
              end if;
            end if;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            y.Value := 1;
            y.Type := 1;
            sendEvent(queue[1],y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase;
            Real sigma;
            Integer elements_chassis;
            Integer elements_body;
            Integer elements_trans;
            Integer elements_engine;
            Real pt;
          end st;

          function initst
            input Real pt;
            output st out;
          algorithm
            out.phase := 1;
            out.sigma := Modelica.Constants.inf; // first internal event never happens
            out.elements_chassis :=0;
            out.elements_body :=0;
            out.elements_trans := 0;
            out.elements_engine := 0;
            out.pt :=pt;
          end initst;
        end FinalAssembly;

        model carfactory

         import DESLib.DEVSLib.SRC.*;
                  // parameter Real p1 = 1;
                  Real petitions = generator.S.gen;
                  Real assembled = display.S.phase;

                  annotation (Diagram,
                    experiment(StopTime=50),
                    experimentSetupOutput,
                    Coordsys(scale=0.1, extent=[-100,-100; 100,100]));
                  Engine.engine engine(piston(process_time=0.2))
                    annotation (extent=[-40,-44; -20,-24]);
                  Chassis.chassis chassis(name="chassis")
                    annotation (extent=[-40,40; -20,60]);
                  Body.body body annotation (extent=[-40,10; -20,30]);
                  Transmission.transmission transmission
                    annotation (extent=[-40,-20; -18,0]);
                  FinalAssembly.finalAssembly finalAssembly
                    annotation (extent=[22,-32; 64,42]);
                  AuxModels.Generator generator(period=3)
                    annotation (extent=[-100,0; -80,20]);
                  AuxModels.DUP dUP annotation (extent=[-18,40; 4,60]);
                  AuxModels.DUP dUP1 annotation (extent=[-18,10; 2,30]);
                  AuxModels.DUP dUP2 annotation (extent=[-18,-20; 2,0]);
                  AuxModels.Display display annotation (extent=[80,20; 100,40]);
                  AuxModels.DUP dUP3 annotation (extent=[-78,4; -68,16]);
                  AuxModels.DUP dUP4 annotation (extent=[-66,-2; -56,10]);
                  AuxModels.DUP dUP5 annotation (extent=[-66,12; -56,22]);
                  AuxModels.DUP dUP6 annotation (extent=[64,0; 76,12]);
                  AuxModels.BreakLoop breakLoop
                    annotation (extent=[-30,30; -10,50], rotation=180);
                  AuxModels.BreakLoop breakLoop1
                    annotation (extent=[-30,0; -10,20], rotation=180);
                  AuxModels.BreakLoop breakLoop2
                    annotation (extent=[-30,-30; -10,-10], rotation=180);
                  AuxModels.BreakLoop breakLoop3
                    annotation (extent=[46,-48; 66,-28], rotation=180);
        equation

         connect(chassis.OUT, dUP.in1) annotation (points=[-21,50; -9.86,50],
                      style(color=3, rgbcolor={0,0,255}));
                  connect(dUP3.out1, dUP5.in1) annotation (points=[-72.7,11.2; -72.7,
                        15.5; -62.3,15.5; -62.3,17],
                                                 style(color=3, rgbcolor={0,0,255}));
                  connect(dUP3.out2, dUP4.in1) annotation (points=[-72.7,8.8; -72.7,3.5;
                        -62.3,3.5; -62.3,4],   style(color=3, rgbcolor={0,0,255}));
                  connect(generator.outPort1, dUP3.in1) annotation (points=[-81,10;
                        -74.3,10],                   style(color=3, rgbcolor={0,0,255}));
                  connect(dUP5.out1, chassis.IN) annotation (points=[-60.7,18; -52,18;
                        -52,54; -39,54], style(color=3, rgbcolor={0,0,255}));
                  connect(dUP5.out2, body.IN) annotation (points=[-60.7,16; -46,16; -46,
                        24; -39,24],     style(color=3, rgbcolor={0,0,255}));
                  connect(dUP4.out1, transmission.IN) annotation (points=[-60.7,5.2;
                        -44,5.2; -44,-6; -38.9,-6],
                                         style(color=3, rgbcolor={0,0,255}));
                  connect(dUP4.out2, engine.inPort1) annotation (points=[-60.7,2.8; -48,
                        2.8; -48,-34; -39,-34],
                                           style(color=3, rgbcolor={0,0,255}));
                  connect(dUP6.out1, display.inPort1) annotation (points=[70.36,7.2;
                        70.36,30.5; 81,30.5; 81,30],
                                               style(color=3, rgbcolor={0,0,255}));
                  connect(finalAssembly.OUT, dUP6.in1) annotation (points=[61.9,5;
                        64.95,5; 64.95,6; 68.44,6],      style(color=3, rgbcolor={0,0,255}));
                  connect(engine.outPort1, finalAssembly.INENGINE) annotation (points=[-21,-34;
                        14,-34; 14,-9.8; 24.1,-9.8],            style(color=3, rgbcolor={
                          0,0,255}));
                  connect(dUP2.out1, finalAssembly.INTRANS) annotation (points=[-7.4,-8;
                        10,-8; 10,5; 24.1,5],  style(color=3, rgbcolor={0,0,255}));
                  connect(dUP1.out1, finalAssembly.INBODY) annotation (points=[-7.4,22;
                        12,22; 12,19.8; 24.1,19.8],style(color=3, rgbcolor={0,0,255}));
                  connect(dUP.out1, finalAssembly.INCHASSIS) annotation (points=[-6.34,52;
                        14,52; 14,34.6; 24.1,34.6],   style(color=3, rgbcolor={0,0,255}));
                  connect(body.OUT, dUP1.in1) annotation (points=[-21,20; -19.5,20;
                        -19.5,20; -10.6,20],
                                     style(color=3, rgbcolor={0,0,255}));
                  connect(transmission.OUT, dUP2.in1) annotation (points=[-19.1,-10;
                        -10.6,-10],            style(color=3, rgbcolor={0,0,255}));
                  connect(dUP.out2, breakLoop.IN) annotation (points=[-6.34,48; 0,48; 0,
                        40; -17.2,40], style(color=0, rgbcolor={0,0,0}));
                  connect(breakLoop.OUT, chassis.DONE) annotation (points=[-20.6,40;
                        -44,40; -44,46; -39,46], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP1.out2, breakLoop1.IN) annotation (points=[-7.4,18; 0,18;
                        0,10; -17.2,10], style(color=0, rgbcolor={0,0,0}));
                  connect(breakLoop1.OUT, body.DONE) annotation (points=[-20.6,10; -42,
                        10; -42,16; -39,16], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP2.out2, breakLoop2.IN) annotation (points=[-7.4,-12; 0,-12;
                        0,-20; -17.2,-20], style(color=0, rgbcolor={0,0,0}));
                  connect(breakLoop2.OUT, transmission.DONE) annotation (points=[-20.6,
                        -20; -42,-20; -42,-14; -38.9,-14], style(color=0, rgbcolor={0,0,
                          0}));
                  connect(dUP6.out2, breakLoop3.IN) annotation (points=[70.36,4.8; 80,
                        4.8; 80,-38; 58.8,-38], style(color=0, rgbcolor={0,0,0}));
                  connect(breakLoop3.OUT, finalAssembly.DONE) annotation (points=[55.4,
                        -38; 20,-38; 20,-24.6; 24.1,-24.6], style(color=0, rgbcolor={0,
                          0,0}));

        end carfactory;

      end CarFactory;

    package HybridONoC
        "Hybrid model of opto-electrical communication interfaces"
    annotation(preferedView="info",
      Documentation(info="<HTML>
<p>
This package includes the implementation of an opto-electrical communication system [1].
The system has been implemented as a hybrid model: the electrical part as a continuous-time model and the optical part as a discrete-event model.
The communication between both parts is performed using the interfaces included in DEVSLib.
</p>

<p>
The transmitter is composed by:
<ul>
<li> Driver, that modulates the electrical signal received from the circuit.
<li> Laser, that generates optical impulses following the electrical current received from the driver.
</ul>

The receiver is composed by;
<ul>
<li> Photodiode, that receives optical impulses and translates them into electrical current.
<li> TIA (TransImpedance Amplifier), that amplifies the generated electrical signal and transmits it to the receiver circuit.
</ul>
</p>

<h3><font color=\"#008000\">Simulation</font></h3>
<p>
The <i>System</i> model implements the basic interfaces where the Driver and the TIA does not modify the electrical current.<br>
The <i>System2</i> model includes electrical circuits for the Driver and the TIA, as described in [2].
</p>

Both models include two variables to facilitate the observation of the simulation results.
<ul>
<li><i>send</i> shows the original electrical signal that will be transmitted.
<li><i>receive</i> shows the electrical signal generated by the receiver.
</ul>

<p>
REFERENCES<br>
[1] V. Sanz, S. Jafer, G. Wainer, G. Nicolescu, A. Urquia and S. Dormido. <a href=http://www.euclides.dia.uned.es/vsanz/publications.html> Hybrid Modeling of Opto-Electrical Interfaces Using DEVS and Modelica.</a> In Proceedings of the Spring Simulation Multiconference, San Diego, CA, USA, 2009.<br>
[2] Ian O'Connor. Optical Solutions for System-Level Interconnect. In Proceedings fo the 2004 International Workshop on System Level Interconnect Prediction, Paris, France, 2004, pp 79-88.
</p>




<br>
</HTML>
"));

          model Laser2
          parameter Real threshold=0.9;
              import DESLib.DEVSLib.SRC.*;
                // parameter Real p1 = 1;

                Interfaces.outPort optical
                                 annotation (extent=[100,-10; 120,10]);
                annotation (Diagram(Text(
                    extent=[-24,16; 4,8],
                    style(color=3, rgbcolor={0,0,255}),
                    string="LaserGen"), Line(points=[-100,0; -22,0], style(
                      color=3,
                      rgbcolor={0,0,255},
                      pattern=3))), Icon);
                Modelica.Electrical.Analog.Interfaces.Pin electrical
                  annotation (extent=[-120,-10; -100,10]);

                AuxModels.CrossUP LaserGen(Value=threshold)
                annotation (extent=[-20,-10; 0,10]);
          equation
                  LaserGen.u = electrical.i;
                  electrical.v = 0;

                connect(LaserGen.outport, optical)
                annotation (points=[0.6,0; 110,0],style(color=0, rgbcolor={0,0,0}));
          end Laser2;

      model Driver
         Modelica.Electrical.Analog.Semiconductors.NPN nPN
                annotation (extent=[-52,-10; -32,10]);
              Modelica.Electrical.Analog.Semiconductors.NPN nPN1
                annotation (extent=[-22,10; -2,-10], rotation=180);
              Modelica.Electrical.Analog.Semiconductors.NPN nPN2
                annotation (extent=[30,-10; 50,10]);
              Modelica.Electrical.Analog.Basic.Ground ground
                annotation (extent=[-10,-96; 10,-76]);
              Modelica.Electrical.Analog.Sources.ConstantCurrent
            constantCurrent
                annotation (extent=[40,40; 60,60], rotation=270);
              Modelica.Electrical.Analog.Interfaces.PositivePin p
                annotation (extent=[-106,-10; -86,10]);
              Modelica.Electrical.Analog.Interfaces.NegativePin n
                annotation (extent=[86,-10; 106,10]);
           annotation (Diagram);
      equation
              connect(ground.p, nPN.E) annotation (points=[0,-76; 0,-60; -28,-60; -28,
                    -5; -32,-5], style(color=3, rgbcolor={0,0,255}));
              connect(ground.p, nPN1.E) annotation (points=[0,-76; 0,-60; -28,
                -60; -28,-5; -22,-5],style(color=3, rgbcolor={0,0,255}));
              connect(nPN1.B, nPN2.B) annotation (points=[-2,1.22461e-015; 16,
                1.22461e-015; 16,0; 30,0],     style(color=3, rgbcolor={0,0,255}));
              connect(ground.p, nPN2.E) annotation (points=[0,-76; 0,-60; 58,-60; 58,
                    -5; 50,-5], style(color=3, rgbcolor={0,0,255}));
              connect(constantCurrent.n, nPN2.C) annotation (points=[50,40; 50,5],
                  style(color=3, rgbcolor={0,0,255}));
              connect(nPN2.B, nPN2.C) annotation (points=[30,0; 30,20; 50,20; 50,5],
                  style(color=3, rgbcolor={0,0,255}));
              connect(constantCurrent.p, ground.p) annotation (points=[50,60; 52,60;
                    52,66; 70,66; 70,-76; 0,-76], style(color=3, rgbcolor={0,0,255}));
              connect(p, nPN.B) annotation (points=[-96,0; -52,0], style(color=3,
                    rgbcolor={0,0,255}));
              connect(n, nPN1.C) annotation (points=[96,0; 80,0; 80,80; -28,80;
                -28,5; -22,5],
                            style(color=3, rgbcolor={0,0,255}));
              connect(n, nPN.C) annotation (points=[96,0; 80,0; 80,80; -28,80; -28,5;
                    -32,5], style(color=3, rgbcolor={0,0,255}));
      end Driver;

          model Transmitter
           annotation (Diagram, Icon(Rectangle(extent=[80,-80; -80,80], style(
                          color=3, rgbcolor={0,0,255})), Text(
                      extent=[-80,40; 80,-20],
                      style(color=3, rgbcolor={0,0,255}),
                      string="%name")));
                SRC.Interfaces.outPort optical   annotation (extent=[80,-10; 100,10]);
                Modelica.Electrical.Analog.Interfaces.PositivePin electrical
                  annotation (extent=[-100,-10; -80,10]);
              Laser2 laser(threshold=0.9) annotation (extent=[6,-20; 46,20]);
          equation
              connect(laser.optical, optical)
                annotation (points=[48,0; 90,0], style(color=0, rgbcolor={0,0,0}));
              connect(electrical, laser.electrical) annotation (points=[-90,0; -44,0; -44,
                    0; 4,0], style(color=3, rgbcolor={0,0,255}));
          end Transmitter;

          model Transmitter2
          annotation (Diagram, Icon(Rectangle(extent=[80,-80; -80,80], style(
                          color=3, rgbcolor={0,0,255})), Text(
                      extent=[-80,40; 80,-20],
                      style(color=3, rgbcolor={0,0,255}),
                      string="%name")));
                SRC.Interfaces.outPort optical   annotation (extent=[80,-10; 100,10]);
                Modelica.Electrical.Analog.Interfaces.PositivePin electrical
                  annotation (extent=[-100,-10; -80,10]);
              Laser2 laser(threshold=0.9) annotation (extent=[6,-20; 46,20]);
              Driver driver annotation (extent=[-54,-20; -14,20]);
          equation
              connect(laser.optical, optical)
                annotation (points=[48,0; 90,0], style(color=0, rgbcolor={0,0,0}));
              connect(driver.n, laser.electrical)
                annotation (points=[-14.8,0; 4,0],
                                                 style(color=3, rgbcolor={0,0,255}));
              connect(electrical, driver.p) annotation (points=[-90,0; -53.2,0],
                                                                               style(
                    color=3, rgbcolor={0,0,255}));
          end Transmitter2;

          model PhotoDiode
          parameter Real threshold=1;
              import DESLib.DEVSLib.SRC.*;
                // parameter Real p1 = 1;

                annotation (Diagram(Line(points=[4,0; 104,0], style(
                      color=3,
                      rgbcolor={0,0,255},
                      pattern=3))),  Icon(Rectangle(extent=[100,-98; -100,100],
                                                                             style(color=3,
                          rgbcolor={0,0,255})), Text(
                      extent=[-80,40; 80,-20],
                      style(color=3, rgbcolor={0,0,255}),
                      string="%name")));
                Modelica.Electrical.Analog.Interfaces.NegativePin electrical
                  annotation (extent=[100,-10; 120,10]);

                SRC.Interfaces.inPort optical   annotation (extent=[-120,-10; -100,10]);
                AuxModels.DICO PDiode
                                    annotation (extent=[-40,-20; 0,20]);
          equation
                when PDiode.change <> pre(PDiode.change) then
                  reinit(electrical.i,PDiode.y);
                end when;
                //electrical.v = 0;

                der(electrical.i) = if electrical.i <= 0 then 0 else -0.5;

                connect(optical, PDiode.inport) annotation (points=[-110,0; -41.2,0],
                              style(color=0, rgbcolor={0,0,0}));
          end PhotoDiode;

      model TIA
          Modelica.Electrical.Analog.Basic.Resistor resistor
                annotation (extent=[-10,70; 10,90]);
              Modelica.Electrical.Analog.Semiconductors.NPN nPN
                annotation (extent=[-40,-40; -20,-20]);
              Modelica.Electrical.Analog.Semiconductors.PNP pNP
                annotation (extent=[-40,40; -20,20]);
              Modelica.Electrical.Analog.Basic.Capacitor capacitor
                annotation (extent=[-40,-10; -20,10]);
              Modelica.Electrical.Analog.Basic.Ground ground
                annotation (extent=[-10,-104; 10,-84]);
              Modelica.Electrical.Analog.Basic.Capacitor capacitor1
                annotation (extent=[-70,-70; -50,-50], rotation=90);
              Modelica.Electrical.Analog.Semiconductors.NPN nPN1
                annotation (extent=[22,20; 42,40]);
              Modelica.Electrical.Analog.Basic.Capacitor capacitor2
                annotation (extent=[50,-70; 70,-50], rotation=90);
              Modelica.Electrical.Analog.Sources.ConstantCurrent
            constantCurrent
                annotation (extent=[-8,46; 12,66], rotation=0);
              Modelica.Electrical.Analog.Basic.Ground ground1
                annotation (extent=[-70,42; -50,62]);
              annotation (Diagram);
              Modelica.Electrical.Analog.Interfaces.PositivePin p
                annotation (extent=[-106,-10; -86,10]);
              Modelica.Electrical.Analog.Interfaces.NegativePin n
                annotation (extent=[86,-10; 106,10]);

      equation
          connect(capacitor1.p, ground.p) annotation (points=[-60,-70; -60,-80;
                0,  -80; 0,-84], style(color=3, rgbcolor={0,0,255}));
              connect(capacitor.p, nPN.B) annotation (points=[-40,0; -50,0; -50,-30;
                    -40,-30], style(color=3, rgbcolor={0,0,255}));
              connect(pNP.B, capacitor.p) annotation (points=[-40,30; -50,30; -50,0;
                    -40,0], style(color=3, rgbcolor={0,0,255}));
              connect(capacitor2.p, ground.p) annotation (points=[60,-70; 60,-80; 0,
                    -80; 0,-84], style(color=3, rgbcolor={0,0,255}));
              connect(ground1.p, constantCurrent.p) annotation (points=[-60,62; -34,
                    62; -34,56; -8,56], style(color=3, rgbcolor={0,0,255}));
              connect(pNP.E, constantCurrent.n) annotation (points=[-20,35; -20,42;
                    20,42; 20,56; 12,56], style(color=3, rgbcolor={0,0,255}));
              connect(nPN1.C, constantCurrent.n) annotation (points=[42,35; 42,56; 12,
                    56], style(color=3, rgbcolor={0,0,255}));
              connect(nPN1.B, constantCurrent.n) annotation (points=[22,30; 22,34; 20,
                    34; 20,56; 12,56], style(color=3, rgbcolor={0,0,255}));
              connect(p, capacitor1.n) annotation (points=[-96,0; -60,0; -60,-50],
                  style(color=3, rgbcolor={0,0,255}));
              connect(p, p) annotation (points=[-96,0; -96,0], style(color=3,
                    rgbcolor={0,0,255}));
              connect(p, resistor.p) annotation (points=[-96,0; -80,0; -80,80; -10,80],
                  style(color=3, rgbcolor={0,0,255}));
              connect(p, capacitor.p) annotation (points=[-96,0; -40,0], style(color=
                      3, rgbcolor={0,0,255}));
              connect(n, capacitor2.n) annotation (points=[96,0; 60,0; 60,-50], style(
                    color=3, rgbcolor={0,0,255}));
              connect(resistor.n, n) annotation (points=[10,80; 80,80; 80,0; 96,0],
                  style(color=3, rgbcolor={0,0,255}));
              connect(n, nPN1.E) annotation (points=[96,0; 60,0; 60,25; 42,25], style(
                    color=3, rgbcolor={0,0,255}));
              connect(n, nPN.C) annotation (points=[96,0; 0,0; 0,-25; -20,-25], style(
                    color=3, rgbcolor={0,0,255}));
              connect(n, capacitor.n) annotation (points=[96,0; -20,0], style(color=3,
                    rgbcolor={0,0,255}));
      end TIA;

          model Receiver
               annotation (Diagram, Icon(Rectangle(extent=[80,-80; -80,80], style(
                                color=3, rgbcolor={0,0,255})), Text(
                            extent=[-80,40; 80,-20],
                            style(color=3, rgbcolor={0,0,255}),
                            string="%name")));
                      SRC.Interfaces.inPort optical   annotation (extent=[-100,-10; -80,10]);
                      Modelica.Electrical.Analog.Interfaces.PositivePin
            electrical  annotation (extent=[80,-10; 100,10]);
                      PhotoDiode photoDiode annotation (extent=[-48,-20; -8,20]);
          equation
                      connect(electrical, electrical) annotation (points=[90,0; 86,0; 86,0;
                            90,0], style(color=3, rgbcolor={0,0,255}));
                      connect(optical, photoDiode.optical)
                        annotation (points=[-90,0; -50,0], style(color=0, rgbcolor={0,0,0}));
                    connect(photoDiode.electrical, electrical) annotation (points=[-6,0; 90,0],
                        style(color=3, rgbcolor={0,0,255}));
          end Receiver;

          model Receiver2

                      annotation (Diagram, Icon(Rectangle(extent=[80,-80; -80,80], style(
                                color=3, rgbcolor={0,0,255})), Text(
                            extent=[-80,40; 80,-20],
                            style(color=3, rgbcolor={0,0,255}),
                            string="%name")));
                      SRC.Interfaces.inPort optical   annotation (extent=[-100,-10; -80,10]);
                      Modelica.Electrical.Analog.Interfaces.PositivePin
            electrical  annotation (extent=[80,-10; 100,10]);
                      PhotoDiode photoDiode annotation (extent=[-48,-20; -8,20]);
                    TIA Tia annotation (extent=[12,-20; 52,20]);
          equation
                      connect(electrical, electrical) annotation (points=[90,0; 86,0; 86,0;
                            90,0], style(color=3, rgbcolor={0,0,255}));
                      connect(optical, photoDiode.optical)
                        annotation (points=[-90,0; -50,0], style(color=0, rgbcolor={0,0,0}));
                    connect(photoDiode.electrical, Tia.p)
                      annotation (points=[-6,0; 12.8,0],
                                                       style(color=3, rgbcolor={0,0,255}));
                    connect(Tia.n, electrical)
                      annotation (points=[51.2,0; 90,0],
                                                       style(color=3, rgbcolor={0,0,255}));

          end Receiver2;

          model System
            Real send = sineCurrent.i;
                    Real receive = currentSensor.i;

                      Transmitter transmitter annotation (extent=[-50,-10; -10,30]);
                      DESLib.DEVSLib.Examples.HybridONoC.Receiver receiver
                                        annotation (extent=[14,-10; 54,30]);
                      annotation (Diagram,
                      experiment(StopTime=100),
                      experimentSetupOutput);
                      Modelica.Electrical.Analog.Sources.SineCurrent
            sineCurrent(                                                         freqHz=
                          0.027)
                        annotation (extent=[-78,0; -58,20],  rotation=0);
                      Modelica.Electrical.Analog.Basic.Ground ground
                        annotation (extent=[-100,-20; -80,0]);
                      Modelica.Electrical.Analog.Basic.Ground ground1
                        annotation (extent=[80,-20; 100,0]);
                      Modelica.Electrical.Analog.Sensors.CurrentSensor
            currentSensor
                        annotation (extent=[58,0; 78,20],  rotation=180);
                      AuxModels.DUP dUP annotation (extent=[-6,-28; 14,-8]);
                      AuxModels.Display display annotation (extent=[22,-38; 42,-18]);
          equation
             connect(sineCurrent.p,ground. p) annotation (points=[-78,10; -90,10;
                          -90,0],     style(
                          color=3,
                          rgbcolor={0,0,255},
                          fillColor=7,
                          rgbfillColor={255,255,255},
                          fillPattern=1));
                      connect(currentSensor.p, ground1.p)
                                                         annotation (points=[78,10;
                90,10; 90,0],     style(color=3, rgbcolor={0,0,255}));
                      connect(sineCurrent.n, transmitter.electrical) annotation (points=[-58,10;
                          -48,10],       style(color=3, rgbcolor={0,0,255}));
                      connect(receiver.electrical, currentSensor.n) annotation (points=[52,10;
                55,10; 55,10; 58,10],             style(color=3, rgbcolor={0,0,255}));
                      connect(transmitter.optical, dUP.in1) annotation (points=[-12,10; -6,
                          10; -6,-18; 1.4,-18],
                                             style(color=0, rgbcolor={0,0,0}));
                      connect(dUP.out1, receiver.optical) annotation (points=[4.6,-16; 4.6,
                          -0.5; 16,-0.5; 16,10],
                                           style(color=0, rgbcolor={0,0,0}));
                      connect(dUP.out2, display.inPort1) annotation (points=[4.6,-20; 17.5,
                          -20; 17.5,-28; 23,-28],
                                               style(color=0, rgbcolor={0,0,0}));
          end System;

          model System2
           Real send = sineCurrent.i;
                    Real receive = currentSensor.i;

                      DESLib.DEVSLib.Examples.HybridONoC.Receiver2 receiver
                                        annotation (extent=[6,-10; 46,30]);
                      annotation (Diagram,
                      experiment(StopTime=100),
                      experimentSetupOutput);
                      Modelica.Electrical.Analog.Sources.SineCurrent
            sineCurrent(                                                         freqHz=
                          0.027)
                        annotation (extent=[-78,0; -58,20],  rotation=0);
                      Modelica.Electrical.Analog.Basic.Ground ground
                        annotation (extent=[-100,-20; -80,0]);
                      Modelica.Electrical.Analog.Basic.Ground ground1
                        annotation (extent=[80,-20; 100,0]);
                      Modelica.Electrical.Analog.Sensors.CurrentSensor
            currentSensor
                        annotation (extent=[58,0; 78,20],  rotation=180);
                      AuxModels.DUP dUP annotation (extent=[-16,-26; 4,-6]);
                      AuxModels.Display display annotation (extent=[22,-38; 42,-18]);
                    Transmitter2 transmitter    annotation (extent=[-54,-10; -14,30]);

          equation
           connect(sineCurrent.p,ground. p) annotation (points=[-78,10; -90,10;
                          -90,0],     style(
                          color=3,
                          rgbcolor={0,0,255},
                          fillColor=7,
                          rgbfillColor={255,255,255},
                          fillPattern=1));
                      connect(currentSensor.p, ground1.p)
                                                         annotation (points=[78,10;
                90,10; 90,0],     style(color=3, rgbcolor={0,0,255}));
                      connect(receiver.electrical, currentSensor.n) annotation (points=[44,10;
                51,10; 51,10; 58,10],             style(color=3, rgbcolor={0,0,255}));
                      connect(dUP.out1, receiver.optical) annotation (points=[-5.4,-14;
                          -5.4,-0.5; 8,-0.5; 8,10],
                                           style(color=0, rgbcolor={0,0,0}));
                      connect(dUP.out2, display.inPort1) annotation (points=[-5.4,-18; 17.5,
                          -18; 17.5,-28; 23,-28],
                                               style(color=0, rgbcolor={0,0,0}));
                    connect(sineCurrent.n, transmitter.electrical)    annotation (points=[-58,10;
                          -52,10],         style(color=3, rgbcolor={0,0,255}));
                    connect(transmitter.optical, dUP.in1)    annotation (points=[-16,10;
                          -14,10; -14,-16; -8.6,-16],style(color=0, rgbcolor={0,0,0}));
          end System2;
    end HybridONoC;

      package QSSIntegration
        "Several systems modeled using the QSS integration methods implemented with DEVSLib"
      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package includes some examples of continuous-time systems modeled using the QSS integrators, implemented using DEVSLib.
</p>

<p>
The <a href=\"Modelica://DESLib.DEVSLib.Examples.QSSIntegration.DEDEVS\">DEDEVS</a> model represents a simple differential equation.<br>
The <a href=\"Modelica://DESLib.DEVSLib.Examples.QSSIntegration.FlybackConverter\">FlybackConverter</a> FlybackConverter implements a DC-DC current converter.<br>
The <a href=\"Modelica://DESLib.DEVSLib.Examples.QSSIntegration.LotkaVolterra\">LotkaVolterra</a> LotkaVolterra model implements a predator-prey interaction system using the equations by Lotka and Volterra.<br>
</p>

<p>
Each system has been modeled using the QSS1 (model named \"system\"), QSS2 (model named \"system2\") and QSS3 (model named \"system3\") integrators.<br>
Other components, like the Add, Multiplier, Gain, etc. are required to model these systems and are included in this package.
</p>



<br>
</HTML>
"));
          package Step

          import DESLib.DEVSLib.SRC.*;
            model step

              parameter Real period = 1;
              parameter Real firstGeneration = 1;
              parameter Integer numGenerations = 0 "0 == infinity";
              parameter Real outValue = 1;
              extends AtomicDEVS(redeclare record State = st);
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(p=period,fg=firstGeneration,val=outValue,ng=numGenerations);
              annotation (Diagram, Icon(
                          Rectangle(extent=[-100,80; 80,-80], style(color=3, rgbcolor={0,0,
                                  255})),
                          Line(points=[-80,-30; -10,-30; -10,30; 60,30], style(color=3,
                                rgbcolor={0,0,255})),
                          Line(points=[-80,60; -80,-50], style(color=10, rgbcolor={95,95,95})),
                          Line(points=[-92,-40; 60,-40], style(color=10, rgbcolor={95,95,95}))));

                        Interfaces.outPort outPort1
                                         annotation (extent=[80,-10; 100,10]);
            equation
              for i in 1:numIn loop
                iEvent[i] = 0;
              end for;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;

            end step;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              sout.gen := sout.gen +1;
              if (sout.gen < s.ng) or (s.ng == 0) then
                sout.sigma := s.p; // period
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
              sout.p := s.p;
              sout.val := s.val;
              sout.port := 1;
            end int;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent Y;
            algorithm
              Y.Value := s.val;
              Y.Type := 1;
              sendEvent(queue[1],Y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase; // 1 = passive, 2 = active
              Real sigma;
              Real p;
              Real fg;
              Real val;
              Integer port;
              Integer ng;
              Integer gen;
            end st;

            function initst
              input Real p;
              input Real fg;
              input Real val;
              input Integer ng;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := fg;
              out.p := p;
              out.val := val;
              out.port := 1;
              out.ng := ng;
              out.gen := 0;
            end initst;
          end Step;

          package Square

          import DESLib.DEVSLib.SRC.*;
            model square
              parameter Real amplitude = 1;
              parameter Real frecuency = 1;
              parameter Real dutyCycle = 50 "(%)";
              extends AtomicDEVS(redeclare record State = st);
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(a=amplitude,f=frecuency,d=dutyCycle);
               annotation (Diagram, Icon(
                          Rectangle(extent=[-100,80; 80,-80], style(color=3, rgbcolor={0,0,
                                  255})),
                          Line(points=[-80,-20; -60,-20; -60,40; -40,40; -40,-20; 0,-20; 0,
                                40; 20,40; 20,-20; 60,-20], style(color=3, rgbcolor={0,0,
                                  255})),
                          Line(points=[-80,60; -80,-50], style(color=10, rgbcolor={95,95,95})),
                          Line(points=[-92,-40; 60,-40], style(color=10, rgbcolor={95,95,95}))));

                        Interfaces.outPort outPort1
                                         annotation (extent=[80,-10; 100,10]);
            equation
              for i in 1:numIn loop
                iEvent[i] = 0;
              end for;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
            end square;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              sout.S := s.lev*s.A;
              sout.sigma := (sout.D*sout.lev+(1-sout.lev)*(100-sout.D))/(100*sout.F);
              sout.lev := 1-sout.lev;
            end int;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent Y;
            algorithm
              Y.Value := s.lev*s.A;
              Y.Type := 1;
              sendEvent(queue[1],Y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase; // 1 = passive, 2 = active
              Real sigma;
              Real A;
              Real F;
              Real D;
              Integer lev;
              Real S;
            end st;

            function initst
              input Real a;
              input Real f;
              input Real d;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := 0;
              out.A := a;
              out.F := f;
              out.D := d;
              out.lev := 1;
              out.S := 0;
            end initst;
          end Square;

          package Constant

          import DESLib.DEVSLib.SRC.*;
            model ConstantG

              parameter Real period = 1;
              parameter Real firstGeneration = 0;
              parameter Integer numGenerations = 1 "0 == infinity";
              parameter Real outValue = 1;
              extends AtomicDEVS(redeclare record State = st);
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(p=period,fg=firstGeneration,val=outValue,ng=numGenerations);
               annotation (Diagram, Icon(
                          Rectangle(extent=[-100,80; 80,-80], style(color=3, rgbcolor={0,0,
                                  255})),
                          Line(points=[-80,20; 60,20], style(color=3, rgbcolor={0,0,255})),
                          Line(points=[-80,60; -80,-50], style(color=10, rgbcolor={95,95,95})),
                          Line(points=[-92,-40; 60,-40], style(color=10, rgbcolor={95,95,95}))));

                        Interfaces.outPort outPort1
                                         annotation (extent=[80,-10; 100,10]);
            equation
              for i in 1:numIn loop
                iEvent[i] = 0;
              end for;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;

            end ConstantG;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              sout.gen := sout.gen +1;
              if (sout.gen < s.ng) or (s.ng == 0) then
                sout.sigma := s.p; // period
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
              sout.p := s.p;
              sout.val := s.val;
              sout.port := 1;
            end int;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent Y;
            algorithm
              Y.Value := s.val;
              Y.Type := 1;
              sendEvent(queue[1],Y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase; // 1 = passive, 2 = active
              Real sigma;
              Real p;
              Real fg;
              Real val;
              Integer port;
              Integer ng;
              Integer gen;
            end st;

            function initst
              input Real p;
              input Real fg;
              input Real val;
              input Integer ng;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := fg;
              out.p := p;
              out.val := val;
              out.port := 1;
              out.ng := ng;
              out.gen := 0;
            end initst;
          end Constant;

        package Add
          import DESLib.DEVSLib.SRC.*;
          model add
          parameter Real c1 = 1;
          parameter Real c2 = -1;
                   extends AtomicDEVS(numIn=2,
          redeclare record State = st);
          redeclare function Fcon = con;
          redeclare function Fint = int;
          redeclare function Fext = ext;
          redeclare function Fout = out;
          redeclare function Fta = ta;
          redeclare function initState = initst(k1=c1,k2=c2);
            annotation (Diagram, Icon(
                        Rectangle(extent=[-80,80; 80,-80], style(color=3, rgbcolor={0,0,
                                255})),
                        Text(
                          extent=[-80,80; 80,-80],
                          string="+",
                          style(
                            color=3,
                            rgbcolor={0,0,255},
                            fillColor=3,
                            rgbfillColor={0,0,255})),
                        Text(
                          extent=[-70,80; -30,20],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%c1"),
                        Text(
                          extent=[-70,-20; -30,-80],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%c2")));
                      Interfaces.outPort y
                                       annotation (extent=[80,-10; 100,10]);
                      Interfaces.inPort u1
                                     annotation (extent=[-100,40; -80,60]);
                      Interfaces.inPort u2
                                     annotation (extent=[-100,-60; -80,-40]);
          equation
              iEvent[1] = u1.event;
              iQueue[1] = u1.queue;
              iEvent[2] = u2.event;
              iQueue[2] = u2.queue;

              oEvent[1] = y.event;
              oQueue[1] = y.queue;

          end add;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),0,bag);
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.S0:= 0;
            sout.S1:= 0;
            sout.S2:= 0;
            sout.S0 := sout.S0 + sout.K1*sout.Xs1;
            sout.S1 := sout.S1 + sout.K1*sout.Mxs1;
            sout.S2 := sout.S2 + sout.K1*sout.Pxs1;
            sout.S0 := sout.S0 + sout.K2*sout.Xs2;
            sout.S1 := sout.S1 + sout.K2*sout.Mxs2;
            sout.S2 := sout.S2 + sout.K2*sout.Pxs2;
            sout.sigma := Modelica.Constants.inf; // processing_time;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
            Boolean p1;
            Boolean p2;
          algorithm
            p1 := false;
            p2 := false;
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              //Modelica.Utilities.Streams.print(String(x.Type)+ " "+String(x.Value));
              if x.Port == 1 then // U1
                p1 := true;
                if x.Type == 1 then
                  sout.Xs1 := x.Value;
                elseif x.Type == 2 then
                  sout.Mxs1 := x.Value;
                elseif x.Type == 3 then
                  sout.Pxs1 := x.Value;
                end if;
              elseif x.Port == 2 then // U2
                p2 := true;
                if x.Type == 1 then
                  sout.Xs2 := x.Value;
                elseif x.Type == 2 then
                  sout.Mxs2 := x.Value;
                elseif x.Type == 3 then
                  sout.Pxs2 := x.Value;
                end if;
              end if;
            end for;
            if not p1 then //and e > 0.0001 then
              sout.Xs1 := sout.Xs1 + sout.Mxs1*e + sout.Pxs1*e*e;
              sout.Mxs1 := sout.Mxs1 + 2*sout.Pxs1*e;
            end if;
            if not p2 then //and e > 0.0001 then
              sout.Xs2 := sout.Xs2 + sout.Mxs2*e + sout.Pxs2*e*e;
              sout.Mxs2 := sout.Mxs2 + 2*sout.Pxs2*e;
            end if;
            sout.sigma := 0; // processing_time
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
            Real S0;
            Real S1;
            Real S2;
          algorithm
              S0:= 0;
              S1:= 0;
              S2:= 0;
              //Modelica.Utilities.Streams.print("ADD INPUTS ARE "+String(s.Xs1)+" "+String(s.Xs2));
              S0 := S0 + s.K1*s.Xs1;
              S1 := S1 + s.K1*s.Mxs1;
              S2 := S2 + s.K1*s.Pxs1;
              S0 := S0 + s.K2*s.Xs2;
              S1 := S1 + s.K2*s.Mxs2;
              S2 := S2 + s.K2*s.Pxs2;
              y.Type := 1;
              y.Value := S0;
              //Modelica.Utilities.Streams.print(" ADD OUTPUT IS: "+String(S0));
              sendEvent(queue[1],y);
              y.Type := 2;
              y.Value := S1;
              //Modelica.Utilities.Streams.print(String(S1));
              sendEvent(queue[1],y);
              y.Type := 3;
              y.Value := S2;
              //Modelica.Utilities.Streams.print(String(S2));
              sendEvent(queue[1],y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real K1;
            Real K2;
            Real Xs1;
            Real Xs2;
            Real Mxs1;
            Real Mxs2;
            Real Pxs1;
            Real Pxs2;
            Real S0;
            Real S1;
            Real S2;
          end st;

          function initst
            input Real k1;
            input Real k2;
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;  //first generation starts at time == 1
            out.K1 := k1;
            out.K2 := k2;
            out.Xs1 := 0;
            out.Xs2 := 0;
            out.Mxs1:= 0;
            out.Mxs2:= 0;
            out.Pxs1:= 0;
            out.Pxs2:= 0;
            out.S0 := 0;
            out.S1:= 0;
            out.S2:= 0;
          end initst;
        end Add;

        package Multiplier
          import DESLib.DEVSLib.SRC.*;
          model multiplier
                   extends AtomicDEVS(numIn=2,
          redeclare record State = st);
          redeclare function Fcon = con;
          redeclare function Fint = int;
          redeclare function Fout = out;
          redeclare function Fext = ext;
          redeclare function Fta = ta;
          redeclare function initState = initst;
            annotation (Diagram, Icon(Rectangle(extent=[-80,80; 80,-80], style(
                              color=3, rgbcolor={0,0,255})), Text(
                          extent=[-80,50; 80,-110],
                          style(color=3, rgbcolor={0,0,255}),
                          string="*")));
                      Interfaces.outPort y
                                       annotation (extent=[80,-10; 100,10]);
                      Interfaces.inPort u1
                                     annotation (extent=[-100,40; -80,60]);
                      Interfaces.inPort u2
                                     annotation (extent=[-100,-60; -80,-40]);
          equation
              iEvent[1] = u1.event;
              iQueue[1] = u1.queue;
              iEvent[2] = u2.event;
              iQueue[2] = u2.queue;

              oEvent[1] = y.event;
              oQueue[1] = y.queue;

          end multiplier;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),0,bag);
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.S := s.u1 * s.u2;
            sout.phase := 1;
            sout.sigma := Modelica.Constants.inf; // processing_time;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
            Real xv3;
            Boolean p1;
            Boolean p2;
          algorithm
            p1 := false;
            p2 := false;
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              //Modelica.Utilities.Streams.print(String(x.Type)+ " "+String(x.Value));
              if x.Port == 1 then
                p1 := true;
                if x.Type == 1 then
                  sout.u1 := x.Value;
                elseif x.Type == 2 then
                  sout.mu1 := x.Value;
                elseif x.Type == 3 then
                  sout.pu1 := x.Value;
                  xv3 := x.Value;
                end if;
              elseif x.Port == 2 then
                p2 := true;
                if x.Type == 1 then
                  sout.u2 := x.Value;
                elseif x.Type == 2 then
                  sout.mu2 := x.Value;
                elseif x.Type == 3 then
                  sout.pu2 := x.Value;
                  xv3 := x.Value;
                end if;
              end if;
            end for;
            if not p2 then
              sout.u2 := sout.u2 + sout.mu2*e + sout.pu2*e*e;
              sout.mu2 := sout.mu2 + 2*sout.pu2*e;
            elseif not p1 then
              sout.u1 := sout.u1 + sout.mu1*e + sout.pu1*e*e;
              sout.mu1 := sout.mu1 + 2*sout.pu1*e;
            end if;
            if xv3 <> 0 then
              sout.nm := 1;
            end if;
            sout.phase := 2;
            sout.sigma := 0;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
            Real S0;
            Real S1;
            Real S2;
          algorithm
              //S0 := 0;
              //S1:= 0;
              //S2:= 0;
              S0 := s.u1 * s.u2;
              S1 := s.mu1*s.u2 + s.mu2*s.u1;
              if s.nm == 1 then
                S2 := s.u1*s.pu2 + s.mu1*s.mu2 + s.pu1*s.u2;
              else
                S2 := 0;
              end if;
              y.Type := 1;
              y.Value := S0;
              //Modelica.Utilities.Streams.print(" MUL OUTPUT IS = "+String(S0));
              sendEvent(queue[1],y);
              y.Type := 2;
              y.Value := S1;
              //Modelica.Utilities.Streams.print(String(S1));
              sendEvent(queue[1],y);
              y.Type := 3;
              y.Value := S2;
              //Modelica.Utilities.Streams.print(String(S2));
              sendEvent(queue[1],y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real u1;
            Real u2;
            Real mu1;
            Real mu2;
            Real pu1;
            Real pu2;
            Integer nm;
            Real S;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;  //first generation starts at time == 1
            out.u1 := 0;
            out.u2 := 0;
            out.mu1:= 0;
            out.mu2:= 0;
            out.pu1:= 0;
            out.pu2:= 0;
            out.nm := 0;
            out.S := 0;
          end initst;
        end Multiplier;

      package Gain

          import DESLib.DEVSLib.SRC.*;
            model gain
              parameter Real K=1;
                       extends AtomicDEVS(numIn=1,
              redeclare record State = st);
              redeclare function Fcon = con;
              redeclare function Fext = ext;
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(k=K);
            annotation (Diagram, Icon(Polygon(points=[-80,80; -80,-78; 80,0;
                            -80,80], style(color=3, rgbcolor={0,0,255})), Text(
                        extent=[-80,40; 0,-40],
                        style(color=3, rgbcolor={0,0,255}),
                        string="%K")));
                    Interfaces.outPort outPort1     annotation (extent=[80,-10; 100,10]);
                    Interfaces.inPort inPort1     annotation (extent=[-100,-10; -80,10]);
            equation
              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
            end gain;

            function con "Confluent Transtition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            algorithm
              sout := ext(int(s),0,bag);
            end con;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              sout.sigma := Modelica.Constants.inf;
            end int;

            function ext "External Transition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
          protected
              Integer numreceived;
              stdEvent x;
            algorithm
              sout := s;
              numreceived := numEvents(bag);
              for i in 1:numreceived loop
                x := getEvent(bag);
                if x.Type == 1 then
                  sout.u1 := x.Value;
                  sout.order := max(sout.order,1);
                elseif x.Type == 2 then
                  sout.u2 := x.Value;
                  sout.order := max(sout.order,2);
                elseif x.Type == 3 then
                  sout.u2 := x.Value;
                  sout.order := max(sout.order,3);
                end if;
              end for;
              sout.sigma := 0;
            end ext;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent y;
            algorithm
              if s.order >= 1 then
                y.Type := 1;
                y.Value := s.K*s.u1;
                sendEvent(queue[1],y);
              elseif s.order >= 2 then
                y.Type := 2;
                y.Value := s.K*s.u2;
                sendEvent(queue[1],y);
              elseif s.order >= 3 then
                y.Type := 3;
                y.Value := s.K*s.u3;
                sendEvent(queue[1],y);
              end if;
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase;
              Real sigma;
              Real u1;
              Real u2;
              Real u3;
              Real K;
              Integer order;
            end st;

            function initst
              input Real k;
              output st out;
            algorithm
              out.phase := 0;
              out.sigma := Modelica.Constants.inf; // first internal event never happens
              out.u1:= 0;
              out.u2 := 0;
              out.u3 := 0;
              out.K := k;
              out.order := 1;
            end initst;
      end Gain;

        package Switch
          import DESLib.DEVSLib.SRC.*;
          model switch
              parameter Real level= 0;
                   extends AtomicDEVS(numIn=3,
          redeclare record State = st);
          redeclare function Fcon = con;
          redeclare function Fint = int;
          redeclare function Fout = out;
          redeclare function Fext = ext;
          redeclare function Fta = ta;
          redeclare function initState = initst(level=level);
           annotation (Diagram, Icon(
                        Rectangle(extent=[-80,80; 80,-80], style(color=3, rgbcolor={0,0,
                                255})),
                        Line(points=[-80,68; 0,68], style(color=3, rgbcolor={0,0,255})),
                        Line(points=[-80,-68; 0,-68], style(color=3, rgbcolor={0,0,255})),
                        Line(points=[80,0; 60,0], style(color=3, rgbcolor={0,0,255})),
                        Line(points=[-80,0; -20,0], style(color=3, rgbcolor={0,0,255})),
                        Line(points=[-20,8; -20,-8], style(color=3, rgbcolor={0,0,255})),
                        Line(points=[60,0; 0,68; 0,66], style(color=3, rgbcolor={0,0,255})),
                        Ellipse(extent=[-2,70; 2,66], style(
                            color=3,
                            rgbcolor={0,0,255},
                            fillColor=3,
                            rgbfillColor={0,0,255})),
                        Ellipse(extent=[-2,-70; 2,-66], style(
                            color=3,
                            rgbcolor={0,0,255},
                            fillColor=3,
                            rgbfillColor={0,0,255}))));

                      Interfaces.outPort y
                                       annotation (extent=[80,-10; 100,10]);
                      Interfaces.inPort u1
                                     annotation (extent=[-100,60; -80,80]);
                      Interfaces.inPort u2
                                     annotation (extent=[-100,-10; -80,10]);
                      Interfaces.inPort u3
                                     annotation (extent=[-100,-80; -80,-60]);
          equation
              iEvent[1] = u1.event;
              iQueue[1] = u1.queue;
              iEvent[2] = u2.event;
              iQueue[2] = u2.queue;
              iEvent[3] = u3.event;
              iQueue[3] = u3.queue;

              oEvent[1] = y.event;
              oQueue[1] = y.queue;

          end switch;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),0,bag);
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            if sout.sigma == 0 then
              sout.sigma := sout.tcross;
            else
              sout.u1 := sout.u1 + sout.mu1*sout.sigma + sout.pu1*sout.sigma*sout.sigma;
              sout.mu1 := sout.mu1 + 2*sout.pu1*sout.sigma;
              sout.u2 := sout.u2 + sout.mu2*sout.sigma + sout.pu2*sout.sigma*sout.sigma;
              sout.mu2 := sout.mu2 + 2*sout.pu2*sout.sigma;
              sout.u3 := sout.u3 + sout.mu3*sout.sigma + sout.pu3*sout.sigma*sout.sigma;
              sout.mu3 := sout.mu3 + 2*sout.pu3*sout.sigma;
              if sout.mu2*sout.pu2 < 0 then
                sout.sigma := -sout.mu2/sout.pu2;
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
              sout.tcross := sout.sigma;
              sout.sw := 2 - sout.sw;
            end if;
            sout.phase := 1;
            sout.ch := 0;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
            Real xv1;
            Real xv2;
            Real xv3;
            Integer port;
            Real a;
            Real b;
            Real c;
            Real s1;
            Real s2;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              //Modelica.Utilities.Streams.print(String(x.Type)+ " "+String(x.Value));
              port := x.Port;
              if x.Type == 1 then
                xv1 := x.Value;
              elseif x.Type == 2 then
                xv2 := x.Value;
              elseif x.Type == 3 then
                xv3 := x.Value;
              end if;
            end for;
            if port == 1 then
              sout.u1 := xv1;
              sout.mu1 := xv2;
              sout.pu1 := xv3;
              sout.u2 := sout.u2 + sout.mu2*e + sout.pu2*e*e;
              sout.mu2 := sout.mu2 + 2*sout.pu2*e;
              sout.u3 := sout.u3 + sout.mu3*e + sout.pu3*e*e;
              sout.mu3 := sout.mu3 + 2*sout.pu3*e;
            elseif port == 2 then
              sout.u2 := xv1;
              sout.mu2 := xv2;
              sout.pu2:= xv3;
              sout.u1 := sout.u1 + sout.mu1*e + sout.pu1*e*e;
              sout.mu1 := sout.mu1 + 2*sout.pu1*e;
              sout.u3 := sout.u3 + sout.mu3*e + sout.pu3*e*e;
              sout.mu3 := sout.mu3 + 2*sout.pu3*e;
            elseif port == 3 then
              sout.u3 := xv1;
              sout.mu3 := xv2;
              sout.pu3:= xv3;
              sout.u1 := sout.u1 + sout.mu1*e + sout.pu1*e*e;
              sout.mu1 := sout.mu1 + 2*sout.pu1*e;
              sout.u2 := sout.u2 + sout.mu2*e + sout.pu2*e*e;
              sout.mu2 := sout.mu2 + 2*sout.pu2*e;
            end if;
            if port == 2 then // calculate new time to cross
              a := sout.pu2;
              b := sout.mu2;
              c := sout.u2 - sout.level;
              if (a == 0) then
                if (b == 0) then
                  s1 := Modelica.Constants.inf;
                  s2 := Modelica.Constants.inf;
                else
                  s1 := -c/b;
                  s2 := Modelica.Constants.inf;
                end if;
              else
                s1 := (-b+sqrt(b*b - 4*a*c))/2/a;
                s2 := (-b-sqrt(b*b - 4*a*c))/2/a;
              end if;
              if (s1>0) and ((s1<s2) or (s2<0)) then
                sout.tcross := s1;
              elseif (s2>0) then
                  sout.tcross := s2;
              else
                  sout.tcross := Modelica.Constants.inf;
              end if;
              if ((sout.sigma == e) and (((sout.u2 > sout.level) and (sout.sw == 0)) or ((sout.u2 < sout.level) and (sout.sw == 2)))) then
                sout.sigma :=0;
              elseif (((sout.u2 > sout.level) and (sout.sw == 2)) or ((sout.u2 < sout.level) and (sout.sw == 0))) then
                  sout.sw := 2-sout.sw;
                  sout.ch := 1;
                  sout.sigma := 0;
              else
                  sout.sigma := sout.tcross;
              end if;
            else // input event
              sout.tcross := sout.tcross -e;
              if (sout.sigma == e) then
                sout.sigma := 0;
              elseif (e == 0) then
                if (sout.sw == port-1) or (sout.ch == 1) then
                  sout.sigma := 0;
                else
                  sout.sigma := sout.tcross;
                end if;
              elseif (sout.u2 > sout.level) and (port == 1) then
                sout.sigma := 0;
                sout.sw := 0;
              elseif (sout.u2 < sout.level) and (port == 3) then
                sout.sigma := 0;
                sout.sw := 2;
              else
                sout.sigma:= sout.tcross;
              end if;
            end if;

          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
            Real S0;
            Real S1;
            Real S2;
          algorithm
              if s.sigma == 0 then
                y.Type := 1;
                if s.sw == 0 then
                  y.Value := s.u1;
                elseif s.sw == 2 then
                  y.Value := s.u3;
                end if;
                sendEvent(queue[1],y);
                y.Type := 2;
                if s.sw == 0 then
                  y.Value := s.mu1;
                elseif s.sw == 2 then
                  y.Value := s.mu3;
                end if;
                sendEvent(queue[1],y);
                y.Type := 3;
                if s.sw == 0 then
                  y.Value := s.pu1;
                elseif s.sw == 2 then
                  y.Value := s.pu3;
                end if;
                sendEvent(queue[1],y);
              else
                y.Type := 1;
                if s.sw == 0 then
                  y.Value := s.u3 + s.mu3*s.sigma + s.pu3*s.sigma*s.sigma;
                elseif s.sw == 2 then
                  y.Value := s.u1+ s.mu1*s.sigma + s.pu1*s.sigma*s.sigma;
                end if;
                sendEvent(queue[1],y);
                y.Type := 2;
                if s.sw == 0 then
                  y.Value := s.mu3 + 2*s.pu3*s.sigma;
                elseif s.sw == 2 then
                  y.Value := s.mu1 + 2*s.pu1*s.sigma;
                end if;
                sendEvent(queue[1],y);
                y.Type := 3;
                if s.sw == 0 then
                  y.Value := s.pu3;
                elseif s.sw == 2 then
                  y.Value := s.pu1;
                end if;
                sendEvent(queue[1],y);
              end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real level;
            Real u1;
            Real u2;
            Real u3;
            Real mu1;
            Real mu2;
            Real mu3;
            Real pu1;
            Real pu2;
            Real pu3;
            Real tcross;
            Integer sw;
            Integer ch;
          end st;

          function initst
            input Real level;
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;  //first generation starts at time == 1
            out.level := level;
            out.u1 := 0;
            out.u2 := 0;
            out.u3 := 0;
            out.mu1:= 0;
            out.mu2:= 0;
            out.mu3:= 0;
            out.pu1:= 0;
            out.pu2:= 0;
            out.pu3:= 0;
            out.tcross := Modelica.Constants.inf;
            out.sw := 0;
            out.ch := 0;
          end initst;
        end Switch;

        package DupQSS
          model dupevent
                     extends SRC.AtomicDEVS(numOut=2,redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst;
            annotation (Diagram, Icon(
                        Rectangle(extent=[-80,80; 80,-80], style(color=3, rgbcolor={0,0,
                                255})),
                        Line(points=[-80,0; 0,0; 80,50], style(
                            color=3,
                            rgbcolor={0,0,255},
                            fillPattern=1)),
                        Line(points=[0,0; 80,-50], style(
                            color=3,
                            rgbcolor={0,0,255},
                            fillPattern=1)),
                        Polygon(points=[74,48; 80,50; 76,46; 74,48], style(
                            color=3,
                            rgbcolor={0,0,255},
                            fillPattern=1)),
                        Polygon(points=[76,-46; 80,-50; 74,-48; 76,-46], style(
                            color=3,
                            rgbcolor={0,0,255},
                            fillColor=3,
                            rgbfillColor={0,0,255},
                            fillPattern=1)),
                        Text(
                          extent=[-80,0; 0,-20],
                          style(
                            color=3,
                            rgbcolor={0,0,255},
                            fillColor=3,
                            rgbfillColor={0,0,255},
                            fillPattern=1),
                          string="QSSDUP")));
                    SRC.Interfaces.outPort out1
                                     annotation (extent=[80,40; 100,60]);
                    SRC.Interfaces.inPort in1
                                   annotation (extent=[-100,-10; -80,10]);
                    SRC.Interfaces.outPort out2
                                     annotation (extent=[80,-60; 100,-40]);

          equation
            iEvent[1] = in1.event;
            iQueue[1] = in1.queue;

            oEvent[1] = out1.event;
            oQueue[1] = out1.queue;
            oEvent[2] = out2.event;
            oQueue[2] = out2.queue;
          end dupevent;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            sout := ext(int(s),0,bag);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            sout.sigma := Modelica.Constants.inf;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            SRC.stdEvent x;
          algorithm
            sout := s;
            numreceived := SRC.numEvents(bag);
            for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              if x.Type == 1 then
                sout.et := x.Type;
                sout.ev1 := x.Value;
              elseif x.Type == 2 then
                sout.ev2 := x.Value;
              elseif x.Type == 3 then
                sout.ev3 := x.Value;
              end if;
            end for;
            sout.sigma := 0;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            SRC.stdEvent y;
          algorithm
            y.Type := 1;
            y.Value := s.ev1;
            SRC.sendEvent(queue[1], y);
            SRC.sendEvent(queue[2], y);
            y.Type := 2;
            y.Value := s.ev2;
            SRC.sendEvent(queue[1], y);
            SRC.sendEvent(queue[2], y);
            y.Type := 3;
            y.Value := s.ev3;
            SRC.sendEvent(queue[1], y);
            SRC.sendEvent(queue[2], y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Integer et;
            Real ev1;
            Real ev2;
            Real ev3;
          end st;

          function initst
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := Modelica.Constants.inf;
            out.et := 0;
            out.ev1 := 0;
            out.ev2 := 0;
            out.ev3 := 0;
          end initst;
        end DupQSS;

        package DEDEVS
        annotation(preferedView="info",
          Window(
            x=0.02,
            y=0.01,
            width=0.2,
            height=0.57,
            library=1,
            autolayout=1),
          version="2.2",
          versionDate="2005-04-15",
          Settings(NewStateSelection=true),
          Documentation(info="<HTML>
<p>
This model represents a simple differential equation:
<br>
<br>
dx/dt=-x + 10<b>s</b>(t-1.76) <br>
<br>
where <b>s</b>() is the unit step function that returns 0 for a negative argument, and 1 for non-negative argument.
<br>
<br>
</p>
<h3><font color=\"#008000\">Simulation</font></h3>
<p>
The <i>System</i> model uses the QSS1 integrator.<br>
The <i>System2</i> model uses the QSS2 integrator.<br>
The <i>System3</i> model uses the QSS3 integrator.<br>
</p>

Each models include a variable, named x, to facilitate the observation of the simulation results.

<br>
</HTML>
"));

          model system
            Real x = QSS1.S.X;

                      annotation (Diagram,
                        experiment(StopTime=10),
                        experimentSetupOutput);
                      Add.add add(name="add") annotation (extent=[-20,0; 0,20]);
                      DESLib.DEVSLib.AuxModels.QSS1 QSS1(quantum=1, name="integ")
                                     annotation (extent=[20,0; 40,20]);
                      Step.step step(
                        firstGeneration=1.76,
                        numGenerations=1,
                        outValue=10,
                        name="step") annotation (extent=[-58,6; -38,26]);
                      AuxModels.BreakLoop breakLoop
                        annotation (extent=[4,-20; 24,0], rotation=180);
          equation
                      connect(add.y, QSS1.inPort1)             annotation (points=[-1,10;
                            21,10],              style(color=0, rgbcolor={0,0,0}));
                      connect(step.outPort1, add.u1)
                        annotation (points=[-39,16; -32,16; -32,15; -19,15],
                                                             style(color=0, rgbcolor={0,0,0}));
                      connect(QSS1.outPort1, breakLoop.IN) annotation (points=[39,10; 52,10; 52,-10;
                            16.8,-10], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop.OUT, add.u2) annotation (points=[13.4,-10; -28,
                            -10; -28,5; -19,5], style(color=0, rgbcolor={0,0,0}));
          end system;

          model system2
          Real x= QSS2.S.X;

                      annotation (Diagram,
                        experiment(StopTime=10),
                        experimentSetupOutput);
                      Add.add add(name="add") annotation (extent=[-20,0; 0,20]);
                      DESLib.DEVSLib.AuxModels.QSS2 QSS2(name="integ")
                        annotation (extent=[20,0; 40,20]);
                      Step.step step(
                        firstGeneration=1.76,
                        numGenerations=1,
                        outValue=10,
                        name="step") annotation (extent=[-58,6; -38,26]);
                      AuxModels.BreakLoop breakLoop
                        annotation (extent=[-6,-18; 14,2], rotation=180);
          equation
                      connect(add.y, QSS2.inPort1)             annotation (points=[-1,10;
                            21,10],               style(color=0, rgbcolor={0,0,0}));
                      connect(step.outPort1, add.u1)
                        annotation (points=[-39,16; -32,16; -32,15; -19,15],
                                                             style(color=0, rgbcolor={0,0,0}));
                      connect(QSS2.outPort1, breakLoop.IN) annotation (points=[39,10;
                  48,10; 48,-8; 6.8,-8],
                                     style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop.OUT, add.u2) annotation (points=[3.4,-8; -30,-8;
                            -30,5; -19,5], style(color=0, rgbcolor={0,0,0}));
          end system2;

          model system3
           Real x = QSS3.S.X;

                      annotation (Diagram,
                        experiment(StopTime=10),
                        experimentSetupOutput);
                      Add.add add(name="add") annotation (extent=[-20,0; 0,20]);
                      DESLib.DEVSLib.AuxModels.QSS3 QSS3(name="Integ")
                                                         annotation (extent=[20,0; 40,20]);
                      Step.step step(
                        firstGeneration=1.76,
                        numGenerations=1,
                        outValue=10,
                        name="step") annotation (extent=[-58,6; -38,26]);
                      AuxModels.BreakLoop breakLoop
                        annotation (extent=[-2,-16; 18,4], rotation=180);
          equation
                      connect(add.y, QSS3.inPort1) annotation (points=[-1,10; 21,10],
                                    style(color=0, rgbcolor={0,0,0}));
                      connect(step.outPort1, add.u1)
                        annotation (points=[-39,16; -32,16; -32,15; -19,15],
                                                             style(color=0, rgbcolor={0,0,0}));
                      connect(QSS3.outPort1, breakLoop.IN) annotation (points=[39,10; 46,10; 46,-6;
                            10.8,-6], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop.OUT, add.u2) annotation (points=[7.4,-6; -26,-6;
                            -26,5; -19,5], style(color=0, rgbcolor={0,0,0}));
          end system3;

        end DEDEVS;

        package FlybackConverter
        annotation(preferedView="info",
          Window(
            x=0.02,
            y=0.01,
            width=0.2,
            height=0.57,
            library=1,
            autolayout=1),
          version="2.2",
          versionDate="2005-04-15",
          Settings(NewStateSelection=true),
          Documentation(info="<HTML>
<p>
This package includes the model of a flyback converter.
</p>

<h3><font color=\"#008000\">Simulation</font></h3>
<p>
The <i>System</i> model uses the QSS1 integrator.<br>
The <i>System2</i> model uses the QSS2 integrator.<br>
The <i>System3</i> model uses the QSS3 integrator.<br>
</p>

Each models include two variables, named x and y, that show the original and converted currents.

<br>
</HTML>
"));
          model system
            Real x = QSS1_1.S.X;
            Real y = QSS1_2.S.X;

           Square.square square(
                        name="square",
                        amplitude=1,
                        dutyCycle=66.6667,
                        frecuency=59880.2)
                                       annotation (extent=[-100,82; -80,102]);
                      Switch.switch switch(name="switch", level=0.5)
                        annotation (extent=[-40,18; -20,38]);
                      Multiplier.multiplier multiplier1(name="mult1")
                        annotation (extent=[-60,-40; -40,-20]);
                      annotation (Diagram,
                        experiment(StopTime=0.001),
                        experimentSetupOutput);
                      Constant.ConstantG constantG(outValue=40, name="constant")
                        annotation (extent=[-68,4; -48,24]);
                      Gain.gain gain1(              name="gain1", K=5000)
                        annotation (extent=[-10,20; 10,40]);
                       DESLib.DEVSLib.AuxModels.QSS1 QSS1_1(
                        quantum=0.1,
                        startX=0,
                        name="integ1") annotation (extent=[20,20; 40,40]);
                      Add.add add(name="add1") annotation (extent=[-26,-40; -6,-20]);
                      Gain.gain gain2(name="gain2", K=45454.5)
                        annotation (extent=[0,-40; 20,-20]);
                       DESLib.DEVSLib.AuxModels.QSS1 QSS1_2(
                        quantum=0.1,
                        startX=0,
                        name="integ2") annotation (extent=[34,-40; 54,-20]);
                      Gain.gain gain4(name="gain4", K=-1)
                        annotation (extent=[-2,54; 18,74], rotation=180);
                      Gain.gain gain3(name="gain3", K=0.2)
                        annotation (extent=[2,-66; 22,-46], rotation=180);
                      DupQSS.dupevent dupevent(name="dupevent")
                                               annotation (extent=[-96,56; -76,76]);
                      DupQSS.dupevent dupevent1(name="dupevent2")
                                                annotation (extent=[58,-40; 78,-20]);
                      AuxModels.BreakLoop breakLoop
                        annotation (extent=[44,-66; 64,-46], rotation=180);
                      AuxModels.BreakLoop breakLoop1
                        annotation (extent=[48,54; 68,74], rotation=180);
          equation
          connect(constantG.outPort1, switch.u3) annotation (points=[-49,14; -45.5,14;
                            -45.5,21; -39,21],           style(color=0, rgbcolor={0,0,0}));
                      connect(switch.y, gain1.inPort1) annotation (points=[-21,28; -16,28;
                            -16,30; -9,30],
                          style(color=0, rgbcolor={0,0,0}));
                      connect(gain1.outPort1, QSS1_1.inPort1) annotation (points=[9,30; 21,
                            30], style(color=0, rgbcolor={0,0,0}));
                      connect(QSS1_1.outPort1, multiplier1.u2) annotation (points=[39,30;
                            62,30; 62,-2; -66,-2; -66,-35; -59,-35],   style(color=0,
                            rgbcolor={0,0,0}));
                      connect(multiplier1.y, add.u1) annotation (points=[-41,-30; -32,-30;
                            -32,-25; -25,-25], style(color=0, rgbcolor={0,0,0}));
                      connect(add.y, gain2.inPort1) annotation (points=[-7,-30; 1,-30],
                          style(color=0, rgbcolor={0,0,0}));
                      connect(gain2.outPort1, QSS1_2.inPort1) annotation (points=[19,-30;
                            35,-30], style(color=0, rgbcolor={0,0,0}));
                      connect(gain4.outPort1, switch.u1) annotation (points=[-1,64; -52,64;
                            -52,35; -39,35], style(color=0, rgbcolor={0,0,0}));
                      connect(gain3.outPort1, add.u2) annotation (points=[3,-56; -30,-56;
                            -30,-35; -25,-35], style(color=0, rgbcolor={0,0,0}));
                      connect(square.outPort1, dupevent.in1) annotation (points=[-81,92;
                            -78,92; -78,78; -100,78; -100,66; -95,66], style(color=0,
                            rgbcolor={0,0,0}));
                      connect(dupevent.out1, switch.u2) annotation (points=[-77,71; -60,71;
                            -60,28; -39,28], style(color=0, rgbcolor={0,0,0}));
                      connect(dupevent.out2, multiplier1.u1) annotation (points=[-77,61;
                            -74,61; -74,-25; -59,-25], style(color=0, rgbcolor={0,0,0}));
                      connect(QSS1_2.outPort1, dupevent1.in1) annotation (points=[53,-30;
                            55,-30; 55,-30; 59,-30], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop.OUT, gain3.inPort1) annotation (points=[53.4,-56; 21,
                            -56], style(color=0, rgbcolor={0,0,0}));
                      connect(dupevent1.out2, breakLoop.IN) annotation (points=[77,-35;
                            82,-35; 82,-56; 56.8,-56], style(color=0, rgbcolor={0,0,0}));
                      connect(gain4.inPort1, breakLoop1.OUT) annotation (points=[17,64;
                            57.4,64], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop1.IN, dupevent1.out1) annotation (points=[60.8,64;
                            82,64; 82,-25; 77,-25], style(color=0, rgbcolor={0,0,0}));

          end system;

          model system2
            Real x = integratorQSS2_1.S.X;
            Real y =  integratorQSS2_2.S.X;

            Square.square square(
                        name="square",
                        amplitude=1,
                        dutyCycle=66.6667,
                        frecuency=59880.2)
                                       annotation (extent=[-100,82; -80,102]);
                      Switch.switch switch(name="switch", level=0.5)
                        annotation (extent=[-40,18; -20,38]);
                      Multiplier.multiplier multiplier1(name="mult1")
                        annotation (extent=[-60,-40; -40,-20]);
                      annotation (Diagram,
                        experiment(StopTime=0.001),
                        experimentSetupOutput);
                      Constant.ConstantG constantG(outValue=40, name="constant")
                        annotation (extent=[-68,4; -48,24]);
                      Gain.gain gain1(              name="gain1", K=5000)
                        annotation (extent=[-10,20; 10,40]);
                      Add.add add(name="add1") annotation (extent=[-26,-40; -6,-20]);
                      Gain.gain gain2(name="gain2", K=45454.5)
                        annotation (extent=[2,-40; 22,-20]);
                      Gain.gain gain4(name="gain4", K=-1)
                        annotation (extent=[-2,54; 18,74], rotation=180);
                      Gain.gain gain3(name="gain3", K=0.2)
                        annotation (extent=[2,-66; 22,-46], rotation=180);
                      DupQSS.dupevent dupevent(name="dupevent")
                                               annotation (extent=[-96,56; -76,76]);
                      DupQSS.dupevent dupevent1(name="dupevent2")
                                                annotation (extent=[58,-40; 78,-20]);
                       DESLib.DEVSLib.AuxModels.QSS2 integratorQSS2_1(
                        quantum=0.1,
                        startX=0,
                        name="integ1") annotation (extent=[20,22; 40,42]);
                       DESLib.DEVSLib.AuxModels.QSS2 integratorQSS2_2(
                        quantum=0.1,
                        startX=0,
                        name="integ2") annotation (extent=[30,-40; 50,-20]);
                      AuxModels.BreakLoop breakLoop1
                        annotation (extent=[52,54; 72,74], rotation=180);
                      AuxModels.BreakLoop breakLoop2
                        annotation (extent=[52,-66; 72,-46], rotation=180);
          equation
                      connect(constantG.outPort1, switch.u3) annotation (points=[-49,14; -45.5,14;
                            -45.5,21; -39,21],           style(color=0, rgbcolor={0,0,0}));
                      connect(switch.y, gain1.inPort1) annotation (points=[-21,28; -16,28;
                            -16,30; -9,30],
                          style(color=0, rgbcolor={0,0,0}));
                      connect(multiplier1.y, add.u1) annotation (points=[-41,-30; -32,-30;
                            -32,-25; -25,-25], style(color=0, rgbcolor={0,0,0}));
                      connect(add.y, gain2.inPort1) annotation (points=[-7,-30; 3,-30],
                          style(color=0, rgbcolor={0,0,0}));
                      connect(gain4.outPort1, switch.u1) annotation (points=[-1,64; -52,64;
                            -52,35; -39,35], style(color=0, rgbcolor={0,0,0}));
                      connect(gain3.outPort1, add.u2) annotation (points=[3,-56; -30,-56;
                            -30,-35; -25,-35], style(color=0, rgbcolor={0,0,0}));
                      connect(square.outPort1, dupevent.in1) annotation (points=[-81,92;
                            -78,92; -78,78; -100,78; -100,66; -95,66], style(color=0,
                            rgbcolor={0,0,0}));
                      connect(dupevent.out1, switch.u2) annotation (points=[-77,71; -60,71;
                            -60,28; -39,28], style(color=0, rgbcolor={0,0,0}));
                      connect(dupevent.out2, multiplier1.u1) annotation (points=[-77,61;
                            -74,61; -74,-25; -59,-25], style(color=0, rgbcolor={0,0,0}));
                      connect(gain1.outPort1, integratorQSS2_1.inPort1) annotation (points=[9,30; 14,
                            30; 14,32; 21,32],           style(color=0, rgbcolor={0,0,0}));
                      connect(integratorQSS2_1.outPort1, multiplier1.u2) annotation (points=[39,32;
                            62,32; 62,-10; -66,-10; -66,-35; -59,-35],        style(color=0,
                            rgbcolor={0,0,0}));
                      connect(gain2.outPort1, integratorQSS2_2.inPort1) annotation (points=[21,-30;
                            26,-30; 26,-30; 31,-30],          style(color=0, rgbcolor={0,0,
                              0}));
                      connect(integratorQSS2_2.outPort1, dupevent1.in1) annotation (points=[49,-30;
                            54.5,-30; 54.5,-30; 59,-30],          style(color=0, rgbcolor={
                              0,0,0}));
                      connect(gain4.inPort1, breakLoop1.OUT) annotation (points=[17,64;
                            61.4,64], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop1.IN, dupevent1.out1) annotation (points=[64.8,64;
                            80,64; 80,-25; 77,-25], style(color=0, rgbcolor={0,0,0}));
                      connect(gain3.inPort1, breakLoop2.OUT) annotation (points=[21,-56;
                            61.4,-56], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop2.IN, dupevent1.out2) annotation (points=[64.8,-56;
                            80,-56; 80,-35; 77,-35], style(color=0, rgbcolor={0,0,0}));
          end system2;

          model system3
            Real x = integratorQSS3_1.S.X;
            Real y = integratorQSS3_2.S.X;

           Square.square square(
                        name="square",
                        amplitude=1,
                        dutyCycle=66.6667,
                        frecuency=59880.2)
                                       annotation (extent=[-100,82; -80,102]);
                      Switch.switch switch(name="switch", level=0.5)
                        annotation (extent=[-40,18; -20,38]);
                      Multiplier.multiplier multiplier1(name="mult1")
                        annotation (extent=[-60,-40; -40,-20]);
                      annotation (Diagram,
                        experiment(StopTime=0.001),
                        experimentSetupOutput);
                      Constant.ConstantG constantG(outValue=40, name="constant")
                        annotation (extent=[-68,4; -48,24]);
                      Gain.gain gain1(              name="gain1", K=5000)
                        annotation (extent=[-10,20; 10,40]);
                      Add.add add(name="add1") annotation (extent=[-26,-40; -6,-20]);
                      Gain.gain gain2(name="gain2", K=45454.5)
                        annotation (extent=[2,-40; 22,-20]);
                      Gain.gain gain4(name="gain4", K=-1)
                        annotation (extent=[-2,54; 18,74], rotation=180);
                      Gain.gain gain3(name="gain3", K=0.2)
                        annotation (extent=[2,-66; 22,-46], rotation=180);
                      DupQSS.dupevent dupevent(name="dupevent")
                                               annotation (extent=[-96,56; -76,76]);
                      DupQSS.dupevent dupevent1(name="dupevent2")
                                                annotation (extent=[58,-40; 78,-20]);
                       DESLib.DEVSLib.AuxModels.QSS3 integratorQSS3_1(
                        quantum=0.1,
                        startX=0,
                        name="integ1") annotation (extent=[20,20; 40,40]);
                       DESLib.DEVSLib.AuxModels.QSS3 integratorQSS3_2(
                        quantum=0.1,
                        startX=0,
                        name="integ1") annotation (extent=[30,-40; 50,-20]);
                      AuxModels.BreakLoop breakLoop1
                        annotation (extent=[56,54; 76,74], rotation=180);
                      AuxModels.BreakLoop breakLoop2
                        annotation (extent=[56,-66; 76,-46], rotation=180);
          equation
                      connect(constantG.outPort1, switch.u3) annotation (points=[-49,14; -45.5,14;
                            -45.5,21; -39,21],           style(color=0, rgbcolor={0,0,0}));
                      connect(switch.y, gain1.inPort1) annotation (points=[-21,28; -16,28;
                            -16,30; -9,30],
                          style(color=0, rgbcolor={0,0,0}));
                      connect(multiplier1.y, add.u1) annotation (points=[-41,-30; -32,-30;
                            -32,-25; -25,-25], style(color=0, rgbcolor={0,0,0}));
                      connect(add.y, gain2.inPort1) annotation (points=[-7,-30; 3,-30],
                          style(color=0, rgbcolor={0,0,0}));
                      connect(gain4.outPort1, switch.u1) annotation (points=[-1,64; -52,64;
                            -52,35; -39,35], style(color=0, rgbcolor={0,0,0}));
                      connect(gain3.outPort1, add.u2) annotation (points=[3,-56; -30,-56;
                            -30,-35; -25,-35], style(color=0, rgbcolor={0,0,0}));
                      connect(square.outPort1, dupevent.in1) annotation (points=[-81,92;
                            -78,92; -78,78; -100,78; -100,66; -95,66], style(color=0,
                            rgbcolor={0,0,0}));
                      connect(dupevent.out1, switch.u2) annotation (points=[-77,71; -60,71;
                            -60,28; -39,28], style(color=0, rgbcolor={0,0,0}));
                      connect(dupevent.out2, multiplier1.u1) annotation (points=[-77,61;
                            -74,61; -74,-25; -59,-25], style(color=0, rgbcolor={0,0,0}));
                      connect(gain1.outPort1, integratorQSS3_1.inPort1) annotation (points=[9,30; 16,
                            30; 16,30; 21,30],           style(color=0, rgbcolor={0,0,0}));
                      connect(gain2.outPort1, integratorQSS3_2.inPort1) annotation (points=[21,-30;
                            26,-30; 26,-30; 31,-30],          style(color=0, rgbcolor={0,0,
                              0}));
                      connect(integratorQSS3_2.outPort1, dupevent1.in1) annotation (points=[49,-30;
                            54.5,-30; 54.5,-30; 59,-30],          style(color=0, rgbcolor={
                              0,0,0}));
                      connect(integratorQSS3_1.outPort1, multiplier1.u2) annotation (points=[39,30;
                            50,30; 50,-2; -68,-2; -68,-35; -59,-35],        style(color=0,
                            rgbcolor={0,0,0}));
                      connect(gain4.inPort1, breakLoop1.OUT) annotation (points=[17,64;
                            65.4,64], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop1.IN, dupevent1.out1) annotation (points=[68.8,64;
                            80,64; 80,-25; 77,-25], style(color=0, rgbcolor={0,0,0}));
                      connect(dupevent1.out2, breakLoop2.IN) annotation (points=[77,-35;
                            80,-35; 80,-56; 68.8,-56], style(color=0, rgbcolor={0,0,0}));
                      connect(breakLoop2.OUT, gain3.inPort1) annotation (points=[65.4,-56;
                            21,-56], style(color=0, rgbcolor={0,0,0}));

          end system3;

          model system3dup
            Real x = integratorQSS3_1.S.X;
            Real y = integratorQSS3_2.S.X;

           Square.square square(
                        name="square",
                        amplitude=1,
                        dutyCycle=66.6667,
                        frecuency=59880.2)
                                       annotation (extent=[-98,16; -78,36]);
          Switch.switch switch(name="switch", level=0.5)
                        annotation (extent=[-40,18; -20,38]);
                      Multiplier.multiplier multiplier1(name="mult1")
                        annotation (extent=[-60,-40; -40,-20]);
                      annotation (Diagram,
                        experiment(StopTime=0.001),
                        experimentSetupOutput);
                      Constant.ConstantG constantG(outValue=40, name="constant")
                        annotation (extent=[-68,4; -48,24]);
           Gain.gain gain1(              name="gain1", K=5000)
                        annotation (extent=[-10,20; 10,40]);
                      Add.add add(name="add1") annotation (extent=[-26,-40; -6,-20]);
                      Gain.gain gain2(name="gain2", K=45454.5)
                        annotation (extent=[2,-40; 22,-20]);
            Gain.gain gain4(name="gain4", K=-1)
                        annotation (extent=[-2,54; 18,74], rotation=180);
                      Gain.gain gain3(name="gain3", K=0.2)
                        annotation (extent=[2,-66; 22,-46], rotation=180);
            DESLib.DEVSLib.AuxModels.QSS3 integratorQSS3_1(
                        quantum=0.1,
                        startX=0,
                        name="integ1") annotation (extent=[20,20; 40,40]);
                       DESLib.DEVSLib.AuxModels.QSS3 integratorQSS3_2(
                        quantum=0.1,
                        startX=0,
                        name="integ1") annotation (extent=[30,-40; 50,-20]);
                      AuxModels.BreakLoop breakLoop1
                        annotation (extent=[56,54; 76,74], rotation=180);
                      AuxModels.BreakLoop breakLoop2
                        annotation (extent=[56,-66; 76,-46], rotation=180);
            AuxModels.DUP dUP annotation (extent=[46,-40;66,-20]);
            AuxModels.DUP dUP1 annotation (extent=[-84,16;-64,36]);
          equation
           connect(constantG.outPort1, switch.u3) annotation (points=[-49,14; -45.5,14;
                            -45.5,21; -39,21],           style(color=0, rgbcolor={0,0,0}));
                      connect(switch.y, gain1.inPort1) annotation (points=[-21,28; -16,28;
                            -16,30; -9,30],
                          style(color=0, rgbcolor={0,0,0}));
                      connect(multiplier1.y, add.u1) annotation (points=[-41,-30; -32,-30;
                            -32,-25; -25,-25], style(color=0, rgbcolor={0,0,0}));
                      connect(add.y, gain2.inPort1) annotation (points=[-7,-30; 3,-30],
                          style(color=0, rgbcolor={0,0,0}));
                      connect(gain4.outPort1, switch.u1) annotation (points=[-1,64; -52,64;
                            -52,35; -39,35], style(color=0, rgbcolor={0,0,0}));
                      connect(gain3.outPort1, add.u2) annotation (points=[3,-56; -30,-56;
                            -30,-35; -25,-35], style(color=0, rgbcolor={0,0,0}));
            connect(gain1.outPort1, integratorQSS3_1.inPort1) annotation (points=[9,30;21,30], style(color=0, rgbcolor={0,0,0}));
            connect(gain2.outPort1, integratorQSS3_2.inPort1) annotation (points=[21,-30;31,-30], style(color=0, rgbcolor={0,0,0}));
            connect(integratorQSS3_1.outPort1, multiplier1.u2) annotation (points=[39,30;50,30;50,-2;-68,-2;-68,-35;-59,-35],style(color=0, rgbcolor={0,0,0}));
            connect(gain4.inPort1, breakLoop1.OUT) annotation (points=[17,64;65.4,64], style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop2.OUT, gain3.inPort1) annotation (points=[65.4,-56;21,-56], style(color=0, rgbcolor={0,0,0}));
            connect(integratorQSS3_2.outPort1, dUP.in1) annotation (points=[49,-30;53.4,-30],style(color=0, rgbcolor={0,0,0}));
            connect(dUP.out1, breakLoop1.IN) annotation (points=[56.6,-28;74,-28;74,64;68.8,64],  style(color=0, rgbcolor={0,0,0}));
            connect(dUP.out2, breakLoop2.IN) annotation (points=[56.6,-32;74,-32;74,-56;68.8,-56], style(color=0, rgbcolor={0,0,0}));
            connect(dUP1.out2, multiplier1.u1) annotation (points=[-73.4,24;-72,24;-72,-25;-59,-25], style(color=0, rgbcolor={0,0,0}));
            connect(dUP1.out1, switch.u2) annotation (points=[-73.4,28;-39,28], style(color=0, rgbcolor={0,0,0}));
            connect(square.outPort1, dUP1.in1) annotation (points=[-79,26;-76.6,26],style(color=0, rgbcolor={0,0,0}));
          end system3dup;
        end FlybackConverter;

        package LotkaVolterra
        annotation(preferedView="info",
          Window(
            x=0.02,
            y=0.01,
            width=0.2,
            height=0.57,
            library=1,
            autolayout=1),
          version="2.2",
          versionDate="2005-04-15",
          Settings(NewStateSelection=true),
          Documentation(info="<HTML>
<p>
This package includes the implementation of the predator-prey interactions described by the Lotka-Volterra equations:
<br>
<br>
dx/dt=A*x-B*x*y
<br>
dy/dt=-C*y+D*x*y
<br>
<br>
<h3><font color=\"#008000\">Simulation</font></h3>
<p>
The <i>System</i> model uses the QSS1 integrator.<br>
The <i>System2</i> model uses the QSS2 integrator.<br>
The <i>System3</i> model uses the QSS3 integrator.<br>
</p>

Each models include two variables, named prey and predator, that show the evolution of the prey and predator populations.


<br>
</HTML>
"));

          model systemdup
            Real prey= QSS1_1.S.y;
            Real predator= QSS1_2.S.y;

            Multiplier.multiplier multiplier(name="mult")
              annotation (extent=[-54,-8;-34,12]);
            Add.add add(
              name="add",
              c1=-0.1,
              c2=0.1) annotation (extent=[-4,14;16,34]);
            Add.add add1(
              c1=0.1,
              c2=-0.1,
              name="add1") annotation (extent=[-4,-18;16,2]);
            DESLib.DEVSLib.AuxModels.QSS1 QSS1_1(
              quantum=0.001,
              startX=0.5,
              name="integ1") annotation (extent=[26, 14;46,34]);
            DESLib.DEVSLib.AuxModels.QSS1 QSS1_2(
              quantum=0.001,
              startX=0.5,
              name="integ2") annotation (extent=[26,-18;46,2]);
            annotation (Diagram,    experiment(StopTime=100),    experimentSetupOutput);
            AuxModels.BreakLoop breakLoop1
              annotation (extent=[10,2;30,24], rotation=180);
            AuxModels.BreakLoop breakLoop2
              annotation (extent=[10,-30;30,-10], rotation=180);
            AuxModels.BreakLoop breakLoop3
              annotation (extent=[-36,-8;-16,12]);
            AuxModels.DUP dUP annotation (extent=[-28,-8;-8,12]);
            AuxModels.DUP dUP1 annotation (extent=[42,14;62,34]);
            AuxModels.DUP dUP2 annotation (extent=[42,-18;62,2]);
          equation

            connect(add.y, QSS1_1.inPort1)     annotation (points=[15,24;27,24], style(color=0, rgbcolor={0,0,0}));
            connect(add1.y, QSS1_2.inPort1)    annotation (points=[15,-8;27,-8], style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop1.OUT, add.u2)    annotation (points=[19.4,13; -8,
                  13; -8,19; -3,19],                                                           style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop2.OUT, add1.u2)   annotation (points=[19.4,-20;-8,-20;-8,-13;-3,-13], style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop3.IN, multiplier.y) annotation (points=[-28.8,2;-35,2], style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop3.OUT, dUP.in1)   annotation (points=[-25.4,2;-20.6,2], style(color=0, rgbcolor={0,0,0}));
            connect(dUP.out1, add.u1)          annotation (points=[-17.4,4;-12,4;-12,29;-3,29], style(color=0, rgbcolor={0,0,0}));
            connect(dUP.out2, add1.u1)         annotation (points=[-17.4,0;-12,0;-12,-3;-3,-3], style(color=0, rgbcolor={0,0,0}));
            connect(QSS1_1.outPort1, dUP1.in1) annotation (points=[45,24;49.4,24], style(color=0, rgbcolor={0,0,0}));
            connect(QSS1_2.outPort1, dUP2.in1) annotation (points=[45,-8;49.4,-8], style(color=0, rgbcolor={0,0,0}));
            connect(dUP2.out2, breakLoop2.IN)  annotation (points=[52.6,-10;54,-10;54,-20;22.8,-20], style(color=0, rgbcolor={0,0,0}));
            connect(dUP1.out2, breakLoop1.IN)  annotation (points=[52.6,22; 54,
                  22; 54,13; 22.8,13],                                                           style(color=0, rgbcolor={0,0,0}));
            connect(dUP1.out1, multiplier.u1)  annotation (points=[52.6,26;56,26;56,36;-56,36;-56,7;-53,7], style(color=0, rgbcolor={0,0,0}));
            connect(dUP2.out1, multiplier.u2)  annotation (points=[52.6,-6;56,-6;56,-26;-56,-26;-56,-3;-53,-3], style(color=0, rgbcolor={0,0,0}));
          end systemdup;

          model system2dup
            Real prey = QSS2_1.S.y0;
            Real predator = QSS2_2.S.y0;

            Multiplier.multiplier multiplier1(
                                             name="mult")
              annotation (extent=[-60,12; -40,32]);
            Add.add add2(
              name="add",
              c1=-0.1,
              c2=0.1) annotation (extent=[-10,34; 10,54]);
            Add.add add3(
              c1=0.1,
              c2=-0.1,
              name="add1") annotation (extent=[-10,2; 10,22]);
            AuxModels.QSS2 QSS2_1(
              quantum=0.001,
              startX=0.5,
              name="integ1") annotation (extent=[20,34; 40,54]);
            AuxModels.QSS2 QSS2_2(
              quantum=0.001,
              startX=0.5,
              name="integ2") annotation (extent=[20,2; 40,22]);
            AuxModels.BreakLoop breakLoop4
              annotation (extent=[4,22; 24,44],rotation=180);
            AuxModels.BreakLoop breakLoop5
              annotation (extent=[4,-10; 24,10],  rotation=180);
            AuxModels.BreakLoop breakLoop6
              annotation (extent=[-42,12; -22,32]);
            AuxModels.DUP dUP3
                              annotation (extent=[-34,12; -14,32]);
            AuxModels.DUP dUP4 annotation (extent=[36,34; 56,54]);
            AuxModels.DUP dUP5 annotation (extent=[36,2; 56,22]);
          equation
            connect(add2.y,QSS2_1. inPort1)    annotation (points=[9,44; 21,44], style(color=0, rgbcolor={0,0,0}));
            connect(add3.y,QSS2_2. inPort1)    annotation (points=[9,12; 21,12], style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop4.OUT, add2.u2)   annotation (points=[13.4,33; -14,33; -14,
                  39; -9,39],                                                                  style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop5.OUT,add3. u2)   annotation (points=[13.4,
                  7.34764e-017; -14,7.34764e-017; -14,7; -9,7],                                    style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop6.IN, multiplier1.y)
                                                 annotation (points=[-34.8,22; -41,22],
                                                                                     style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop6.OUT, dUP3.in1)  annotation (points=[-31.4,22; -26.6,22],
                                                                                     style(color=0, rgbcolor={0,0,0}));
            connect(dUP3.out1, add2.u1)        annotation (points=[-23.4,24; -18,24; -18,
                  49; -9,49],                                                                   style(color=0, rgbcolor={0,0,0}));
            connect(dUP3.out2, add3.u1)        annotation (points=[-23.4,20; -18,20; -18,
                  17; -9,17],                                                                   style(color=0, rgbcolor={0,0,0}));
            connect(QSS2_1.outPort1,dUP4. in1) annotation (points=[39,44; 43.4,44],style(color=0, rgbcolor={0,0,0}));
            connect(QSS2_2.outPort1,dUP5. in1) annotation (points=[39,12; 43.4,12],style(color=0, rgbcolor={0,0,0}));
            connect(dUP5.out2,breakLoop5. IN)  annotation (points=[46.6,10; 48,
                  10; 48,-3.4289e-016; 16.8,-3.4289e-016],                                           style(color=0, rgbcolor={0,0,0}));
            connect(dUP4.out2,breakLoop4. IN)  annotation (points=[46.6,42; 48,42; 48,33;
                  16.8,33],                                                                      style(color=0, rgbcolor={0,0,0}));
            connect(dUP4.out1, multiplier1.u1) annotation (points=[46.6,46; 50,46; 50,56;
                  -62,56; -62,27; -59,27],                                                                  style(color=0, rgbcolor={0,0,0}));
            connect(dUP5.out1, multiplier1.u2) annotation (points=[46.6,14; 50,14; 50,-6;
                  -62,-6; -62,17; -59,17],                                                                      style(color=0, rgbcolor={0,0,0}));
          end system2dup;

          model system3dup
            Real prey = QSS3_1.S.y0;
            Real predator = QSS3_2.S.y0;

            Multiplier.multiplier multiplier1(
                                             name="mult")
              annotation (extent=[-60,-12; -40,8]);
            Add.add add2(
              name="add",
              c1=-0.1,
              c2=0.1) annotation (extent=[-10,10; 10,30]);
            Add.add add3(
              c1=0.1,
              c2=-0.1,
              name="add1") annotation (extent=[-10,-22; 10,-2]);
            AuxModels.QSS3 QSS3_1(
              quantum=0.001,
              startX=0.5,
              name="integ1") annotation (extent=[20,10; 40,30]);
            AuxModels.QSS3 QSS3_2(
              quantum=0.001,
              startX=0.5,
              name="integ2") annotation (extent=[20,-22; 40,-2]);
            AuxModels.BreakLoop breakLoop4
              annotation (extent=[4,-2; 24,20],rotation=180);
            AuxModels.BreakLoop breakLoop5
              annotation (extent=[4,-34; 24,-14], rotation=180);
            AuxModels.BreakLoop breakLoop6
              annotation (extent=[-42,-12; -22,8]);
            AuxModels.DUP dUP3
                              annotation (extent=[-34,-12; -14,8]);
            AuxModels.DUP dUP4 annotation (extent=[36,10; 56,30]);
            AuxModels.DUP dUP5 annotation (extent=[36,-22; 56,-2]);
          equation
            connect(add2.y,QSS3_1. inPort1)    annotation (points=[9,20; 21,20], style(color=0, rgbcolor={0,0,0}));
            connect(add3.y,QSS3_2. inPort1)    annotation (points=[9,-12; 21,-12],
                                                                                 style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop4.OUT,add2. u2)   annotation (points=[13.4,9; -14,9; -14,15;
                  -9,15],                                                                      style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop5.OUT,add3. u2)   annotation (points=[13.4,-24; -14,-24; -14,
                  -17; -9,-17],                                                                    style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop6.IN,multiplier1. y)
                                                 annotation (points=[-34.8,-2; -41,-2],
                                                                                     style(color=0, rgbcolor={0,0,0}));
            connect(breakLoop6.OUT,dUP3. in1)  annotation (points=[-31.4,-2; -26.6,-2],
                                                                                     style(color=0, rgbcolor={0,0,0}));
            connect(dUP3.out1,add2. u1)        annotation (points=[-23.4,0; -18,0; -18,25;
                  -9,25],                                                                       style(color=0, rgbcolor={0,0,0}));
            connect(dUP3.out2,add3. u1)        annotation (points=[-23.4,-4; -18,-4; -18,
                  -7; -9,-7],                                                                   style(color=0, rgbcolor={0,0,0}));
            connect(QSS3_1.outPort1,dUP4. in1) annotation (points=[39,20; 43.4,20],style(color=0, rgbcolor={0,0,0}));
            connect(QSS3_2.outPort1,dUP5. in1) annotation (points=[39,-12; 43.4,-12],
                                                                                   style(color=0, rgbcolor={0,0,0}));
            connect(dUP5.out2,breakLoop5. IN)  annotation (points=[46.6,-14; 48,-14; 48,
                  -24; 16.8,-24],                                                                    style(color=0, rgbcolor={0,0,0}));
            connect(dUP4.out2,breakLoop4. IN)  annotation (points=[46.6,18; 48,18; 48,9;
                  16.8,9],                                                                       style(color=0, rgbcolor={0,0,0}));
            connect(dUP4.out1,multiplier1. u1) annotation (points=[46.6,22; 50,22; 50,32;
                  -62,32; -62,3; -59,3],                                                                    style(color=0, rgbcolor={0,0,0}));
            connect(dUP5.out1,multiplier1. u2) annotation (points=[46.6,-10; 50,-10; 50,
                  -30; -62,-30; -62,-7; -59,-7],                                                                style(color=0, rgbcolor={0,0,0}));
          end system3dup;
        end LotkaVolterra;
      end QSSIntegration;

      package ControlledTanks
        "Hybrid model of a system with two tanks controlled by a discrete controller"
      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package includes the implementation of the <a href=\"Modelica://Modelica.StateGraph.Examples.ControlledTanks\">Controlled Tanks</a> model described in the StateGraph library.
It is composed of two tanks and three valves, implemented as continuous-time models, and a discrete-event controller.
</p>

<p>
The behavior of the system is detailed in the StateGraph library. <br>
Basically the controller fills the first tank, then the second tank (while the first gets empty) and finally empties the second tank. The normal process can be interrupted with the start, stop and shut buttons.
</p>

<p>
The controller in DEVSLib has been implemented in two ways:
<ul>
<li> As a Atomic DEVS model, including the logic of the controller in the internal and external transition functions.
<li> As a Coupled DEVS model, describing the actions of the controller with smaller and simpler components.
</ul>

<h3><font color=\"#008000\">Simulation</font></h3>
<p>
The <i>systemAtomic</i> model includes the implementation of the system usign the atomic discrete controller.<br>
The <i>systemCoupled</i> model includes the implementation using the coupled discrete controller.
</p>

Both models include two variables to facilitate the observation of the simulation results, and check the level of both tanks.



<br>
</HTML>
"));
        import DEVSLib;

      package Utilities "Utility components for the examples"
        model TankController "Controller for tank system"
            extends Modelica.StateGraph.Interfaces.PartialStateGraphIcon;
                parameter Modelica.StateGraph.Temporary.SetRealParameter limit=0.98
              "Limit level of tank 1"
                  annotation (extent=[-64,76; -44,96]);
                parameter Modelica.StateGraph.Temporary.SetRealParameter
              waitTime =                                                          3
              "Wait time"
                  annotation (extent=[-91,76; -71,96]);

                Modelica.StateGraph.InitialStep s1(nIn=2)
                               annotation (extent=[-72,30; -52,50]);
                MakeProduct makeProduct(limit=limit, waitTime=waitTime)
                  annotation (extent=[-20,25; 10,55]);
                Modelica.StateGraph.Transition T1(condition=start)
                                               annotation (extent=[-50,50; -30,30]);
                Modelica.StateGraph.Transition T2(condition=level2 < 0.001)
                  annotation (extent=[27,50; 47,30]);
                Modelica.StateGraph.Transition T3(condition=stop)
                  annotation (extent=[-33,-11; -13,9],     rotation=-90);
                Modelica.StateGraph.Step s2(nOut=2)
                        annotation (extent=[-50,-60; -30,-40]);
                Modelica.StateGraph.Transition T4(condition=start)
                  annotation (extent=[0,-11; 20,9],     rotation=90);
                Modelica.StateGraph.Transition T5(condition=shut)
                                              annotation (extent=[-6,-60; 14,-40]);
                Modelica.StateGraph.Step emptyTanks
                                annotation (extent=[22,-60; 42,-40]);
                Modelica.StateGraph.Transition T6(condition=level1 + level2 < 0.001)
                  annotation (extent=[45,-60; 65,-40]);
                Modelica.Blocks.Interfaces.BooleanInput start
                  annotation (extent=[-120, 50; -100, 70]);
                Modelica.Blocks.Interfaces.BooleanInput stop
                  annotation (extent=[-120, -10; -100, 10]);
                Modelica.Blocks.Interfaces.BooleanInput shut
                  annotation (extent=[-120, -70; -100, -50]);
                Modelica.Blocks.Interfaces.RealInput level1
                  annotation (extent=[-70, -120; -50, -100], rotation=90);
                Modelica.Blocks.Interfaces.RealInput level2
                  annotation (extent=[50, -120; 70, -100], rotation=90);
                Modelica.Blocks.Interfaces.BooleanOutput valve1
                  annotation (extent=[100, 55; 110, 65]);
                Modelica.Blocks.Interfaces.BooleanOutput valve2
                  annotation (extent=[100, -5; 110, 5]);
                Modelica.Blocks.Interfaces.BooleanOutput valve3
                  annotation (extent=[100, -65; 110, -55]);
                Modelica.Blocks.Sources.BooleanExpression setValve1(y=makeProduct.fillTank1.active)
                  annotation (extent=[20,73; 80,92]);
                Modelica.Blocks.Sources.BooleanExpression setValve2(y=makeProduct.fillTank2.active or emptyTanks.active)
                  annotation (extent=[-25,-89; 80,-68]);
                Modelica.Blocks.Sources.BooleanExpression setValve3(y=makeProduct.emptyTank2.active or emptyTanks.active)
                  annotation (extent=[-26,-100; 80,-80]);
        equation

                annotation (
                  Diagram(Rectangle(extent=[-100,100; 100,-100],   style(color=0,
                          rgbcolor={0,0,0}))),
                  Icon(
                    Text(
                      extent=[-100, 68; -32, 54],
                      style(
                        color=0,
                        fillColor=0,
                        fillPattern=1),
                      string="start"),
                    Text(
                      extent=[-100, 6; -32, -8],
                      style(
                        color=0,
                        fillColor=0,
                        fillPattern=1),
                      string="stop"),
                    Text(
                      extent=[-100, -54; -32, -68],
                      style(
                        color=0,
                        fillColor=0,
                        fillPattern=1),
                      string="shut"),
                    Text(
                      extent=[-94, -82; -18, -96],
                      style(
                        color=0,
                        fillColor=0,
                        fillPattern=1),
                      string="level1"),
                    Text(
                      extent=[24, -84; 96, -98],
                      style(
                        color=0,
                        fillColor=0,
                        fillPattern=1),
                      string="level2"),
                    Text(
                      extent=[31, 68; 99, 54],
                      style(
                        color=0,
                        fillColor=0,
                        fillPattern=1),
                      string="valve1"),
                    Text(
                      extent=[33, 9; 101, -5],
                      style(
                        color=0,
                        fillColor=0,
                        fillPattern=1),
                      string="valve2"),
                    Text(
                      extent=[34, -53; 102, -67],
                      style(
                        color=0,
                        fillColor=0,
                        fillPattern=1),
                      string="valve3")),
                  Coordsys(grid=[1, 1], component=[20, 20]));
                connect(s1.outPort[1], T1.inPort)
                                               annotation (points=[-51.5,40; -44,40],
                    style(
                    color=0,
                    fillColor=0,
                    fillPattern=1));
                connect(T1.outPort, makeProduct.inPort) annotation (points=[-38.5,40; -21,
                      40],  style(
                    color=0,
                    fillColor=0,
                    fillPattern=1));
                connect(makeProduct.outPort, T2.inPort) annotation (points=[10.5,40; 33,40],
                            style(
                    color=0,
                    fillColor=0,
                    fillPattern=1));
                connect(T5.outPort, emptyTanks.inPort[1])
                                                       annotation (points=[5.5,-50; 21,
                      -50],  style(
                    color=0,
                    fillColor=0,
                    fillPattern=1));
                connect(emptyTanks.outPort[1], T6.inPort)
                                                       annotation (points=[42.5,-50; 51,
                      -50],  style(
                    color=0,
                    fillColor=0,
                    fillPattern=1));
                connect(setValve1.y, valve1)
                  annotation (points=[83,82.5; 90,82.5; 90,60; 105,60], style(color=5));
                connect(setValve2.y, valve2)
                  annotation (points=[85.25,-78.5; 90,-78.5; 90,0; 105,0],
                                                                        style(color=5));
                connect(setValve3.y, valve3) annotation (points=[85.3,-90; 95,
                  -90; 95,-60; 105,-60],
                                  style(color=5));
                connect(makeProduct.suspend[1], T3.inPort)
                                                        annotation (points=[-12.5,24.5;
                        -12.5,12; -23,12; -23,3],
                                               style(color=0, rgbcolor={0,0,0}));
                connect(T3.outPort, s2.inPort[1])
                                               annotation (points=[-23,-2.5; -23,-20; -66,
                      -20; -66,-50; -51,-50],        style(color=0, rgbcolor={0,0,0}));
                connect(T4.outPort, makeProduct.resume[1])
                                                        annotation (points=[10,0.5; 10,15;
                        2.5,15; 2.5,24],       style(color=0, rgbcolor={0,0,0}));
                connect(level1, makeProduct.level1) annotation(points=[-60,-110; -60,-80;
                      -80,-80; -80,20; -30,20; -30,28; -22,28], style(color=3, rgbcolor={
                        0,0,255}));
                connect(s2.outPort[1], T5.inPort) annotation(points=[-29.5,-49.75; -30,
                      -49.75; -30,-50; 0,-50], style(color=0, rgbcolor={0,0,0}));
                connect(s2.outPort[2], T4.inPort) annotation(points=[-29.5,-50.25; -29,
                      -50; -8,-50; -8,-25; 10,-25; 10,-5],
                                                      style(color=0, rgbcolor={0,0,0}));
                connect(T2.outPort, s1.inPort[1]) annotation(points=[38.5,40; 70,40; 70,
                      70; -84,70; -84,40; -73,40; -73,40.5],
                                                       style(color=0, rgbcolor={0,0,0}));
                connect(T6.outPort, s1.inPort[2]) annotation(points=[56.5,-50; 70,-50; 70,
                      70; -84,70; -84,40; -74,40; -73,39.5],
                                                       style(color=0, rgbcolor={0,0,0}));

        end TankController;

        model MakeProduct
         extends Modelica.StateGraph.PartialCompositeStep;
                  parameter Modelica.StateGraph.Temporary.SetRealParameter
              limit =                                                            0.98
              "Limit level of tank 1"
                    annotation (extent=[-60,40; -20,60]);
                  parameter Modelica.StateGraph.Temporary.SetRealParameter
              waitTime =                                                            3
              "Wait time"
                    annotation (extent=[-120,40; -80,60]);

                  Modelica.Blocks.Interfaces.RealInput level1
                    annotation (extent=[-190,-140; -150,-100]);
                  annotation (Diagram);
                  Modelica.StateGraph.Step fillTank1
                                 annotation (extent=[-140,-10; -120,10]);
                  Modelica.StateGraph.Transition T1(condition=level1 > limit)
                    annotation (extent=[-110,-10; -90,10]);
                  Modelica.StateGraph.Step fillTank2
                                 annotation (extent=[-10,-10; 10,10]);
                  Modelica.StateGraph.Transition T3(condition=level1 < 0.001)
                    annotation (extent=[20,-10; 40,10]);
                  Modelica.StateGraph.Step emptyTank2
                                  annotation (extent=[120,-10; 140,10]);
                  Modelica.StateGraph.Step wait1
                             annotation (extent=[-80,-10; -60,10]);
                  Modelica.StateGraph.Transition T2(enableTimer=true, waitTime=waitTime)
                    annotation (extent=[-50,-10; -30,10]);
                  Modelica.StateGraph.Step wait2
                             annotation (extent=[54,-10; 74,10]);
                  Modelica.StateGraph.Transition T4(enableTimer=true, waitTime=waitTime)
                    annotation (extent=[82,-10; 102,10]);
        equation
                  connect(fillTank1.inPort[1], inPort)
                                                    annotation (points=[-141,0; -160,0],
                      style(
                      color=0,
                      fillColor=0,
                      fillPattern=1));
                  connect(fillTank1.outPort[1], T1.inPort)
                                                        annotation (points=[-119.5,0; -104,
                        0],  style(
                      color=0,
                      fillColor=0,
                      fillPattern=1));
                  connect(fillTank2.outPort[1], T3.inPort)
                                                        annotation (points=[10.5,0; 26,0],
                       style(
                      color=0,
                      fillColor=0,
                      fillPattern=1));
                  connect(emptyTank2.outPort[1], outPort)
                                                       annotation (points=[140.5,0; 155,0],
                       style(
                      color=0,
                      fillColor=0,
                      fillPattern=1));
                  connect(wait1.outPort[1], T2.inPort)
                                                    annotation (points=[-59.5,0; -44,0],
                      style(color=0, rgbcolor={0,0,0}));
                  connect(T2.outPort, fillTank2.inPort[1])
                                                        annotation (points=[-38.5,0; -11,0],
                            style(color=0, rgbcolor={0,0,0}));
                  connect(T1.outPort, wait1.inPort[1])
                                                    annotation (points=[-98.5,0; -81,0],
                       style(color=0, rgbcolor={0,0,0}));
                  connect(wait2.outPort[1], T4.inPort)
                                                    annotation (points=[74.5,0; 88,0],
                      style(color=0, rgbcolor={0,0,0}));
                  connect(T3.outPort, wait2.inPort[1])
                    annotation (points=[31.5,0; 53,0],   style(color=0, rgbcolor={0,0,0}));
                  connect(T4.outPort,emptyTank2.inPort[1])
                                                         annotation (points=[93.5,0; 119,0],
                             style(color=0, rgbcolor={0,0,0}));

        end MakeProduct;

        connector inflow
            "Inflow connector (this is a copy from Isolde Dressler's master thesis project)"

                    import Units = Modelica.SIunits;

                  Units.VolumeFlowRate Fi "inflow";
                  annotation (Icon(Polygon(points=[-100, -100; 0, 100; 100, -100; -100, -100],
                           style(
                          color=0,
                          thickness=2,
                          fillColor=7,
                          fillPattern=1))));
        end inflow;

        connector outflow
            "Outflow connector (this is a copy from Isolde Dressler's master thesis project)"

                    import Units = Modelica.SIunits;

                  Units.VolumeFlowRate Fo "outflow";
                  Boolean open "valve open";
                  annotation (Icon(Polygon(points=[-100, 100; 0, -100; 100, 100; -100, 100],
                           style(
                          color=0,
                          thickness=2,
                          fillColor=7,
                          fillPattern=1))));
        end outflow;

        model valve
            "Simple valve model (this is a copy from Isolde Dressler's master thesis project)"

                  annotation (
                    Diagram(Line(points=[0, -60; 0, 0], style(color=5))),
                    Icon(
                      Line(points=[20, 20; 20, 20], style(
                          color=0,
                          thickness=2,
                          fillColor=86,
                          fillPattern=1)),
                      Text(
                        extent=[-131, 125; 136, 67],
                        string="%name",
                        style(
                          color=3,
                          rgbcolor={0,0,255},
                          fillColor=86,
                          rgbfillColor={191,0,95},
                          fillPattern=1)),
                      Line(points=[0, 0; 0, -60], style(color=5))),
                    Coordsys(grid=[1, 1], component=[20, 20]));
                  Modelica.Blocks.Interfaces.BooleanInput valveControl
                    annotation (extent=[-20, -100; 20, -60], rotation=90);
                  inflow inflow1 annotation (extent=[0, -50; 100, 50], rotation=90);
                  outflow outflow1 annotation (extent=[-100, -50; 0, 50], rotation=90);
        equation
                  outflow1.Fo = inflow1.Fi;
                  outflow1.open = valveControl;
        end valve;

        model Tank
            "Simple tank model (this is a copy from Isolde Dressler's master thesis project)"

                  Modelica.Blocks.Interfaces.RealOutput levelSensor
                    annotation (extent=[-61, -30; -81, -10]);

                  inflow inflow1 annotation (extent=[-55, 60; -45, 70]);
                  outflow outflow1 annotation (extent=[55, -50; 65, -40]);
                  annotation (
                    Diagram,
                    Icon(
                      Text(
                        extent=[-122, -82; 88, -42],
                        style(fillColor=86, fillPattern=1),
                        string="%name"),
                      Rectangle(extent=[-60, 60; 80, -40], style(
                          color=0,
                          thickness=2,
                          fillColor=7,
                          fillPattern=1)),
                      Rectangle(extent=DynamicSelect([-60, -40; -60, -40], [-60, -40; 80, (
                            -40 + level*100)]), style(
                          color=0,
                          gradient=2,
                          thickness=2,
                          fillColor=86,
                          fillPattern=1))),
                    Coordsys(grid=[1, 1], component=[20, 20]));
                  Real level "Tank level in % of max height";
                  parameter Real A=1 "ground area of tank in m²";
                  parameter Real a=0.2 "area of drain hole in m²";
                  parameter Real hmax=1 "max height of tank in m";
                  constant Real g=Modelica.Constants.g_n;
        equation
                  der(level) = (inflow1.Fi - outflow1.Fo)/(hmax*A);
                  if outflow1.open then
                    outflow1.Fo = sqrt(2*g*hmax*level)*a;
                  else
                    outflow1.Fo = 0;
                  end if;
                  levelSensor = level;
        end Tank;

        model Source
            "Simple source model (this is a copy from Isolde Dressler's master thesis project)"

                  outflow outflow1 annotation (extent=[-10, -60; 10, -40]);
                  parameter Real maxflow=1 "maximal flow out of source";
                  annotation (Icon(Rectangle(extent=[-80, 40; 80, -40], style(
                          color=0,
                          thickness=2,
                          fillColor=7,
                          fillPattern=1)), Text(
                        extent=[-144, 54; 152, 114],
                        style(fillColor=7, fillPattern=1),
                        string="%name")));
        equation
                  if outflow1.open then
                    outflow1.Fo = maxflow;
                  else
                    outflow1.Fo = 0;
                  end if;
        end Source;

      end Utilities;

        package TankControllerAtomic

          import DESLib.DEVSLib.SRC.*;
          model tankController
            parameter Real limit = 0.98;
            parameter Real waitTime = 3;
                     extends AtomicDEVS(numIn=5,numOut=3,
            redeclare record State = st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst(l=limit,wt=waitTime);
            annotation (Diagram, Icon);
                    Interfaces.outPort Valve1       annotation (extent=[80,60; 100,80]);
                    Interfaces.inPort START       annotation (extent=[-100,60; -80,80]);
                    Interfaces.inPort STOP        annotation (extent=[-100,0; -80,20]);
                    Interfaces.inPort SHUT        annotation (extent=[-100,-60; -80,-40]);
                    Interfaces.outPort Valve2       annotation (extent=[80,0; 100,20]);
                    Interfaces.outPort Valve3       annotation (extent=[80,-60; 100,-40]);
                    Interfaces.inPort L1          annotation (extent=[-80,-100; -60,-80]);
                    Interfaces.inPort L2          annotation (extent=[0,-100; 20,-80]);
          equation
            iEvent[1] = START.event;
            iQueue[1] = START.queue;
            iEvent[2] = STOP.event;
            iQueue[2] = STOP.queue;
            iEvent[3] = SHUT.event;
            iQueue[3] = SHUT.queue;
            iEvent[4] = L1.event;
            iQueue[4] = L1.queue;
            iEvent[5] = L2.event;
            iQueue[5] = L2.queue;

            oEvent[1] = Valve1.event;
            oQueue[1] = Valve1.queue;
            oEvent[2] = Valve2.event;
            oQueue[2] = Valve2.queue;
            oEvent[3] = Valve3.event;
            oQueue[3] = Valve3.queue;

          end tankController;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          algorithm
            // first internal
            sout := ext(int(s),0,bag);
            //----------------
            // first external
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            if s.phase == 221 then // T1 full
              sout.sigma := s.wt;
              sout.phase := 22;
              sout.l1f := true;
              sout.l1e := false;
              sout.l2f := false;
              sout.l2e := true;
            elseif s.phase == 231 then // T2 full
              sout.sigma := s.wt;
              sout.phase := 23;
              sout.l1f := false;
              sout.l1e := true;
              sout.l2f := true;
              sout.l2e := false;
            else
              if s.phase == 21 then // start filling T1
                sout.l1e := false;
                sout.l1f := false;
                sout.l2e := true;
                sout.l2f := false;
              elseif s.phase == 22 then // start filling T2 (T1 drains)
                sout.l1f := false;
                sout.l1e := false;
                sout.l2e := false;
                sout.l2f := false;
              elseif s.phase == 23 then // start draining T2
                sout.l2e := false;
                sout.l2f := false;
                sout.l1f := false;
                sout.l1e := true;
              end if;
              sout.sigma := Modelica.Constants.inf;
            end if;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := getEvent(bag);
              if x.Port == 1 then // START
                if x.Value > 0 then
                  sout.start := true;
                  sout.stop := false;
                  sout.shut := false;
                else
                  sout.start := false;
                end if;
                if sout.start and (s.phase == 1 or s.phase == 3 or s.phase == 4) then
                  sout.phase := 21; // fill T1
                  sout.sigma := 0;
                end if;
              elseif x.Port == 2 then  // STOP
                if x.Value > 0 then
                  sout.stop := true;
                  sout.start := false;
                else
                  sout.stop := false;
                end if;
                if sout.stop and (s.phase == 21 or s.phase == 22 or s.phase == 23) then
                  sout.phase := 3; // stop
                  sout.sigma := 0;
                end if;
              elseif x.Port == 3 then // SHUT
                if x.Value > 0 then
                  sout.shut := true;
                  sout.start := false;
                else
                  sout.shut := false;
                end if;
                if sout.shut and (s.phase == 21 or s.phase == 22 or s.phase == 23 or s.phase == 3) then
                  sout.phase := 4;
                  sout.sigma := 0;
                end if;
              elseif x.Port == 4 then // L1
                if x.Type == 1 then // FULL
                  sout.l1f := true;
                  sout.l1e := false;
                  sout.sigma := 0;
                  sout.phase := 221; // stop filling T1, before filling T2
                elseif x.Type == 2 then // EMPTY
                  sout.l1f := false;
                  sout.l1e := true;
                  if sout.l2e and sout.start then
                    sout.sigma := 0;
                    sout.phase := 21; // restart filling
                  elseif sout.l2e then
                    sout.sigma := 0;
                    sout.phase := 1; // tanks empty, wait.
                  elseif not sout.shut then // T2 full
                    sout.l2f := true;
                    sout.l2e := false;
                    sout.sigma := 0;
                    sout.phase := 231; // stop filling T2, before draining it
                  end if;
                end if;
              elseif x.Port == 5 then // L2
                /*if x.Type == 1 then // FULL
        sout.l2f := true;
        sout.l2e := false;
        sout.sigma := 0;
        sout.phase := 231; // stop filling T2, before draining it
      else*/
                if x.Type == 2 then // EMPTY
                  sout.l2f := false;
                  sout.l2e := true;
                  if sout.l1e and sout.start then
                    sout.sigma := 0;
                    sout.phase := 21; // restart filling
                  elseif sout.l1e then
                    sout.sigma := 0;
                    sout.phase := 1; // tanks empty, wait.
                  end if;
                end if;
              end if;
            end for;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y1;
            stdEvent y2;
            stdEvent y3;
          algorithm
            if s.phase == 21 then
              y1.Value := 1;
              y2.Value := 0;
              y3.Value := 0;
            elseif s.phase == 22 then
              y1.Value := 0;
              y2.Value := 1;
              y3.Value := 0;
            elseif s.phase == 23 then
              y1.Value := 0;
              y2.Value := 0;
              y3.Value := 1;
            elseif s.phase == 4 then
              y1.Value := 0;
              y2.Value := 1;
              y3.Value := 1;
            else
              y1.Value := 0;
              y2.Value := 0;
              y3.Value := 0;
            end if;
            sendEvent(queue[1],y1);
            sendEvent(queue[2],y2);
            sendEvent(queue[3],y3);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase;
            Real sigma;
            Boolean start;
            Boolean stop;
            Boolean shut;
            Boolean l1f;
            Boolean l1e;
            Boolean l2f;
            Boolean l2e;
            Real l;
            Real wt;
          end st;

          function initst
            input Real l;
            input Real wt;
            output st out;
          algorithm
            out.phase := 1; // 1 wait, 21 fill 1, 22 fill 2, 23 empty 2, 3 stop, 4 shut.
            out.sigma := Modelica.Constants.inf; // first internal event never happens
            out.start := false;
            out.stop := false;
            out.shut := false;
            out.l1f := false;
            out.l1e := true;
            out.l2f := false;
            out.l2e := true;
            out.l := l;
            out.wt := wt;
          end initst;
        end TankControllerAtomic;

        model TankControllerBigAtomic
         DESLib.DEVSLib.Examples.ControlledTanks.TankControllerAtomic.tankController
            tankController
                    annotation (extent=[-58,-20; 2,40]);
                  DESLib.DEVSLib.AuxModels.DiBO diBo       annotation (extent=[16,22; 36,42]);
                  DESLib.DEVSLib.AuxModels.DiBO diBo1       annotation (extent=[16,2; 36,22]);
                  DESLib.DEVSLib.AuxModels.DiBO diBo2       annotation (extent=[16,-18; 36,2]);
                  annotation (Diagram, Icon(Rectangle(extent=[-80,100; 80,-60], style(
                            color=3, rgbcolor={0,0,255})), Text(
                        extent=[-76,96; 80,-60],
                        style(color=3, rgbcolor={0,0,255}),
                        string="%name")));
                  SRC.Interfaces.inPort START annotation (extent=[-100,60; -80,80]);
                  SRC.Interfaces.inPort STOP annotation (extent=[-100,20; -80,40]);
                  SRC.Interfaces.inPort SHUT annotation (extent=[-100,-20; -80,0]);
                  Modelica.Blocks.Interfaces.BooleanOutput V1
                    annotation (extent=[80,60; 100,80]);
                  Modelica.Blocks.Interfaces.BooleanOutput V2
                    annotation (extent=[80,20; 100,40]);
                  Modelica.Blocks.Interfaces.BooleanOutput V3
                    annotation (extent=[80,-20; 100,0]);
                  Modelica.Blocks.Interfaces.RealInput L1
                    annotation (extent=[-60,-100; -20,-60], rotation=90);
                  Modelica.Blocks.Interfaces.RealInput L2
                    annotation (extent=[0,-100; 40,-60], rotation=90);
                  SRC.Hybrid.CrossDOWN crossDOWN(Value=0.001, EType=2)
                    annotation (extent=[-12,-52; 8,-32], rotation=0);
                  SRC.Hybrid.CrossDOWN crossDOWN1(Value=0.001, EType=2)
                    annotation (extent=[-76,-54; -56,-34], rotation=0);
                  SRC.Hybrid.CrossUP crossUP(Value=0.98)
                    annotation (extent=[-52,-54; -32,-34], rotation=0);
        equation
        connect(tankController.Valve3, diBo2.inport) annotation (points=[-1,-5;
                7.5,-5; 7.5,-8; 15.4,-8],           style(color=0, rgbcolor={0,0,0}));
                  connect(tankController.Valve1, diBo.inport) annotation (points=[-1,31;
                8.5,31; 8.5,32; 15.4,32],       style(color=0, rgbcolor={0,0,0}));
                  connect(START, tankController.START) annotation (points=[-90,70; -90,
                        51; -55,51; -55,31], style(color=0, rgbcolor={0,0,0}));
                  connect(STOP, tankController.STOP) annotation (points=[-90,30; -74,30;
                        -74,13; -55,13], style(color=0, rgbcolor={0,0,0}));
                  connect(SHUT, tankController.SHUT) annotation (points=[-90,-10; -74,
                        -10; -74,-5; -55,-5], style(color=0, rgbcolor={0,0,0}));
                  connect(diBo.change, V1) annotation (points=[37,32; 59.5,32; 59.5,
                        70; 90,70],
                                style(color=5, rgbcolor={255,0,255}));
                  connect(diBo1.change, V2) annotation (points=[37,12; 66,12; 66,30;
                        90,30],
                             style(color=5, rgbcolor={255,0,255}));
                  connect(diBo2.change, V3) annotation (points=[37,-8; 60.5,-8; 60.5,
                        -10; 90,-10], style(color=5, rgbcolor={255,0,255}));
                  connect(tankController.Valve2, diBo1.inport) annotation (points=[-1,13;
                7.5,13; 7.5,12; 15.4,12],           style(color=0, rgbcolor={0,0,0}));
                  connect(L1, crossUP.u) annotation (points=[-40,-80; -40,-67; -53,-67; -53,-44],
                              style(color=74, rgbcolor={0,0,127}));
                  connect(L1, crossDOWN1.u) annotation (points=[-40,-80; -77,-80; -77,-44],
                              style(color=74, rgbcolor={0,0,127}));
                  connect(crossDOWN1.outport, tankController.L1) annotation (points=[-55.4,-44;
                        -56,-44; -56,-28; -49,-28; -49,-17],          style(color=0,
                        rgbcolor={0,0,0}));
                  connect(crossUP.outport, tankController.L1) annotation (points=[-31.4,-44;
                        -34,-44; -34,-28; -49,-28; -49,-17],      style(color=0,
                        rgbcolor={0,0,0}));
                  connect(L2, crossDOWN.u) annotation (points=[20,-80; -18,-80; -18,-42; -13,
                        -42],     style(color=74, rgbcolor={0,0,127}));
                  connect(crossDOWN.outport, tankController.L2) annotation (points=[8.6,-42;
                10,-42; 10,-26; -25,-26; -25,-17],              style(color=0, rgbcolor=
                         {0,0,0}));
        end TankControllerBigAtomic;

        model systemAtomic
        Real Tank1Level = tank1.levelSensor;
        Real Tank2Level = tank2.levelSensor;

          annotation (Diagram,
                    experiment(StopTime=30),
                    experimentSetupOutput);
                  Utilities.Tank tank1 annotation (extent=[18,18; 68,68]);
                  Utilities.Tank tank2 annotation (extent=[42,-62; 92,-12]);
                  Utilities.valve valve1
                    annotation (extent=[25,64.5; 36,75.5],   rotation=-90);
                  Utilities.Source source annotation (extent=[20.5,78.5; 40.5,98.5]);
                  Utilities.valve valve2 annotation (extent=[46.5,4; 62.5,18],   rotation=-90);
                  Utilities.valve valve3
                    annotation (extent=[73.5,-86; 89.5,-72],   rotation=-90);
                  DESLib.DEVSLib.AuxModels.Generator start(
                    period=14,
                    firstGeneration=1,
                    name="start",
                    numGenerations=2)
                                  annotation (extent=[-100,50; -80,70]);
                  DESLib.DEVSLib.AuxModels.Generator stop(
                    period=6,
                    numGenerations=2,
                    name="stop",
                    firstGeneration=13)
                                 annotation (extent=[-100,4; -80,24]);
                  DESLib.DEVSLib.AuxModels.Generator shut(
                    period=1,
                    name="shut",
                    numGenerations=1,
                    firstGeneration=21,
                    outValue=1)  annotation (extent=[-100,-18; -80,2]);

            DESLib.DEVSLib.Examples.ControlledTanks.TankControllerBigAtomic
            TankControllerAtomic                   annotation (extent=[-68,-42; 4,48]);
                  DESLib.DEVSLib.AuxModels.Generator start1(
                    name="start",
                    firstGeneration=100,
                    numGenerations=1)
                                  annotation (extent=[-100,28; -80,48]);

        equation
          connect(tank1.outflow1,valve2. outflow1) annotation (points=[58,31.75;
                        58,24.875; 54.5,24.875; 54.5,14.5], style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=2,
                      fillColor=0,
                      rgbfillColor={0,0,0},
                      fillPattern=1));
                  connect(tank2.inflow1,valve2. inflow1) annotation (points=[54.5,
                        -20.75; 54.5,7.5],
                                    style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=2,
                      fillColor=0,
                      rgbfillColor={0,0,0},
                      fillPattern=1));
                  connect(tank2.outflow1,valve3. outflow1) annotation (points=[82,
                        -48.25; 82,-75.5; 81.5,-75.5],
                                              style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=2,
                      fillColor=0,
                      rgbfillColor={0,0,0},
                      fillPattern=1));
                  connect(tank1.inflow1,valve1. inflow1) annotation (points=[30.5,59.25;
                        30.5,67.25],  style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=2,
                      fillColor=0,
                      rgbfillColor={0,0,0},
                      fillPattern=1));
                  connect(source.outflow1,valve1. outflow1) annotation (points=[30.5,
                        83.5; 30.5,72.75],
                                      style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=2));

                  connect(start.outPort1, TankControllerAtomic.START)
                                                                annotation (points=[-81,60;
                        -72,60; -72,34.5; -64.4,34.5],     style(color=0, rgbcolor={0,0,
                          0}));
                  connect(stop.outPort1, TankControllerAtomic.STOP)
                                                              annotation (points=[-81,
                        14; -74,14; -74,16.5; -64.4,16.5], style(color=0, rgbcolor={0,0,
                          0}));
                  connect(shut.outPort1, TankControllerAtomic.SHUT)
                                                              annotation (points=[-81,
                        -8; -74,-8; -74,-1.5; -64.4,-1.5], style(color=0, rgbcolor={0,0,
                          0}));
                  connect(TankControllerAtomic.V1, valve1.valveControl)
                                                                  annotation (points=[0.4,34.5;
                10,34.5; 10,70; 26.1,70],                   style(color=5, rgbcolor={
                          255,0,255}));
                  connect(TankControllerAtomic.V2, valve2.valveControl)
                                                                  annotation (points=[0.4,16.5;
                24.2,16.5; 24.2,11; 48.1,11],                   style(color=5, rgbcolor=
                         {255,0,255}));
                  connect(TankControllerAtomic.V3, valve3.valveControl)
                                                                  annotation (points=[0.4,-1.5;
                24,-1.5; 24,-79; 75.1,-79],                   style(color=5, rgbcolor={
                          255,0,255}));
                  connect(tank1.levelSensor, TankControllerAtomic.L1)
                                                                annotation (points=[
                        25.25,38; 18,38; 18,-52; -46.4,-52; -46.4,-33], style(color=74,
                        rgbcolor={0,0,127}));
                  connect(tank2.levelSensor, TankControllerAtomic.L2)
                                                                annotation (points=[
                        49.25,-42; 38,-42; 38,-56; -24.8,-56; -24.8,-33], style(color=
                          74, rgbcolor={0,0,127}));
                  connect(start1.outPort1, TankControllerAtomic.START) annotation (
                      points=[-81,38; -72,38; -72,34.5; -64.4,34.5], style(color=0,
                        rgbcolor={0,0,0}));

        end systemAtomic;

        model tankControllerCoupled
          parameter Real limit = 0.98;
          parameter Real waitTime = 3;

          annotation (Diagram, Icon);
                  SRC.Interfaces.outPort Valve1   annotation (extent=[80,60; 100,80]);
                  SRC.Interfaces.inPort START   annotation (extent=[-100,60; -80,80]);
                  SRC.Interfaces.inPort STOP    annotation (extent=[-100,0; -80,20]);
                  SRC.Interfaces.inPort SHUT    annotation (extent=[-100,-60; -80,-40]);
                  SRC.Interfaces.outPort Valve2   annotation (extent=[80,0; 100,20]);
                  SRC.Interfaces.outPort Valve3   annotation (extent=[80,-60; 100,-40]);
                  SRC.Interfaces.inPort L1      annotation (extent=[-40,-100; -20,-80]);
                  SRC.Interfaces.inPort L2      annotation (extent=[0,-100; 20,-80]);
                  SimpleModels.SetValue.setValue setValue
                    annotation (extent=[52,70; 72,90]);
                  SimpleModels.SetValue.setValue setValue1(Value=0)
                    annotation (extent=[52,50; 72,70]);
                  SimpleModels.SetValue.setValue setValue2
                    annotation (extent=[52,10; 72,30]);
                  SimpleModels.SetValue.setValue setValue3(Value=0)
                    annotation (extent=[52,-10; 72,10]);
                  SimpleModels.SetValue.setValue setValue4
                    annotation (extent=[52,-50; 72,-30]);
                  SimpleModels.SetValue.setValue setValue5(Value=0)
                    annotation (extent=[52,-70; 72,-50]);
                  SimpleModels.IFType.ifType IF_TankFull
                    annotation (extent=[-32,-74; -10,-54], rotation=0);
                  AuxModels.DUP dUP annotation (extent=[-22,-50; 0,-30],   rotation=90);
                  AuxModels.DUP dUP1 annotation (extent=[-2,-46; 12,-36],   rotation=90);
                  AuxModels.DUP dUP6
                                    annotation (extent=[-66,-54; -58,-44], rotation=0);
                  AuxModels.DUP3 dUP3_1 annotation (extent=[-74,0; -54,20]);
                  SimpleModels.Wait.wait wait(processTime=3) annotation (extent=[24,10; 44,30]);
                  SimpleModels.Wait.wait wait1(processTime=3) annotation (extent=[24,-50; 44,-30]);
                  AuxModels.DUP3 dUP3_2 annotation (extent=[-86,-6; -66,14]);
                  SimpleModels.Storage.storage Tank1
                    annotation (extent=[-78,80; -58,100]);
                  SimpleModels.IFType.ifType ifType(condition=0)
                    annotation (extent=[-56,80; -36,100]);
                  AuxModels.DUP dUP2
                                    annotation (extent=[-46,-86; -24,-66], rotation=90);
                  SimpleModels.Storage.storage Tank2_1
                    annotation (extent=[-28,80; -8,100]);
                  SimpleModels.IFType.ifType ifType1(condition=0)
                    annotation (extent=[-6,80; 14,100]);
                  AuxModels.DUP dUP3 annotation (extent=[6,-74; 20,-64],    rotation=90);
                  SimpleModels.Storage.storage shut_st
                    annotation (extent=[-78,-94; -64,-82]);
                  SimpleModels.SetValue.setValue setValue6
                    annotation (extent=[-98,-92; -86,-76]);
                  SimpleModels.SetValue.setValue setValue7(Value=0)
                    annotation (extent=[-98,-102; -86,-86]);
                  SimpleModels.IFType.ifType ifType2(condition=1)
                    annotation (extent=[-62,-94; -50,-84]);
                  AuxModels.DUP dUP5
                                    annotation (extent=[-86,-60; -64,-40], rotation=0);
                  AuxModels.Display display annotation (extent=[34,-104; 54,-84]);
                  AuxModels.DUP dUP7
                                    annotation (extent=[0,-86; 22,-66],    rotation=90);
                  AuxModels.DUP3 dUP3_3 annotation (extent=[-100,74; -80,94], rotation=
                        90);

        equation
           connect(setValue.outPort1, Valve1) annotation (points=[71,80; 80,80; 80,70;
                        90,70], style(color=0, rgbcolor={0,0,0}));
                  connect(setValue1.outPort1, Valve1) annotation (points=[71,60; 80,60;
                        80,70; 90,70], style(color=0, rgbcolor={0,0,0}));
                  connect(setValue2.outPort1, Valve2) annotation (points=[71,20; 80,20;
                        80,10; 90,10], style(color=0, rgbcolor={0,0,0}));
                  connect(setValue3.outPort1, Valve2) annotation (points=[71,0; 80,0;
                        80,10; 90,10], style(color=0, rgbcolor={0,0,0}));
                  connect(setValue4.outPort1, Valve3) annotation (points=[71,-40; 80,
                        -40; 80,-50; 90,-50], style(color=0, rgbcolor={0,0,0}));
                  connect(setValue5.outPort1, Valve3) annotation (points=[71,-60; 80,
                        -60; 80,-50; 90,-50], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP.out1, setValue1.inPort1) annotation (points=[-13.2,-39.4;
                        -13.2,60; 53,60], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP1.out1, setValue3.inPort1) annotation (points=[3.6,
                -40.7; 3.6,0; 53,0],    style(color=0, rgbcolor={0,0,0}));
                  connect(IF_TankFull.out1, dUP.in1)
                                                annotation (points=[-11.1,-60; -11.1,
                        -58.45; -11,-58.45; -11,-42.6],
                                                      style(color=0, rgbcolor={0,0,0}));
                  connect(IF_TankFull.out2, dUP1.in1)
                                                 annotation (points=[-11.1,-69;
                -11.1,-68.45; 5,-68.45; 5,-42.3],   style(color=0, rgbcolor={0,0,0}));
                  connect(dUP6.out2, setValue4.inPort1) annotation (points=[-61.76,-50;
                        -34,-50; -34,-24; 48,-24; 48,-40; 53,-40],   style(color=0,
                        rgbcolor={0,0,0}));
                  connect(dUP6.out1, setValue2.inPort1) annotation (points=[-61.76,-48;
                        -38,-48; -38,-22; 48,-22; 48,20; 53,20], style(color=0,
                        rgbcolor={0,0,0}));
                  connect(dUP3_1.out1, setValue1.inPort1) annotation (points=[-62.6,14;
                        -16,14; -16,60; 53,60], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP3_1.out2, setValue3.inPort1) annotation (points=[-62.6,10;
                        -16,10; -16,0; 53,0], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP3_1.out3, setValue5.inPort1) annotation (points=[-62.6,6;
                        -54,6; -54,2; 50,2; 50,-60; 53,-60],   style(color=0, rgbcolor=
                          {0,0,0}));
                  connect(wait.outPort1, setValue2.inPort1) annotation (points=[43,20;
                        53,20], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP.out2, wait.IN) annotation (points=[-8.8,-39.4; -8.8,20;
                        25,20], style(color=0, rgbcolor={0,0,0}));
                  connect(wait1.outPort1, setValue4.inPort1) annotation (points=[43,-40;
                        53,-40], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP1.out2, wait1.IN) annotation (points=[6.4,-40.7;
                14.2,-40.7; 14.2,-40; 25,-40],    style(color=0, rgbcolor={0,0,0}));
                  connect(STOP, dUP3_2.IN) annotation (points=[-90,10; -85,10; -85,4;
                        -81,4], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP3_2.out1, dUP3_1.IN) annotation (points=[-74.6,8; -72,8;
                        -72,10; -69,10], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP3_2.out2, wait.STOP) annotation (points=[-74.6,4; 18,4; 18,
                        13; 25,13], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP3_2.out3, wait1.STOP) annotation (points=[-74.6,0; -28,0;
                        -28,-2; 18,-2; 18,-47; 25,-47], style(color=0, rgbcolor={0,0,0}));
                  connect(Tank1.outPort1, ifType.in1) annotation (points=[-59,90; -58,
                        90; -58,94; -55,94], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP2.out2, IF_TankFull.in1) annotation (points=[-32.8,-75.4;
                        -32.8,-67.7; -30.9,-67.7; -30.9,-60], style(color=0, rgbcolor={
                          0,0,0}));
                  connect(L1, dUP2.in1) annotation (points=[-30,-90; -32,-90; -32,-78.6;
                        -35,-78.6], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP2.out1, Tank1.STORE) annotation (points=[-37.2,-75.4; -52,
                        -75.4; -52,20; -78,20; -78,85; -77,85], style(color=0, rgbcolor=
                         {0,0,0}));
                  connect(Tank2_1.outPort1, ifType1.in1) annotation (points=[-9,90; -8,
                        90; -8,94; -5,94], style(color=0, rgbcolor={0,0,0}));
                  connect(ifType.out1, Tank2_1.CHECK) annotation (points=[-37,94; -32,
                        94; -32,95; -27,95], style(color=0, rgbcolor={0,0,0}));
                  connect(ifType1.out1, setValue.inPort1) annotation (points=[13,94; 48,
                        94; 48,80; 53,80], style(color=0, rgbcolor={0,0,0}));
                  connect(ifType1.out2, wait1.IN) annotation (points=[13,85; 16,85; 16,
                        -40; 25,-40], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP3.out2, setValue5.inPort1) annotation (points=[14.4,-68.7;
                        33.2,-68.7; 33.2,-60; 53,-60], style(color=0, rgbcolor={0,0,0}));
                  connect(setValue6.outPort1, shut_st.STORE) annotation (points=[-86.6,
                        -84; -84,-84; -84,-91; -77.3,-91], style(color=0, rgbcolor={0,0,
                          0}));
                  connect(setValue7.outPort1, shut_st.STORE) annotation (points=[-86.6,
                        -94; -84,-94; -84,-91; -77.3,-91], style(color=0, rgbcolor={0,0,
                          0}));
                  connect(shut_st.outPort1, ifType2.in1) annotation (points=[-64.7,-88;
                        -64,-88; -64,-87; -61.4,-87], style(color=0, rgbcolor={0,0,0}));
                  connect(SHUT, dUP5.in1) annotation (points=[-90,-50; -77.86,-50],
                      style(color=0, rgbcolor={0,0,0}));
                  connect(dUP5.out1, dUP6.in1) annotation (points=[-74.34,-48; -68.69,
                        -48; -68.69,-49; -63.04,-49], style(color=0, rgbcolor={0,0,0}));
                  connect(shut_st.CHECK, dUP3.out1) annotation (points=[-77.3,-85; -80,
                        -85; -80,-100; 20,-100; 20,-66; 11.6,-66; 11.6,-68.7], style(
                        color=0, rgbcolor={0,0,0}));
                  connect(ifType2.out2, display.inPort1) annotation (points=[-50.6,
                        -91.5; -44,-91.5; -44,-102; 28,-102; 28,-94; 35,-94], style(
                        color=0, rgbcolor={0,0,0}));
                  connect(dUP5.out2, setValue7.inPort1) annotation (points=[-74.34,-52;
                        -72,-52; -72,-74; -102,-74; -102,-94; -97.4,-94], style(color=0,
                        rgbcolor={0,0,0}));
                  connect(ifType.out2, wait.IN) annotation (points=[-37,85; -34,85; -34,
                        66; 12,66; 12,20; 25,20], style(color=0, rgbcolor={0,0,0}));
                  connect(L2, dUP7.in1) annotation (points=[10,-90; 11,-90; 11,-78.6],
                      style(color=0, rgbcolor={0,0,0}));
                  connect(dUP7.out2, dUP3.in1) annotation (points=[13.2,-75.4; 13.2,
                        -73.7; 13,-73.7; 13,-70.3], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP7.out1, Tank2_1.STORE) annotation (points=[8.8,-75.4; 8.8,
                        -72; -4,-72; -4,76; -30,76; -30,85; -27,85], style(color=0,
                        rgbcolor={0,0,0}));
                  connect(ifType2.out1, setValue.inPort1) annotation (points=[-50.6,-87;
                        -44,-87; -44,54; 38,54; 38,80; 53,80], style(color=0, rgbcolor=
                          {0,0,0}));
                  connect(START, dUP3_3.IN) annotation (points=[-90,70; -90,79], style(
                        color=0, rgbcolor={0,0,0}));
                  connect(dUP3_3.out3, Tank1.CHECK) annotation (points=[-86,85.4; -86,
                        95; -77,95], style(color=0, rgbcolor={0,0,0}));
                  connect(dUP3_3.out1, setValue6.inPort1) annotation (points=[-94,85.4;
                        -94,90; -100,90; -100,-84; -97.4,-84], style(color=0, rgbcolor=
                          {0,0,0}));
                  connect(dUP3_3.out2, setValue5.inPort1) annotation (points=[-90,85.4;
                        -90,96; -104,96; -104,-104; 56,-104; 56,-60; 53,-60], style(
                        color=0, rgbcolor={0,0,0}));
        end tankControllerCoupled;

        model TankControllerBigCoupled

            DESLib.DEVSLib.Examples.ControlledTanks.tankControllerCoupled
            tankController
                    annotation (extent=[-58,-20; 2,40]);
                  DESLib.DEVSLib.AuxModels.DiBO diBo       annotation (extent=[16,22; 36,42]);
                  DESLib.DEVSLib.AuxModels.DiBO diBo1       annotation (extent=[16,2; 36,22]);
                  DESLib.DEVSLib.AuxModels.DiBO diBo2       annotation (extent=[16,-18; 36,2]);
                  annotation (Diagram, Icon(Rectangle(extent=[-80,100; 80,-60], style(
                            color=3, rgbcolor={0,0,255})), Text(
                        extent=[-76,96; 80,-60],
                        style(color=3, rgbcolor={0,0,255}),
                        string="%name")));
                  SRC.Interfaces.inPort START annotation (extent=[-100,60; -80,80]);
                  SRC.Interfaces.inPort STOP annotation (extent=[-100,20; -80,40]);
                  SRC.Interfaces.inPort SHUT annotation (extent=[-100,-20; -80,0]);
                  Modelica.Blocks.Interfaces.BooleanOutput V1
                    annotation (extent=[80,60; 100,80]);
                  Modelica.Blocks.Interfaces.BooleanOutput V2
                    annotation (extent=[80,20; 100,40]);
                  Modelica.Blocks.Interfaces.BooleanOutput V3
                    annotation (extent=[80,-20; 100,0]);
                  Modelica.Blocks.Interfaces.RealInput L1
                    annotation (extent=[-60,-100; -20,-60], rotation=90);
                  Modelica.Blocks.Interfaces.RealInput L2
                    annotation (extent=[0,-100; 40,-60], rotation=90);
                  SRC.Hybrid.CrossDOWN crossDOWN(Value=0.001, EType=0)
                    annotation (extent=[-12,-52; 8,-32], rotation=0);
                  SRC.Hybrid.CrossDOWN crossDOWN1(Value=0.001, EType=0)
                    annotation (extent=[-76,-54; -56,-34], rotation=0);
                  SRC.Hybrid.CrossUP crossUP(Value=0.98)
                    annotation (extent=[-52,-54; -32,-34], rotation=0);
        equation
                  connect(tankController.Valve3, diBo2.inport) annotation (points=[-1,-5;
                7.5,-5; 7.5,-8; 15.4,-8],           style(color=0, rgbcolor={0,0,0}));
                  connect(tankController.Valve1, diBo.inport) annotation (points=[-1,31;
                8.5,31; 8.5,32; 15.4,32],       style(color=0, rgbcolor={0,0,0}));
                  connect(START, tankController.START) annotation (points=[-90,70; -90,
                        51; -55,51; -55,31], style(color=0, rgbcolor={0,0,0}));
                  connect(STOP, tankController.STOP) annotation (points=[-90,30; -74,30;
                        -74,13; -55,13], style(color=0, rgbcolor={0,0,0}));
                  connect(SHUT, tankController.SHUT) annotation (points=[-90,-10; -74,
                        -10; -74,-5; -55,-5], style(color=0, rgbcolor={0,0,0}));
                  connect(diBo.change, V1) annotation (points=[37,32; 59.5,32; 59.5,70; 90,70],
                                style(color=5, rgbcolor={255,0,255}));
                  connect(diBo1.change, V2) annotation (points=[37,12; 66,12; 66,30; 90,30],
                             style(color=5, rgbcolor={255,0,255}));
                  connect(diBo2.change, V3) annotation (points=[37,-8; 60.5,-8; 60.5,-10; 90,
                        -10],         style(color=5, rgbcolor={255,0,255}));
                  connect(tankController.Valve2, diBo1.inport) annotation (points=[-1,13;
                7.2,13; 7.2,12; 15.4,12],           style(color=0, rgbcolor={0,0,0}));
                  connect(L1, crossUP.u) annotation (points=[-40,-80; -40,-67; -53,
                        -67; -53,-44],
                              style(color=74, rgbcolor={0,0,127}));
                  connect(L1, crossDOWN1.u) annotation (points=[-40,-80; -77,-80; -77,
                        -44], style(color=74, rgbcolor={0,0,127}));
                  connect(crossDOWN1.outport, tankController.L1) annotation (points=[-55.4,
                        -44; -56,-44; -56,-28; -37,-28; -37,-17],     style(color=0,
                        rgbcolor={0,0,0}));
                  connect(crossUP.outport, tankController.L1) annotation (points=[-31.4,
                        -44; -34,-44; -34,-28; -37,-28; -37,-17], style(color=0,
                        rgbcolor={0,0,0}));
                  connect(L2, crossDOWN.u) annotation (points=[20,-80; -18,-80; -18,
                        -42; -13,-42],
                                  style(color=74, rgbcolor={0,0,127}));
                  connect(crossDOWN.outport, tankController.L2) annotation (points=[8.6,-42;
                10,-42; 10,-26; -25,-26; -25,-17],              style(color=0, rgbcolor=
                         {0,0,0}));
        end TankControllerBigCoupled;

        model systemCoupled
        Real Tank1Level = tank1.levelSensor;
        Real Tank2Level = tank2.levelSensor;

          annotation (Diagram,
                    experiment(StopTime=30),
                    experimentSetupOutput);
                  Utilities.Tank tank1 annotation (extent=[18,18; 68,68]);
                  Utilities.Tank tank2 annotation (extent=[42,-62; 92,-12]);
                  Utilities.valve valve1
                    annotation (extent=[25,64.5; 36,75.5],   rotation=-90);
                  Utilities.Source source annotation (extent=[20.5,78.5; 40.5,98.5]);
                  Utilities.valve valve2 annotation (extent=[46.5,4; 62.5,18],   rotation=-90);
                  Utilities.valve valve3
                    annotation (extent=[73.5,-86; 89.5,-72],   rotation=-90);
                  DESLib.DEVSLib.AuxModels.Generator start(
                    period=14,
                    firstGeneration=1,
                    name="start",
                    numGenerations=2)
                                  annotation (extent=[-100,54; -80,74]);
                  DESLib.DEVSLib.AuxModels.Generator stop(
                    period=6,
                    numGenerations=2,
                    name="stop",
                    firstGeneration=13)
                                 annotation (extent=[-100,8; -80,28]);
                  DESLib.DEVSLib.AuxModels.Generator shut(
                    period=1,
                    name="shut",
                    numGenerations=1,
                    firstGeneration=21)
                                 annotation (extent=[-100,-16; -80,4]);

            DESLib.DEVSLib.Examples.ControlledTanks.TankControllerBigCoupled
            TankControllerCoupled                  annotation (extent=[-68,-42; 4,48]);
                  AuxModels.Generator start1(
                    name="start",
                    firstGeneration=100,
                    numGenerations=1)
                                  annotation (extent=[-100,34; -80,54]);
        equation
                  connect(tank1.outflow1,valve2. outflow1) annotation (points=[58,31.75;
                        58,24.875; 54.5,24.875; 54.5,14.5], style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=2,
                      fillColor=0,
                      rgbfillColor={0,0,0},
                      fillPattern=1));
                  connect(tank2.inflow1,valve2. inflow1) annotation (points=[54.5,
                        -20.75; 54.5,7.5],
                                    style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=2,
                      fillColor=0,
                      rgbfillColor={0,0,0},
                      fillPattern=1));
                  connect(tank2.outflow1,valve3. outflow1) annotation (points=[82,
                        -48.25; 82,-75.5; 81.5,-75.5],
                                              style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=2,
                      fillColor=0,
                      rgbfillColor={0,0,0},
                      fillPattern=1));
                  connect(tank1.inflow1,valve1. inflow1) annotation (points=[30.5,59.25;
                        30.5,67.25],  style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=2,
                      fillColor=0,
                      rgbfillColor={0,0,0},
                      fillPattern=1));
                  connect(source.outflow1,valve1. outflow1) annotation (points=[30.5,
                        83.5; 30.5,72.75],
                                      style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=2));

                  connect(start.outPort1, TankControllerCoupled.START)
                                                                annotation (points=[-81,64;
                        -72,64; -72,34.5; -64.4,34.5],     style(color=0, rgbcolor={0,0,
                          0}));
                  connect(stop.outPort1, TankControllerCoupled.STOP)
                                                              annotation (points=[-81,18;
                        -74,18; -74,16.5; -64.4,16.5],     style(color=0, rgbcolor={0,0,
                          0}));
                  connect(shut.outPort1, TankControllerCoupled.SHUT)
                                                              annotation (points=[-81,-6;
                        -74,-6; -74,-1.5; -64.4,-1.5],     style(color=0, rgbcolor={0,0,
                          0}));
                  connect(TankControllerCoupled.V1, valve1.valveControl)
                                                                  annotation (points=[0.4,34.5;
                10,34.5; 10,70; 26.1,70],                   style(color=5, rgbcolor={
                          255,0,255}));
                  connect(TankControllerCoupled.V2, valve2.valveControl)
                                                                  annotation (points=[0.4,16.5;
                24.2,16.5; 24.2,11; 48.1,11],                   style(color=5, rgbcolor=
                         {255,0,255}));
                  connect(TankControllerCoupled.V3, valve3.valveControl)
                                                                  annotation (points=[0.4,-1.5;
                24,-1.5; 24,-79; 75.1,-79],                   style(color=5, rgbcolor={
                          255,0,255}));
                  connect(tank1.levelSensor, TankControllerCoupled.L1)
                                                                annotation (points=[
                        25.25,38; 18,38; 18,-52; -46.4,-52; -46.4,-33], style(color=74,
                        rgbcolor={0,0,127}));
                  connect(tank2.levelSensor, TankControllerCoupled.L2)
                                                                annotation (points=[
                        49.25,-42; 38,-42; 38,-56; -24.8,-56; -24.8,-33], style(color=
                          74, rgbcolor={0,0,127}));
                  connect(start1.outPort1, TankControllerCoupled.START) annotation (
                      points=[-81,44; -72,44; -72,34.5; -64.4,34.5], style(color=0,
                        rgbcolor={0,0,0}));

        end systemCoupled;

      end ControlledTanks;

      package PetriNetsExamples "Model of an MM1 queue system"
      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package includes the model of an MM1 queue system.
This system is also implemented in the Modelica Extended Petri Nets library.
</p>

<p>
The system has been implemented in two ways:
<ul>
<li> The MM1 model contains an implementation of the system using the random number and variates generation functions included in the RandomLib package.
<li> The MM12 model contains an implementation of the system that uses the same random number and variates generation function used in the Extended Petri Nets library.
</ul>

In both models, the variable QAVG has been included to show the average number of elements in the queue during the simulation.


<br>
</HTML>
"));
          package Interarrival

          annotation(preferedView="info",
            Window(
              x=0.02,
              y=0.01,
              width=0.2,
              height=0.57,
              library=1,
              autolayout=1),
            version="2.2",
            versionDate="2005-04-15",
            Settings(NewStateSelection=true),
            Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica library for modeling and simulation systems using the DEVS formalism.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
            model interarrival
            //    Real ia "Interarrival time";
              parameter Real period = 1;
              parameter Real firstGeneration = 1;
              parameter Integer numGenerations = 0 "0 == infinity";
              parameter Real outValue = 1;
              extends AtomicDEVS(redeclare record State = st);
              redeclare function Fint = int;//(ia=ia);
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(p=period,fg=firstGeneration,val=outValue,ng=numGenerations);
              annotation (Diagram);
              Interfaces.outPort outPort1
                               annotation (extent=[80,-10;100,10]);
            equation
              for i in 1:numIn loop
                iEvent[i] = 0;
              end for;
              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;

            end interarrival;

            function int "Internal Transition Function"
            import RandomLib;
              input st s;
              //input Real ia;
              output st sout;
            algorithm
              sout := s;
              sout.gen := sout.gen +1;
              if (sout.gen < s.ng) or (s.ng == 0) then
              (sout.sigma,sout.Cg1,sout.Cg2,sout.Cg3,sout.Cg4,sout.Cg5,sout.Cg6,
                sout.Bg1,sout.Bg2,sout.Bg3,sout.Bg4,sout.Bg5,sout.Bg6,sout.Ig1,
                sout.Ig2,sout.Ig3,sout.Ig4,sout.Ig5,sout.Ig6,sout.Anti,sout.IncPrec)
                := DESLib.RandomLib.Variates.GenerateVariateWrap(
                            sout.Cg1,
                            sout.Cg2,
                            sout.Cg3,
                            sout.Cg4,
                            sout.Cg5,
                            sout.Cg6,
                            sout.Bg1,
                            sout.Bg2,
                            sout.Bg3,
                            sout.Bg4,
                            sout.Bg5,
                            sout.Bg6,
                            sout.Ig1,
                            sout.Ig2,
                            sout.Ig3,
                            sout.Ig4,
                            sout.Ig5,
                            sout.Ig6,
                            sout.Anti,
                            sout.IncPrec,
                            9,
                            10);
                //sout.sigma := ia; // period
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
              sout.p := s.p;
              sout.val := s.val;
              sout.port := 1;
            end int;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent Y;
            algorithm
              Y.Value := s.val;
              Y.Type := 1;
              sendEvent(queue[1],Y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase; // 1 = passive, 2 = active
              Real sigma;
              Real p;
              Real fg;
              Real val;
              Integer port;
              Integer ng;
              Integer gen;
              Real Cg1;
              Real Cg2;
              Real Cg3;
              Real Cg4;
              Real Cg5;
              Real Cg6;
              Real Bg1;
              Real Bg2;
              Real Bg3;
              Real Bg4;
              Real Bg5;
              Real Bg6;
              Real Ig1;
              Real Ig2;
              Real Ig3;
              Real Ig4;
              Real Ig5;
              Real Ig6;
              Integer Anti;
              Integer IncPrec;
            end st;

            function initst
            import RandomLib;
              input Real p;
              input Real fg;
              input Real val;
              input Integer ng;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := fg;
              out.p := p;
              out.val := val;
              out.port := 1;
              out.ng := ng;
              out.gen := 0;
            (out.Cg1,out.Cg2,out.Cg3,out.Cg4,out.Cg5,out.Cg6,out.Bg1,out.Bg2,
              out.Bg3,out.Bg4,out.Bg5,out.Bg6,out.Ig1,out.Ig2,out.Ig3,out.Ig4,
              out.Ig5,out.Ig6,out.Anti,out.IncPrec) :=
              DESLib.RandomLib.Variates.initGeneratorWrap();
            end initst;
          end Interarrival;

        package ProcessorWithQueue
          import DESLib.DEVSLib.SRC.*;
          model Qprocessor
          //Real pt "random processing time";
          Real T = time "time";
                   extends AtomicDEVS(
          redeclare record State = st);
          redeclare function Fcon = con(t=T);
          redeclare function Fint = int(t=T);
          redeclare function Fext = ext(t=T);
          redeclare function Fout = out;
          redeclare function Fta = ta;
          redeclare function initState = initst;
            annotation (Diagram(graphics));
              Interfaces.outPort outPort1
                               annotation (extent=[80,-10;100,10]);
              Interfaces.inPort inPort1
                             annotation (extent=[-100,-10;-80,10]);
          equation
              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;

          end Qprocessor;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            input Real t;
            output st sout;
          algorithm
            sout := ext(int(s,t=t),0,bag, t=t);
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            import RandomLib;
            input st s;
            input Real t;
            output st sout;
          protected
            stdEvent x;
          algorithm
            sout := s;
            if sout.NQ == 0 then // no jobs in queue
              sout.phase := 1;
              sout.sigma := Modelica.Constants.inf; // processing_time;
              sout.job := 0;
            else // take job from queue
              sout.Qsum := sout.Qsum + sout.NQ*(t-sout.Qlast);
              sout.Qavg := sout.Qsum/t;
              sout.Qlast := t;
              sout.NQ := sout.NQ - 1;
              x := DEVSLib.SRC.getEvent(sout.Q);
              sout.job := x.Value;
              sout.phase := 2;
              (sout.sigma,sout.Cg1,sout.Cg2,sout.Cg3,sout.Cg4,sout.Cg5,sout.Cg6,
                sout.Bg1,sout.Bg2,sout.Bg3,sout.Bg4,sout.Bg5,sout.Bg6,sout.Ig1,
                sout.Ig2,sout.Ig3,sout.Ig4,sout.Ig5,sout.Ig6,sout.Anti,sout.IncPrec)
                := DESLib.RandomLib.Variates.GenerateVariateWrap(
                          sout.Cg1,
                          sout.Cg2,
                          sout.Cg3,
                          sout.Cg4,
                          sout.Cg5,
                          sout.Cg6,
                          sout.Bg1,
                          sout.Bg2,
                          sout.Bg3,
                          sout.Bg4,
                          sout.Bg5,
                          sout.Bg6,
                          sout.Ig1,
                          sout.Ig2,
                          sout.Ig3,
                          sout.Ig4,
                          sout.Ig5,
                          sout.Ig6,
                          sout.Anti,
                          sout.IncPrec,
                          9,
                          8);
              //sout.sigma := pt;
            end if;
          end int;

          function ext "External Transition Function"
            import RandomLib;
            input st s;
            input Real e;
            input Integer bag;
            //input Real pt;
            input Real t;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              sout.received := sout.received +1;
              if sout.phase == 1 then // passive
                sout.job := x.Value;
                sout.phase := 2;
                (sout.sigma,sout.Cg1,sout.Cg2,sout.Cg3,sout.Cg4,sout.Cg5,sout.Cg6,
                  sout.Bg1,sout.Bg2,sout.Bg3,sout.Bg4,sout.Bg5,sout.Bg6,sout.Ig1,
                  sout.Ig2,sout.Ig3,sout.Ig4,sout.Ig5,sout.Ig6,sout.Anti,sout.IncPrec)
                  := DESLib.RandomLib.Variates.GenerateVariateWrap(
                            sout.Cg1,
                            sout.Cg2,
                            sout.Cg3,
                            sout.Cg4,
                            sout.Cg5,
                            sout.Cg6,
                            sout.Bg1,
                            sout.Bg2,
                            sout.Bg3,
                            sout.Bg4,
                            sout.Bg5,
                            sout.Bg6,
                            sout.Ig1,
                            sout.Ig2,
                            sout.Ig3,
                            sout.Ig4,
                            sout.Ig5,
                            sout.Ig6,
                            sout.Anti,
                            sout.IncPrec,
                            9,
                            8);
                // sout.sigma := pt; // random processing_time
              else // processing
                sout.Qsum := sout.Qsum + sout.NQ*(t-sout.Qlast);
                sout.Qavg := sout.Qsum/t;
                sout.Qlast := t;
                sout.NQ := sout.NQ +1;
                sendEvent(sout.Q,x);
                sout.sigma := s.sigma -e;
              end if;
            end for;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            if s.phase == 2 then
              y.Type := 1;
              y.Value := s.job;
              sendEvent(queue[1],y);
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real job;
            Integer received;
            Integer Q;
            Integer NQ;
            Real Qavg;
            Real Qsum;
            Real Qlast;
              Real Cg1;
              Real Cg2;
              Real Cg3;
              Real Cg4;
              Real Cg5;
              Real Cg6;
              Real Bg1;
              Real Bg2;
              Real Bg3;
              Real Bg4;
              Real Bg5;
              Real Bg6;
              Real Ig1;
              Real Ig2;
              Real Ig3;
              Real Ig4;
              Real Ig5;
              Real Ig6;
              Integer Anti;
              Integer IncPrec;
          end st;

          function initst
            import RandomLib;
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 0; //Modelica.Constants.inf;  //first generation starts at time == 1
            out.job := 0;
            out.received := 0;
            out.Q := DESLib.DEVSLib.SRC.CreateQueue();
            out.NQ := 0;
            out.Qavg := 0;
            out.Qsum := 0;
            out.Qlast := 0;
            (out.Cg1,out.Cg2,out.Cg3,out.Cg4,out.Cg5,out.Cg6,out.Bg1,out.Bg2,
              out.Bg3,out.Bg4,out.Bg5,out.Bg6,out.Ig1,out.Ig2,out.Ig3,out.Ig4,
              out.Ig5,out.Ig6,out.Anti,out.IncPrec) :=
              DESLib.RandomLib.Variates.initGeneratorWrap();
          end initst;
        end ProcessorWithQueue;

        model MM1
          Real QAVG;

                  Interarrival.interarrival IA annotation (extent=[-80,0; -60,20]);
                  AuxModels.Display display annotation (extent=[40,0; 60,20]);
                  ProcessorWithQueue.Qprocessor QP annotation (extent=[-20,0; 0,20]);

        equation
                  QAVG = QP.S.Qavg;

                  when terminal() then
                    Modelica.Utilities.Streams.print("Qavg = "+String(QAVG));
                  end when;

                  annotation (Diagram,
                    experiment(StopTime=1e+006),
                    experimentSetupOutput);
                  connect(IA.outPort1, QP.inPort1)
                    annotation (points=[-61,10; -19,10], style(color=0, rgbcolor={0,0,0}));
                  connect(QP.outPort1, display.inPort1)
                    annotation (points=[-1,10; 41,10], style(color=0, rgbcolor={0,0,0}));

        end MM1;

          package Interarrival2

          annotation(preferedView="info",
            Window(
              x=0.02,
              y=0.01,
              width=0.2,
              height=0.57,
              library=1,
              autolayout=1),
            version="2.2",
            versionDate="2005-04-15",
            Settings(NewStateSelection=true),
            Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica library for modeling and simulation systems using the DEVS formalism.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
            model interarrival
              parameter Real mean = 10;
              parameter Integer seed = 123134123;
              parameter Real period = 1;
              parameter Real firstGeneration = 1;
              parameter Integer numGenerations = 0 "0 == infinity";
              parameter Real outValue = 1;
              extends AtomicDEVS(redeclare record State = st);
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(p=period,fg=firstGeneration,val=outValue,ng=numGenerations,m=mean,s=seed);
              annotation (Diagram);
              Interfaces.outPort outPort1
                               annotation (extent=[80,-10;100,10]);
            equation
              for i in 1:numIn loop
                iEvent[i] = 0;
              end for;
              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;

            end interarrival;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              sout.gen := sout.gen +1;
              if (sout.gen < s.ng) or (s.ng == 0) then
                sout.Xiplus := mod(sout.a*sout.Xi,sout.m);
                sout.sigma := -sout.mean * ln(sout.Xiplus/sout.m); // random processing_time
                sout.Xi := sout.Xiplus;
              else
                sout.sigma := Modelica.Constants.inf;
              end if;
              sout.p := s.p;
              sout.val := s.val;
              sout.port := 1;
            end int;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent Y;
            algorithm
              Y.Value := s.val;
              Y.Type := 1;
              sendEvent(queue[1],Y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase; // 1 = passive, 2 = active
              Real sigma;
              Real p;
              Real fg;
              Real val;
              Integer port;
              Integer ng;
              Integer gen;
              Real mean;
              Integer s;
              Integer Xi;
              Integer Xiplus;
              Integer m;
              Integer a;
            end st;

            function initst
              input Real p;
              input Real fg;
              input Real val;
              input Integer ng;
              input Real m;
              input Integer s;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := fg;
              out.p := p;
              out.val := val;
              out.port := 1;
              out.ng := ng;
              out.gen := 0;
              out.mean := m;
              out.s := s;
              out.Xi := s;
              out.Xiplus := 0;
              out.m := 2147483647;
              out.a := 7^5;
            end initst;
          end Interarrival2;

        package ProcessorWithQueue2
          import DESLib.DEVSLib.SRC.*;
          model Qprocessor
            Real T= time;
            parameter Real mean = 8;
            parameter Integer seed = 1324123;
                   extends AtomicDEVS(
          redeclare record State = st);
          redeclare function Fcon = con(t=T);
          redeclare function Fint = int(t=T);
          redeclare function Fext = ext(t=T);
          redeclare function Fout = out;
          redeclare function Fta = ta;
          redeclare function initState = initst(m=mean,s=seed);
            annotation (Diagram);
              Interfaces.outPort outPort1
                               annotation (extent=[80,-10;100,10]);
              Interfaces.inPort inPort1
                             annotation (extent=[-100,-10;-80,10]);
          equation
              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;

          end Qprocessor;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            input Real t;
            output st sout;
          algorithm
            sout := ext(int(s,t=t),0,bag,t=t);
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            input Real t;
            output st sout;
          protected
            stdEvent x;
          algorithm
            sout := s;
            if sout.NQ == 0 then // no jobs in queue
              sout.phase := 1;
              sout.sigma := Modelica.Constants.inf; // processing_time;
              sout.job := 0;
            else // take job from queue
              sout.Qsum := sout.Qsum + sout.NQ*(t-sout.Qlast);
              sout.Qavg := sout.Qsum/t;
              sout.Qlast := t;
              sout.NQ := sout.NQ - 1;
              x := DEVSLib.SRC.getEvent(sout.Q);
              sout.job := x.Value;
              sout.phase := 2;
              sout.Xiplus := mod(sout.a*sout.Xi,sout.m);
              sout.sigma := -sout.mean * ln(sout.Xiplus/sout.m); // random processing_time
              sout.Xi := sout.Xiplus;
            end if;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            input Real t;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := DEVSLib.SRC.getEvent(bag);
              sout.received := sout.received +1;
              if sout.phase == 1 then // passive
                sout.job := x.Value;
                sout.phase := 2;
                sout.Xiplus := mod(sout.a*sout.Xi,sout.m);
                sout.sigma := -sout.mean * ln(sout.Xiplus/sout.m); // random processing_time
                sout.Xi := sout.Xiplus;
              else // processing
                sout.Qsum := sout.Qsum + sout.NQ*(t-sout.Qlast);
                sout.Qavg := sout.Qsum/t;
                sout.Qlast := t;
                sout.NQ := sout.NQ +1;
                sendEvent(sout.Q,x);
                sout.sigma := s.sigma -e;
              end if;
            end for;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
          algorithm
            if s.phase == 2 then
              y.Type := 1;
              y.Value := s.job;
              sendEvent(queue[1],y);
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st
            Integer phase; // 1 = passive, 2 = active
            Real sigma;
            Real job;
            Integer received;
            Integer Q;
            Integer NQ;
            Real Qavg;
            Real Qsum;
            Real Qlast;
            Real mean;
            Integer s;
            Integer Xi;
            Integer Xiplus;
            Integer m;
            Integer a;
          end st;

          function initst
            input Real m;
            input Integer s;
            output st out;
          algorithm
            out.phase := 1; // passive
            out.sigma := 0; //Modelica.Constants.inf;  //first generation starts at time == 1
            out.job := 0;
            out.received := 0;
            out.Q := DEVSLib.SRC.CreateQueue();
            out.NQ := 0;
            out.Qavg := 0;
            out.Qsum := 0;
            out.Qlast := 0;
            out.mean := m;
            out.s := s;
            out.Xi := s;
            out.Xiplus := 0;
            out.m := 2147483647;
            out.a := 7^5;
          end initst;
        end ProcessorWithQueue2;

        model MM12
        Real QAVG;

                  AuxModels.Display display annotation (extent=[40,0; 60,20]);
                  ProcessorWithQueue2.Qprocessor qprocessor(seed=4556)
                    annotation (extent=[-20,0; 0,20]);
                  Interarrival2.interarrival interarrival(seed=998611)
                    annotation (extent=[-80,0; -60,20]);

        equation
                  QAVG = qprocessor.S.Qavg;

                  when terminal() then
                    Modelica.Utilities.Streams.print("Qavg = "+String(QAVG));
                  end when;

                  annotation (Diagram,
                    experiment(StopTime=1e+006),
                    experimentSetupOutput);
                  connect(interarrival.outPort1, qprocessor.inPort1) annotation (points=[-61,10;
                        -19,10],                     style(color=0, rgbcolor={0,0,0}));
                  connect(qprocessor.outPort1, display.inPort1) annotation (points=[-1,10; 41,
                        10],                 style(color=0, rgbcolor={0,0,0}));
        end MM12;
      end PetriNetsExamples;

      package CraneCrab2006
        "Model of the ARGESIM benchmark system C13 'crane and embedded control`"
      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package includes the implementation of the C13 comparison system \"Crane and Embedded Control\" proposed by ARGESIM.<br>
A detailed description of the system can be found in [1].
</p>

<p>
The system is composed by a crane that moves along a horizontal rail transporting a load, which has been modeled as a continuous-time system.<br>
The crane system has been modeled as a continuous-time model, using the linear and non-linear equations described in [1].
</p>
<p>
That crane is controlled by a discrete regulator, modeled using DEVSLib and components from the Modelica Standard Library.<br>
The communication between the crane system and the controller has been performed using the interface models included in DEVSLib.
</p>

<p>
As described in the system definition, three tasks have been performed in order to evaluate its results and compare them with previous implementations.
<ul>
<li>Task A compares the linear and the non-linear implementation of the crane system, using multiple disturbances.
<li>Task B evaluates the response of the system without using the brake and the emergency stop.
<li>Task C evaluates the behavior of the brake and the emergency stop.
</ul>

Task B and Task C models include variables to facilitate the observation of the car and load positions during the simulation.
</p>

<p>
REFERENCES:<br>
[1] Joachim Scheikl, Felix Breitenecher and Ingrid Bausch-Gall. \"Crane and Embedded Control\" - Definition of an ARGESIM Benchmark with Implicit Modelling, Digital Control and Sensor Action. Revised Definition - C13 revised. <i>Simulation News Europe </i>, 16(1), 2006.

<br>
</HTML>
"));

        package CraneSystem
        annotation (uses(Modelica(version="2.2.1")));

        model LinearCrane
            import Modelica.SIunits.*;
          parameter Mass mc = 10 "mass of the car";
          parameter Mass ml = 100 "mass of the load";
          parameter Length r = 5 "lenght of the cable that holds the load";
          parameter Force g = 9.81 "force of gravity";
          parameter Real dcnormal = 0.5;
          parameter Real dcbrake = 100000;
          parameter Real dl = 0.01;
          parameter Voltage VMax = 40 "max voltage for the DC motor";
          parameter Position PosCarMax = 5
              "maximum limit for the position of the car";
          parameter Position PosCarMin = -5
              "minimum limit for the position of the car";

         Angle alpha;
                  Position loadPos;
                  Modelica.Blocks.Interfaces.RealOutput posCar
                    annotation (extent=[100,70; 120,90]);
                  Modelica.Blocks.Interfaces.BooleanOutput SwPosCarMin
                    annotation (extent=[100,-90; 120,-70]);
                  Modelica.Blocks.Interfaces.RealInput fcdesired
              "input voltage for the DC motor"
                    annotation (extent=[-140,-20; -100,20]);
                  Modelica.Blocks.Interfaces.BooleanOutput SwPosCarMax
                    annotation (extent=[100,-50; 120,-30]);
                Modelica.Blocks.Interfaces.RealInput fd
                    annotation (extent=[-40,-140; 0,-100], rotation=90);
          protected
          Real dc;

          Position xl;
          Position xc;
          Force fc;
          Real fc1;
          Real xc1;
          Real xc2;
          Real alpha2;
          Real alpha1;
          public
          Modelica.Blocks.Interfaces.BooleanInput brake
                    annotation (extent=[-140,-100; -100,-60]);
        equation
        // Car and Load
          xc2 = fc/mc + g*(ml/mc)*alpha - (dc/mc)*xc1;
          r*alpha2 = -g*(1+ml/mc)*alpha + (dc/mc - dl/ml)*xc1 - r*(dl/ml)*alpha1 - fc/mc + fd/ml;
          xl = xc + r*alpha;
          alpha1 = der(alpha);
          alpha2 = der(alpha1);
          xc1 = der(xc);
          xc2 = der(xc1);
        // DC motor
          fc1 = der(fc);
          fc1 = -4*(fc - fcdesired);
          //  tm*fc1 + fc = km*v;
          //v = if abs(VC) <= VMax then VC else 40*sign(VC);

          SwPosCarMin = if posCar < PosCarMin then true else false;
          SwPosCarMax = if posCar > PosCarMax then true else false;
          posCar = xc;
          loadPos = xl;
          dc = if brake then dcbrake else dcnormal;

          annotation (Diagram, Icon(Bitmap(extent=[-70,58; 70,-62], name=
                              "Figs/cranesystem.png"),
                        Text(
                        extent=[-98,98; 98,52],
                        style(color=3, rgbcolor={0,0,255}),
                        string="Linear")));
        end LinearCrane;

        model NonLinearCrane
            import Modelica.SIunits.*;
          parameter Mass mc = 10 "mass of the car";
          parameter Length r = 5 "lenght of the cable that holds the load";
          parameter Force g = 9.81 "force of gravity";
          parameter Real dcnormal = 0.5;
          parameter Real dcbrake = 100000;
          parameter Real dl = 0.01;
          parameter Mass ml = 100 "mass of the load";
          parameter Voltage VMax = 40 "max voltage for the DC motor";
          parameter Position PosCarMax = 5
              "maximum limit for the position of the car";
          parameter Position PosCarMin = -5
              "minimum limit for the position of the car";

        Position loadPos;
                  Angle alpha;
                  Modelica.Blocks.Interfaces.RealOutput posCar
                    annotation (extent=[100,70; 120,90]);
                  Modelica.Blocks.Interfaces.BooleanOutput SwPosCarMin
                    annotation (extent=[100,-90; 120,-70]);
                  Modelica.Blocks.Interfaces.RealInput fcdesired
              "input voltage for the DC motor"
                    annotation (extent=[-140,-20; -100,20]);
                  Modelica.Blocks.Interfaces.BooleanOutput SwPosCarMax
                    annotation (extent=[100,-50; 120,-30]);
                  Modelica.Blocks.Interfaces.RealInput fd
                    annotation (extent=[-40,-140; 0,-100], rotation=90);

          protected
          Real dc;
          Position xl;
          Position xc;
          Force fc;
          Real fc1;
          Real xc1;
          Real xc2;
          Real alpha2;
          Real alpha1;

          public
        Modelica.Blocks.Interfaces.BooleanInput brake
                    annotation (extent=[-140,-100; -100,-60]);
        equation
        //Car and Load
          xc2*(mc + ml*sin(alpha)^2) = -dc*xc1 + fc + fd*sin(alpha)^2
          + ml*sin(alpha)*(r*alpha1^2 + g*cos(alpha)) - dl*xc1*sin(alpha)^2;
          alpha2*r^2*(ml*sin(alpha)^2 + mc) = (fd*mc/ml - fc + dc*xc1)*r*cos(alpha)
          - (g*(ml + mc) + ml*r*alpha1^2*cos(alpha))*r*sin(alpha)
          - dl*((mc/ml)*(xc1*r*cos(alpha) + r^2*alpha1) + r^2*alpha1*sin(alpha)^2);
          xl = xc + r*sin(alpha);
          alpha1 = der(alpha);
          alpha2 = der(alpha1);
          xc1 = der(xc);
          xc2 = der(xc1);
        // DC motor
          fc1 = der(fc);
          fc1 = -4*(fc -fcdesired);

          SwPosCarMin = if posCar < PosCarMin then true else false;
          SwPosCarMax = if posCar > PosCarMax then true else false;
          posCar = xc;
          loadPos = xl;
          dc = if brake then dcbrake else dcnormal;

         annotation (Diagram, Icon(Bitmap(extent=[-70,60; 70,-60], name=
                              "Figs/cranesystem.png"),
                                            Text(
                          extent=[-100,100; 100,60],
                          style(color=3, rgbcolor={0,0,255}),
                          string="Non-Linear")));

        end NonLinearCrane;

        end CraneSystem;

        package Controller

        model DEVSController

                  Modelica.Blocks.Interfaces.RealInput posCar
                    annotation (extent=[-106,20; -66,60],  rotation=0);
                  Modelica.Blocks.Interfaces.BooleanInput swPosCarMax
                    annotation (extent=[-106,-40; -66,0],   rotation=0);
                  Modelica.Blocks.Interfaces.BooleanInput swPosCarMin
                    annotation (extent=[-106,-72; -66,-32],  rotation=0);
                  annotation (Diagram);
                  Modelica.Blocks.Interfaces.RealOutput fcdesired
                    annotation (extent=[86,-10; 106,10]);

              DESLib.DEVSLib.Examples.CraneCrab2006.Controller.PositionController.positionController
              PositionController
                    annotation (extent=[-48,-14; 14,46]);
                  AuxModels.DICO controlSignal
                                      annotation (extent=[20,8; 44,24]);
                  Modelica.Blocks.Interfaces.RealInput desiredPosition
                    annotation (extent=[-36,62; 4,102],   rotation=270);
                  Modelica.Blocks.Interfaces.BooleanOutput brake
                    annotation (extent=[86,-60; 106,-40]);
                  DESLib.DEVSLib.Examples.CraneCrab2006.Controller.Diagnosis
              DIAGNOSIS
                    annotation (extent=[30,-56; 74,-12]);
        equation
        connect(posCar, PositionController.posCar)         annotation (points=[-86,40;
                          -54.2,40],                 style(color=74, rgbcolor={0,0,127}));
                    connect(PositionController.outPort1, controlSignal.inport)
                                                                  annotation (points=[15.86,16;
                          19.28,16],                 style(color=0, rgbcolor={0,0,0}));
                    connect(swPosCarMax, DIAGNOSIS.swPosCarMax)         annotation (
                      points=[-86,-20; -41,-20; -41,-27.4; 30,-27.4],
                                                                 style(color=5, rgbcolor=
                          {255,0,255}));
                    connect(swPosCarMin, DIAGNOSIS.swPosCarMin)         annotation (
                      points=[-86,-52; -32,-52; -32,-40.6; 29.56,-40.6],
                                                                 style(color=5, rgbcolor=
                          {255,0,255}));
                    connect(controlSignal.y, DIAGNOSIS.fcdesired)           annotation (points=[45.2,16;
                          47.6,16; 47.6,-12],  style(color=74, rgbcolor={0,0,127}));
                    connect(DIAGNOSIS.brake, brake)           annotation (points=[76.2,
                          -44.56; 80,-44.56; 80,-50; 96,-50],style(color=5, rgbcolor={
                            255,0,255}));
                    connect(DIAGNOSIS.fcdesiredOUT, fcdesired) annotation (points=[
                          76.42,-31.8; 79.75,-31.8; 79.75,0; 96,0], style(color=74,
                          rgbcolor={0,0,127}));
                    connect(desiredPosition, PositionController.desiredPosition)
                      annotation (points=[-16,82; -16,64; -17,64; -17,52], style(color=
                            74, rgbcolor={0,0,127}));
        end DEVSController;

        package PositionController

        annotation(preferedView="info",
          Window(
            x=0.02,
            y=0.01,
            width=0.2,
            height=0.57,
            library=1,
            autolayout=1),
          version="2.2",
          versionDate="2005-04-15",
          Settings(NewStateSelection=true),
          Documentation(info="<HTML>
<p>
<b>DEVSLib</b> is a Modelica library for modeling and simulation systems using the DEVS formalism.
</p>


</HTML>
"));
            import DESLib.DEVSLib.SRC.*;
          model positionController
            parameter Real ForceMax = 160;
            parameter Real V = 109.5;
            parameter Real cycle = 0.01;
            Real T = time;

                     extends AtomicDEVS(numIn=1,
            redeclare record State = st);
            //redeclare function Fcon = con(pc=posCar,pos=desiredPosition);
            //redeclare function Fext = ext(pc=posCar,pos=desiredPosition);
            redeclare function Fint = int(pc=posCar,pos=desiredPosition,t=T);
            redeclare function Fout = out;
            redeclare function Fta = ta;
            redeclare function initState = initst(pos=desiredPosition,fmax = ForceMax,cycle=cycle,V=V);

              annotation (Diagram, Icon(Text(
                          extent=[-80,78; 80,0],
                          style(color=3, rgbcolor={0,0,255}),
                          string="%name"), Rectangle(extent=[-100,100; 100,-100], style(
                              color=3, rgbcolor={0,0,255}))));
                    Interfaces.outPort outPort1     annotation (extent=[96,-10; 116,10]);
                    //Interfaces.inPort setPos annotation 8;

                    Modelica.Blocks.Interfaces.RealInput posCar
                      annotation (extent=[-140,60; -100,100]);
                    Modelica.Blocks.Interfaces.RealInput desiredPosition
                      annotation (extent=[-20,100; 20,140], rotation=270);
          equation
            iEvent[1] = 0;//setPos.event;
            //iQueue[1] = 0;//setPos.queue;

            oEvent[1] = outPort1.event;
            oQueue[1] = outPort1.queue;
          end positionController;

        /* function con "Confluent Transtition Function"
    input st s;
    input Real e;
    input Integer bag;
    input Real pc;
    input Real pos;
    output st sout;
    output st soutput;
  algorithm
    // first internal
    soutput := s;
    sout := ext(int(s,pc=pc),0,bag,pc=pc,pos=pos);
    //----------------
    // first external
    //soutput := ext(s,e,bag);
    //sout := int(soutput);
  end con;*/

          function int "Internal Transition Function"
            input st s;
            input Real pc;
            input Real pos;
            input Real t;
            output st sout;
          algorithm
            sout := s;
            sout.q1 := (s.M11 - s.d1*s.c1)*s.q1 + (s.M12 - s.d1*s.c2)*s.q2 + (s.M13 - s.d1*s.c3)*s.q3 + (s.M14 - s.d1*s.c4)*s.q4 + (s.M15 - s.d1*s.c5)*s.q5  + pc*s.d1 + s.fcdesired*s.b1;
            sout.q2 := (s.M21 - s.d2*s.c1)*s.q1 + (s.M22 - s.d2*s.c2)*s.q2 + (s.M23 - s.d2*s.c3)*s.q3 + (s.M24 - s.d2*s.c4)*s.q4 + (s.M25 - s.d2*s.c5)*s.q5  + pc*s.d2 + s.fcdesired*s.b2;
            sout.q3 := (s.M31 - s.d3*s.c1)*s.q1 + (s.M32 - s.d3*s.c2)*s.q2 + (s.M33 - s.d3*s.c3)*s.q3 + (s.M34 - s.d3*s.c4)*s.q4 + (s.M35 - s.d3*s.c5)*s.q5  + pc*s.d3 + s.fcdesired*s.b3;
            sout.q4 := (s.M41 - s.d4*s.c1)*s.q1 + (s.M42 - s.d4*s.c2)*s.q2 + (s.M43 - s.d4*s.c3)*s.q3 + (s.M44 - s.d4*s.c4)*s.q4 + (s.M45 - s.d4*s.c5)*s.q5  + pc*s.d4 + s.fcdesired*s.b4;
            sout.q5 := (s.M51 - s.d5*s.c1)*s.q1 + (s.M52 - s.d5*s.c2)*s.q2 + (s.M53 - s.d5*s.c3)*s.q3 + (s.M54 - s.d5*s.c4)*s.q4 + (s.M55 - s.d5*s.c5)*s.q5  + pc*s.d5 + s.fcdesired*s.b5;
            sout.u := s.V*pos - (s.h1*sout.q1 + s.h2*sout.q2 + s.h3*sout.q3 + s.h4*sout.q4 +s.h5*sout.q5);
            sout.fcdesired := max(min(sout.u,s.fmax),-s.fmax);
            sout.sigma := s.cycle;
            sout.pos := pos;
            sout.pc := pc;
            //LogVariable(pos);
          end int;

         /* function ext "External Transition Function"
    input st s;
    input Real e;
    input Integer bag;
    input Real pc;
    output st sout;
  protected
    Integer numreceived;
    stdEvent x;
  algorithm
    sout := s;
    numreceived := numEvents(bag);
    for i in 1:numreceived loop
      x := getEvent(bag);
      // change in desired position
      sout.pos := x.Value;
      sout.sigma := 0;
    end for;
  end ext;*/

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
            protected
            stdEvent y;
          algorithm
            y.Type := integer(s.fcdesired);
            y.Value := s.fcdesired;
            sendEvent(queue[1],y);
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st "state of the model"
            Integer phase;
            Real sigma;
            Real fmax;
            Real cycle;
            Real pos;
            Real q1;
            Real q2;
            Real q3;
            Real q4;
            Real q5;
            Real M11;
            Real M12;
            Real M13;
            Real M14;
            Real M15;
            Real M21;
            Real M22;
            Real M23;
            Real M24;
            Real M25;
            Real M31;
            Real M32;
            Real M33;
            Real M34;
            Real M35;
            Real M41;
            Real M42;
            Real M43;
            Real M44;
            Real M45;
            Real M51;
            Real M52;
            Real M53;
            Real M54;
            Real M55;
            Real c1;
            Real c2;
            Real c3;
            Real c4;
            Real c5;
            Real d1;
            Real d2;
            Real d3;
            Real d4;
            Real d5;
            Real b1;
            Real b2;
            Real b3;
            Real b4;
            Real b5;
            Real h1;
            Real h2;
            Real h3;
            Real h4;
            Real h5;
            Real u;
            Real fcdesired;
            Real V;
            Real pc;
          end st;

          function initst "state initalization function"
            input Real pos;
            input Real fmax;
            input Real cycle;
            input Real V;
            output st out;
          algorithm
            out.phase := 0;
            out.sigma := 0;
            out.fmax := fmax;
            out.cycle := cycle;
            out.pos := pos;
            out.q1 := 0;
            out.q2 := 0;
            out.q3 := 0;
            out.q4 := 0;
            out.q5 := 0;
            out.M11 := 0.96;
            out.M12 := 0;
            out.M13 := 0;
            out.M14 := 0;
            out.M15 := 0;
            out.M21 := 0;
            out.M22 := 1;
            out.M23 := 0.01;
            out.M24 := 0;
            out.M25 := 0;
            out.M31 := 0.001;
            out.M32 := 0;
            out.M33 := 0.9995;
            out.M34 := 0.981;
            out.M35 := 0;
            out.M41 := 0;
            out.M42 := 0;
            out.M43 := 0;
            out.M44 := 1;
            out.M45 := 0.01;
            out.M51 := -0.0002;
            out.M52 := 0;
            out.M53 := 0.0001;
            out.M54 := -0.2158;
            out.M55 := 1;
            out.c1 := 0;
            out.c2 := 1;
            out.c3 := 0;
            out.c4 := 0;
            out.c5 := 0;
            out.d1 := 34.5724;
            out.d2 := 0.2395;
            out.d3 := 2.0322;
            out.d4 := 0.0164;
            out.d5 := -0.1979;
            out.b1 := 0.04;
            out.b2 := 0;
            out.b3 := 0;
            out.b4 := 0;
            out.b5 := 0;
            out.h1 := 2.9;
            out.h2 := 109.5;
            out.h3 := 286.0;
            out.h4 := 1790.6;
            out.h5 := 44.5;
            out.u := 0;
            out.fcdesired := 0;
            out.V := V;
            out.pc := 0;
          end initst;
        end PositionController;

        model Diagnosis
            import DESLib.DEVSLib.SRC.*;
          // parameter Real p1 = 1;

          annotation (Diagram);
                  Modelica.Blocks.Interfaces.BooleanInput swPosCarMin
                    annotation (extent=[-122,-50; -82,-10]);
                  Modelica.Blocks.Interfaces.BooleanInput swPosCarMax
                    annotation (extent=[-120,10; -80,50]);
                  Modelica.Blocks.Logical.Or emergencyMODE
                                                 annotation (extent=[-76,-10; -48,10]);
                  Modelica.Blocks.Interfaces.BooleanOutput brake
                    annotation (extent=[100,-58; 120,-38]);
                  Modelica.Blocks.Interfaces.RealInput fcdesired
                    annotation (extent=[-40,80; 0,120],  rotation=270);
                  Modelica.Blocks.Interfaces.RealOutput fcdesiredOUT
                    annotation (extent=[100,0; 122,20]);
                  Modelica.Blocks.Logical.Or ActivateBrake
                    annotation (extent=[2,-2; 22,18]);
                  Modelica.Blocks.Logical.Switch switch1 annotation (extent=[58,-12; 78,
                          8]);
                  Modelica.Blocks.Sources.Constant const(k=0)
                    annotation (extent=[36,4; 46,16]);
                  Modelica.Blocks.Logical.Timer timer annotation (extent=[42,60; 62,80]);
                  Modelica.Blocks.Math.Abs abs  annotation (extent=[-12,60; 8,80]);
                  Modelica.Blocks.Logical.Less less annotation (extent=[16,60; 36,80]);
                  Modelica.Blocks.Sources.Constant brakeCondition(k=0.01)
                    annotation (extent=[-14,32; 10,52]);
                  Modelica.Blocks.Logical.Greater greater
                    annotation (extent=[70,60; 90,80]);
                  Modelica.Blocks.Sources.Constant timeWaiting(k=3)
                    annotation (extent=[42,32; 62,52]);
                  setTrue setTrue1 annotation (extent=[-44,-10; -24,10]);
        equation
                  connect(swPosCarMin, emergencyMODE.u2)
                                               annotation (points=[-102,-30; -84,-30;
                          -84,-8; -78.8,-8],
                                 style(color=5, rgbcolor={255,0,255}));
                  connect(ActivateBrake.y, switch1.u2) annotation (points=[23,8; 32,8;
                          32,-2; 56,-2],
                                    style(color=5, rgbcolor={255,0,255}));
                  connect(ActivateBrake.y, brake) annotation (points=[23,8; 32,8; 32,
                          -48; 110,-48],
                                  style(color=5, rgbcolor={255,0,255}));
                    connect(fcdesired, abs.u)
                                             annotation (points=[-20,100; -20,70; -14,
                          70],
                      style(color=74, rgbcolor={0,0,127}));
                  connect(less.y, timer.u) annotation (points=[37,70; 40,70], style(color=
                         5, rgbcolor={255,0,255}));
                    connect(abs.y, less.u1)
                                           annotation (points=[9,70; 14,70],  style(color=
                         74, rgbcolor={0,0,127}));
                  connect(brakeCondition.y, less.u2) annotation (points=[11.2,42; 14,42;
                          14,62],
                             style(color=74, rgbcolor={0,0,127}));
                  connect(timer.y, greater.u1) annotation (points=[63,70; 68,70], style(
                        color=74, rgbcolor={0,0,127}));
                  connect(greater.y, ActivateBrake.u1) annotation (points=[91,70; 94,70;
                          94,24; -14,24; -14,8; 0,8],
                                                  style(color=5, rgbcolor={255,0,255}));
                  connect(timeWaiting.y, greater.u2) annotation (points=[63,42; 64,42;
                          64,62; 68,62],
                                    style(color=74, rgbcolor={0,0,127}));
                  connect(const.y, switch1.u1) annotation (points=[46.5,10; 50,10; 50,6;
                          56,6],                                                    style(
                        color=74, rgbcolor={0,0,127}));
                  connect(fcdesired, switch1.u3) annotation (points=[-20,100; -20,-10;
                          56,-10],                style(color=74, rgbcolor={0,0,127}));
                  connect(emergencyMODE.y, setTrue1.u1) annotation (points=[-46.6,0;
                          -46,0],
                      style(color=5, rgbcolor={255,0,255}));
                  connect(setTrue1.y, ActivateBrake.u2) annotation (points=[-23,0; 0,0],
                      style(color=5, rgbcolor={255,0,255}));
                    connect(switch1.y, fcdesiredOUT) annotation (points=[79,-2; 86,-2;
                          86,10; 111,10], style(color=74, rgbcolor={0,0,127}));
                    connect(brake, brake) annotation (points=[110,-48; 110,-48], style(
                          color=5, rgbcolor={255,0,255}));
                    connect(swPosCarMax, emergencyMODE.u1) annotation (points=[-100,30;
                          -84,30; -84,0; -78.8,0], style(color=5, rgbcolor={255,0,255}));

        end Diagnosis;

        model setTrue
            Modelica.Blocks.Interfaces.BooleanOutput y
                    annotation (extent=[100,-10; 120,10]);
                  annotation (Diagram);
                  Modelica.Blocks.Interfaces.BooleanInput u1
                    annotation (extent=[-140,-20; -100,20]);
        initial equation
                  y = false;
        equation
                  when u1 then
                    y = true;
                  end when;
        end setTrue;

        end Controller;

        package SIMANLibController

          model SLibController

            Modelica.Blocks.Interfaces.RealInput posCar
              annotation (extent=[-120,30;-80,70]);
            Modelica.Blocks.Interfaces.BooleanInput swPosCarMax
              annotation (extent=[-120,-30;-80,10]);
            Modelica.Blocks.Interfaces.BooleanInput swPosCarMin
              annotation (extent=[-120,-62;-80,-22]);
            Modelica.Blocks.Interfaces.RealInput desiredPosition
              annotation (extent=[-26,70;14,112], rotation=270);
            Modelica.Blocks.Interfaces.RealOutput fcdesired
              annotation (extent=[96,0;116,20]);
            Modelica.Blocks.Interfaces.BooleanOutput brake
              annotation (extent=[96,-50;116,-30]);
            Controller.Diagnosis diagnosis annotation (extent=[40,-40;60,-20]);
            annotation (Diagram, Icon(Text(
                    extent=[-80,72;96,-60],
                    style(color=3,rgbcolor={0,0,255}),
                    string="%name"), Rectangle(extent=[-80,72;96,-80],style(color=3,rgbcolor={0,0,255}))));
            DESLib.DEVSLib.Examples.CraneCrab2006.SIMANLibController.SLibPosControl
              sLibPosControl               annotation (extent=[-34,12;38,54]);
          equation
            connect(diagnosis.fcdesiredOUT, fcdesired)
            annotation (points=[61.1,-29;79.55,-29;79.55,10;106,10], style(color=0,rgbcolor={0,0,127}));
            connect(diagnosis.brake, brake)
            annotation (points=[61,-34.8;79.5,-34.8;79.5,-40;106,-40], style(color=0,rgbcolor={255,0,255}));
            connect(diagnosis.swPosCarMin, swPosCarMin)
            annotation (points=[39.8,-33;-24.1,-33;-24.1,-42;-100,-42],style(color=0,rgbcolor={255,0,255}));
            connect(diagnosis.swPosCarMax, swPosCarMax)
            annotation (points=[40,-27;-25,-27;-25,-10;-100,-10],style(color=0,rgbcolor={255,0,255}));
            connect(posCar, sLibPosControl.posCar)
            annotation (points=[-100,50;-66,50;-66,33;-34,33],style(color=0,rgbcolor={0,0,127}));
            connect(diagnosis.fcdesired, sLibPosControl.controlsignal)
            annotation (points=[48,-20;48,28.8;38,28.8],style(color=0,rgbcolor={0,0,127}));
            connect(desiredPosition, sLibPosControl.desiredPosition)
            annotation (points=[-6,92;-6,66;-46,66;-46,24.6;-34,24.6],style(color=0,rgbcolor={0,0,127}));
          end SLibController;

          model SLibPosControl
             extends DESLib.SIMANLib.SRC.Draft;
            import DESLib.SIMANLib.SRC.Elements.Attributes.get;
            import DESLib.SIMANLib.SRC.Elements.Variables;

            annotation (Diagram, Icon(Text(
                    extent=[-80,60;80,-60],
                    style(rgbcolor={0,0,255}),
                    string="%name"), Rectangle(extent=[-80,60;90,-80],
                    style(rgbcolor={0,0,255}))));

           parameter Real ForceMax = 160;
           parameter Real V = 109.5;
           parameter Real M[5,5]= [0.96,0,0,0,0;0,1,0.01,0,0;0.001,0,0.9995,0.981,0;0,0,0,1,0.01;-0.0002,0,0.0001,-0.2158,1];
           parameter Real c[5] = {0,1,0,0,0};
           parameter Real d[5] = {34.5724,0.2395,2.0322,0.0164,-0.1979};
           parameter Real b[5] = {0.04,0,0,0,0};
           parameter Real h[5] = {2.9,109.5,286.0,1790.6,44.5};
           Real fcdes;

            Modelica.Blocks.Interfaces.RealInput posCar
              annotation (extent=[-120,-20;-80,20]);
            Modelica.Blocks.Interfaces.RealInput desiredPosition
              annotation (extent=[-120,-60;-80,-20]);
            annotation (Diagram);
            Modelica.Blocks.Interfaces.RealOutput controlsignal
              annotation (extent=[90,-30;110,-10]);
            DEVSLib.AuxModels.BreakLoop breakLoop annotation (extent=[-16,-72;4,-52],rotation=180);
            SIMANLib.Blocks.Create create(
              EntityType=entity,
              Interval=1,
              Maximum_batches=1,
              First_creation=0)
              annotation (extent=[-102,32;-76,52]);
            SIMANLib.Elements.EntityType entity
              annotation (extent=[-100,-88;-80,-68]);
            SIMANLib.Elements.Attribute Q(rows=5, InitialValue=[0; 0; 0; 0; 0])
              annotation (extent=[-60,-88;-40,-68]);
            SIMANLib.Elements.Attribute aQ(
              rows=5,
              InitialValue=[0; 0; 0; 0; 0],
              Number=2)
              annotation (extent=[-40,-88;-20,-68]);
            SIMANLib.Elements.Attribute U(Number=3)
              annotation (extent=[-20,-88;0,-68]);
            SIMANLib.Elements.Variable fcdesired(Number=4)
              annotation (extent=[0,-88;20,-68]);
            SIMANLib.Blocks.Assign calculate_aQ1(O=aQ, Value=(M[1, 1] - d[1]*c[1])*get(cycles.S.e,1,1)
                   + (M[1, 2] - d[1]*c[2])*get(cycles.S.e,1,2) + (M[1, 3] - d[1]*c[3])*get(cycles.S.e,1,3)
                   + (M[1, 4] - d[1]*c[4])*get(cycles.S.e,1,4) + (M[1, 5] - d[1]*c[5])*get(cycles.S.e,1,5)
                   + posCar*d[1] + fcdes*b[1])
              annotation (extent=[-50,32;-24,52]);
            SIMANLib.Blocks.Assign calculate_aQ2(
              O=aQ,
              Value=(M[2, 1] - d[2]*c[1])*get(cycles.S.e,1,1) + (M[2, 2] - d[2]*c[2])*get(cycles.S.e,1,2) + (
                  M[2, 3] - d[2]*c[3])*get(cycles.S.e,1,3) + (M[2, 4] - d[2]*c[4])*get(cycles.S.e,1,4) + (
                  M[2, 5] - d[2]*c[5])*get(cycles.S.e,1,5) + posCar*d[2] + fcdes*b[2],
              row=2) annotation (extent=[-50,10;-24,30]);

            SIMANLib.Blocks.Assign calculate_aQ3(
              O=aQ,
              row=3,
              Value=(M[3, 1] - d[3]*c[1])*get(cycles.S.e,1,1) + (M[3, 2] - d[3]*c[2])*get(cycles.S.e,1,2) + (
                  M[3, 3] - d[3]*c[3])*get(cycles.S.e,1,3) + (M[3, 4] - d[3]*c[4])*get(cycles.S.e,1,4) + (
                  M[3, 5] - d[3]*c[5])*get(cycles.S.e,1,5) + posCar*d[3] + fcdes*b[3])
              annotation (extent=[-50,-12;-24,8]);
            SIMANLib.Blocks.Assign calculate_aQ4(
              O=aQ,
              row=4,
              Value=(M[4, 1] - d[4]*c[1])*get(cycles.S.e,1,1) + (M[4, 2] - d[4]*c[2])*get(cycles.S.e,1,2) + (
                  M[4, 3] - d[4]*c[3])*get(cycles.S.e,1,3) + (M[4, 4] - d[4]*c[4])*get(cycles.S.e,1,4) + (
                  M[4, 5] - d[4]*c[5])*get(cycles.S.e,1,5) + posCar*d[4] + fcdes*b[4])
              annotation (extent=[-50,-34;-24,-14]);
            SIMANLib.Blocks.Assign calculate_aQ5(
              O=aQ,
              Value=(M[5, 1] - d[5]*c[1])*get(cycles.S.e,1,1) + (M[5, 2] - d[5]*c[2])*get(cycles.S.e,1,2) + (
                  M[5, 3] - d[5]*c[3])*get(cycles.S.e,1,3) + (M[5, 4] - d[5]*c[4])*get(cycles.S.e,1,4) + (
                  M[5, 5] - d[5]*c[5])*get(cycles.S.e,1,5) + posCar*d[5] + fcdes*b[5],
              row=5) annotation (extent=[-50,-56;-24,-36]);

            SIMANLib.Blocks.Assign update_Q1(O=Q, Value=get(calculate_aQ5.S.e,2,1))
              annotation (extent=[-10,32;16,52]);
            SIMANLib.Blocks.Assign update_Q2(
              O=Q,
              row=2,
              Value=get(calculate_aQ5.S.e,2,2))
              annotation (extent=[-10,10;16,30]);
            SIMANLib.Blocks.Assign update_Q3(
              O=Q,
              row=3,
              Value=get(calculate_aQ5.S.e,2,3))
              annotation (extent=[-10,-12;16,8]);
            SIMANLib.Blocks.Assign update_Q4(
              O=Q,
              row=4,
              Value=get(calculate_aQ5.S.e,2,4))
              annotation (extent=[-10,-34;16,-14]);
            SIMANLib.Blocks.Assign update_Q5(
              O=Q,
              row=5,
              Value=get(calculate_aQ5.S.e,2,5))
              annotation (extent=[-10,-56;16,-36]);
            SIMANLib.Blocks.Assign calculate_U(O=U, Value=V*desiredPosition - (h[1]*get(update_Q5.S.e,1,1) + h[2]*get(update_Q5.S.e,1,2) + h[3]*get(update_Q5.S.e,1,3) + h[4]*get(update_Q5.S.e,1,4) + h[5]*get(update_Q5.S.e,1,5)))
              annotation (extent=[24,32;50,52]);
            SIMANLib.Blocks.ExternalAssign ControlSignal(O=fcdesired, Value=max(min(get(calculate_U.S.e,3),ForceMax), -ForceMax))
              annotation (extent=[50,32;76,52]);
            SIMANLib.Blocks.Delay delay1(Duration=1, p1=0.01)
              annotation (extent=[76,32;98,52]);
            SIMANLib.Elements.Counter counter
              annotation (extent=[20,-88;40,-68]);
            SIMANLib.Blocks.Count cycles(Counter=counter)
              annotation (extent=[-72,32;-50,52]);
            Modelica.Blocks.Discrete.ZeroOrderHold zeroOrderHold(samplePeriod=
                  0.01) annotation (extent=[68,-28;86,-10]);
          equation
           //when sample(0,0.01) then
            fcdes = ControlSignal.y;
           //end when;

            connect(calculate_U.OUT, ControlSignal.IN)
            annotation (points=[48.18,42;51.82,42],style(color=0,rgbcolor={0,0,0}));
            connect(calculate_aQ5.OUT, update_Q1.IN)
            annotation (points=[-25.82,-46;-18,-46;-18,42;-8.18,42],style(color=0,rgbcolor={0,0,0}));
            connect(update_Q5.OUT, calculate_U.IN) annotation (points=[14.18,-46;22,-46;22,42;25.82,42],style(color=0,rgbcolor={0,0,0}));
            connect(ControlSignal.OUT, delay1.IN) annotation (points=[74.18,42;77.54,42],style(color=0,rgbcolor={0,0,0}));
            connect(cycles.OUT, calculate_aQ1.IN) annotation (points=[-51.54,42;-48.18,42],style(color=0,rgbcolor={0,0,0}));
            connect(breakLoop.OUT, cycles.IN) annotation (points=[-6.6,-62;-74,-62;-74,42;-70.46,42],style(color=0,rgbcolor={0,0,0}));
            connect(zeroOrderHold.y, controlsignal) annotation (points=[86.9,-19;92,-19;92,-20;100,-20],style(color=0,rgbcolor={0,0,127}));
            connect(zeroOrderHold.u, ControlSignal.y) annotation (points=[66.2,-19;53.9,-19;53.9,35],style(color=0,rgbcolor={0,0,127}));
            connect(delay1.OUT, breakLoop.IN) annotation (points=[96.46,42;98,42;98,12;46,12;46,-62;-3.2,-62],style(color=0,rgbcolor={0,0,0}));

            connect(create.OUT, cycles.IN) annotation (points=[-77.82,42;-70.46,42],style(color=0,rgbcolor={0,0,0}));
            connect(calculate_aQ1.OUT, calculate_aQ2.IN) annotation (points=[-25.82,42;-22,42;-22,32;-54,32;-54,20;-48.18,20],style(color=0,rgbcolor={0,0,0}));
            connect(calculate_aQ2.OUT, calculate_aQ3.IN) annotation (points=[-25.82,20;-22,20;-22,10;-54,10;-54,-2;-48.18,-2],style(color=0,rgbcolor={0,0,0}));
            connect(calculate_aQ3.OUT, calculate_aQ4.IN)
          annotation (points=[-25.82,-2;-22,-2;-22,-12;-54,-12;-54,-24;-48.18,-24],style(color=0,rgbcolor={0,0,0}));
            connect(calculate_aQ4.OUT, calculate_aQ5.IN)
          annotation (points=[-25.82,-24;-22,-24;-22,-34;-54,-34;-54,-46;-48.18,-46],style(color=0,rgbcolor={0,0,0}));
            connect(update_Q1.OUT, update_Q2.IN) annotation (points=[14.18,42;18,42;18,32;-12,32;-12,20;-8.18,20],style(color=0,rgbcolor={0,0,0}));

            connect(update_Q2.OUT, update_Q3.IN) annotation (points=[14.18,20;18,20;18,10;-12,10;-12,-2;-8.18,-2],style(color=0,rgbcolor={0,0,0}));

            connect(update_Q3.OUT, update_Q4.IN) annotation (points=[14.18,-2;18,-2;18,-12;-12,-12;-12,-24;-8.18,-24],style(color=0,rgbcolor={0,0,0}));
            connect(update_Q4.OUT, update_Q5.IN)
          annotation (points=[14.18,-24;18,-24;18,-34;-12,-34;-12,-46;-8.18,-46],style(color=0,rgbcolor={0,0,0}));
          end SLibPosControl;
        end SIMANLibController;

        package Tasks

          package LinearCrane
            model TaskA

                Modelica.Blocks.Sources.Step setFC(
                  startTime=15,
                  height=-160,
                  offset=160)   annotation (extent=[-36,26;-16,46]);
                annotation (Diagram,    experiment(StopTime=2000),    experimentSetupOutput);
                Modelica.Blocks.Sources.Trapezoid Dest1(
                  width=3,
                  nperiod=1,
                  startTime=4,
                  amplitude=-750,
                  period=1000) annotation (extent=[-36,-36;-16,-16]);
                Modelica.Blocks.Sources.BooleanConstant brake(k=false)
                  annotation (extent=[-36,-6;-16,14]);
              CraneSystem.LinearCrane linearCrane
                annotation (extent=[-2,2;18,22]);
            equation
              connect(brake.y, linearCrane.brake) annotation (points=[-15,4;-4,4],style(color=0,rgbcolor={255,0,255}));
              connect(setFC.y, linearCrane.fcdesired) annotation (points=[-15,36;-10,36;-10,12;-4,12],style(color=0,rgbcolor={0,0,127}));
              connect(Dest1.y, linearCrane.fd) annotation (points=[-15,-26;6,-26;6,0],style(color=0,rgbcolor={0,0,127}));
            end TaskA;

            model TaskA2

                Modelica.Blocks.Sources.Step setFC(
                  startTime=15,
                  height=-160,
                  offset=160)   annotation (extent=[-30,16;-10,36]);
                annotation (Diagram,    experiment(StopTime=2000),    experimentSetupOutput);
                Modelica.Blocks.Sources.Trapezoid Dest2(
                  width=3,
                  nperiod=1,
                  startTime=4,
                  period=1000,
                  amplitude=-800) annotation (extent=[-28,-44;-8,-24]);
                Modelica.Blocks.Sources.BooleanConstant brake(k=false)
                  annotation (extent=[-28,-14;-8,6]);
              CraneSystem.LinearCrane linearCrane
                annotation (extent=[4,-6;24,14]);
            equation
              connect(brake.y, linearCrane.brake) annotation (points=[-7,-4;2,-4], style(color=0,rgbcolor={255,0,255}));
              connect(setFC.y, linearCrane.fcdesired) annotation (points=[-9,26;-4,26;-4,4;2,4], style(color=0,rgbcolor={0,0,127}));
              connect(Dest2.y, linearCrane.fd) annotation (points=[-7,-34;12,-34;12,-8], style(color=0,rgbcolor={0,0,127}));
            end TaskA2;

            model TaskA3

                Modelica.Blocks.Sources.Step setFC(
                  startTime=15,
                  height=-160,
                  offset=160)   annotation (extent=[-34,24;-14,44]);
                annotation (Diagram,    experiment(StopTime=2000),    experimentSetupOutput);
                Modelica.Blocks.Sources.Trapezoid Dest3(
                  width=3,
                  nperiod=1,
                  startTime=4,
                  period=1000,
                  amplitude=-850) annotation (extent=[-34,-38;-14,-18]);
                Modelica.Blocks.Sources.BooleanConstant brake(k=false)
                  annotation (extent=[-34,-8;-14,12]);
              CraneSystem.LinearCrane linearCrane
                annotation (extent=[2,0;22,20]);
            equation
              connect(brake.y, linearCrane.brake) annotation (points=[-13,2;0,2], style(color=0,rgbcolor={255,0,255}));
              connect(setFC.y, linearCrane.fcdesired) annotation (points=[-13,34;-8,34;-8,10;0,10], style(color=0,rgbcolor={0,0,127}));
              connect(Dest3.y, linearCrane.fd) annotation (points=[-13,-28;10,-28;10,-2], style(color=0,rgbcolor={0,0,127}));
            end TaskA3;

          model TaskB
            Real Carpos = linearCrane.posCar;
            Real Loadpos = linearCrane.loadPos;

            annotation (Diagram,    experiment(StopTime=60),    experimentSetupOutput);
            Modelica.Blocks.Sources.Trapezoid trapezoid(
              amplitude=-200,
              width=1,
              period=1000,
              nperiod=1,
              startTime=42) annotation (extent=[-22,-50;-2,-30]);
            Modelica.Blocks.Sources.TimeTable desiredPosition(table=[0,3; 16,3; 16,-0.5; 36,-0.5; 36,3.8; 60,3.8])
              annotation (extent=[-60,40;-40,60]);
              DESLib.DEVSLib.Examples.CraneCrab2006.Controller.DEVSController
                dEVSController
                annotation (extent=[-40,0;-20,20]);
              CraneSystem.LinearCrane linearCrane
                annotation (extent=[-2,0;18,20]);
          equation
              connect(desiredPosition.y, dEVSController.desiredPosition)    annotation (points=[-39,50;
                    -31.6,50; -31.6,18.2],                                                                                  style(color=0,rgbcolor={0,0,127}));
              connect(dEVSController.fcdesired, linearCrane.fcdesired)    annotation (points=[-20.4,10;-4,10],style(color=0,rgbcolor={0,0,127}));
              connect(dEVSController.brake, linearCrane.brake)    annotation (points=[-20.4,5;-14.5,5;-14.5,2;-4,2],style(color=0,rgbcolor={255,0,255}));
              connect(trapezoid.y, linearCrane.fd) annotation (points=[-1,-40;6,-40;6,-2],style(color=0,rgbcolor={0,0,127}));
              connect(linearCrane.posCar, dEVSController.posCar)    annotation (points=[19,18;24,18;24,24;-50,24;-50,14;-38.6,14], style(color=0,rgbcolor={0,0,127}));
              connect(linearCrane.SwPosCarMin, dEVSController.swPosCarMin)
                annotation (points=[19,2;26,2;26,-6;-50,-6;-50,4.8;-38.6,4.8],style(color=0,rgbcolor={255,0,255}));
              connect(linearCrane.SwPosCarMax, dEVSController.swPosCarMax)
                annotation (points=[19,6;32,6;32,-10;-54,-10;-54,8;-38.6,8],style(color=0,rgbcolor={255,0,255}));
          end TaskB;

          model TaskC
            Real Carpos = linearCrane.posCar;
            Real Loadpos = linearCrane.loadPos;

            annotation (Diagram,    experiment(StopTime=60),    experimentSetupOutput);
            Modelica.Blocks.Sources.Trapezoid trapezoid1(
              width=1,
              period=1000,
              nperiod=1,
              startTime=42,
                amplitude=200)
                            annotation (extent=[-12,-50;8,-30]);
            Modelica.Blocks.Sources.TimeTable desiredPosition1(
                                                              table=[0,3; 16,3; 16,
                  -0.5; 36,-0.5; 36,3.8; 60,3.8])
              annotation (extent=[-50,40;-30,60]);
              DESLib.DEVSLib.Examples.CraneCrab2006.Controller.DEVSController
                dEVSController
                annotation (extent=[-30,0;-10,20]);
              CraneSystem.LinearCrane linearCrane
                annotation (extent=[8,0;28,20]);
            Modelica.Blocks.Sources.TimeTable DisturbingForce(table=[0,0; 42,0; 42,-200;
                  43,-200; 43,0; 44,0; 46,0; 46,200; 47,200; 47,0; 48,0])
                                               annotation (extent=[-10,-84;10,-64]);
          equation
              connect(desiredPosition1.y, dEVSController.desiredPosition)   annotation (points=[-29,50;
                    -21.6,50; -21.6,18.2],                                                                                  style(color=0,rgbcolor={0,0,127}));
              connect(dEVSController.fcdesired, linearCrane.fcdesired)    annotation (points=[-10.4,10;
                    6,10],                                                                                   style(color=0,rgbcolor={0,0,127}));
              connect(dEVSController.brake, linearCrane.brake)    annotation (points=[-10.4,5;
                    -4.5,5; -4.5,2; 6,2],                                                                        style(color=0,rgbcolor={255,0,255}));
              connect(linearCrane.posCar, dEVSController.posCar)    annotation (points=[29,18;34,18;34,24;-40,24;-40,14;-28.6,14],style(color=0,rgbcolor={0,0,127}));
              connect(linearCrane.SwPosCarMin, dEVSController.swPosCarMin)
                annotation (points=[29,2;36,2;36,-6;-40,-6;-40,4.8;-28.6,4.8],style(color=0,rgbcolor={255,0,255}));
              connect(linearCrane.SwPosCarMax, dEVSController.swPosCarMax)
                annotation (points=[29,6;42,6;42,-10;-44,-10;-44,8;-28.6,8],style(color=0,rgbcolor={255,0,255}));
              connect(DisturbingForce.y, linearCrane.fd) annotation (points=[11,-74;16,-74;16,-2],style(color=0,rgbcolor={0,0,127}));
          end TaskC;

          end LinearCrane;

          package NonLinearCrane
            model TaskA

                Modelica.Blocks.Sources.Step setFC(
                  startTime=15,
                  height=-160,
                  offset=160)   annotation (extent=[-36,26;-16,46]);
                annotation (Diagram,    experiment(StopTime=2000),    experimentSetupOutput);
                Modelica.Blocks.Sources.Trapezoid Dest1(
                  width=3,
                  nperiod=1,
                  startTime=4,
                  amplitude=-750,
                  period=1000) annotation (extent=[ -36,-36;-16,-16]);
                Modelica.Blocks.Sources.BooleanConstant brake(k=false)
                  annotation (extent=[-36,-6;-16,14]);
              CraneSystem.NonLinearCrane NonLinearCrane annotation (extent=[2,2;22,22]);
            equation
              connect(brake.y, NonLinearCrane.brake)
                annotation (points=[-15,4;0,4],style(color=0,rgbcolor={255,0,255}));
              connect(setFC.y, NonLinearCrane.fcdesired) annotation (points=[-15,36;-10,36;-10,12;0,12],style(color=0,rgbcolor={0,0,127}));
              connect(Dest1.y, NonLinearCrane.fd) annotation (points=[-15,-26;10,-26;10,0],style(color=0,rgbcolor={0,0,127}));
            end TaskA;

            model TaskA2

                Modelica.Blocks.Sources.Step setFC(
                  startTime=15,
                  height=-160,
                  offset=160)   annotation (extent=[-30,16;-10,36]);
                annotation (Diagram,    experiment(StopTime=2000),   experimentSetupOutput);
                Modelica.Blocks.Sources.Trapezoid Dest2(
                  width=3,
                  nperiod=1,
                  startTime=4,
                  period=1000,
                  amplitude=-800) annotation (extent=[-28,-44;-8,-24]);
                Modelica.Blocks.Sources.BooleanConstant brake(k=false)
                  annotation (extent=[-28,-14;-8,6]);
              CraneSystem.NonLinearCrane NonLinearCrane annotation (extent=[4,-6;24,14]);
            equation
              connect(brake.y, NonLinearCrane.brake)
                annotation (points=[-7,-4;2,-4], style(color=0, rgbcolor={255,0,255}));
              connect(setFC.y, NonLinearCrane.fcdesired) annotation (points=[-9,26;-4,26;-4,4;2,4], style(color=0, rgbcolor={0,0,127}));
              connect(Dest2.y, NonLinearCrane.fd) annotation (points=[-7,-34;12,-34;12,-8], style(color=0, rgbcolor={0,0,127}));
            end TaskA2;

            model TaskA3

                Modelica.Blocks.Sources.Step setFC(
                  startTime=15,
                  height=-160,
                  offset=160)   annotation (extent=[-34,24;-14,44]);
                annotation (Diagram, experiment(StopTime=2000), experimentSetupOutput);
                Modelica.Blocks.Sources.Trapezoid Dest3(
                  width=3,
                  nperiod=1,
                  startTime=4,
                  period=1000,
                  amplitude=-850) annotation (extent=[-34,-38;-14,-18]);
                Modelica.Blocks.Sources.BooleanConstant brake(k=false)
                  annotation (extent=[-34,-8;-14,12]);
              CraneSystem.NonLinearCrane NonLinearCrane annotation (extent=[2,0;22,20]);
            equation
              connect(brake.y, NonLinearCrane.brake)
                annotation (points=[-13,2;0,2], style(color=0, rgbcolor={255,0,255}));
              connect(setFC.y, NonLinearCrane.fcdesired) annotation (points=[-13,34;-8,34;-8,10;0,10], style(color=0, rgbcolor={0,0,127}));
              connect(Dest3.y, NonLinearCrane.fd) annotation (points=[-13,-28;10,-28;10,-2], style(color=0, rgbcolor={0,0,127}));
            end TaskA3;

          model TaskB
            Real Carpos = NonLinearCrane.posCar;
            Real Loadpos = NonLinearCrane.loadPos;

            annotation (Diagram,      experiment(StopTime=60),      experimentSetupOutput);
            Modelica.Blocks.Sources.Trapezoid trapezoid(
              amplitude=-200,
              width=1,
              period=1000,
              nperiod=1,
              startTime=42) annotation (extent=[-44,-46;-24,-26]);
            Modelica.Blocks.Sources.TimeTable desiredPosition(table=[0,3; 16,3; 16,
                  -0.5; 36,-0.5; 36,3.8; 60,3.8])
              annotation (extent=[-74,48;-44,72]);
              DESLib.DEVSLib.Examples.CraneCrab2006.Controller.DEVSController
                dEVSController
                annotation (extent=[-40,0;-20,20]);
              CraneSystem.NonLinearCrane NonLinearCrane annotation (extent=[-2,0;18,20]);
          equation
              connect(desiredPosition.y, dEVSController.desiredPosition)
                annotation (points=[-42.5,60; -31.6,60; -31.6,18.2],
                                                                  style(color=0,rgbcolor={0,0,127}));
              connect(dEVSController.fcdesired, NonLinearCrane.fcdesired)
                annotation (points=[-20.4,10;-4,10],style(color=0,rgbcolor={0,0,127}));
              connect(dEVSController.brake, NonLinearCrane.brake)
                annotation (points=[-20.4,5;-14.5,5;-14.5,2;-4,2],style(color=0,rgbcolor={255,0,255}));
              connect(NonLinearCrane.posCar, dEVSController.posCar)
                annotation (points=[19,18;24,18;24,24;-50,24;-50,14;-38.6,14],style(color=0,rgbcolor={0,0,127}));
              connect(NonLinearCrane.SwPosCarMin, dEVSController.swPosCarMin)
                annotation (points=[19,2;26,2;26,-6;-50,-6;-50,4.8;-38.6,4.8],style(color=0,rgbcolor={255,0,255}));
              connect(NonLinearCrane.SwPosCarMax, dEVSController.swPosCarMax)
                annotation (points=[19,6;32,6;32,-10;-54,-10;-54,8;-38.6,8],style(color=0,rgbcolor={255,0,255}));
              connect(trapezoid.y, NonLinearCrane.fd) annotation (points=[-23,-36;6,-36;6,-2],style(color=0,rgbcolor={0,0,127}));
          end TaskB;

          model TaskC
            Real Carpos = NonLinearCrane.posCar;
            Real Loadpos = NonLinearCrane.loadPos;

            annotation (Diagram,    experiment(StopTime=60),   experimentSetupOutput);
            Modelica.Blocks.Sources.Trapezoid Disturbance(
                width=1,
                period=1000,
                nperiod=1,
                startTime=42,
                amplitude=200)
                            annotation (extent=[-12,-46;8,-26]);
            Modelica.Blocks.Sources.TimeTable desiredPosition1(table=[0,3; 16,3; 16,-0.5; 36,-0.5; 36,3.8; 60,3.8])
              annotation (extent=[-50,40;-30,60]);
              DESLib.DEVSLib.Examples.CraneCrab2006.Controller.DEVSController
                dEVSController
                annotation (extent=[-30,0;-10,20]);
              CraneSystem.NonLinearCrane NonLinearCrane annotation (extent=[8,0;28,20]);
            Modelica.Blocks.Sources.TimeTable DisturbingForce(table=[0,0; 42,0; 42,-200;
                  43,-200; 43,0; 44,0; 46,0; 46,200; 47,200; 47,0; 48,0])
                                               annotation (extent=[-12,-82;8,-62]);
          equation
              connect(desiredPosition1.y, dEVSController.desiredPosition)   annotation (points=[-29,50;
                    -21.6,50; -21.6,18.2],                                                                                   style(color=0,rgbcolor={0,0,127}));
              connect(dEVSController.fcdesired, NonLinearCrane.fcdesired)
                annotation (points=[-10.4,10; 6,10],style(color=0,rgbcolor={0,0,127}));
              connect(dEVSController.brake, NonLinearCrane.brake)    annotation (points=[-10.4,5;
                    -4.5,5; -4.5,2; 6,2],                                                                            style(color=0,rgbcolor={255,0,255}));
              connect(NonLinearCrane.posCar, dEVSController.posCar)    annotation (points=[29,18;34,18;34,24;-40,24;-40,14;-28.6,14], style(color=0,rgbcolor={0,0,127}));
              connect(NonLinearCrane.SwPosCarMin, dEVSController.swPosCarMin)
           annotation (points=[29,2;36,2;36,-6;-40,-6;-40,4.8;-28.6,4.8], style(color=0,rgbcolor={255,0,255}));
              connect(NonLinearCrane.SwPosCarMax, dEVSController.swPosCarMax)
           annotation (points=[29,6;42,6;42,-10;-44,-10;-44,8;-28.6,8], style(color=0,rgbcolor={255,0,255}));
              connect(DisturbingForce.y, NonLinearCrane.fd) annotation (points=[9,-72;16,-72;16,-2], style(color=0,rgbcolor={0,0,127}));
          end TaskC;

          end NonLinearCrane;

          package NonLinearCraneSLibControl

          model TaskB
            Real Carpos = NonLinearCrane.posCar;
            Real Loadpos = NonLinearCrane.loadPos;

            annotation (Diagram,      experiment(StopTime=60),      experimentSetupOutput);
            Modelica.Blocks.Sources.Trapezoid trapezoid(
              amplitude=-200,
              width=1,
              period=1000,
              nperiod=1,
              startTime=42) annotation (extent=[-44,-46;-24,-26]);
            Modelica.Blocks.Sources.TimeTable desiredPosition(table=[0,3; 16,3; 16,
                  -0.5; 36,-0.5; 36,3.8; 60,3.8])
              annotation (extent=[-74,48;-44,72]);
              CraneSystem.NonLinearCrane NonLinearCrane annotation (extent=[-2,0;18,20]);
            SIMANLibController.SLibController sLibController
              annotation (extent=[-40,-2;-20,18]);
          equation
              connect(trapezoid.y, NonLinearCrane.fd)
          annotation (points=[-23,-36;6,-36;6,-2], style(color=0,rgbcolor={0,0,127}));
            connect(sLibController.fcdesired, NonLinearCrane.fcdesired)
          annotation (points=[-19.4,9;-12.7,9;-12.7,10;-4,10], style(color=0,rgbcolor={0,0,127}));
            connect(sLibController.brake, NonLinearCrane.brake)
          annotation (points=[-19.4,4;-12,4;-12,2;-4,2], style(color=0,rgbcolor={255,0,255}));
            connect(NonLinearCrane.SwPosCarMax, sLibController.swPosCarMax)
          annotation (points=[19,6;32,6;32,-10;-52,-10;-52,7;-40,7], style(color=0,rgbcolor={255,0,255}));
            connect(NonLinearCrane.SwPosCarMin, sLibController.swPosCarMin)
          annotation (points=[19,2;26,2;26,-6;-46,-6;-46,3.8;-40,3.8], style(color=0,rgbcolor={255,0,255}));
            connect(desiredPosition.y, sLibController.desiredPosition)
          annotation (points=[-42.5,60; -30.6,60; -30.6,17.1],
                                                             style(color=0,rgbcolor={0,0,127}));
            connect(NonLinearCrane.posCar, sLibController.posCar)
          annotation (points=[19,18;24,18;24,24;-46,24;-46,13;-40,13], style(color=0,rgbcolor={0,0,127}));
          end TaskB;

          model TaskB2
            Real Carpos = NonLinearCrane.posCar;
            Real Loadpos = NonLinearCrane.loadPos;

            annotation (Diagram,      experiment(StopTime=60),      experimentSetupOutput);
            Modelica.Blocks.Sources.Trapezoid trapezoid(
              amplitude=-200,
              width=1,
              period=1000,
              nperiod=1,
              startTime=42) annotation (extent=[-44,-46;-24,-26]);
            Modelica.Blocks.Sources.TimeTable desiredPosition(table=[0,3; 16,3; 16,
                  -0.5; 36,-0.5; 36,3.8; 60,3.8])
              annotation (extent=[-74,48;-44,72]);
              CraneSystem.NonLinearCrane NonLinearCrane annotation (extent=[-2,0;18,20]);
            SIMANLibController.SLibController2 sLibController
              annotation (extent=[-40,-2;-20,18]);
          equation
              connect(trapezoid.y, NonLinearCrane.fd) annotation (points=[-23,-36;6,-36;6,-2], style(color=0, rgbcolor={0,0,127}));
            connect(sLibController.fcdesired, NonLinearCrane.fcdesired)
          annotation (points=[-19.4,9;-12.7,9;-12.7,10;-4,10], style(color=0, rgbcolor={0,0,127}));
            connect(sLibController.brake, NonLinearCrane.brake) annotation (points=[-19.4,4;-12,4;-12,2;-4,2], style(color=0, rgbcolor={255,0,255}));
            connect(NonLinearCrane.SwPosCarMax, sLibController.swPosCarMax)
          annotation (points=[19,6;32,6;32,-10;-52,-10;-52,7;-40,7], style(color=0, rgbcolor={255,0,255}));
            connect(NonLinearCrane.SwPosCarMin, sLibController.swPosCarMin)
          annotation (points=[19,2;26,2;26,-6;-46,-6;-46,3.8;-40,3.8], style(color=0, rgbcolor={255,0,255}));
            connect(desiredPosition.y, sLibController.desiredPosition) annotation (points=[-42.5,60;-30.6,60;-30.6,17.2], style(color=0, rgbcolor={0,0,127}));
            connect(NonLinearCrane.posCar, sLibController.posCar)
          annotation (points=[19,18;24,18;24,24;-46,24;-46,13;-40,13], style(color=0, rgbcolor={0,0,127}));
          end TaskB2;

          model TaskC
            Real Carpos = NonLinearCrane.posCar;
            Real Loadpos = NonLinearCrane.loadPos;

            annotation (Diagram,      experiment(StopTime=60),      experimentSetupOutput);
            Modelica.Blocks.Sources.Trapezoid Disturbance(
                width=1,
                period=1000,
                nperiod=1,
                startTime=42,
                amplitude=200)
                            annotation (extent=[-12,-46;8,-26]);
            Modelica.Blocks.Sources.TimeTable desiredPosition1(table=[0,3; 16,3;
                    16,-0.5; 36,-0.5; 36,3.8; 60,3.8])
              annotation (extent=[-50,40;-30,60]);
              CraneSystem.NonLinearCrane NonLinearCrane annotation (extent=[8,0;28,20]);
            Modelica.Blocks.Sources.TimeTable DisturbingForce(table=[0,0; 42,0; 42,-200;
                  43,-200; 43,0; 44,0; 46,0; 46,200; 47,200; 47,0; 48,0])
                                               annotation (extent=[-12,-82;8,-62]);
            DESLib.DEVSLib.Examples.CraneCrab2006.SIMANLibController.SLibController
                sLibController
              annotation (extent=[-30,-2;-10,20]);
          equation
              connect(DisturbingForce.y, NonLinearCrane.fd) annotation (points=[9,-72;16,-72;16,-2], style(color=0,rgbcolor={0,0,127}));
            connect(sLibController.fcdesired, NonLinearCrane.fcdesired)
          annotation (points=[-9.4,10.1; -1.7,10.1; -1.7,10; 6,10],
                                                                 style(color=0,rgbcolor={0,0,127}));
            connect(sLibController.brake, NonLinearCrane.brake) annotation (points=[-9.4,4.6;
                    -1.7,4.6; -1.7,2; 6,2],                                                                        style(color=0,rgbcolor={255,0,255}));
            connect(NonLinearCrane.SwPosCarMin, sLibController.swPosCarMin)
          annotation (points=[29,2;34,2;34,-6;-38,-6;-38,4.38;-30,4.38], style(color=0,rgbcolor={255,0,255}));
            connect(NonLinearCrane.SwPosCarMax, sLibController.swPosCarMax)
          annotation (points=[29,6;42,6;42,-12;-46,-12;-46,7.9;-30,7.9], style(color=0,rgbcolor={255,0,255}));
            connect(NonLinearCrane.posCar, sLibController.posCar)
          annotation (points=[29,18;36,18;36,28;-40,28;-40,14.5;-30,14.5], style(color=0,rgbcolor={0,0,127}));
            connect(desiredPosition1.y, sLibController.desiredPosition) annotation (points=[-29,50;
                    -20.6,50; -20.6,19.01],                                                                               style(color=0,rgbcolor={0,0,127}));
          end TaskC;

          model TaskCcomparison
            Real CarposDEVS = NonLinearCrane1.posCar;
            Real LoadposDEVS = NonLinearCrane1.loadPos;
            Real CarposSIMAN = NonLinearCrane.posCar;
            Real LoadposSIMAN = NonLinearCrane.loadPos;
            Real errorC;
            Real errorL;

            annotation (Diagram,      experiment(StopTime=60),      experimentSetupOutput);
            Modelica.Blocks.Sources.Trapezoid Disturbance(
                width=1,
                period=1000,
                nperiod=1,
                startTime=42,
                amplitude=200)
                            annotation (extent=[-62,-44;-42,-24]);
            Modelica.Blocks.Sources.TimeTable desiredPosition1(table=[0,3; 16,3;
                    16,-0.5; 36,-0.5; 36,3.8; 60,3.8])
              annotation (extent=[-100,42;-80,62]);
              CraneSystem.NonLinearCrane NonLinearCrane annotation (extent=[-42,2;-22,22]);
            Modelica.Blocks.Sources.TimeTable DisturbingForce(table=[0,0; 42,0; 42,-200;
                  43,-200; 43,0; 44,0; 46,0; 46,200; 47,200; 47,0; 48,0])
                                               annotation (extent=[-62,-80;-42,-60]);
            DESLib.DEVSLib.Examples.CraneCrab2006.SIMANLibController.SLibController
                sLibController
              annotation (extent=[-82,2;-54,22]);
            Modelica.Blocks.Sources.Trapezoid Disturbance1(
                width=1,
                period=1000,
                nperiod=1,
                startTime=42,
                amplitude=200)
                            annotation (extent=[36,-44;56,-24]);
            Modelica.Blocks.Sources.TimeTable desiredPosition2(table=[0,3; 16,3;
                    16,-0.5; 36,-0.5; 36,3.8; 60,3.8])
              annotation (extent=[-2,42;18,62]);
              Controller.DEVSController dEVSController
                annotation (extent=[16,-4;48,28]);
              CraneSystem.NonLinearCrane NonLinearCrane1
                                                        annotation (extent=[56,2;76,22]);
            Modelica.Blocks.Sources.TimeTable DisturbingForce1(
                                                              table=[0,0; 42,0; 42,-200;
                  43,-200; 43,0; 44,0; 46,0; 46,200; 47,200; 47,0; 48,0])
                                               annotation (extent=[36,-80;56,-60]);
          equation
            errorC = (CarposDEVS-CarposSIMAN);
            errorL = (LoadposDEVS-LoadposSIMAN);

              connect(DisturbingForce.y, NonLinearCrane.fd) annotation (points=[-41,-70;-34,-70;-34,0], style(color=0,rgbcolor={0,0,127}));
            connect(sLibController.fcdesired, NonLinearCrane.fcdesired)
          annotation (points=[-53.16,13;-51.7,13;-51.7,12;-44,12], style(color=0,rgbcolor={0,0,127}));
            connect(sLibController.brake, NonLinearCrane.brake) annotation (points=[-53.16,8;-49.7,8;-49.7,4;-44,4], style(color=0,rgbcolor={255,0,255}));
            connect(NonLinearCrane.SwPosCarMin, sLibController.swPosCarMin)
          annotation (points=[-21,4;-16,4;-16,-4;-88,-4;-88,7.8;-82,7.8], style(color=0,rgbcolor={255,0,255}));
            connect(NonLinearCrane.SwPosCarMax, sLibController.swPosCarMax)
          annotation (points=[-21,8;-8,8;-8,-10;-96,-10;-96,11;-82,11], style(color=0,rgbcolor={255,0,255}));
            connect(NonLinearCrane.posCar, sLibController.posCar)
          annotation (points=[-21,20;-14,20;-14,30;-90,30;-90,17;-82,17], style(color=0,rgbcolor={0,0,127}));
            connect(desiredPosition1.y, sLibController.desiredPosition)
          annotation (points=[-79,52; -68.84,52; -68.84,21.1],
                                                             style(color=0,rgbcolor={0,0,127}));
              connect(desiredPosition2.y,dEVSController. desiredPosition)
          annotation (points=[19,52; 29.44,52; 29.44,25.12],
                                                           style(color=0,rgbcolor={0,0,127}));
            connect(NonLinearCrane1.posCar, dEVSController.posCar)
          annotation (points=[77,20; 82,20; 82,26; 8,26; 8,18.4; 18.24,18.4],
                                                                        style(color=0,rgbcolor={0,0,127}));
            connect(NonLinearCrane1.SwPosCarMin, dEVSController.swPosCarMin)
          annotation (points=[77,4; 84,4; 84,-4; 8,-4; 8,3.68; 18.24,3.68],
                                                                      style(color=0,rgbcolor={255,0,255}));
            connect(NonLinearCrane1.SwPosCarMax, dEVSController.swPosCarMax)
           annotation (points=[77,8; 90,8; 90,-8; 4,-8; 4,8.8; 18.24,8.8],
                                                                     style(color=0,rgbcolor={255,0,255}));
            connect(DisturbingForce1.y, NonLinearCrane1.fd)
          annotation (points=[57,-70;64,-70;64,0], style(color=0,rgbcolor={0,0,127}));
              connect(dEVSController.fcdesired, NonLinearCrane1.fcdesired)
                annotation (points=[47.36,12;54,12], style(color=0,rgbcolor={0,0,127}));
              connect(dEVSController.brake, NonLinearCrane1.brake)
          annotation (points=[47.36,4;54,4], style(color=0,rgbcolor={255,0,255}));
          end TaskCcomparison;
          end NonLinearCraneSLibControl;
        end Tasks;
      end CraneCrab2006;

      package Supermarket

        package DEVSControl
          model DEVSAirController
            parameter Real Tairmax = 1
              "Max temperature threshold for the controller";
            parameter Real Tairmin = 1
              "Min temperature threshold for the controller";
            Modelica.Blocks.Interfaces.RealInput Tair
              annotation (extent=[-80,-10;-60,10]);
            AuxModels.CrossUP crossUP(Value=Tairmax, EType=1)
              annotation (extent=[-40,26;-20,46]);
            annotation (Diagram, Icon(
                    Rectangle(
                    extent=[-60,40;60,-40], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)), Text(
                    extent=[-58,20;58,-20], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1),
                    string="AirTemp Control")));
            AuxModels.CrossDOWN crossDOWN(Value=Tairmin, EType=0)
              annotation (extent=[-40,-46;-20,-26]);
            AuxModels.DiBO diBO(name="diboair")
                                annotation (extent=[20,-10;40,10]);
            Modelica.Blocks.Interfaces.BooleanOutput ValveOpen
              annotation (extent=[60,-10;80,10]);
            SimpleModels.SetValue.setValue setValue
              annotation (extent=[-14,26;6,46]);
            SimpleModels.SetValue.setValue setValue1(Value=0)
              annotation (extent=[-14,-46;6,-26]);
            SRC.Hybrid.CondGen condGen(Condition=(Tair > Tairmax) and (time > 0)
                   and (time < 0.00001))
                                        annotation (extent=[-40,0;-20,20]);
            SRC.Hybrid.CondGen condGen1(
              Condition=(Tair < Tairmin) and (time > 0) and (time < 0.00001),
              Type=0,
              Value=0)                  annotation (extent=[-40,-20;-20,0]);
          equation
            connect(Tair, crossUP.u) annotation (points=[-70,0;-60,0;-60,36;-41,36], style(color=0, rgbcolor={0,0,127}));
            connect(Tair, crossDOWN.u) annotation (points=[-70,0;-60,0;-60,-36;-41,-36], style(color=0, rgbcolor={0,0,127}));
            connect(crossDOWN.outport, setValue1.inPort1) annotation (points=[-19.4,-36;-13,-36], style(color=0, rgbcolor={0,0,0}));
            connect(setValue1.outPort1, diBO.inport) annotation (points=[5,-36;12,-36;12,0;19.4,0], style(color=0, rgbcolor={0,0,0}));
            connect(crossUP.outport, setValue.inPort1) annotation (points=[-19.4,36;-13,36], style(color=0, rgbcolor={0,0,0}));
            connect(setValue.outPort1, diBO.inport) annotation (points=[5,36;12,36;12,0;19.4,0], style(color=0, rgbcolor={0,0,0}));
            connect(condGen.outport, diBO.inport) annotation (points=[-19.4,10;0,10;0,0;19.4,0], style(color=0, rgbcolor={0,0,0}));
            connect(diBO.change, ValveOpen) annotation (points=[41,0;70,0], style(color=0, rgbcolor={255,0,255}));
            connect(condGen1.outport, diBO.inport) annotation (points=[-19.4,-10;0,-10;0,0;19.4,0], style(color=0, rgbcolor={0,0,0}));
          end DEVSAirController;

          model DEVSControlledDisplayCase

            parameter Real UAgoodsair = 1
              "Heat transfer coeficient between goods and air";
            parameter Real Mgoods = 1 "Mass of the goods";
            parameter Real CPgoods = 1 "Heat capacity of goods";
            parameter Real UAairwall = 1
              "Heat transfer coeficient between air and wall";
            parameter Real Mwall = 1 "Mass of the wall";
            parameter Real CPwall = 1 "Heat capacity of wall";
            parameter Real UAwallrefmax = 1
              "Maximum heat transfer coeficient between wall and refrigerant";
            parameter Real Mrefmax = 1 "Maximum mass of refrigerant";
            parameter Real Mair = 1 "Mass of the air";
            parameter Real CPair = 1 "Heat capacity of air";
            parameter Real Taufill = 1 "Refrigerator filling time";
            parameter Real Tairmax = 1
              "Max temperature threshold for the controller";
            parameter Real Tairmin = 1
              "Min temperature threshold for the controller";
            parameter Real Twallstart = 1;
            parameter Real Tairstart = 1;
            parameter Real Tgoodsstart = 1;
            parameter Real Mrefstart = 1;
            Continuous.DisplayCase displayCase(
              UAgoodsair=UAgoodsair,
              Mgoods=Mgoods,
              CPgoods=CPgoods,
              UAairwall=UAairwall,
              Mwall=Mwall,
              CPwall=CPwall,
              UAwallrefmax=UAwallrefmax,
              Mrefmax=Mrefmax,
              Mair=Mair,
              CPair=CPair,
              Taufill=Taufill,
              Tgoods(start=Tgoodsstart),
              Twall(start=Twallstart),
              Tair(start=Tairstart),
              Mref(start=Mrefstart),
              Tgoods_initial=Tgoodsstart,
              Twall_initial=Twallstart,
              Tair_initial=Tairstart,
              Mref_initial=Mrefstart)
                               annotation (extent=[-74,-78;66,102]);
            DEVSAirController airController(
                Tairmax=Tairmax, Tairmin=Tairmin)
              annotation (extent=[-82,-100;-36,-62]);
            annotation (Diagram, Icon(Bitmap(
                      extent=[-100,100;80,-100], name="Figs/displaycase.png")));
            Modelica.Blocks.Interfaces.RealInput Qairload
              annotation (extent=[80,30;100,50], rotation=180);
            Modelica.Blocks.Interfaces.RealInput Psuc
              annotation (extent=[80,-68;100,-48], rotation=180);
            flow Modelica.Blocks.Interfaces.RealOutput m
              annotation (extent=[80,-90;100,-70]);
            Modelica.Blocks.Interfaces.RealOutput Tair
              annotation (extent=[-120,30;-100,50], rotation=180);
            Modelica.Blocks.Interfaces.RealOutput valveOpen
              annotation (extent=[-10,-120;10,-100], rotation=270);
          equation
           valveOpen = if airController.ValveOpen then 1 else 0;
            connect(displayCase.AirTemp, airController.Tair) annotation (points=[-67,12;-84,12;-84,-81;-75.1,-81], style(rgbcolor={0,0,127}));
            connect(Qairload, displayCase.Qairload) annotation (points=[90,40;73.1,40;73.1,42.6;59,42.6], style(rgbcolor={0,0,127}));
            connect(displayCase.Psuc, Psuc) annotation (points=[59,-33;79.5,-33;79.5,-58;90,-58], style(rgbcolor={0,0,127}));
            connect(displayCase.m, m) annotation (points=[59,-51;72.5,-51;72.5,-80;90,-80], style(rgbcolor={0,0,127}));
            connect(airController.ValveOpen, displayCase.valveOpen) annotation (points=[-42.9,-81;-30.45,-81;-30.45,-69;-30.6,-69], style(rgbcolor={255,0,255}));
            connect(displayCase.AirTemp, Tair) annotation (points=[-67,12;-84,12;-84,40;-110,40], style(rgbcolor={0,0,127}));
          end DEVSControlledDisplayCase;

          model CompressorRack2C
            parameter Real DB = 1 "Controller Dead-Band";
            parameter Real comp = 1 "Compressor capacity";
            parameter Real Nuvol = 1 "Compressor volumetric efficiency";
            parameter Real Vsl = 1 "Compressor total displacement volume";
            parameter Real Kp = 1 "PI gain";
            parameter Real Taoi = 1 "PI time constant";
            Continuous.Compressor compressor(
              comp=comp,
              Nuvol=Nuvol,
              Vsl=Vsl)            annotation (extent=[-20,40;0,60]);
            Continuous.Compressor compressor1(
              comp=comp,
              Nuvol=Nuvol,
              Vsl=Vsl)             annotation (extent=[74,40;94,60]);
            Modelica.Blocks.Interfaces.RealOutput Vcomp
              annotation (extent=[-10,80;10,100], rotation=90);
            annotation (Diagram,Icon(Ellipse(
                    extent=[-72,80;88,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-52,52;-12,-76], style( rgbcolor={0,0,0})),
                  Line(points=[68,52;28,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-52,60;68,-60], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1),
                    string="T"),
                  Ellipse(
                    extent=[-84,80;76,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-64,52;-24,-76], style(rgbcolor={0,0,0})),
                  Line(points=[56,52;16,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-64,60;56,-60], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1),
                    string="T")));
            Modelica.Blocks.Continuous.PI PI(k=Kp, T=Taoi)
              annotation (extent=[38,-18;58,2]);
            Modelica.Blocks.Sources.Constant const(k=0)
              annotation (extent=[-10,-34;-2,-28]);
            Modelica.Blocks.Sources.Constant const1(k=DB)
              annotation (extent=[-42,-32;-32,-24]);
            Modelica.Blocks.Logical.Switch switch1
              annotation (extent=[8,-18;28,2]);
            Modelica.Blocks.Logical.Greater greater
              annotation (extent=[-22,-18;-2,2]);
            Modelica.Blocks.Interfaces.RealInput Psuc
              annotation (extent=[-10,-100;10,-80], rotation=90);
            Modelica.Blocks.Math.Abs abs1 annotation (extent=[-52,-10;-32,10]);
            Modelica.Blocks.Math.Feedback feedback
              annotation (extent=[-78,-10;-58,10]);
            AuxModels.DiBO diBO annotation (extent=[-38,40;-24,60]);
            AuxModels.CrossDOWN crossDOWNC1(EType=0, Value=comp/2)
              annotation (extent=[-64,42;-56,50]);
            AuxModels.CrossUP crossUPC1(EType=1, Value=comp/2)
              annotation (extent=[-64,50;-56,58]);
            AuxModels.DiBO diBO1 annotation (extent=[54,40;68,60]);
            AuxModels.CrossDOWN crossDOWNC2(EType=0, Value=comp + (comp/2))
              annotation (extent=[24,42;32,50]);
            AuxModels.CrossUP crossUPC2(EType=1, Value=comp + (comp/2))
              annotation (extent=[24,50;32,58]);
            Modelica.Blocks.Interfaces.RealInput SetPoint
              annotation (extent=[-100,-10;-80,10]);
            SimpleModels.SetValue.setValue setValue
              annotation (extent=[-52,50;-44,58]);
            SimpleModels.SetValue.setValue setValue1(Value=0)
              annotation (extent=[-52,42;-44,50]);
            SimpleModels.SetValue.setValue setValue2
              annotation (extent=[36,50;44,58]);
            SimpleModels.SetValue.setValue setValue3(Value=0)
              annotation (extent=[36,42;44,50]);
          protected
            Real comp1ON = if diBO.change then 1 else 0;
            Real comp2ON = if diBO1.change then 1 else 0;
          public
            Modelica.Blocks.Math.Sum sum1(nin=2)
              annotation (extent=[-4,64;4,72], rotation=90);
            Modelica.Blocks.Discrete.Sampler sampler(samplePeriod=60)
              annotation (extent=[66,-12;74,-4]);
          public
            Modelica.Blocks.Interfaces.RealOutput numCompON
                                                    annotation (extent=[100,-10;120,10]);
          equation
            numCompON = comp1ON + comp2ON;
            connect(const.y, switch1.u3) annotation (points=[-1.6,-31; 2,-31; 2,
                  -16; 6,-16],                                                            style(rgbcolor={0,0,127}));
            connect(const1.y, greater.u2) annotation (points=[-31.5,-28;-28,-28;-28,-16;-24,-16], style(rgbcolor={0,0,127}));
            connect(greater.y, switch1.u2) annotation (points=[-1,-8;6,-8], style(rgbcolor={255,0,255}));
            connect(abs1.y, greater.u1) annotation (points=[-31,0;-28,0;-28,-8;-24,-8], style(rgbcolor={0,0,127}));
            connect(feedback.y, abs1.u) annotation (points=[-59,0;-54,0], style(rgbcolor={0,0,127}));
            connect(diBO1.change, compressor1.ON) annotation (points=[68.7,50;75,50], style(rgbcolor={255,0,255}));
            connect(diBO.change, compressor.ON) annotation (points=[-23.3,50;-19,50], style(rgbcolor={255,0,255}));
            connect(crossUPC1.outport, setValue.inPort1) annotation (points=[-55.76,54;-51.6,54], style(rgbcolor={0,0,0}));
            connect(crossDOWNC1.outport, setValue1.inPort1) annotation (points=[-55.76,46;-51.6,46], style(rgbcolor={0,0,0}));
            connect(setValue.outPort1, diBO.inport) annotation (points=[-44.4,54;-42,54;-42,50;-38.42,50], style(rgbcolor={0,0,0}));
            connect(setValue1.outPort1, diBO.inport) annotation (points=[-44.4,46;-42,46;-42,50;-38.42,50], style(rgbcolor={0,0,0}));
            connect(crossUPC2.outport, setValue2.inPort1) annotation (points=[32.24,54;36.4,54], style(rgbcolor={0,0,0}));
            connect(crossDOWNC2.outport, setValue3.inPort1) annotation (points=[32.24,46;36.4,46], style(rgbcolor={0,0,0}));
            connect(setValue2.outPort1, diBO1.inport) annotation (points=[43.6,54;48,54;48,50;53.58,50], style(rgbcolor={0,0,0}));
            connect(setValue3.outPort1, diBO1.inport) annotation (points=[43.6,46;48,46;48,50;53.58,50], style(rgbcolor={0,0,0}));
            connect(sum1.y, Vcomp) annotation (points=[2.69413e-016,72.4;
                  2.69413e-016,81.2; 0,81.2; 0,90],                                                      style(rgbcolor={0,0,127}));
            connect(compressor.Vcomp, sum1.u[1]) annotation (points=[-10,59;0,59;0,63.2;0.4,63.2], style(rgbcolor={0,0,127}));
            connect(compressor1.Vcomp, sum1.u[2]) annotation (points=[84,59;0,59;0,63.2;-0.4,63.2], style(rgbcolor={0,0,127}));
            connect(feedback.y, switch1.u1) annotation (points=[-59,0;-59,16;2,16;2,0;6,0], style(rgbcolor={0,0,127}));
            connect(SetPoint, feedback.u1) annotation (points=[-90,0;-76,0], style(rgbcolor={0,0,127}));
            connect(switch1.y, PI.u) annotation (points=[29,-8;36,-8], style(rgbcolor={0,0,127}));
            connect(PI.y, sampler.u) annotation (points=[59,-8;65.2,-8], style(rgbcolor={0,0,127}));
            connect(sampler.y, crossDOWNC2.u) annotation (points=[74.4,-8;80,-8;80,32;4,32;4,46;23.6,46], style(rgbcolor={0,0,127}));
            connect(sampler.y, crossUPC2.u) annotation (points=[74.4,-8;80,-8;80,32;4,32;4,54;23.6,54], style(rgbcolor={0,0,127}));
            connect(sampler.y, crossDOWNC1.u) annotation (points=[74.4,-8;80,-8;80,32;-84,32;-84,46;-64.4,46], style(rgbcolor={0,0,127}));
            connect(sampler.y, crossUPC1.u) annotation (points=[74.4,-8;80,-8;80,32;-84,32;-84,54;-64.4,54], style(rgbcolor={0,0,127}));
            connect(Psuc, feedback.u2) annotation (points=[0,-90;0,-60;-68,-60;-68,-8], style(rgbcolor={0,0,127}));
          end CompressorRack2C;

          model CompressorRack3C
            parameter Real DB = 1 "Controller Dead-Band";
            parameter Real comp = 1 "Compressor capacity";
            parameter Real Kp = 1 "PI gain";
            parameter Real Taoi = 1 "PI time constant";
            parameter Real Nuvol = 1 "Compressor volumetric efficiency";
            parameter Real Vsl = 1 "Compressor total displacement volume";
            Continuous.Compressor compressor(
              comp=comp,
              Nuvol=Nuvol,
              Vsl=Vsl) annotation (extent=[-20,40;0,60]);
            Continuous.Compressor compressor1(
              comp=comp,
              Nuvol=Nuvol,
              Vsl=Vsl) annotation (extent=[18,22;38,42]);
            Modelica.Blocks.Interfaces.RealOutput Vcomp
              annotation (extent=[-10,80;10,100], rotation=90);
            annotation (Diagram, Icon(
                  Ellipse(
                    extent=[-60,80;100,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-40,52;0,-76], style(rgbcolor={0,0,0})),
                  Line(points=[80,52;40,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-40,60;80,-60], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1),
                    string="T"),
                  Ellipse(
                    extent=[-72,80;88,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1),
                  Line(points=[-52,52;-12,-76], style(rgbcolor={0,0,0})),
                  Line(points=[68,52;28,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-52,60;68,-60], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1),
                    string="T"),
                  Ellipse(
                    extent=[-84,80;76,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-64,52;-24,-76], style(rgbcolor={0,0,0})),
                  Line(points=[56,52;16,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-64,60;56,-60], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1),
                    string="T"))));
            Modelica.Blocks.Continuous.PI PI(k=Kp, T=Taoi)
              annotation (extent=[36,-46;56,-26]);
            Modelica.Blocks.Sources.Constant const(k=0)
              annotation (extent=[-14,-56;-6,-50]);
            Modelica.Blocks.Sources.Constant const1(k=DB)
              annotation (extent=[-42,-56;-32,-48]);
            Modelica.Blocks.Logical.Switch switch1
              annotation (extent=[8,-46;28,-26]);
            Modelica.Blocks.Logical.Greater greater
              annotation (extent=[-24,-46;-4,-26]);
            Modelica.Blocks.Interfaces.RealInput Psuc
              annotation (extent=[-10,-100;10,-80], rotation=90);
            Modelica.Blocks.Math.Abs abs1 annotation (extent=[-52,-38;-32,-18]);
            Modelica.Blocks.Math.Feedback feedback
              annotation (extent=[-84,-38;-64,-18]);
            AuxModels.DiBO diBO annotation (extent=[-42,40;-28,60]);
            AuxModels.CrossDOWN crossDOWN(Value=comp/2, EType=0)
              annotation (extent=[-64,42;-56,50]);
            AuxModels.CrossUP crossUP(Value=comp/2, EType=1)
              annotation (extent=[-64,50;-56,58]);
            AuxModels.DiBO diBO1 annotation (extent=[2,22;16,42]);
            AuxModels.CrossDOWN crossDOWN1(Value=comp + comp/2, EType=0)
              annotation (extent=[-20,24; -12,32]);
            AuxModels.CrossUP crossUP1(Value=comp + comp/2, EType=1)
              annotation (extent=[-20,32;-12,40]);
            Continuous.Compressor compressor2(
              comp=comp,
              Nuvol=Nuvol,
              Vsl=Vsl) annotation (extent=[66,2;86,22]);
            AuxModels.DiBO diBO2 annotation (extent=[50,2;64,22]);
            AuxModels.CrossDOWN crossDOWN2(  EType=0, Value=comp + comp + comp/2)
              annotation (extent=[28,4;36,12]);
            AuxModels.CrossUP crossUP2(                     EType=1, Value=comp
                   + comp + comp/2)
              annotation (extent=[28,12;36,20]);
          protected
            Real comp1ON = if diBO.change then 1 else 0;
            Real comp2ON = if diBO1.change then 1 else 0;
            Real comp3ON = if diBO2.change then 1 else 0;
          public
            Modelica.Blocks.Math.Sum sum1(nin=3)
              annotation (extent=[-4,70;4,78], rotation=90);
            SimpleModels.SetValue.setValue setValue
              annotation (extent=[-54,50;-46,58]);
            SimpleModels.SetValue.setValue setValue1(Value=0)
              annotation (extent=[-54,42;-46,50]);
            SimpleModels.SetValue.setValue setValue2
              annotation (extent=[-10,32;-2,40]);
            SimpleModels.SetValue.setValue setValue3(Value=0)
              annotation (extent=[-10,24;-2,32]);
            SimpleModels.SetValue.setValue setValue4
              annotation (extent=[38,12;46,20]);
            SimpleModels.SetValue.setValue setValue5(Value=0)
              annotation (extent=[38,4;46,12]);
            Modelica.Blocks.Interfaces.RealInput SetPoint
              annotation (extent=[-104,-10;-84,10]);
            Modelica.Blocks.Discrete.Sampler sampler(samplePeriod=60)
              annotation (extent=[62,-40;70,-32]);
          public
            Modelica.Blocks.Interfaces.RealOutput numCompON
                                                    annotation (extent=[100,-10;120,10]);
            Modelica.Blocks.Discrete.Sampler sampler1(samplePeriod=60)
              annotation (extent=[-26,-82;-18,-74], rotation=180);
          equation
           numCompON = comp1ON + comp2ON + comp3ON;
            connect(const.y, switch1.u3) annotation (points=[-5.6,-53;0,-53;0,-44;6,-44], style(rgbcolor={0,0,127}));
            connect(greater.y, switch1.u2) annotation (points=[-3,-36;6,-36], style(rgbcolor={255,0,255}));
            connect(switch1.y, PI.u) annotation (points=[29,-36;34,-36], style(rgbcolor={0,0,127}));
            connect(abs1.y, greater.u1) annotation (points=[-31,-28;-28,-28;-28,-36;-26,-36], style(rgbcolor={0,0,127}));
            connect(feedback.y, abs1.u) annotation (points=[-65,-28;-54,-28], style(rgbcolor={0,0,127}));
            connect(diBO1.change, compressor1.ON) annotation (points=[16.7,32;19,32], style(rgbcolor={255,0,255}));
            connect(diBO.change, compressor.ON) annotation (points=[-27.3,50;-19,50], style(rgbcolor={255,0,255}));
            connect(diBO2.change,compressor2. ON) annotation (points=[64.7,12;67,12], style(rgbcolor={255,0,255}));
            connect(sum1.y, Vcomp) annotation (points=[2.69413e-016,78.4; 0,
                  78.4; 0,90],                                                         style(rgbcolor={0,0,127}));
            connect(compressor.Vcomp, sum1.u[1]) annotation (points=[-10,59;-10,64;0.533333,64;0.533333,69.2], style(rgbcolor={0,0,127}));
            connect(compressor1.Vcomp, sum1.u[2]) annotation (points=[28,41;28,64;-3.05311e-016,64;-3.05311e-016,69.2], style(rgbcolor={0,0,127}));
            connect(compressor2.Vcomp, sum1.u[3]) annotation (points=[76,21;76,64;-0.533333,64;-0.533333,69.2], style(rgbcolor={0,0,127}));
            connect(crossUP.outport,setValue. inPort1) annotation (points=[-55.76,54;-53.6,54], style(rgbcolor={0,0,0}));
            connect(crossDOWN.outport,setValue1. inPort1) annotation (points=[-55.76,46;-53.6,46], style(rgbcolor={0,0,0}));
            connect(setValue.outPort1, diBO.inport) annotation (points=[-46.4,54;-46,54;-46,50;-42.42,50], style(rgbcolor={0,0,0}));
            connect(setValue1.outPort1, diBO.inport) annotation (points=[-46.4,46;-46,46;-46,50;-42.42,50], style(rgbcolor={0,0,0}));
            connect(crossUP1.outport, setValue2.inPort1) annotation (points=[-11.76,36;-9.6,36], style(rgbcolor={0,0,0}));
            connect(crossDOWN1.outport, setValue3.inPort1) annotation (points=[-11.76,
                  28; -9.6,28],                                                                    style(rgbcolor={0,0,0}));
            connect(setValue2.outPort1, diBO1.inport) annotation (points=[-2.4,36;
                  0,36; 0,32; 1.58,32],                                                               style(rgbcolor={0,0,0}));
            connect(setValue3.outPort1, diBO1.inport) annotation (points=[-2.4,28;
                  0,28; 0,32; 1.58,32],                                                               style(rgbcolor={0,0,0}));
            connect(crossUP2.outport, setValue4.inPort1) annotation (points=[36.24,16;38.4,16], style(rgbcolor={0,0,0}));
            connect(setValue4.outPort1, diBO2.inport) annotation (points=[45.6,16;48,16;48,12;49.58,12], style(rgbcolor={0,0,0}));
            connect(setValue5.outPort1, diBO2.inport) annotation (points=[45.6,8;48,8;48,12;49.58,12], style(rgbcolor={0,0,0}));
            connect(SetPoint, feedback.u1)    annotation (points=[-94,0;-88,0;-88,-28;-82,-28], style(rgbcolor={0,0,127}));
            connect(PI.y, sampler.u) annotation (points=[57,-36;61.2,-36], style(rgbcolor={0,0,127}));
            connect(greater.u2, const1.y) annotation (points=[-26,-44;-28,-44;-28,-52;-31.5,-52], style(rgbcolor={0,0,127}));
            connect(feedback.y, switch1.u1) annotation (points=[-65,-28;-58,-28;-58,-12;0,-12;0,-28;6,-28], style(rgbcolor={0,0,127}));
            connect(crossDOWN2.outport, setValue5.inPort1)
              annotation (points=[36.24,8;38.4,8], style(rgbcolor={0,0,0}));
            connect(sampler.y, crossDOWN2.u) annotation (points=[70.4,-36;80,-36;80,-2;24,-2;24,8;27.6,8], style(rgbcolor={0,0,127}));
            connect(sampler.y, crossUP2.u) annotation (points=[70.4,-36;80,-36;80,-2;24,-2;24,16;27.6,16], style(rgbcolor={0,0,127}));
            connect(sampler.y, crossDOWN1.u) annotation (points=[70.4,-36; 80,
                  -36; 80,-2; -26,-2; -26,28; -20.4,28],                                                        style(rgbcolor={0,0,127}));
            connect(sampler.y, crossUP1.u) annotation (points=[70.4,-36;80,-36;80,-2;-26,-2;-26,36;-20.4,36], style(rgbcolor={0,0,127}));
            connect(sampler.y, crossDOWN.u) annotation (points=[70.4,-36;80,-36;80,-2;-72,-2;-72,46;-64.4,46], style(rgbcolor={0,0,127}));
            connect(sampler.y, crossUP.u) annotation (points=[70.4,-36;80,-36;80,-2;-72,-2;-72,54;-64.4,54], style(rgbcolor={0,0,127}));
            connect(Psuc, sampler1.u) annotation (points=[0,-90;0,-78;-17.2,-78], style(rgbcolor={0,0,127}));
            connect(sampler1.y, feedback.u2) annotation (points=[-26.4,-78;-74,-78;-74,-36], style(rgbcolor={0,0,127}));
          end CompressorRack3C;

          package DEVSPresureControl2
            "Draft package to construct new atomic DEVS models"

          annotation(preferedView="info",
            Documentation(info="<HTML>
<p>
This package contains the basic structures needed in atomic models.
It can be duplicated, renamed and modified to construct new atomic DEVS models.
</p>
<p>
The user can implement the behavior of a new DEVS model modifying the con, int, ext, out, and ta functions.<br>
The state can be defined using the st record, and initialized using the initst function.
</p>


</HTML>
"));
            import DESLib.DEVSLib.SRC.*;
            model devsPressureControl2
              "Draft model to construct new atomic DEVS models"
              parameter Real sampleTime = 1;
              parameter Real Kp = 1;
              parameter Real Ki = 1;
              parameter Real DB = 1;
              parameter Real comp = 1;
                       extends AtomicDEVS(numIn=1,numOut=2,
              redeclare record State = st);
              redeclare function Fint = int(psuc=Psuc,setpoint=SetPoint);
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(sampletime = sampleTime,kp = Kp,ki = Ki, db = DB,comp = comp);
              Interfaces.outPort comp1        annotation (extent=[96,30;116,50]);
              Interfaces.outPort comp2        annotation (extent=[96,-50;116,-30]);
              Modelica.Blocks.Interfaces.RealInput Psuc
                annotation (extent=[-140,-60;-100,-20]);
              Modelica.Blocks.Interfaces.RealInput SetPoint
                annotation (extent=[-140,20;-100,60]);
            equation
              // INPUT PORTS
              iEvent[1] = 0;
              // OUTPUT PORTS
              oEvent[1] = comp1.event;
              oQueue[1] = comp1.queue;
              oEvent[2] = comp2.event;
              oQueue[2] = comp2.queue;
              annotation (Diagram(graphics));
            end devsPressureControl2;

            function int "Internal Transition Function"
              input st s;
              input Real psuc;
              input Real setpoint;
              output st sout;
              //Real u;
            algorithm
              sout := s;
              if sout.phase == 1 then
                //sout.I := sout.I + sout.e*sout.sampletime;
                //sout.I := sout.I + ((sout.kp/sout.ki)*sout.e) * sout.sampletime;
                if abs(setpoint - psuc) > sout.db then
                  sout.e := setpoint - psuc;
                  //sout.I := sout.I + ((sout.kp/sout.ki)*sout.e) * sout.sampletime;
                else
                  sout.e := 0;
                end if;
                sout.I := sout.I + ((sout.kp/sout.ki)*sout.e) * sout.sampletime;
                // PI
                sout.u := sout.kp * sout.e + sout.I;
                //sout.u := sout.kp * (sout.e + sout.I/sout.ki);
                // first compressor
                if sout.u > sout.comp/2 then
                  sout.c1 := true;
                else
                  sout.c1 := false;
                end if;
                // second compressor
                if sout.u > (sout.comp + sout.comp/2) then
                  sout.c2 := true;
                else
                  sout.c2 := false;
                end if;
                // SATURACION DEL ERROR
                if abs(setpoint-psuc) < sout.db  and sout.I < 0 then
                   sout.I := 0;
                elseif abs(setpoint-psuc) < sout.db and sout.I > 100 then
                   sout.I := 100;
                end if;
                sout.phase := 2;
                sout.sigma := 0;
              elseif sout.phase == 2 then
                sout.phase := 1;
                sout.sigma := sout.sampletime; // sample time
              end if;

            end int;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
            protected
              stdEvent y;
            algorithm
              if s.phase == 2 then
                if s.c1 then
                  y.Value := 1;
                  y.Type := 1;
                else
                  y.Value := 0;
                  y.Type := 0;
                end if;
                sendEvent(queue[1],y);
                if s.c2 then
                  y.Value := 1;
                  y.Type := 1;
                else
                  y.Value := 0;
                  y.Type := 0;
                end if;
                sendEvent(queue[2],y);
              end if;
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st "state of the model"
              Integer phase;
              Real sigma;
              Real sampletime;
              Real kp;
              Real ki;
              Real db;
              Real e;
              Real I;
              Real comp;
              Boolean c1;
              Boolean c2;
              Real u;
            end st;

            function initst "state initalization function"
              input Real sampletime;
              input Real kp;
              input Real ki;
              input Real db;
              input Real comp;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := 0;
              out.sampletime := sampletime;
              out.kp := kp;
              out.ki := ki;
              out.db := db;
              out.e := 0;
              out.I := 0;
              out.comp := comp;
              out.c1 := false;
              out.c2 := false;
              out.u := 0;
            end initst;
          end DEVSPresureControl2;

          model DEVSCompressorRack2C
            import DESLib;
            parameter Real DB = 1 "Controller Dead-Band";
            parameter Real comp = 1 "Compressor capacity";
            parameter Real Nuvol = 1 "Compressor volumetric efficiency";
            parameter Real Vsl = 1 "Compressor total displacement volume";
            parameter Real Kp = 1 "PI gain";
            parameter Real Taoi = 1 "PI time constant";
            Continuous.Compressor compressor(
              comp=comp,
              Nuvol=Nuvol,
              Vsl=Vsl)            annotation (extent=[-18,32;2,52]);
            Continuous.Compressor compressor1(
              comp=comp,
              Nuvol=Nuvol,
              Vsl=Vsl)             annotation (extent=[52,32;72,52]);
            Modelica.Blocks.Interfaces.RealOutput Vcomp
              annotation (extent=[-6,82; 6,94],    rotation=90);
            annotation (Diagram, Icon(
                  Ellipse(
                    extent=[-72,80;88,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-52,52;-12,-76], style(rgbcolor={0,0,0})),
                  Line(points=[68,52;28,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-52,60;68,-60], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1),
                    string="T"),
                  Ellipse(
                    extent=[-84,80;76,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-64,52;-24,-76], style(rgbcolor={0,0,0})),
                  Line(points=[56,52;16,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-64,60;56,-60], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1),
                    string="T")));
            Modelica.Blocks.Interfaces.RealInput Psuc
              annotation (extent=[-10,-60; 10,-80],rotation=90);
            Modelica.Blocks.Interfaces.RealInput SetPoint
              annotation (extent=[-104,-10;-84,10]);
          protected
            Real comp1ON = if diBO.change then 1 else 0;
            Real comp2ON = if diBO1.change then 1 else 0;
          public
            Modelica.Blocks.Math.Sum sum1(nin=2)
              annotation (extent=[-4,62;4,72], rotation=90);
          public
            Modelica.Blocks.Interfaces.RealOutput numCompON
                                                    annotation (extent=[88,-10;108,10]);
            DEVSPresureControl2.devsPressureControl2 devsPressureControl(    sampleTime=60,
              Kp=Kp,
              DB=DB,
              comp=comp,
              Ki=Taoi,
              name="DEVS PI")
                       annotation (extent=[-38,-32;16,14]);
            AuxModels.DiBO diBO(name="dibo1")
                                annotation (extent=[-36,36;-24,48]);
            AuxModels.DiBO diBO1(name="dibo2")
                                 annotation (extent=[34,36;46,48]);
          equation
          numCompON = comp1ON + comp2ON;
            connect(sum1.y, Vcomp) annotation (points=[2.69413e-016,72.5;
                  2.69413e-016,81.2; 0,81.2; 0,88],                                                      style(rgbcolor={0,0,127}));
            connect(compressor.Vcomp, sum1.u[1]) annotation (points=[-8,51; 0,
                  51; 0,61; 0.4,61],                                                              style(rgbcolor={0,0,127}));
            connect(compressor1.Vcomp, sum1.u[2]) annotation (points=[62,51; 0,
                  51; 0,61; -0.4,61],                                                               style(rgbcolor={0,0,127}));
            connect(SetPoint, devsPressureControl.SetPoint) annotation (points=[-94,0;
                  -68,0; -68,0.2; -43.4,0.2],                                                                   style(rgbcolor={0,0,127}));
            connect(Psuc, devsPressureControl.Psuc) annotation (points=[0,-70; 0,
                  -44; -56,-44; -56,-18.2; -43.4,-18.2],                                                            style(rgbcolor={0,0,127}));
            connect(diBO.change, compressor.ON) annotation (points=[-23.4,42;-17,42], style(rgbcolor={255,0,255}));
            connect(diBO1.change, compressor1.ON) annotation (points=[46.6,42;53,42], style(rgbcolor={255,0,255}));
            connect(devsPressureControl.comp1, diBO.inport) annotation (points=[17.62,
                  0.2; 24,0.2; 24,26; -52,26; -52,42; -36.36,42],                                                                style(rgbcolor={0,0,0}));
            connect(devsPressureControl.comp2, diBO1.inport) annotation (points=[17.62,
                  -18.2; 28,-18.2; 28,42; 33.64,42],                                                                   style(rgbcolor={0,0,0}));
          end DEVSCompressorRack2C;

          package DEVSPresureControl3
            "Draft package to construct new atomic DEVS models"

          annotation(preferedView="info",
            Documentation(info="<HTML>
<p>
This package contains the basic structures needed in atomic models.
It can be duplicated, renamed and modified to construct new atomic DEVS models.
</p>
<p>
The user can implement the behavior of a new DEVS model modifying the con, int, ext, out, and ta functions.<br>
The state can be defined using the st record, and initialized using the initst function.
</p>


</HTML>
"));
            import DESLib.DEVSLib.SRC.*;
            model devsPressureControl3
              "Draft model to construct new atomic DEVS models"
              parameter Real sampleTime = 1;
              parameter Real Kp = 1;
              parameter Real Ki = 1;
              parameter Real DB = 1;
              parameter Real comp = 1;
                       extends AtomicDEVS(numIn=1,numOut=3,
              redeclare record State = st);
              redeclare function Fint = int(psuc=Psuc,setpoint=SetPoint);
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(sampletime = sampleTime,kp = Kp,ki = Ki, db = DB,comp = comp);
              Interfaces.outPort comp1        annotation (extent=[96,50;116,70]);
              Interfaces.outPort comp2        annotation (extent=[96,-10;116,10]);
              Modelica.Blocks.Interfaces.RealInput Psuc
                annotation (extent=[-140,-60;-100,-20]);
              Modelica.Blocks.Interfaces.RealInput SetPoint
                annotation (extent=[-140,20;-100,60]);
              Interfaces.outPort comp3        annotation (extent=[96,-70;116,-50]);
            equation
              // INPUT PORTS
              iEvent[1] = 0;
              // OUTPUT PORTS
              oEvent[1] = comp1.event;
              oQueue[1] = comp1.queue;
              oEvent[2] = comp2.event;
              oQueue[2] = comp2.queue;
              oEvent[3] = comp3.event;
              oQueue[3] = comp3.queue;
              annotation (Diagram(graphics));
            end devsPressureControl3;

            function int "Internal Transition Function"
              input st s;
              input Real psuc;
              input Real setpoint;
              output st sout;
              //Real u;
            algorithm
              sout := s;
              if sout.phase == 1 then
                //sout.I := sout.I + sout.e*sout.sampletime;
                //sout.I := sout.I + ((sout.kp/sout.ki)*sout.e) * sout.sampletime;
                if abs(setpoint - psuc) > sout.db then
                  sout.e := setpoint - psuc;
                  //sout.I := sout.I + ((sout.kp/sout.ki)*sout.e) * sout.sampletime;
                else
                  sout.e := 0;
                end if;
                sout.I := sout.I + ((sout.kp/sout.ki)*sout.e) * sout.sampletime;
                // PI
                sout.u := sout.kp * sout.e + sout.I;
                //sout.u := sout.kp * (sout.e + sout.I/sout.ki);
                // first compressor
                if sout.u > sout.comp/2 then
                  sout.c1 := true;
                else
                  sout.c1 := false;
                end if;
                // second compressor
                if sout.u > (sout.comp + sout.comp/2) then
                  sout.c2 := true;
                else
                  sout.c2 := false;
                end if;
                if sout.u > (sout.comp + sout.comp + sout.comp/2) then
                  sout.c3 := true;
                else
                  sout.c3 := false;
                end if;
                // SATURACION DEL ERROR
                if abs(setpoint-psuc) < sout.db  and sout.I < 0 then
                   sout.I := 0;
                elseif abs(setpoint-psuc) < sout.db and sout.I > 100 then
                   sout.I := 100;
                end if;
                sout.phase := 2;
                sout.sigma := 0;
              elseif sout.phase == 2 then
                sout.phase := 1;
                sout.sigma := sout.sampletime; // sample time
              end if;

            end int;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
            protected
              stdEvent y;
            algorithm
              if s.c1 then
                y.Value := 1;
                y.Type := 1;
              else
                y.Value := 0;
                y.Type := 0;
              end if;
              sendEvent(queue[1],y);
              if s.c2 then
                y.Value := 1;
                y.Type := 1;
              else
                y.Value := 0;
                y.Type := 0;
              end if;
              sendEvent(queue[2],y);
              if s.c3 then
                y.Value := 1;
                y.Type := 1;
              else
                y.Value := 0;
                y.Type := 0;
              end if;
              sendEvent(queue[3],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st "state of the model"
              Integer phase;
              Real sigma;
              Real sampletime;
              Real kp;
              Real ki;
              Real db;
              Real e;
              Real I;
              Real comp;
              Boolean c1;
              Boolean c2;
              Boolean c3;
              Real u;
            end st;

            function initst "state initalization function"
              input Real sampletime;
              input Real kp;
              input Real ki;
              input Real db;
              input Real comp;
              output st out;
            algorithm
              out.phase := 1;
              out.sigma := 0;
              out.sampletime := sampletime;
              out.kp := kp;
              out.ki := ki;
              out.db := db;
              out.e := 0;
              out.I := 0;
              out.comp := comp;
              out.c1 := false;
              out.c2 := false;
              out.c3 := false;
              out.u := 0;
            end initst;
          end DEVSPresureControl3;

          model DEVSCompressorRack3C
            import DESLib;
            parameter Real DB = 1 "Controller Dead-Band";
            parameter Real comp = 1 "Compressor capacity";
            parameter Real Nuvol = 1 "Compressor volumetric efficiency";
            parameter Real Vsl = 1 "Compressor total displacement volume";
            parameter Real Kp = 1 "PI gain";
            parameter Real Taoi = 1 "PI time constant";
            Continuous.Compressor compressor(
              comp=comp,
              Nuvol=Nuvol,
              Vsl=Vsl)            annotation (extent=[-46,32;-26,52]);
            Continuous.Compressor compressor1(
              comp=comp,
              Nuvol=Nuvol,
              Vsl=Vsl)             annotation (extent=[4,32;24,52]);
            Modelica.Blocks.Interfaces.RealOutput Vcomp
              annotation (extent=[-10,80;10,100], rotation=90);
            annotation (Diagram, Icon(
                  Ellipse(
                    extent=[-72,80;88,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-52,52;-12,-76], style(rgbcolor={0,0,0})),
                  Line(points=[68,52;28,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-52,60;68,-60], string="T", style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Ellipse(
                    extent=[-84,80;76,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-64,52;-24,-76], style(rgbcolor={0,0,0})),
                  Line(points=[56,52;16,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-64,60;56,-60], string="T", style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1))));
            Modelica.Blocks.Interfaces.RealInput Psuc
              annotation (extent=[-10,-100;10,-80], rotation=90);
            Modelica.Blocks.Interfaces.RealInput SetPoint
              annotation (extent=[-104,-10;-84,10]);
          protected
            Real comp1ON = if diBO.change then 1 else 0;
            Real comp2ON = if diBO1.change then 1 else 0;
          public
            Modelica.Blocks.Math.Sum sum1(nin=3)
              annotation (extent=[-4,64;4,72], rotation=90);
          public
            Modelica.Blocks.Interfaces.RealOutput numCompON
                                                    annotation (extent=[88,-10;108,10]);
            DEVSPresureControl3.devsPressureControl3 devsPressureControl(
              sampleTime=60,
              Kp=Kp,
              DB=DB,
              comp=comp,
              Ki=Taoi) annotation (extent=[-38,-32;16,14]);
            AuxModels.DiBO diBO annotation (extent=[-64,36;-52,48]);
            AuxModels.DiBO diBO1 annotation (extent=[-14,36;-2,48]);
            Continuous.Compressor compressor2(
              comp=comp,
              Nuvol=Nuvol,
              Vsl=Vsl)             annotation (extent=[54,32;74,52]);
            AuxModels.DiBO diBO2 annotation (extent=[36,36;48,48]);
          equation
          numCompON = comp1ON + comp2ON;
            connect(sum1.y, Vcomp) annotation (points=[2.69413e-016,72.4;
                  2.69413e-016,81.2; 0,81.2; 0,90],                                                      style(rgbcolor={0,0,127}));
            connect(SetPoint, devsPressureControl.SetPoint) annotation (points=[-94,0;
                  -68,0; -68,0.2; -43.4,0.2],                                                                   style(rgbcolor={0,0,127}));
            connect(Psuc, devsPressureControl.Psuc) annotation (points=[0,-90; 0,
                  -44; -56,-44; -56,-18.2; -43.4,-18.2],                                                            style(rgbcolor={0,0,127}));
            connect(diBO.change, compressor.ON) annotation (points=[-51.4,42;-45,42], style(rgbcolor={255,0,255}));
            connect(diBO1.change, compressor1.ON) annotation (points=[-1.4,42;5,42], style(rgbcolor={255,0,255}));
            connect(diBO2.change,compressor2. ON) annotation (points=[48.6,42;55,42], style(rgbcolor={255,0,255}));
            connect(compressor.Vcomp, sum1.u[1]) annotation (points=[-36,51;-36,56;0.533333,56;0.533333,63.2], style(rgbcolor={0,0,127}));
            connect(compressor1.Vcomp, sum1.u[2]) annotation (points=[14,51;14,56;-3.05311e-016,56;-3.05311e-016,63.2], style(rgbcolor={0,0,127}));
            connect(compressor2.Vcomp, sum1.u[3]) annotation (points=[64,51;64,56;-0.533333,56;-0.533333,63.2], style(rgbcolor={0,0,127}));
            connect(devsPressureControl.comp1, diBO.inport) annotation (points=[17.62,
                  4.8; 28,4.8; 28,22; -68,22; -68,42; -64.36,42],                                                                style(rgbcolor={0,0,0}));
            connect(devsPressureControl.comp2, diBO1.inport) annotation (points=[17.62,-9;38,-9;38,28;-22,28;-22,42;-14.36,42], style(rgbcolor={0,0,0}));
            connect(devsPressureControl.comp3, diBO2.inport) annotation (points=[17.62,-22.8;48,-22.8;48,34;26,34;26,42;35.64,42], style(rgbcolor={0,0,0}));
          end DEVSCompressorRack3C;

          model system2D2C
            Real Tair1 = displayCase.displayCase.Tair;
            Real Tair2 = displayCase1.displayCase.Tair;
            Real Psuc = suctionMainfold.Psuc;
            Real Ycon = constraintSat.Ycon;
            Real Yswitch = numberSwitches.Yswitch;
            Real Ypow = powerConsup.Ypow;
            annotation (Diagram(graphics), experiment(StopTime=14400),  experimentSetupOutput);
            DEVSControlledDisplayCase displayCase(
              UAgoodsair=300,
              Mgoods=200,
              CPgoods=1000,
              UAairwall=500,
              Mwall=260,
              CPwall=385,
              UAwallrefmax=4000,
              Mrefmax=1,
              Mair=50,
              CPair=1000,
              Taufill=40,
              Tairmax=5,
              Tairmin=2,
              Twallstart=0,
              Tairstart=5.1,
              Tgoodsstart=2,
              Mrefstart=0)          annotation (extent=[-74,20;-40,60]);
            Continuous.SuctionMainfold suctionMainfold(Vsuc=5, Psuc(start=1.4))
                                            annotation (extent=[34,-26;86,36]);
            DEVSControlledDisplayCase displayCase1(
              UAgoodsair=300,
              Mgoods=200,
              CPgoods=1000,
              UAairwall=500,
              Mwall=260,
              CPwall=385,
              UAwallrefmax=4000,
              Mrefmax=1,
              Mair=50,
              CPair=1000,
              Taufill=40,
              Tairmax=5,
              Tairmin=2,
              Twallstart=0,
              Tairstart=0,
              Tgoodsstart=2,
              Mrefstart=1)          annotation (extent=[-74,-20;-40,20]);
            CompressorRack2C comprossorRack2C(
              DB=0.2,
              comp=50,
              Kp=-75,
              Taoi=50,
              Nuvol=0.81,
              Vsl=0.08)                       annotation (extent=[78,-70;98,-50]);
            Modelica.Blocks.Sources.Step AirLoad(
              height=-1200,
              offset=3000,
              startTime=7200) annotation (extent=[-64,60;-50,68]);
            Modelica.Blocks.Sources.Step Mrefconst(
              startTime=7200,
              height=-0.2,
              offset=0.2) annotation (extent=[40,52; 54,66]);
            Modelica.Blocks.Sources.Step Psucdesired(
              startTime=7200,
              height=0.2,
              offset=1.4) annotation (extent=[66,-64;74,-56]);
            Continuous.ConstraintSat constraintSat(
              NumDisp=2,
              MaxTair={5,5},
              MinTair={2,2}) annotation (extent=[-16,-46;4,-26]);
            Modelica.Blocks.Sources.Step MaxPsuc(
              startTime=7200,
              height=0.2,
              offset=1.7)     annotation (extent=[-38,-38;-28,-28]);
            Continuous.NumberSwitches numberSwitches
                                          annotation (extent=[-16,-70;4,-50]);
            Modelica.Blocks.Math.Sum sum1(nin=2) annotation (extent=[-40,-70;-30,-60]);
            Continuous.PowerConsup powerConsup
                                    annotation (extent=[52,-48;72,-28]);
          equation
            connect(displayCase.m, suctionMainfold.m) annotation (points=[-41.7,24;-22,24;-22,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
            connect(displayCase1.m, suctionMainfold.m) annotation (points=[-41.7,-16;-22,-16;-22,1.9;41.8,1.9], style(rgbcolor={0,0,
                    127}));
            connect(comprossorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[88,-51;88,5;78.2,5], style(rgbcolor={0,0,127}));
            connect(AirLoad.y, displayCase.Qairload) annotation (points=[-49.3,64;-32,64;-32,48;-41.7,48], style(rgbcolor={0,0,127}));
            connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-49.3,64;-32,64;-32,8;-41.7,8], style(rgbcolor={0,0,127}));
            connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[54.7,59;
                  60.26,59; 60.26,32.9],                                                                      style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[41.8,8.1;-26.1,8.1;-26.1,28.4;-41.7,28.4], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[41.8,8.1;-26,8.1;-26,-11.6;-41.7,-11.6], style(rgbcolor={0,0,127}));
            connect(Psucdesired.y, comprossorRack2C.SetPoint) annotation (points=[74.4,-60;79,-60], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc, comprossorRack2C.Psuc) annotation (points=[41.8,8.1;16,8.1;16,-76;88,-76;88,-69], style(rgbcolor={0,0,127}));
            connect(displayCase.Tair,constraintSat. Tair[1]) annotation (points=[-75.7,48;-94,48;-94,-45;-18,-45], style(rgbcolor={0,0,127}));
            connect(displayCase1.Tair,constraintSat. Tair[2]) annotation (points=[-75.7,8;-94,8;-94,-43;-18,-43], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc,constraintSat. Psuc) annotation (points=[41.8,8.1;-22,8.1;-22,-28;-18,-28], style(rgbcolor={0,0,127}));
            connect(MaxPsuc.y,constraintSat. MaxPsuc) annotation (points=[-27.5,-33;-24,-33;-24,-34;-18,-34], style(rgbcolor={0,0,127}));
            connect(sum1.y,numberSwitches. NumValve) annotation (points=[-29.5,-65;-21.75,-65;-21.75,-66;-18,-66], style(rgbcolor={0,0,127}));
            connect(displayCase.valveOpen, sum1.u[1]) annotation (points=[-57,18;-80,18;-80,-65.5;-41,-65.5], style(rgbcolor={0,0,127}));
            connect(sum1.u[2], displayCase1.valveOpen) annotation (points=[-41,-64.5;-57,-64.5;-57,-22], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc,powerConsup. Psuc) annotation (points=[41.8,8.1;34,8.1;34,-44;50,-44], style(rgbcolor={0,0,127}));
            connect(comprossorRack2C.Vcomp, powerConsup.Fcomp) annotation (points=[88,-51;88,-24;44,-24;44,-32;50,-32], style(rgbcolor={0,0,127}));
            connect(comprossorRack2C.numCompON, numberSwitches.NumComp)
              annotation (points=[99,-60;106,-60;106,-84;-26,-84;-26,-54;-18,-54], style(rgbcolor={0,0,127}));
          end system2D2C;

          model system3D3C
            Real Tair1 = displayCase1.displayCase.Tair;
            Real Tair2 = displayCase2.displayCase.Tair;
            Real Tair3 = displayCase3.displayCase.Tair;
            Real Psuc = suctionMainfold.Psuc;

            annotation (Diagram(graphics));
            DEVSControlledDisplayCase displayCase1(
              UAgoodsair=300,
              Mgoods=200,
              CPgoods=1000,
              UAairwall=500,
              Mwall=260,
              CPwall=385,
              UAwallrefmax=4000,
              Mrefmax=1,
              Mair=50,
              CPair=1000,
              Taufill=40,
              Tairmax=5,
              Tairmin=2,
              Twallstart=0,
              Tairstart=5.1,
              Tgoodsstart=2,
              Mrefstart=0)          annotation (extent=[-74,12;-34,52]);
            DEVSControlledDisplayCase displayCase2(
              UAgoodsair=300,
              Mgoods=200,
              CPgoods=1000,
              UAairwall=500,
              Mwall=260,
              CPwall=385,
              UAwallrefmax=4000,
              Mrefmax=1,
              Mair=50,
              CPair=1000,
              Taufill=40,
              Tairmax=5,
              Tairmin=2,
              Twallstart=0,
              Tairstart=0,
              Tgoodsstart=2,
              Mrefstart=1)           annotation (extent=[-74,-34;-34,6]);
            Continuous.SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                            annotation (extent=[38,-44;90,18]);
            DEVSControlledDisplayCase displayCase3(
              UAgoodsair=300,
              Mgoods=200,
              CPgoods=1000,
              UAairwall=500,
              Mwall=260,
              CPwall=385,
              UAwallrefmax=4000,
              Twallstart=0,
              Tairstart=2.5,
              Tgoodsstart=2,
              Mrefstart=0.5,
              Mrefmax=1,
              Mair=50,
              CPair=1000,
              Taufill=40,
              Tairmax=5,
              Tairmin=2)             annotation (extent=[-74,-84;-34,-44]);
            Modelica.Blocks.Sources.Step AirLoad(
              height=-1200,
              offset=3000,
              startTime=7200) annotation (extent=[-64,62;-44,82]);
            CompressorRack3C comprossorRack3C(
              DB=0.2,
              comp=100/3,
              Vsl=0.095,
              Kp=-75,
              Taoi=50,
              Nuvol=0.81)                     annotation (extent=[76,-74;96,-54]);
            Modelica.Blocks.Sources.Step Mrefconst(
              startTime=7200,
              height=-0.2,
              offset=0.2) annotation (extent=[30,60;50,80]);
            Modelica.Blocks.Sources.Step Psucdesired(
              startTime=7200,
              height=0.2,
              offset=1.4) annotation (extent=[44,-74;64,-54]);
          equation
            connect(suctionMainfold.Psuc, displayCase1.Psuc)
                                                            annotation (points=[45.8,-9.9;-24,-9.9;-24,20.4;-36,20.4], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc,displayCase2. Psuc) annotation (points=[45.8,-9.9;-24,-9.9;-24,-25.6;-36,-25.6], style(rgbcolor={0,0,127}));
            connect(displayCase1.m, suctionMainfold.m)
                                                      annotation (points=[-36,16;-26,16;-26,-16.1;45.8,-16.1], style(rgbcolor={0,0,127}));
            connect(displayCase2.m,suctionMainfold. m) annotation (points=[-36,-30;-26,-30;-26,-16.1;45.8,-16.1], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc,displayCase3. Psuc) annotation (points=[45.8,-9.9;-24,-9.9;-24,-75.6;-36,-75.6], style(rgbcolor={0,0,127}));
            connect(displayCase3.m, suctionMainfold.m) annotation (points=[-36,-80;-26,-80;-26,-16.1;45.8,-16.1], style(rgbcolor={0,0,127}));
            connect(AirLoad.y, displayCase1.Qairload)
                                                     annotation (points=[-43,72;-30,72;-30,40;-36,40], style(rgbcolor={0,0,127}));
            connect(AirLoad.y,displayCase2. Qairload) annotation (points=[-43,72;-30,72;-30,-6;-36,-6], style(rgbcolor={0,0,127}));
            connect(AirLoad.y,displayCase3. Qairload) annotation (points=[-43,72;-30,72;-30,-56;-36,-56], style(rgbcolor={0,0,127}));
            connect(comprossorRack3C.Vcomp, suctionMainfold.Vcomp) annotation (points=[86,-55;86,-12;82.2,-12;82.2,-13], style(rgbcolor={0,0,127}));
            connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[51,70;64.26,70;64.26,14.9], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc, comprossorRack3C.Psuc) annotation (points=[45.8,-9.9;34,-9.9;34,-94;86,-94;86,-73], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc, displayCase3.Psuc) annotation (points=[45.8,-9.9;-24,-9.9;-24,-75.6;-36,-75.6], style(rgbcolor={0,0,127}));
            connect(displayCase3.m, suctionMainfold.m) annotation (points=[-36,-80;-26,-80;-26,-16.1;45.8,-16.1], style(rgbcolor={0,0,127}));
            connect(AirLoad.y, displayCase3.Qairload) annotation (points=[-43,72;-30,72;-30,-56;-36,-56], style(rgbcolor={0,0,127}));
            connect(Psucdesired.y, comprossorRack3C.SetPoint) annotation (points=[65,-64;76.6,-64], style(rgbcolor={0,0,127}));
          end system3D3C;

          model DEVSsystem2D2C
            Real Tair1 = displayCase.displayCase.Tair;
            Real Tair2 = displayCase1.displayCase.Tair;
            Real Psuc = suctionMainfold.Psuc;
            Real Ycon = constraintSat.Ycon;
            Real Yswitch = numberSwitches.Yswitch;
            Real Ypow = powerConsup.Ypow;
            annotation (Diagram(graphics), experiment(StopTime=14400), experimentSetupOutput);
            DEVSControlledDisplayCase displayCase(
              UAgoodsair=300,
              Mgoods=200,
              CPgoods=1000,
              UAairwall=500,
              Mwall=260,
              CPwall=385,
              UAwallrefmax=4000,
              Mrefmax=1,
              Mair=50,
              CPair=1000,
              Taufill=40,
              Tairmax=5,
              Tairmin=2,
              Twallstart=0,
              Tairstart=5.1,
              Tgoodsstart=2,
              Mrefstart=0)          annotation (extent=[-74,20;-40,60]);
            Continuous.SuctionMainfold suctionMainfold(Vsuc=5, Psuc(start=1.4),
              Psucstart=1.4)                annotation (extent=[34,-26;86,36]);
            DEVSControlledDisplayCase displayCase1(
              UAgoodsair=300,
              Mgoods=200,
              CPgoods=1000,
              UAairwall=500,
              Mwall=260,
              CPwall=385,
              UAwallrefmax=4000,
              Mrefmax=1,
              Mair=50,
              CPair=1000,
              Taufill=40,
              Tairmax=5,
              Tairmin=2,
              Twallstart=0,
              Tairstart=0,
              Tgoodsstart=2,
              Mrefstart=1)          annotation (extent=[-74,-20;-40,20]);
            DEVSCompressorRack2C comprossorRack2C(
              DB=0.2,
              comp=50,
              Kp=-75,
              Taoi=50,
              Nuvol=0.81,
              Vsl=0.08)                       annotation (extent=[78,-68;98,-48]);
            Modelica.Blocks.Sources.Step AirLoad(
              height=-1200,
              offset=3000,
              startTime=7200) annotation (extent=[-64,60;-50,70]);
            Modelica.Blocks.Sources.Step Mrefconst(
              startTime=7200,
              height=-0.2,
              offset=0.2) annotation (extent=[40,36;54,46]);
            Modelica.Blocks.Sources.Step Psucdesired(
              startTime=7200,
              height=0.2,
              offset=1.4) annotation (extent=[60,-64;74,-56]);
            Continuous.ConstraintSat constraintSat(
              NumDisp=2,
              MaxTair={5,5},
              MinTair={2,2}) annotation (extent=[-16,-46;4,-26]);
            Modelica.Blocks.Sources.Step MaxPsuc(
              startTime=7200,
              height=0.2,
              offset=1.7)     annotation (extent=[-42,-38;-28,-28]);
            Continuous.NumberSwitches numberSwitches
                                          annotation (extent=[-16,-70;4,-50]);
            Modelica.Blocks.Math.Sum sum1(nin=2) annotation (extent=[-40,-70;-30,-60]);
            Continuous.PowerConsup powerConsup
                                    annotation (extent=[52,-48;72,-28]);
          equation
            connect(displayCase.m, suctionMainfold.m) annotation (points=[-41.7,24;-22,24;-22,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
            connect(displayCase1.m, suctionMainfold.m) annotation (points=[-41.7,-16;-22,-16;-22,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
            connect(comprossorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[88,-49;88,5;78.2,5], style(rgbcolor={0,0,127}));
            connect(AirLoad.y, displayCase.Qairload) annotation (points=[-49.3,65;-32,65;-32,48;-41.7,48], style(rgbcolor={0,0,127}));
            connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-49.3,65;-32,65;-32,8;-41.7,8], style(rgbcolor={0,0,127}));
            connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[54.7,41;60.26,41;60.26,32.9], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[41.8,8.1;-26.1,8.1;-26.1,28.4;-41.7,28.4], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[41.8,8.1;-26,8.1;-26,-11.6;-41.7,-11.6], style(rgbcolor={0,0,127}));
            connect(Psucdesired.y, comprossorRack2C.SetPoint) annotation (points=[74.7,-60;76,-60;76,-58;78.6,-58], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc, comprossorRack2C.Psuc) annotation (points=[41.8,8.1;16,8.1;16,-76;88,-76;88,-67], style(rgbcolor={0,0,127}));
            connect(displayCase.Tair,constraintSat. Tair[1]) annotation (points=[-75.7,48;-94,48;-94,-45;-18,-45], style(rgbcolor={0,0,127}));
            connect(displayCase1.Tair,constraintSat. Tair[2]) annotation (points=[-75.7,8;-94,8;-94,-43;-18,-43], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc,constraintSat. Psuc) annotation (points=[41.8,8.1;-22,8.1;-22,-28;-18,-28], style(rgbcolor={0,0,127}));
            connect(MaxPsuc.y,constraintSat. MaxPsuc) annotation (points=[-27.3,-33;-24,-33;-24,-34;-18,-34], style(rgbcolor={0,0,127}));
            connect(sum1.y,numberSwitches. NumValve) annotation (points=[-29.5,-65;-21.75,-65;-21.75,-66;-18,-66], style(rgbcolor={0,0,127}));
            connect(displayCase.valveOpen, sum1.u[1]) annotation (points=[-57,18;-80,18;-80,-65.5;-41,-65.5], style(rgbcolor={0,0,127}));
            connect(sum1.u[2], displayCase1.valveOpen) annotation (points=[-41,-64.5;-57,-64.5;-57,-22], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc,powerConsup. Psuc) annotation (points=[41.8,8.1;34,8.1;34,-44;50,-44], style(rgbcolor={0,0,127}));
            connect(comprossorRack2C.Vcomp, powerConsup.Fcomp) annotation (points=[88,-49;88,-24;44,-24;44,-32;50,-32], style(rgbcolor={0,0,127}));
            connect(comprossorRack2C.numCompON, numberSwitches.NumComp)
              annotation (points=[97.8,-58;104,-58;104,-80;-26,-80;-26,-54;-18,-54], style(rgbcolor={0,0,127}));
          end DEVSsystem2D2C;

          model DEVSsystem3D3C
            Real Tair1 = displayCase1.displayCase.Tair;
            Real Tair2 = displayCase2.displayCase.Tair;
            Real Tair3 = displayCase3.displayCase.Tair;
            Real Psuc = suctionMainfold.Psuc;

            annotation (Diagram(graphics));
            DEVSControlledDisplayCase displayCase1(
              UAgoodsair=300,
              Mgoods=200,
              CPgoods=1000,
              UAairwall=500,
              Mwall=260,
              CPwall=385,
              UAwallrefmax=4000,
              Mrefmax=1,
              Mair=50,
              CPair=1000,
              Taufill=40,
              Tairmax=5,
              Tairmin=2,
              Twallstart=0,
              Tairstart=5.1,
              Tgoodsstart=2,
              Mrefstart=0)          annotation (extent=[-74,12;-34,52]);
            DEVSControlledDisplayCase displayCase2(
              UAgoodsair=300,
              Mgoods=200,
              CPgoods=1000,
              UAairwall=500,
              Mwall=260,
              CPwall=385,
              UAwallrefmax=4000,
              Mrefmax=1,
              Mair=50,
              CPair=1000,
              Taufill=40,
              Tairmax=5,
              Tairmin=2,
              Twallstart=0,
              Tairstart=0,
              Tgoodsstart=2,
              Mrefstart=1)           annotation (extent=[-74,-34;-34,6]);
            Continuous.SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                            annotation (extent=[38,-44;90,18]);
            DEVSControlledDisplayCase displayCase3(
              UAgoodsair=300,
              Mgoods=200,
              CPgoods=1000,
              UAairwall=500,
              Mwall=260,
              CPwall=385,
              UAwallrefmax=4000,
              Twallstart=0,
              Tairstart=2.5,
              Tgoodsstart=2,
              Mrefstart=0.5,
              Mrefmax=1,
              Mair=50,
              CPair=1000,
              Taufill=40,
              Tairmax=5,
              Tairmin=2)             annotation (extent=[-74,-84;-34,-44]);
            Modelica.Blocks.Sources.Step AirLoad(
              height=-1200,
              offset=3000,
              startTime=7200) annotation (extent=[-64,62;-44,82]);
            DEVSCompressorRack3C comprossorRack3C(
              DB=0.2,
              comp=100/3,
              Vsl=0.095,
              Kp=-75,
              Taoi=50,
              Nuvol=0.81)                     annotation (extent=[76,-74;96,-54]);
            Modelica.Blocks.Sources.Step Mrefconst(
              startTime=7200,
              height=-0.2,
              offset=0.2) annotation (extent=[30,60;50,80]);
            Modelica.Blocks.Sources.Step Psucdesired(
              startTime=7200,
              height=0.2,
              offset=1.4) annotation (extent=[44,-74;64,-54]);
          equation
            connect(suctionMainfold.Psuc, displayCase1.Psuc)
                                                            annotation (points=[45.8,-9.9;-24,-9.9;-24,20.4;-36,20.4], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc,displayCase2. Psuc) annotation (points=[45.8,-9.9;-24,-9.9;-24,-25.6;-36,-25.6], style(rgbcolor={0,0,127}));
            connect(displayCase1.m, suctionMainfold.m)
                                                      annotation (points=[-36,16;-26,16;-26,-16.1;45.8,-16.1], style(rgbcolor={0,0,127}));
            connect(displayCase2.m,suctionMainfold. m) annotation (points=[-36,-30;-26,-30;-26,-16.1;45.8,-16.1], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc,displayCase3. Psuc) annotation (points=[45.8,-9.9;-24,-9.9;-24,-75.6;-36,-75.6], style(rgbcolor={0,0,127}));
            connect(displayCase3.m, suctionMainfold.m) annotation (points=[-36,-80;-26,-80;-26,-16.1;45.8,-16.1], style(rgbcolor={0,0,127}));
            connect(AirLoad.y, displayCase1.Qairload)
                                                     annotation (points=[-43,72;-30,72;-30,40;-36,40], style(rgbcolor={0,0,127}));
            connect(AirLoad.y,displayCase2. Qairload) annotation (points=[-43,72;-30,72;-30,-6;-36,-6], style(rgbcolor={0,0,127}));
            connect(AirLoad.y,displayCase3. Qairload) annotation (points=[-43,72;-30,72;-30,-56;-36,-56], style(rgbcolor={0,0,127}));
            connect(comprossorRack3C.Vcomp, suctionMainfold.Vcomp) annotation (points=[86,-55;86,-12;82.2,-12;82.2,-13], style(rgbcolor={0,0,127}));
            connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[51,70;64.26,70;64.26,14.9], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc, comprossorRack3C.Psuc) annotation (points=[45.8,-9.9;34,-9.9;34,-94;86,-94;86,-73], style(rgbcolor={0,0,127}));
            connect(suctionMainfold.Psuc, displayCase3.Psuc) annotation (points=[45.8,-9.9;-24,-9.9;-24,-75.6;-36,-75.6], style(rgbcolor={0,0,127}));
            connect(displayCase3.m, suctionMainfold.m) annotation (points=[-36,-80;-26,-80;-26,-16.1;45.8,-16.1], style(rgbcolor={0,0,127}));
            connect(AirLoad.y, displayCase3.Qairload) annotation (points=[-43,72;-30,72;-30,-56;-36,-56], style(rgbcolor={0,0,127}));
            connect(Psucdesired.y, comprossorRack3C.SetPoint) annotation (points=[65,-64;76.6,-64], style(rgbcolor={0,0,127}));
          end DEVSsystem3D3C;
        end DEVSControl;

        package Continuous
        model DisplayCase
                  parameter Real UAgoodsair
              "Heat transfer coeficient between goods and air";
                  parameter Real Mgoods "Mass of the goods";
                  parameter Real CPgoods "Heat capacity of goods";
                  parameter Real UAairwall
              "Heat transfer coeficient between air and wall";
                  parameter Real Mwall "Mass of the wall";
                  parameter Real CPwall "Heat capacity of wall";
                  parameter Real UAwallrefmax
              "Maximum heat transfer coeficient between wall and refrigerant";
                  parameter Real Mrefmax "Maximum mass of refrigerant";
                  parameter Real Mair "Mass of the air";
                  parameter Real CPair "Heat capacity of air";
                  parameter Real Taufill "Refrigerator filling time";

                  parameter Real Tgoods_initial;
                  parameter Real Twall_initial;
                  parameter Real Tair_initial;
                  parameter Real Mref_initial;

                  Real Tgoods( start = Tgoods_initial)
              "Temperature of the goods in the display case";
                  Real Twall( start = Twall_initial)
              "Temperature of the evaporator wall";
                  Real Tair( start = Tair_initial)
              "Temperature of the air in the display case";
                  Real Mref( start = Mref_initial)
              "Mass of liquified refrigerante in the evaporator";

                  Modelica.Blocks.Interfaces.RealInput Psuc "Suction Pressure"
                    annotation (extent=[80,-60;100,-40], rotation=180);
                  Modelica.Blocks.Interfaces.BooleanInput valveOpen
              "State of the valve (open,closed)"
                    annotation (extent=[-48,-100;-28,-80], rotation=90);
                  Modelica.Blocks.Interfaces.RealInput Qairload
              "External air disturbance"
                    annotation (extent=[80,24;100,44], rotation=180);
                  flow Modelica.Blocks.Interfaces.RealOutput m
        annotation (extent=[80,-80;100,-60]);
          protected
                Real Qgoodsair;
                Real Qairwall;
                Real Qe "heat";
                Real Te "refrigerant evaporation temperature";
                Real UAwallref;
                Real IncHlg;
          public
                  Modelica.Blocks.Interfaces.RealOutput AirTemp
                    annotation (extent=[-100,-10;-80,10], rotation=180);
        equation
                AirTemp = Tair;
                // Eq. (1)
                der(Tgoods) = - Qgoodsair / (Mgoods * CPgoods);
                // Eq. (2)
                der(Twall) = (Qairwall - Qe) / (Mwall * CPwall);
                // Eq. (3)
                der(Tair) = (Qgoodsair + Qairload - Qairwall) / (Mair*CPair);
                // Eq. (4)
                Qgoodsair = UAgoodsair * (Tgoods - Tair);
                // Eq. (5)
                Qairwall = UAairwall * (Tair - Twall);
                // Eq. (6)
                Qe = UAwallref * (Twall - Te);
                // Eq. (7)
                UAwallref = UAwallrefmax * Mref / Mrefmax;
                // Eq. (8)
                der(Mref) = if valveOpen then
                                 (Mrefmax - Mref) / Taufill else
                                 if Mref > 0 then - Qe / IncHlg else
                                                  0;
                // Eq. (9)
                m = - Qe / IncHlg;
                // Appendix
                Te = -4.3544*Psuc^2  + 29.2240*Psuc - 51.2005;
                IncHlg = (0.0217*Psuc^2 - 0.1704*Psuc +2.2988)*10^5;

                  annotation (Diagram, Icon(Rectangle(
                    extent=[-80,80;80,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                    Text(
                    extent=[-78,78;78,0], string="DisplayCase", style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1))));

        end DisplayCase;

                model AirController
                  parameter Real Tairmax
              "Max temperature threshold for the controller";
                  parameter Real Tairmin
              "Min temperature threshold for the controller";

                  Modelica.Blocks.Interfaces.RealInput Tair
                    annotation (extent=[-80,-10;-60,10]);
                  Modelica.Blocks.Interfaces.BooleanOutput ValveOpen
                    annotation (extent=[60,-10;80,10]);

                   parameter Real sampleTime;

                  annotation (Diagram, Icon(
                    Rectangle(
                    extent=[-60,40;60,-40], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                    Text(
                    extent=[-58,20;58,-20], string="AirTemp Control", style(
                    rgbolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1))));

                equation
                   when sample(0,sampleTime) then
                      ValveOpen = if Tair > Tairmax then  true else
                                       if Tair < Tairmin then false else
                                       pre(ValveOpen);
                   end when;

                end AirController;

                model ControlledDisplayCase
                  parameter Real UAgoodsair
              "Heat transfer coeficient between goods and air";
                  parameter Real Mgoods "Mass of the goods";
                  parameter Real CPgoods "Heat capacity of goods";
                  parameter Real UAairwall
              "Heat transfer coeficient between air and wall";
                  parameter Real Mwall "Mass of the wall";
                  parameter Real CPwall "Heat capacity of wall";
                  parameter Real UAwallrefmax
              "Maximum heat transfer coeficient between wall and refrigerant";
                  parameter Real Mrefmax "Maximum mass of refrigerant";
                  parameter Real Mair "Mass of the air";
                  parameter Real CPair "Heat capacity of air";
                  parameter Real Taufill "Refrigerator filling time";
                  parameter Real Tairmax
              "Max temperature threshold for the controller";
                  parameter Real Tairmin
              "Min temperature threshold for the controller";
                  // Valores iniciales
                  parameter Real Twallstart;
                  parameter Real Tairstart;
                  parameter Real Tgoodsstart;
                  parameter Real Mrefstart;
                  // Muestreo controlador
                  parameter Real sampleTime = 1;

                  DisplayCase displayCase(
                    UAgoodsair=UAgoodsair,
                    Mgoods=Mgoods,
                    CPgoods=CPgoods,
                    UAairwall=UAairwall,
                    Mwall=Mwall,
                    CPwall=CPwall,
                    UAwallrefmax=UAwallrefmax,
                    Mrefmax=Mrefmax,
                    Mair=Mair,
                    CPair=CPair,
                    Taufill=Taufill,
                    Tgoods_initial=Tgoodsstart,
                    Twall_initial=Twallstart,
                    Tair_initial=Tairstart,
                    Mref_initial=Mrefstart)
                                     annotation (extent=[-74,-80;66,100]);
                  AirController airController(
                      Tairmax=Tairmax, Tairmin=Tairmin, sampleTime=sampleTime)
                    annotation (extent=[-96,-108;-50,-70]);
                  annotation (Diagram, Icon(Bitmap(extent=[-100,100;80,-100], name="Figs/displaycase.png")));
                  Modelica.Blocks.Interfaces.RealInput Qairload
                    annotation (extent=[80,30;100,50], rotation=180);
                  Modelica.Blocks.Interfaces.RealInput Psuc
                    annotation (extent=[80,-68;100,-40], rotation=180);
                  flow Modelica.Blocks.Interfaces.RealOutput m
                    annotation (extent=[80,-90;100,-70]);
                  Modelica.Blocks.Interfaces.RealOutput Tair
                    annotation (extent=[-120,30;-100,50], rotation=180);
                  Modelica.Blocks.Interfaces.RealOutput valveOpen
                    annotation (extent=[-10,-120;10,-100], rotation=270);
                equation
                  valveOpen = if airController.ValveOpen then 1 else 0;

                  connect(displayCase.AirTemp, airController.Tair) annotation (points=[-67,10;-94,10;-94,-89;-89.1,-89], style(rgbcolor={0,0,127}));
                  connect(Qairload, displayCase.Qairload) annotation (points=[90,40;
                  73.1,40; 73.1,40.6; 59,40.6],                                                                 style(rgbcolor={0,0,127}));
                  connect(displayCase.Psuc, Psuc) annotation (points=[59,-35;
                  79.5,-35; 79.5,-54; 90,-54],                                                                     style(rgbcolor={0,0,127}));
                  connect(displayCase.m, m) annotation (points=[59,-53; 72.5,
                  -53; 72.5,-80; 90,-80],                                                         style(rgbcolor={0,0,127}));
                  connect(airController.ValveOpen, displayCase.valveOpen) annotation (points=[-56.9,
                  -89; -30.45,-89; -30.45,-71; -30.6,-71],                                                                                style(rgbcolor={255,0,255}));
                  connect(displayCase.AirTemp, Tair) annotation (points=[-67,10;-74,10;-74,40;-110,40], style(rgbcolor={0,0,127}));
                end ControlledDisplayCase;

                model SuctionMainfold
                  parameter Real Vsuc "Total volume";
                  parameter Real Psucstart "Initial value of Psuc";

                  Modelica.Blocks.Interfaces.RealOutput Psuc( start = Psucstart)
                    annotation (extent=[-80,0;-60,20], rotation=180);

                  flow Modelica.Blocks.Interfaces.RealInput m
                    annotation (extent=[-80,-20;-60,0]);

                  Modelica.Blocks.Interfaces.RealInput Vcomp
                    annotation (extent=[60,-10;80,10], rotation=180);

          protected
                  Real Rosuc "density of the vapor";
                  Real dersuc;
          public
                  Modelica.Blocks.Interfaces.RealInput Mrefconst
                    annotation (extent=[-9,79;11,101], rotation=270);
                equation
                  // Eq. (10)
                  der(Psuc) = (m + Mrefconst - Vcomp*Rosuc) / (Vsuc*dersuc);
                  // Appendix
                  Rosuc = 4.6073*Psuc+0.3798;
                  dersuc = -0.0329*Psuc^3 + 0.2161*Psuc^2 -0.4742*Psuc + 5.4817;

                  annotation (Icon(
                  Ellipse(extent=[-60,80;-20,40], style(rgbcolor={95,95,95})),
                  Ellipse(extent=[20,80;60,40], style(rgbcolor={95,95,95})),
                  Ellipse(extent=[-60,-40;-20,-80], style(rgbcolor={95,95,95})),
                  Ellipse(extent=[20,-40;60,-80], style(rgbcolor={95,95,95})),
                  Rectangle(
                    extent=[-40,80;40,-80], style(
                    rgbcolor={255,255,255},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Rectangle(
                    extent=[-60,60;60,-60], style(
                    rgbcolor={255,255,255},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-60,60;-60,-60], style(rgbcolor={95,95,95})),
                  Line(points=[-40,80;40,80], style(rgbcolor={95,95,95})),
                  Line(points=[60,60;60,-60], style(rgbcolor={95,95,95})),
                  Line(points=[-40,-80;40,-80], style(rgbcolor={95,95,95})),
                  Text(
                    extent=[-58,60;58,20],string="Suction Mainfold", style(
                    rgbcolor={95,95,95},
                    rgbfillColor={255,255,255},
                    fillPattern=1))), Diagram);

                end SuctionMainfold;

                model Compressor
                  parameter Real comp "Capacity";
                  parameter Real Nuvol "Volumetric efficiency";
                  parameter Real Vsl "Total displacement volume";

                  Modelica.Blocks.Interfaces.RealOutput Vcomp
                    annotation (extent=[-10,80;10,100], rotation=90);

                  Modelica.Blocks.Interfaces.BooleanInput ON
                    annotation (extent=[-100,-10;-80,10]);

                  annotation (Icon(
                  Ellipse(
                    extent=[-80,80;80,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-60,52;-20,-76], style(rgbcolor={0,0,0})),
                  Line(points=[60,52;20,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-60,60;60,-60], string="T", style(
                    rgbolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1))),Diagram);

                equation
                  // Eq. (11)
                  Vcomp = if ON then comp*(1/100)*Nuvol*Vsl else 0;
                end Compressor;

                model CompressorRack

                  // Numero de compresores
                  parameter Integer NUM_COMPRESSORS = 2;

                  parameter Real DB "Controller Dead-Band";
                  parameter Real comp "Compressor capacity";
                  parameter Real Nuvol "Compressor volumetric efficiency";
                  parameter Real Vsl "Compressor total displacement volume";
                  parameter Real Kp "PI gain";
                  parameter Real Taoi "PI time constant";

                  // Capacidad de cada compresor, por si quisiera asignarse a cada uno una capacidad diferente
                  parameter Real comps[NUM_COMPRESSORS] = ones(NUM_COMPRESSORS)*comp;

                  // Variables del controlador: error (e), trmino integral (I) y salida (u)
                  Real e;
                  Real I;
                  Real u;
                  Real C[NUM_COMPRESSORS];  // Variable auxiliar
                  parameter Real sampleTime = 60;

                  Compressor compressor[NUM_COMPRESSORS]( each comp=comp,
                    each Nuvol=Nuvol,
                    each Vsl=Vsl);
                  Modelica.Blocks.Interfaces.RealOutput Vcomp
                    annotation (extent=[-10,80;10,100], rotation=90);

                  Modelica.Blocks.Interfaces.RealInput Psuc
                    annotation (extent=[-10,-100;10,-80], rotation=90);

                  Modelica.Blocks.Interfaces.RealInput SetPoint
                    annotation (extent=[-100,-10;-80,10]);

          protected
                  Boolean compON[NUM_COMPRESSORS];

          public
                  Modelica.Blocks.Interfaces.RealOutput numCompON
                                                          annotation (extent=[88,-10;108,10]);
                equation
                // Nmero de compresores que estn a ON
                   numCompON = sum(if compON[i] then 1 else 0 for i in 1:NUM_COMPRESSORS);

                // Controlador
                   when sample(0,sampleTime) then
                      e = if abs(SetPoint-Psuc)>DB then SetPoint-Psuc else 0;
                      I = pre(I)+pre(e)*sampleTime;
                      u = Kp*(e+I/Taoi);
                   end when;

                  C[1] = comps[1]/2;
                  for i in 2:NUM_COMPRESSORS loop
                     C[i] = C[i-1] + comps[i-1]/2 + comps[i]/2;
                  end for;
                  for i in 1:NUM_COMPRESSORS loop
                     compON[i] = u > C[i];
                  end for;
                // Conexin del controlador a los compresores
                  for i in 1:NUM_COMPRESSORS loop
                     compressor[i].ON = compON[i];
                  end for;

                // Suma del flujo total de salida de los compresores
                 Vcomp = sum(compressor[i].Vcomp for i in 1:NUM_COMPRESSORS);

                  annotation (Diagram, Icon(
                  Ellipse(
                    extent=[-72,80;88,-80],
                style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-52,52;-12,-76], style(rgbcolor={0,0,0})),
                  Line(points=[68,52;28,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-52,60;68,-60],
                string="T", style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Ellipse(
                    extent=[-84,80;76,-80],
                    style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-64,52;-24,-76], style(rgbcolor={0,0,0})),
                  Line(points=[56,52;16,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-64,60;56,-60],
                    string="T", style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1))));
                end CompressorRack;

                model CompressorRack_2

                  // Numero de compresores
                  parameter Integer NUM_COMPRESSORS = 2;

                  parameter Real DB "Controller Dead-Band";
                  parameter Real comp "Compressor capacity";
                  parameter Real Nuvol "Compressor volumetric efficiency";
                  parameter Real Vsl "Compressor total displacement volume";
                  parameter Real Kp "PI gain";
                  parameter Real Taoi "PI time constant";

                  // Capacidad de cada compresor, por si quisiera asignarse a cada uno una capacidad diferente
                  parameter Real comps[NUM_COMPRESSORS] = ones(NUM_COMPRESSORS)*comp;

                  // Variables del controlador: error (e), trmino integral (I) y salida (u)
                  Real e;
                  Real I;
                  Real u;
                  Real u1;
                  Real C[NUM_COMPRESSORS];  // Variable auxiliar
                  parameter Real sampleTime = 60;

                  Compressor compressor[NUM_COMPRESSORS]( each comp=comp,
                    each Nuvol=Nuvol,
                    each Vsl=Vsl);
                  Modelica.Blocks.Interfaces.RealOutput Vcomp
                    annotation (extent=[-10,80;10,100], rotation=90);

                  Modelica.Blocks.Interfaces.RealInput Psuc
                    annotation (extent=[-10,-100;10,-80], rotation=90);

                  Modelica.Blocks.Interfaces.RealInput SetPoint
                    annotation (extent=[-100,-10;-80,10]);

          protected
                  Boolean compON[NUM_COMPRESSORS];

          public
                  Modelica.Blocks.Interfaces.RealOutput numCompON
                                                          annotation (extent=[88,-10;108,10]);
                equation
                // Nmero de compresores que estn a ON
                   numCompON = sum(if compON[i] then 1 else 0 for i in 1:NUM_COMPRESSORS);

                // My controller
                   e = if abs(SetPoint-Psuc)>DB then SetPoint - Psuc else 0;
                   der(I) = e/Taoi;
                   u1 = Kp*(e+I);
                   when sample(0,sampleTime) then
                    u = u1;
                   end when;

                  C[1] = comps[1]/2;
                  for i in 2:NUM_COMPRESSORS loop
                     C[i] = C[i-1] + comps[i-1]/2 + comps[i]/2;
                  end for;
                  for i in 1:NUM_COMPRESSORS loop
                     compON[i] = u > C[i];
                  end for;
                // Conexin del controlador a los compresores
                  for i in 1:NUM_COMPRESSORS loop
                     compressor[i].ON = compON[i];
                  end for;

                // Suma del flujo total de salida de los compresores
                 Vcomp = sum(compressor[i].Vcomp for i in 1:NUM_COMPRESSORS);

                  annotation (Diagram, Icon(
                  Ellipse(
                    extent=[-72,80;88,-80],
                style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-52,52;-12,-76], style(rgbcolor={0,0,0})),
                  Line(points=[68,52;28,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-52,60;68,-60],
                string="T", style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Ellipse(
                    extent=[-84,80;76,-80],
                    style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-64,52;-24,-76], style(rgbcolor={0,0,0})),
                  Line(points=[56,52;16,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-64,60;56,-60],
                    string="T", style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1))));
                end CompressorRack_2;

          model ConstraintSat
            parameter Integer NumDisp = 1 "Number of display cases";
            parameter Real MaxTair[NumDisp] = {1}
              "Maximum air temperatures for the displays";
            parameter Real MinTair[NumDisp] = {1}
              "Minimum air temperatures for the displays";

            Real EPsuc;
            Real ETair[NumDisp];

            Modelica.Blocks.Interfaces.RealOutput Ycon
              annotation (extent=[98,-10;118,10]);
            Modelica.Blocks.Interfaces.RealInput Psuc
              annotation (extent=[-140,60;-100,100]);
            Modelica.Blocks.Interfaces.RealInput Tair[NumDisp]
              annotation (extent=[-140,-100;-100,-60]);
           Modelica.Blocks.Interfaces.RealInput MaxPsuc
              annotation (extent=[-140,0;-100,40]);
            Real Yday;
            Real Ynight;
          protected
            Real y;
          equation

          der(y) = EPsuc^2 + (1/NumDisp) * sum(ETair)^2;
          Ycon = if time <= 0 then 0 else y / time;

          when time > 7200 then
          Yday = y/7200;
          end when;

          when terminal() then
          Ynight = y/7200;
          end when;

          EPsuc = if Psuc > MaxPsuc then Psuc - MaxPsuc else 0;
          for i in 1:NumDisp loop
            ETair[i] = if Tair[i] > MaxTair[i] then Tair[i] - MaxTair[i] else
                               if Tair[i] < MinTair[i] then MinTair[i] - Tair[i] else
                                       0;
          end for;

            annotation (Diagram);
          end ConstraintSat;

          model NumberSwitches
            Modelica.Blocks.Interfaces.RealInput NumComp
              annotation (extent=[-140,40;-100,80]);
            Modelica.Blocks.Interfaces.RealInput NumValve
              annotation (extent=[-140,-80;-100,-40]);
            Modelica.Blocks.Interfaces.RealOutput Yswitch
              annotation (extent=[100,-10;120,10]);
            annotation (Diagram(graphics));

          protected
            Real y;
          equation
          der(y) = NumComp + NumValve/100;
          Yswitch = if time <= 0 then 0 else y/time;

          end NumberSwitches;

          model PowerConsup

            Modelica.Blocks.Interfaces.RealInput Fcomp
              annotation (extent=[-140,40;-100,80]);
            Modelica.Blocks.Interfaces.RealOutput Ypow
              annotation (extent=[100,-10;120,10]);
          Modelica.Blocks.Interfaces.RealInput Psuc
              annotation (extent=[-140,-80;-100,-40]);
            annotation (Diagram(graphics));

          protected
            Real powcomp;
            Real Rosuc = 4.6073*Psuc+0.3798;
          //  Real nuis;
          //  Real his;
          //  Real hic;
          //  Real hoc;
            Real ro;
          equation

          //der(powcomp) = Fcomp*Rosuc*(his - hic)/nuis;
          der(powcomp) = Fcomp*Rosuc*ro;
          Ypow = if time <= 0 then 0 else powcomp/time;

          //nuis = (his - hic) / (hoc - hic);
          ro = (0.0265*Psuc^3 - 0.4346*Psuc^2 + 2.4923*Psuc + 1.2189) * 10^5;
          end PowerConsup;

                model system2D2C
                  Real Tair1 = displayCase.displayCase.Tair;
                  Real Tair2 = displayCase1.displayCase.Tair;
                  Real Psuc = suctionMainfold.Psuc;
                  Real Ycon = constraintSat.Ycon;
                  Real Yswitch = numberSwitches.Yswitch;
                  Real Ypow = powerConsup.Ypow;

                  parameter Integer NUM_COMPRESSORS = 2;

                  ControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Tgoodsstart=2,
                    Mrefstart=0)          annotation (extent=[-68,-20;-26,26]);
                  SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[32,-26;84,36]);
                  ControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-68,36;-26,78]);
                  CompressorRack compressorRack2C(
                    NUM_COMPRESSORS = NUM_COMPRESSORS,
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[74,-72;94,-52]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-30,82;-18,94]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[42,38;54,50]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[58,-68;70,-56]);
                  ConstraintSat constraintSat(
                    NumDisp=2,
                    MaxTair={5,5},
                    MinTair={2,2}) annotation (extent=[-10,-44;10,-24]);
                  Modelica.Blocks.Sources.Step MaxPsuc(
                    startTime=7200,
                    height=0.2,
                    offset=1.7)     annotation (extent=[-32,-36;-22,-26]);
                  NumberSwitches numberSwitches annotation (extent=[-10,-68;10,-48]);
                  Modelica.Blocks.Math.Sum sum1(nin=2) annotation (extent=[-34,-68;-24,-58]);
                  PowerConsup powerConsup annotation (extent=[50,-48;70,-28]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-28.1,
                  -15.4; -8,-15.4; -8,1.9; 39.8,1.9],                                                                  style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-28.1,40.2;2,40.2;2,1.9;39.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[84,-53;84,5;76.2,5], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-17.4,88;
                  -12,88; -12,12.2; -28.1,12.2],                                                                     style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-17.4,88;-12,88;-12,65.4;-28.1,65.4], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[54.6,44;58.26,44;58.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[39.8,8.1;
                  13.9,8.1; 13.9,-9.42; -28.1,-9.42],                                                                              style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[39.8,8.1;
                  14,8.1; 14,45.66; -28.1,45.66],                                                                             style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[70.6,-62;75,-62], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[39.8,8.1;32,8.1;32,-76;84,-76;84,-71], style(rgbcolor={0,0,127}));

                  annotation (Diagram(graphics), experiment(StopTime=14400), experimentSetupOutput);
                  connect(displayCase.Tair, constraintSat.Tair[1]) annotation (points=[-70.1,
                  12.2; -88,12.2; -88,-43; -12,-43],                                                                         style(rgbcolor={0,0,127}));
                  connect(displayCase1.Tair, constraintSat.Tair[2]) annotation (points=[-70.1,65.4;-88,65.4;-88,-41;-12,-41], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, constraintSat.Psuc) annotation (points=[39.8,8.1;-16,8.1;-16,-26;-12,-26], style(rgbcolor={0,0,127}));
                  connect(MaxPsuc.y, constraintSat.MaxPsuc) annotation (points=[-21.5,-31;-18,-31;-18,-32;-12,-32], style(rgbcolor={0,0,127}));
                  connect(sum1.u[1], displayCase.valveOpen) annotation (points=[-35,-63.5;-47,-63.5;-47,-22.3], style(rgbcolor={0,0,127}));
                  connect(sum1.y, numberSwitches.NumValve) annotation (points=[-23.5,-63;-15.75,-63;-15.75,-64;-12,-64], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.numCompON, numberSwitches.NumComp) annotation (points=[93.8,-62;94,-62;94,-82;-20,-82;-20,-52;-12,-52], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, powerConsup.Psuc) annotation (points=[39.8,8.1;32,8.1;32,-44;48,-44], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, powerConsup.Fcomp) annotation (points=[84,-53;84,-24;42,-24;42,-32;48,-32], style(rgbcolor={0,0,127}));
                  connect(displayCase1.valveOpen, sum1.u[2]) annotation (points=[-47,33.9;
                  -47,26; -74,26; -74,-62.5; -35,-62.5],                                                                      style(rgbcolor={0,0,127}));
                end system2D2C;

                model system2D2CDEVSaircontrol
                  Real Tair1 = displayCase.displayCase.Tair;
                  Real Tair2 = displayCase1.displayCase.Tair;
                  Real Psuc = suctionMainfold.Psuc;
                  Real Ycon = constraintSat.Ycon;
                  Real Yswitch = numberSwitches.Yswitch;
                  Real Ypow = powerConsup.Ypow;

                  parameter Integer NUM_COMPRESSORS = 2;

                  DEVSControl.DEVSControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Tgoodsstart=2,
                    Mrefstart=0)          annotation (extent=[-68,-20;-26,26]);
                  SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[32,-26;84,36]);
                  DEVSControl.DEVSControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-70,36;-28,78]);
                  CompressorRack compressorRack2C(
                    NUM_COMPRESSORS = NUM_COMPRESSORS,
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[74,-72;94,-52]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-30,82;-18,94]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[42,38;54,50]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[58,-68;70,-56]);
                  ConstraintSat constraintSat(
                    NumDisp=2,
                    MaxTair={5,5},
                    MinTair={2,2}) annotation (extent=[-10,-44;10,-24]);
                  Modelica.Blocks.Sources.Step MaxPsuc(
                    startTime=7200,
                    height=0.2,
                    offset=1.7)     annotation (extent=[-32,-36;-22,-26]);
                  NumberSwitches numberSwitches annotation (extent=[-10,-68;10,-48]);
                  Modelica.Blocks.Math.Sum sum1(nin=2) annotation (extent=[-34,-68;-24,-58]);
                  PowerConsup powerConsup annotation (extent=[50,-48;70,-28]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-28.1,
                  -15.4; -8,-15.4; -8,1.9; 39.8,1.9],                                                                  style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-30.1,40.2;2,40.2;2,1.9;39.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[84,-53;84,5;76.2,5], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-17.4,88;
                  -12,88; -12,12.2; -28.1,12.2],                                                                     style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-17.4,88;-12,88;-12,65.4;-30.1,65.4], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[54.6,44;58.26,44;58.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[39.8,8.1;13.9,8.1;13.9,-10.34;-28.1,-10.34], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[39.8,8.1;14,8.1;14,44.82;-30.1,44.82], style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[70.6,-62;75,-62], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[39.8,8.1;32,8.1;32,-76;84,-76;84,-71], style(rgbcolor={0,0,127}));

                  annotation (Diagram(graphics),    experiment(StopTime=14400),    experimentSetupOutput);
                  connect(displayCase.Tair, constraintSat.Tair[1]) annotation (points=[-70.1,
                  12.2; -88,12.2; -88,-43; -12,-43],                                                                         style(rgbcolor={0,0,127}));
                  connect(displayCase1.Tair, constraintSat.Tair[2]) annotation (points=[-72.1,65.4;-88,65.4;-88,-41;-12,-41], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, constraintSat.Psuc) annotation (points=[39.8,8.1;-16,8.1;-16,-26;-12,-26], style(rgbcolor={0,0,127}));
                  connect(MaxPsuc.y, constraintSat.MaxPsuc) annotation (points=[-21.5,-31;-18,-31;-18,-32;-12,-32], style(rgbcolor={0,0,127}));
                  connect(sum1.u[1], displayCase.valveOpen) annotation (points=[-35,-63.5;-47,-63.5;-47,-22.3], style(rgbcolor={0,0,127}));
                  connect(sum1.y, numberSwitches.NumValve) annotation (points=[-23.5,-63;-15.75,-63;-15.75,-64;-12,-64], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.numCompON, numberSwitches.NumComp) annotation (points=[93.8,-62;94,-62;94,-82;-20,-82;-20,-52;-12,-52], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, powerConsup.Psuc) annotation (points=[39.8,8.1;32,8.1;32,-44;48,-44], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, powerConsup.Fcomp) annotation (points=[84,-53;84,-24;42,-24;42,-32;48,-32], style(rgbcolor={0,0,127}));
                  connect(displayCase1.valveOpen, sum1.u[2]) annotation (points=[-49,33.9;
                  -49,26; -74,26; -74,-62.5; -35,-62.5],                                                                      style(rgbcolor={0,0,127}));
                end system2D2CDEVSaircontrol;

                model system2D2C_2
                  Real Tair1 = displayCase.displayCase.Tair;
                  Real Tair2 = displayCase1.displayCase.Tair;
                  Real Psuc = suctionMainfold.Psuc;
                  Real Ycon = constraintSat.Ycon;
                  Real Yswitch = numberSwitches.Yswitch;
                  Real Ypow = powerConsup.Ypow;

                  parameter Integer NUM_COMPRESSORS = 2;

                  ControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Tgoodsstart=2,
                    Mrefstart=0)          annotation (extent=[-68,-20;-26,26]);
                  SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[32,-26;84,36]);
                  ControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-68,36;-26,78]);
                  CompressorRack_2 compressorRack2C(
                    NUM_COMPRESSORS = NUM_COMPRESSORS,
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[74,-72;94,-52]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-30,82;-18,94]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[42,38;54,50]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[58,-68;70,-56]);
                  ConstraintSat constraintSat(
                    NumDisp=2,
                    MaxTair={5,5},
                    MinTair={2,2}) annotation (extent=[-10,-44;10,-24]);
                  Modelica.Blocks.Sources.Step MaxPsuc(
                    startTime=7200,
                    height=0.2,
                    offset=1.7)     annotation (extent=[-32,-36;-22,-26]);
                  NumberSwitches numberSwitches annotation (extent=[-10,-68;10,-48]);
                  Modelica.Blocks.Math.Sum sum1(nin=2) annotation (extent=[-34,-68;-24,-58]);
                  PowerConsup powerConsup annotation (extent=[50,-48;70,-28]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-28.1,
                  -15.4; -8,-15.4; -8,1.9; 39.8,1.9],                                                                  style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-28.1,40.2;2,40.2;2,1.9;39.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[84,-53;84,5;76.2,5], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-17.4,88;
                  -12,88; -12,12.2; -28.1,12.2],                                                                     style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-17.4,88;-12,88;-12,65.4;-28.1,65.4], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[54.6,44;58.26,44;58.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[39.8,8.1;
                  13.9,8.1; 13.9,-9.42; -28.1,-9.42],                                                                              style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[39.8,8.1;
                  14,8.1; 14,45.66; -28.1,45.66],                                                                             style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[70.6,-62;75,-62], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[39.8,8.1;32,8.1;32,-76;84,-76;84,-71], style(rgbcolor={0,0,127}));

                  annotation (Diagram(graphics),  experiment(StopTime=14400),  experimentSetupOutput);
                  connect(displayCase.Tair, constraintSat.Tair[1]) annotation (points=[-70.1,
                  12.2; -88,12.2; -88,-43; -12,-43],                                                                         style(rgbcolor={0,0,127}));
                  connect(displayCase1.Tair, constraintSat.Tair[2]) annotation (points=[-70.1,65.4;-88,65.4;-88,-41;-12,-41], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, constraintSat.Psuc) annotation (points=[39.8,8.1;-16,8.1;-16,-26;-12,-26], style(rgbcolor={0,0,127}));
                  connect(MaxPsuc.y, constraintSat.MaxPsuc) annotation (points=[-21.5,-31;-18,-31;-18,-32;-12,-32], style(rgbcolor={0,0,127}));
                  connect(sum1.u[1], displayCase.valveOpen) annotation (points=[-35,-63.5;-47,-63.5;-47,-22.3], style(rgbcolor={0,0,127}));
                  connect(sum1.y, numberSwitches.NumValve) annotation (points=[-23.5,-63;-15.75,-63;-15.75,-64;-12,-64], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.numCompON, numberSwitches.NumComp) annotation (points=[93.8,-62;94,-62;94,-82;-20,-82;-20,-52;-12,-52], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, powerConsup.Psuc) annotation (points=[39.8,8.1;32,8.1;32,-44;48,-44], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, powerConsup.Fcomp) annotation (points=[84,-53;84,-24;42,-24;42,-32;48,-32], style(rgbcolor={0,0,127}));
                  connect(displayCase1.valveOpen, sum1.u[2]) annotation (points=[-47,33.9;
                  -47,26; -74,26; -74,-62.5; -35,-62.5],                                                                      style(rgbcolor={0,0,127}));
                end system2D2C_2;

                model system2D2C_2DEVSaircontrol
                  Real Tair1 = displayCase.displayCase.Tair;
                  Real Tair2 = displayCase1.displayCase.Tair;
                  Real Psuc = suctionMainfold.Psuc;
                  Real Ycon = constraintSat.Ycon;
                  Real Yswitch = numberSwitches.Yswitch;
                  Real Ypow = powerConsup.Ypow;

                  parameter Integer NUM_COMPRESSORS = 2;

                  DEVSControl.DEVSControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Tgoodsstart=2,
                    Mrefstart=0)          annotation (extent=[-68,-20;-26,26]);
                  SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[32,-26;84,36]);
                  DEVSControl.DEVSControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-70,36;-28,78]);
                  CompressorRack_2 compressorRack2C(
                    NUM_COMPRESSORS = NUM_COMPRESSORS,
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[74,-72;94,-52]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-30,82;-18,94]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[42,38;54,50]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[58,-68;70,-56]);
                  ConstraintSat constraintSat(
                    NumDisp=2,
                    MaxTair={5,5},
                    MinTair={2,2}) annotation (extent=[-10,-44;10,-24]);
                  Modelica.Blocks.Sources.Step MaxPsuc(
                    startTime=7200,
                    height=0.2,
                    offset=1.7)     annotation (extent=[-32,-36;-22,-26]);
                  NumberSwitches numberSwitches annotation (extent=[-10,-68;10,-48]);
                  Modelica.Blocks.Math.Sum sum1(nin=2) annotation (extent=[-34,-68;-24,-58]);
                  PowerConsup powerConsup annotation (extent=[50,-48;70,-28]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-28.1,
                  -15.4; -8,-15.4; -8,1.9; 39.8,1.9],                                                                  style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-30.1,40.2;2,40.2;2,1.9;39.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[84,-53;84,5;76.2,5], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-17.4,88;
                  -12,88; -12,12.2; -28.1,12.2],                                                                     style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-17.4,88;-12,88;-12,65.4;-30.1,65.4], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[54.6,44;58.26,44;58.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[39.8,8.1;13.9,8.1;13.9,-10.34;-28.1,-10.34], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[39.8,8.1;14,8.1;14,44.82;-30.1,44.82], style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[70.6,-62;75,-62], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[39.8,8.1;32,8.1;32,-76;84,-76;84,-71], style(rgbcolor={0,0,127}));

                  annotation (Diagram(graphics),  experiment(StopTime=14400),  experimentSetupOutput);
                  connect(displayCase.Tair, constraintSat.Tair[1]) annotation (points=[-70.1,
                  12.2; -88,12.2; -88,-43; -12,-43],                                                                         style(rgbcolor={0,0,127}));

                  connect(displayCase1.Tair, constraintSat.Tair[2]) annotation (points=[-72.1,65.4;-88,65.4;-88,-41;-12,-41], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, constraintSat.Psuc) annotation (points=[39.8,8.1;-16,8.1;-16,-26;-12,-26], style(rgbcolor={0,0,127}));
                  connect(MaxPsuc.y, constraintSat.MaxPsuc) annotation (points=[-21.5,-31;-18,-31;-18,-32;-12,-32], style(rgbcolor={0,0,127}));
                  connect(sum1.u[1], displayCase.valveOpen) annotation (points=[-35,-63.5;-47,-63.5;-47,-22.3], style(rgbcolor={0,0,127}));
                  connect(sum1.y, numberSwitches.NumValve) annotation (points=[-23.5,-63;-15.75,-63;-15.75,-64;-12,-64], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.numCompON, numberSwitches.NumComp) annotation (points=[93.8,-62;94,-62;94,-82;-20,-82;-20,-52;-12,-52], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, powerConsup.Psuc) annotation (points=[39.8,8.1;32,8.1;32,-44;48,-44], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, powerConsup.Fcomp) annotation (points=[84,-53;84,-24;42,-24;42,-32;48,-32], style(rgbcolor={0,0,127}));
                  connect(displayCase1.valveOpen, sum1.u[2]) annotation (points=[-49,33.9;
                  -49,26; -74,26; -74,-62.5; -35,-62.5],                                                                      style(rgbcolor={0,0,127}));
                end system2D2C_2DEVSaircontrol;

                model system2D2C_CEP
                  Real Tair1 = displayCase.displayCase.Tair;
                  Real Tair2 = displayCase1.displayCase.Tair;
                  Real Psuc = suctionMainfold.Psuc;

                  parameter Integer NUM_COMPRESSORS = 2;

                  ControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Mrefstart=0,
                    Tgoodsstart=4,
                    UAwallrefmax=900,
                    Mrefmax=0.6)
                           annotation (extent=[-80,-4;-20,56]);
                  SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[32,-26;84,36]);
                  ControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Mrefstart=1,
                    Tgoodsstart=4,
                    UAwallrefmax=900,
                    Mrefmax=0.6)
                           annotation (extent=[-80,-62;-20,-2]);
                  CompressorRack compressorRack2C(
                    NUM_COMPRESSORS = NUM_COMPRESSORS,
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[78,-60;98,-40]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-66,70;-46,90]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[34,70;54,90]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[54,-60;74,-40]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-23,2;-12,2;-12,1.9;39.8,1.9], style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-23,-56;-12,-56;-12,1.9;39.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[88,-41;88,5;76.2,5], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-45,80;-16,80;-16,38;-23,38], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-45,80;-16,80;-16,-20;-23,-20], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[55,80;58.26,80;58.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[39.8,8.1;
                  9.9,8.1; 9.9,9.8; -23,9.8],                                                                            style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[39.8,8.1;
                  10,8.1; 10,-48.2; -23,-48.2],                                                                             style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[75,-50;79,-50], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[39.8,8.1;32,8.1;32,-70;88,-70;88,-59], style(rgbcolor={0,0,127}));

                  annotation (Diagram(graphics),  experiment(StopTime=14400),  experimentSetupOutput);

                end system2D2C_CEP;
        end Continuous;

        package Continuous2

        annotation(uses(Modelica(version="2.2.1")));

        model DisplayCase
                  parameter Real UAgoodsair
              "Heat transfer coeficient between goods and air";
                  parameter Real Mgoods "Mass of the goods";
                  parameter Real CPgoods "Heat capacity of goods";
                  parameter Real UAairwall
              "Heat transfer coeficient between air and wall";
                  parameter Real Mwall "Mass of the wall";
                  parameter Real CPwall "Heat capacity of wall";
                  parameter Real UAwallrefmax
              "Maximum heat transfer coeficient between wall and refrigerant";
                  parameter Real Mrefmax "Maximum mass of refrigerant";
                  parameter Real Mair "Mass of the air";
                  parameter Real CPair "Heat capacity of air";
                  parameter Real Taufill "Refrigerator filling time";

                  parameter Real Tgoods_initial;
                  parameter Real Twall_initial;
                  parameter Real Tair_initial;
                  parameter Real Mref_initial;

                  Real Tgoods( start = Tgoods_initial)
              "Temperature of the goods in the display case";
                  Real Twall( start = Twall_initial)
              "Temperature of the evaporator wall";
                  Real Tair( start = Tair_initial)
              "Temperature of the air in the display case";
                  Real Mref( start = Mref_initial)
              "Mass of liquified refrigerante in the evaporator";

                  Modelica.Blocks.Interfaces.RealInput Psuc "Suction Pressure"
         annotation (extent=[80,-60;100,-40], rotation=180);
                  Modelica.Blocks.Interfaces.BooleanInput valveOpen
              "State of the valve (open,closed)"
                                                annotation (extent=[-48,-100;-28,-80], rotation=90);
                  Modelica.Blocks.Interfaces.RealInput Qairload
              "External air disturbance"
                    annotation (extent=[80,24;100,44], rotation=180);

                  flow Modelica.Blocks.Interfaces.RealOutput m
        annotation (extent=[80,-80;100,-60]);
          protected
                Real Qgoodsair;
                Real Qairwall;
                Real Qe "heat";
                Real Te "refrigerant evaporation temperature";
                Real UAwallref;
                Real IncHlg;
          public
                  Modelica.Blocks.Interfaces.RealOutput AirTemp
         annotation (extent=[-100,-10;-80,10], rotation=180);
        equation
                AirTemp = Tair;
                // Eq. (1)
                der(Tgoods) = - Qgoodsair / (Mgoods * CPgoods);
                // Eq. (2)
                der(Twall) = (Qairwall - Qe) / (Mwall * CPwall);
                // Eq. (3)
                der(Tair) = (Qgoodsair + Qairload - Qairwall) / (Mair*CPair);
                // Eq. (4)
                Qgoodsair = UAgoodsair * (Tgoods - Tair);
                // Eq. (5)
                Qairwall = UAairwall * (Tair - Twall);
                // Eq. (6)
                Qe = UAwallref * (Twall - Te);
                // Eq. (7)
                UAwallref = UAwallrefmax * Mref / Mrefmax;
                // Eq. (8)
                der(Mref) = if valveOpen then
                                 (Mrefmax - Mref) / Taufill else
                                 if Mref > 0 then - Qe / IncHlg else
                                                  0;
                // Eq. (9)
                m = - Qe / IncHlg;
                // Appendix
                Te = -4.3544*Psuc^2  + 29.2240*Psuc - 51.2005;
                IncHlg = (0.0217*Psuc^2 - 0.1704*Psuc +2.2988)*10^5;

         annotation (Diagram, Icon(Rectangle(
                    extent=[-80,80;80,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                    Text(
                    extent=[-78,78;78,0], string="DisplayCase", style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1))));

        end DisplayCase;

                model AirController
                  parameter Real Tairmax
              "Max temperature threshold for the controller";
                  parameter Real Tairmin
              "Min temperature threshold for the controller";

                  Modelica.Blocks.Interfaces.RealInput Tair
                 annotation (extent=[-80,-10;-60,10]);
                  Modelica.Blocks.Interfaces.BooleanOutput ValveOpen
                annotation (extent=[60,-10;80,10]);

                   parameter Real sampleTime;

                 annotation (Diagram, Icon(
                    Rectangle(
                    extent=[-60,40;60,-40], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                    Text(
                    extent=[-58,20;58,-20], string="AirTemp Control", style(
                    rgbolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1))));

                equation
                   when sample(0,sampleTime) then
                      ValveOpen = if Tair > Tairmax then  true else
                                       if Tair < Tairmin then false else
                                       pre(ValveOpen);
                   end when;

                end AirController;

                model ControlledDisplayCase
                  parameter Real UAgoodsair
              "Heat transfer coeficient between goods and air";
                  parameter Real Mgoods "Mass of the goods";
                  parameter Real CPgoods "Heat capacity of goods";
                  parameter Real UAairwall
              "Heat transfer coeficient between air and wall";
                  parameter Real Mwall "Mass of the wall";
                  parameter Real CPwall "Heat capacity of wall";
                  parameter Real UAwallrefmax
              "Maximum heat transfer coeficient between wall and refrigerant";
                  parameter Real Mrefmax "Maximum mass of refrigerant";
                  parameter Real Mair "Mass of the air";
                  parameter Real CPair "Heat capacity of air";
                  parameter Real Taufill "Refrigerator filling time";
                  parameter Real Tairmax
              "Max temperature threshold for the controller";
                  parameter Real Tairmin
              "Min temperature threshold for the controller";
                  // Valores iniciales
                  parameter Real Twallstart;
                  parameter Real Tairstart;
                  parameter Real Tgoodsstart;
                  parameter Real Mrefstart;
                  // Muestreo controlador
                  parameter Real sampleTime = 1;

                  DisplayCase displayCase(
                    UAgoodsair=UAgoodsair,
                    Mgoods=Mgoods,
                    CPgoods=CPgoods,
                    UAairwall=UAairwall,
                    Mwall=Mwall,
                    CPwall=CPwall,
                    UAwallrefmax=UAwallrefmax,
                    Mrefmax=Mrefmax,
                    Mair=Mair,
                    CPair=CPair,
                    Taufill=Taufill,
                    Tgoods_initial=Tgoodsstart,
                    Twall_initial=Twallstart,
                    Tair_initial=Tairstart,
                    Mref_initial=Mrefstart)
                 annotation (extent=[-74,-80;66,100]);
                  AirController airController(
                      Tairmax=Tairmax, Tairmin=Tairmin, sampleTime=sampleTime)
                 annotation (extent=[-96,-108;-50,-70]);
                annotation (Diagram, Icon(Bitmap(extent=[-100,100;80,-100], name="Figs/displaycase.png")));
                Modelica.Blocks.Interfaces.RealInput Qairload
                 annotation (extent=[80,30;100,50], rotation=180);
                  Modelica.Blocks.Interfaces.RealInput Psuc
                 annotation (extent=[80,-68;100,-40], rotation=180);
                  flow Modelica.Blocks.Interfaces.RealOutput m
                annotation (extent=[80,-90;100,-70]);
                equation
                  connect(displayCase.AirTemp, airController.Tair) annotation (points=[-67,10;-94,10;-94,-89;-89.1,-89], style(rgbcolor={0,0,127}));
                  connect(Qairload, displayCase.Qairload) annotation (points=[90,40;
                  73.1,40; 73.1,40.6; 59,40.6],                                                                        style(rgbcolor={0,0,127}));
                connect(displayCase.Psuc, Psuc) annotation (points=[59,-35;
                  79.5,-35; 79.5,-54; 90,-54],                                                                   style(rgbcolor={0,0,127}));
                  connect(displayCase.m, m) annotation (points=[59,-53; 72.5,
                  -53; 72.5,-80; 90,-80],                                                         style(rgbcolor={0,0,127}));
                  connect(airController.ValveOpen, displayCase.valveOpen) annotation (points=[-56.9,
                  -89; -30.45,-89; -30.45,-71; -30.6,-71],                                                                                style(rgbcolor={255,0,255}));
                end ControlledDisplayCase;

                model SuctionMainfold
                  parameter Real Vsuc "Total volume";
                  parameter Real Psucstart "Initial value of Psuc";

                  Modelica.Blocks.Interfaces.RealOutput Psuc( start = Psucstart)
                 annotation (extent=[-80,0;-60,20], rotation=180);

                  flow Modelica.Blocks.Interfaces.RealInput m
                  annotation (extent=[-80,-20;-60,0]);

                  Modelica.Blocks.Interfaces.RealInput Vcomp
                    annotation (extent=[60,-10;80,10], rotation=180);

          protected
                  Real Rosuc "density of the vapor";
                  Real dersuc;
          public
                  Modelica.Blocks.Interfaces.RealInput Mrefconst
                 annotation (extent=[-9,79;11,101], rotation=270);
                equation
                  // Eq. (10)
                  der(Psuc) = (m + Mrefconst - Vcomp*Rosuc) / (Vsuc*dersuc);
                  // Appendix
                  Rosuc = 4.6073*Psuc+0.3798;
                  dersuc = -0.0329*Psuc^3 + 0.2161*Psuc^2 -0.4742*Psuc + 5.4817;

                 annotation (Icon(
                  Ellipse(extent=[-60,80;-20,40], style(rgbcolor={95,95,95})),
                  Ellipse(extent=[20,80;60,40], style(rgbcolor={95,95,95})),
                  Ellipse(extent=[-60,-40;-20,-80], style(rgbcolor={95,95,95})),
                  Ellipse(extent=[20,-40;60,-80], style(rgbcolor={95,95,95})),
                  Rectangle(
                    extent=[-40,80;40,-80], style(
                    rgbcolor={255,255,255},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Rectangle(
                    extent=[-60,60;60,-60], style(
                    rgbcolor={255,255,255},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-60,60;-60,-60], style(rgbcolor={95,95,95})),
                  Line(points=[-40,80;40,80], style(rgbcolor={95,95,95})),
                  Line(points=[60,60;60,-60], style(rgbcolor={95,95,95})),
                  Line(points=[-40,-80;40,-80], style(rgbcolor={95,95,95})),
                  Text(
                    extent=[-58,60;58,20],string="Suction Mainfold", style(
                    rgbcolor={95,95,95},
                    rgbfillColor={255,255,255},
                    fillPattern=1))), Diagram);

                end SuctionMainfold;

                model Compressor
                  parameter Real comp "Capacity";
                  parameter Real Nuvol "Volumetric efficiency";
                  parameter Real Vsl "Total displacement volume";

                  Modelica.Blocks.Interfaces.RealOutput Vcomp
                 annotation (extent=[-10,80;10,100], rotation=90);

                  Modelica.Blocks.Interfaces.BooleanInput ON
                 annotation (extent=[-100,-10;-80,10]);

                  annotation (Icon(
                  Ellipse(
                    extent=[-80,80;80,-80], style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-60,52;-20,-76], style(rgbcolor={0,0,0})),
                  Line(points=[60,52;20,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-60,60;60,-60], string="T", style(
                    rgbolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1))),Diagram);

                equation
                  // Eq. (11)
                  Vcomp = if ON then comp*(1/100)*Nuvol*Vsl else 0;
                end Compressor;

                model CompressorRack

                  // Numero de compresores
                  parameter Integer NUM_COMPRESSORS = 2;

                  parameter Real DB "Controller Dead-Band";
                  parameter Real comp "Compressor capacity";
                  parameter Real Nuvol "Compressor volumetric efficiency";
                  parameter Real Vsl "Compressor total displacement volume";
                  parameter Real Kp "PI gain";
                  parameter Real Taoi "PI time constant";

                  // Capacidad de cada compresor, por si quisiera asignarse a cada uno una capacidad diferente
                  parameter Real comps[NUM_COMPRESSORS] = ones(NUM_COMPRESSORS)*comp;

                  // Variables del controlador: error (e), trmino integral (I) y salida (u)
                  Real e;
                  Real I;
                  Real u;
                  Real C[NUM_COMPRESSORS];  // Variable auxiliar
                  parameter Real sampleTime = 60;

                  Compressor compressor[NUM_COMPRESSORS]( each comp=comp,
                    each Nuvol=Nuvol,
                    each Vsl=Vsl);
                  Modelica.Blocks.Interfaces.RealOutput Vcomp
                annotation (extent=[-10,80;10,100], rotation=90);

                  Modelica.Blocks.Interfaces.RealInput Psuc
                   annotation (extent=[-10,-100;10,-80], rotation=90);

                  Modelica.Blocks.Interfaces.RealInput SetPoint
                 annotation (extent=[-100,-10;-80,10]);

          protected
                  Boolean compON[NUM_COMPRESSORS];

          public
                  Modelica.Blocks.Interfaces.RealOutput numCompON
                 annotation (extent=[88,-10;108,10]);

                equation
                // Nmero de compresores que estn a ON
                   numCompON = sum(if compON[i] then 1 else 0 for i in 1:NUM_COMPRESSORS);

                // Controlador

                // ---------------------------
                // Modificado
                //   when sample(0,sampleTime) then
                //      e = if abs(SetPoint-Psuc)>DB then SetPoint-Psuc else 0;
                //      I = pre(I)+pre(e)*sampleTime;
                //      u = Kp*(e+I/Taoi);
                //   end when;
                   e = if abs(SetPoint-Psuc)>DB then SetPoint-Psuc else 0;
                   der(I) =  (Kp/Taoi) * e;
                   u = Kp * e + I;
                   when sample(0,sampleTime) then
                      // Saturacin de los errores
                      reinit(I,
                             if abs(SetPoint-Psuc)<DB and pre(I)<0 then
                                  0 else
                                  if abs(SetPoint-Psuc)<DB and pre(I)>100 then
                                       100 else
                                       pre(I));
                      // Activacin de los compresores
                     for i in 1:NUM_COMPRESSORS loop
                        compON[i] = u > C[i];
                     end for;
                   end when;
                // --------------------------------

                  C[1] = comps[1]/2;
                  for i in 2:NUM_COMPRESSORS loop
                     C[i] = C[i-1] + comps[i-1]/2 + comps[i]/2;
                  end for;
                // Movido dentro del bloque when
                //  for i in 1:NUM_COMPRESSORS loop
                //     compON[i] = u > C[i];
                //  end for;
                // Conexin del controlador a los compresores
                  for i in 1:NUM_COMPRESSORS loop
                     compressor[i].ON = compON[i];
                  end for;
                // Suma del flujo total de salida de los compresores
                 Vcomp = sum(compressor[i].Vcomp for i in 1:NUM_COMPRESSORS);

                 annotation (Diagram, Icon(
                  Ellipse(
                    extent=[-72,80;88,-80],
                style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-52,52;-12,-76], style(rgbcolor={0,0,0})),
                  Line(points=[68,52;28,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-52,60;68,-60],
                string="T", style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Ellipse(
                    extent=[-84,80;76,-80],
                    style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1)),
                  Line(points=[-64,52;-24,-76], style(rgbcolor={0,0,0})),
                  Line(points=[56,52;16,-76], style(rgbcolor={0,0,0})),
                  Text(
                    extent=[-64,60;56,-60],
                    string="T", style(
                    rgbcolor={0,0,0},
                    rgbfillColor={255,255,255},
                    fillPattern=1))));

                end CompressorRack;

                model system2D2C

                  parameter Integer NUM_COMPRESSORS = 2;

                  ControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Tgoodsstart=2,
                    Mrefstart=0)          annotation (extent=[-80,-4;-20,56]);
                  SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[34,-26;86,36]);
                  ControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-80,-60;-20,0]);
                  CompressorRack compressorRack2C(
                    NUM_COMPRESSORS = NUM_COMPRESSORS,
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[78,-58;98,-38]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-66,70;-46,90]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[34,70;54,90]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[54,-60;74,-40]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-23,2;-12,2;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-23,-54;-12,-54;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[88,-39;88,5;78.2,5], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-45,80;-16,80;-16,38;-23,38], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-45,80;-16,80;-16,-18;-23,-18], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[55,80;60.26,80;60.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[41.8,8.1;
                  9.9,8.1; 9.9,9.8; -23,9.8],                                                                            style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[41.8,8.1;
                  10,8.1; 10,-46.2; -23,-46.2],                                                                             style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[75,-50;78,-50;78,-48;79,-48], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[41.8,8.1;32,8.1;32,-70;88,-70;88,-57], style(rgbcolor={0,0,127}));

                  annotation (Diagram(graphics), experiment(StopTime=14400), experimentSetupOutput);

                end system2D2C;

                model system2D2CDEVSair
                  Real Tair1 = displayCase.displayCase.Tair;
                  Real Tair2 = displayCase1.displayCase.Tair;
                  Real Psuc = suctionMainfold.Psuc;

                  parameter Integer NUM_COMPRESSORS = 2;

                  DEVSControl.DEVSControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Tgoodsstart=2,
                    Mrefstart=0)          annotation (extent=[-80,20;-40,60]);
                  SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[34,-26;86,36]);
                  DEVSControl.DEVSControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-80,-28;-40,12]);
                  CompressorRack compressorRack2C(
                    NUM_COMPRESSORS = NUM_COMPRESSORS,
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[78,-58;98,-38]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-66,70;-46,90]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[34,70;54,90]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[54,-60;74,-40]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-42,24;-12,24;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-42,-24;-12,-24;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[88,-39;88,5;78.2,5], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-45,80;-16,80;-16,48;-42,48], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-45,80;-16,80;-16,0;-42,0], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[55,80;60.26,80;60.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[41.8,8.1;9.9,8.1;9.9,28.4;-42,28.4], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[41.8,8.1;10,8.1;10,-19.6;-42,-19.6], style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[75,-50;78,-50;78,-48;79,-48], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[41.8,8.1;32,8.1;32,-70;88,-70;88,-57], style(rgbcolor={0,0,127}));

                  annotation (Diagram(graphics),
                    experiment(StopTime=14400),
                    experimentSetupOutput);

                end system2D2CDEVSair;

                model system2D2C_ECO

                  parameter Integer NUM_COMPRESSORS = 2;

                  ControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1,
                    Tairstart=5.1)
                             annotation (extent=[-80,-4;-20,56]);
                  SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[34,-26;86,36]);
                  ControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tgoodsstart=2,
                    Tairstart=2.5,
                    Mrefstart=0.5)
                             annotation (extent=[-80,-60;-20,0]);
                  CompressorRack compressorRack2C(
                    NUM_COMPRESSORS = NUM_COMPRESSORS,
                    DB=0.2,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    comp=100/2,
                    Vsl=0.08)
                        annotation (extent=[78,-60;98,-40]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-66,70;-46,90]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[34,70;54,90]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[54,-60;74,-40]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-23,2;-12,2;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-23,-54;-12,-54;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[88,-41;88,5;78.2,5], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-45,80;-16,80;-16,38;-23,38], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-45,80;-16,80;-16,-18;-23,-18], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[55,80;60.26,80;60.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[41.8,8.1;
                  9.9,8.1; 9.9,9.8; -23,9.8],                                                                            style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[41.8,8.1;
                  10,8.1; 10,-46.2; -23,-46.2],                                                                             style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[75,-50;79,-50], style(rgbcolor={0,0,127}));
                 connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[41.8,8.1;32,8.1;32,-70;88,-70;88,-59], style(rgbcolor={0,0,127}));

                  annotation (Diagram(graphics), experiment(StopTime=12000), experimentSetupOutput);

                end system2D2C_ECO;

                model system3D3CDEVSair
                  Real Tair1 = displayCase.displayCase.Tair;
                  Real Tair2 = displayCase1.displayCase.Tair;
                  Real Psuc = suctionMainfold.Psuc;

                  parameter Integer NUM_COMPRESSORS = 3;

                  DEVSControl.DEVSControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Tgoodsstart=2,
                    Mrefstart=0)          annotation (extent=[-80,12;-40,52]);
                  SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[34,-26;86,36]);
                  DEVSControl.DEVSControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-80,-32;-40,8]);
                  CompressorRack compressorRack2C(
                    NUM_COMPRESSORS = NUM_COMPRESSORS,
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[78,-58;98,-38]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-66,70;-46,90]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[34,70;54,90]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[54,-60;74,-40]);
                  DEVSControl.DEVSControlledDisplayCase displayCase2(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-80,-80;-40,-40]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-42,16;-12,16;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-42,-28;-12,-28;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[88,-39;88,5;78.2,5], style(rgbcolor={0,0,127}));
                 connect(AirLoad.y, displayCase.Qairload) annotation (points=[-45,80;-16,80;-16,40;-42,40], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-45,80;-16,80;-16,-4;-42,-4], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[55,80;60.26,80;60.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[41.8,8.1;9.9,8.1;9.9,20.4;-42,20.4], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[41.8,8.1;10,8.1;10,-23.6;-42,-23.6], style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[75,-50;78,-50;78,-48;79,-48], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[41.8,8.1;32,8.1;32,-70;88,-70;88,-57], style(rgbcolor={0,0,127}));

                  annotation (Diagram(graphics),  experiment(StopTime=14400),  experimentSetupOutput);
                  connect(suctionMainfold.Psuc, displayCase2.Psuc) annotation (points=[41.8,8.1;10,8.1;10,-71.6;-42,-71.6], style(rgbcolor={0,0,127}));
                  connect(displayCase2.m, suctionMainfold.m) annotation (points=[-42,-76;-12,-76;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase2.Qairload) annotation (points=[-45,80;-16,80;-16,-52;-42,-52], style(rgbcolor={0,0,127}));
                end system3D3CDEVSair;

          model ConstraintSat
            parameter Integer NumDisp = 1 "Number of display cases";
            parameter Real MaxTair[NumDisp] = {1}
              "Maximum air temperatures for the displays";
            parameter Real MinTair[NumDisp] = {1}
              "Minimum air temperatures for the displays";

            Real EPsuc;
            Real ETair[NumDisp];

            Modelica.Blocks.Interfaces.RealOutput Ycon
              annotation (extent=[98,-10;118,10]);
            Modelica.Blocks.Interfaces.RealInput Psuc
              annotation (extent=[-140,60;-100,100]);
            Modelica.Blocks.Interfaces.RealInput Tair[NumDisp]
              annotation (extent=[-140,-100;-100,-60]);
           Modelica.Blocks.Interfaces.RealInput MaxPsuc
              annotation (extent=[-140,0;-100,40]);
            Real Yday;
            Real Ynight;
          protected
            Real y;
          equation

          der(y) = EPsuc^2 + (1/NumDisp) * sum(ETair)^2;
          Ycon = if time <= 0 then 0 else y / time;

          when time > 7200 then
          Yday = y/7200;
          end when;

          when terminal() then
          Ynight = y/7200;
          end when;

          EPsuc = if Psuc > MaxPsuc then Psuc - MaxPsuc else 0;
          for i in 1:NumDisp loop
            ETair[i] = if Tair[i] > MaxTair[i] then Tair[i] - MaxTair[i] else
                               if Tair[i] < MinTair[i] then MinTair[i] - Tair[i] else
                                       0;
          end for;

            annotation (Diagram(graphics));
          end ConstraintSat;

          model NumberSwitches
            Modelica.Blocks.Interfaces.RealInput NumComp
              annotation (extent=[-140,40;-100,80]);
            Modelica.Blocks.Interfaces.RealInput NumValve
              annotation (extent=[-140,-80;-100,-40]);
            Modelica.Blocks.Interfaces.RealOutput Yswitch
              annotation (extent=[100,-10;120,10]);
            annotation (Diagram(graphics));

          protected
            Real y;
          equation
          der(y) = NumComp + NumValve/100;
          Yswitch = if time <= 0 then 0 else y/time;

          end NumberSwitches;

          model PowerConsup

            Modelica.Blocks.Interfaces.RealInput Fcomp
              annotation (extent=[-140,40;-100,80]);
            Modelica.Blocks.Interfaces.RealOutput Ypow
              annotation (extent=[100,-10;120,10]);
          Modelica.Blocks.Interfaces.RealInput Psuc
              annotation (extent=[-140,-80;-100,-40]);
            annotation (Diagram(graphics));

          protected
            Real powcomp;
            Real Rosuc = 4.6073*Psuc+0.3798;
          //  Real nuis;
          //  Real his;
          //  Real hic;
          //  Real hoc;
            Real ro;
          equation

          //der(powcomp) = Fcomp*Rosuc*(his - hic)/nuis;
          der(powcomp) = Fcomp*Rosuc*ro;
          Ypow = if time <= 0 then 0 else powcomp/time;

          //nuis = (his - hic) / (hoc - hic);
          ro = (0.0265*Psuc^3 - 0.4346*Psuc^2 + 2.4923*Psuc + 1.2189) * 10^5;
          end PowerConsup;
        end Continuous2;

        package Experiments
                model modelicaDEVSair
                  Real Tair1 = displayCase.displayCase.Tair;
                  Real Tair2 = displayCase1.displayCase.Tair;
                  Real Psuc = suctionMainfold.Psuc;

                  parameter Integer NUM_COMPRESSORS = 2;

                  DEVSControl.DEVSControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Tgoodsstart=2,
                    Mrefstart=0)          annotation (extent=[-80,20;-40,60]);
                  Continuous2.SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[34,-26;86,36]);
                  DEVSControl.DEVSControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-80,-40;-40,0]);
                  Continuous2.CompressorRack compressorRack2C(
                    NUM_COMPRESSORS=NUM_COMPRESSORS,
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[78,-58;98,-38]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-66,70;-46,90]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[34,70;54,90]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[54,-60;74,-40]);
                  Continuous2.ConstraintSat constraintSat(
                    NumDisp=2,
                    MaxTair={5,5},
                    MinTair={2,2})
                             annotation (extent=[-16,-70;4,-50]);
                  Continuous2.NumberSwitches numberSwitches annotation (extent=[-16,-96;4,-76]);
                  Continuous2.PowerConsup powerConsup annotation (extent=[52,-94;72,-74]);
                  Modelica.Blocks.Sources.Step MaxPsuc(
                    startTime=7200,
                    height=0.2,
                    offset=1.7)     annotation (extent=[-42,-64;-30,-52]);
                  Modelica.Blocks.Math.Sum sum1(nin=2) annotation (extent=[-34,-98;-24,-86]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-42,24;-12,24;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-42,-36;-12,-36;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[88,-39;88,5;78.2,5], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-45,80;-16,80;-16,48;-42,48], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-45,80;-16,80;-16,-12;-42,-12], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[55,80;60.26,80;60.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[41.8,8.1;9.9,8.1;9.9,28.4;-42,28.4], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[41.8,8.1;10,8.1;10,-31.6;-42,-31.6], style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[75,-50;78,-50;78,-48;79,-48], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[41.8,8.1;32,8.1;32,-70;88,-70;88,-57], style(rgbcolor={0,0,127}));

                  annotation (Diagram,    experiment(StopTime=14400),    experimentSetupOutput);
                  connect(suctionMainfold.Psuc, constraintSat.Psuc) annotation (points=[41.8,8.1;10,8.1;10,-48;-18,-48;-18,-52], style(rgbcolor={0,0,127}));

                  connect(MaxPsuc.y, constraintSat.MaxPsuc)  annotation (points=[-29.4,-58;-18,-58], style(rgbcolor={0,0,127}));
                  connect(displayCase.Tair, constraintSat.Tair[1]) annotation (points=[-82,48;-90,48;-90,-69;-18,-69], style(rgbcolor={0,0,127}));
                  connect(displayCase1.Tair, constraintSat.Tair[2]) annotation (points=[-82,-12;-88,-12;-88,-67;-18,-67], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.numCompON, numberSwitches.NumComp) annotation (points=[97.8,-48;102,-48;102,-100;-42,-100;-42,-80;-18,-80], style(rgbcolor={0,0,127}));
                  connect(displayCase1.valveOpen, sum1.u[2]) annotation (points=[-60,-42;-60,-91.4;-35,-91.4], style(rgbcolor={0,0,127}));
                  connect(displayCase.valveOpen, sum1.u[1]) annotation (points=[-60,18;-96,18;-96,-92.6;-35,-92.6], style(rgbcolor={0,0,127}));
                  connect(sum1.y, numberSwitches.NumValve)  annotation (points=[-23.5,-92;-18,-92], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, powerConsup.Fcomp) annotation (points=[88,-39;88,-28;44,-28;44,-78;50,-78], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, powerConsup.Psuc) annotation (points=[41.8,8.1;32,8.1;32,-90;50,-90], style(rgbcolor={0,0,127}));
                end modelicaDEVSair;

                model devslibmsl
                  Real Tair1 = displayCase.displayCase.Tair;
                  Real Tair2 = displayCase1.displayCase.Tair;
                  Real Psuc = suctionMainfold.Psuc;

                  DEVSControl.DEVSControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Tgoodsstart=2,
                    Mrefstart=0)          annotation (extent=[-80,20;-40,60]);
                  Continuous2.SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[34,-26;86,36]);
                  DEVSControl.DEVSControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-80,-40;-40,0]);
                  DEVSControl.CompressorRack2C compressorRack2C(
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[78,-58;98,-38]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-66,70;-46,90]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[34,70;54,90]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[54,-60;74,-40]);
                  Continuous2.ConstraintSat constraintSat(NumDisp=2,
                    MaxTair={5,5},
                    MinTair={2,2}) annotation (extent=[-16,-70;4,-50]);
                  Continuous2.NumberSwitches numberSwitches annotation (extent=[-16,-96;4,-76]);
                  Continuous2.PowerConsup powerConsup annotation (extent=[52,-94;72,-74]);
                  Modelica.Blocks.Sources.Step MaxPsuc(
                    startTime=7200,
                    height=0.2,
                    offset=1.7)     annotation (extent=[-42,-64;-30,-52]);
                  Modelica.Blocks.Math.Sum sum1(nin=2) annotation (extent=[-34,-98;-24,-86]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-42,24;-12,24;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-42,-36;-12,-36;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[88,-39;88,5;78.2,5], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-45,80;-16,80;-16,48;-42,48], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-45,80;-16,80;-16,-12;-42,-12], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[55,80;60.26,80;60.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[41.8,8.1;9.9,8.1;9.9,28.4;-42,28.4], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[41.8,8.1;10,8.1;10,-31.6;-42,-31.6], style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[75,-50;78,-50;78,-48;79,-48], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[41.8,8.1;32,8.1;32,-70;88,-70;88,-57], style(rgbcolor={0,0,127}));

                  annotation (Diagram,    experiment(StopTime=14400),    experimentSetupOutput);
                  connect(suctionMainfold.Psuc, constraintSat.Psuc) annotation (points=[41.8,8.1;10,8.1;10,-48;-18,-48;-18,-52], style(rgbcolor={0,0,127}));
                  connect(MaxPsuc.y, constraintSat.MaxPsuc) annotation (points=[-29.4,-58;-18,-58], style(rgbcolor={0,0,127}));
                  connect(displayCase.Tair, constraintSat.Tair[1]) annotation (points=[-82,48;-90,48;-90,-69;-18,-69], style(rgbcolor={0,0,127}));
                  connect(displayCase1.Tair, constraintSat.Tair[2]) annotation (points=[-82,-12;-88,-12;-88,-67;-18,-67], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.numCompON, numberSwitches.NumComp) annotation (points=[99,-48;102,-48;102,-100;-42,-100;-42,-80;-18,-80], style(rgbcolor={0,0,127}));
                  connect(displayCase1.valveOpen, sum1.u[2]) annotation (points=[-60,-42;-60,-91.4;-35,-91.4], style(rgbcolor={0,0,127}));
                  connect(displayCase.valveOpen, sum1.u[1]) annotation (points=[-60,18;-96,18;-96,-92.6;-35,-92.6], style(rgbcolor={0,0,127}));
                  connect(sum1.y, numberSwitches.NumValve) annotation (points=[-23.5,-92;-18,-92], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, powerConsup.Fcomp) annotation (points=[88,-39;88,-28;44,-28;44,-78;50,-78], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, powerConsup.Psuc) annotation (points=[41.8,8.1;32,8.1;32,-90;50,-90], style(rgbcolor={0,0,127}));
                end devslibmsl;

                model atomicdevslib
                  Real Tair1 = displayCase.displayCase.Tair;
                  Real Tair2 = displayCase1.displayCase.Tair;
                  Real Psuc = suctionMainfold.Psuc;

                  DEVSControl.DEVSControlledDisplayCase displayCase(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=5.1,
                    Tgoodsstart=2,
                    Mrefstart=0)          annotation (extent=[-80,20;-40,60]);
                  Continuous2.SuctionMainfold suctionMainfold(Vsuc=5, Psucstart=1.4)
                                                  annotation (extent=[34,-26;86,36]);
                  DEVSControl.DEVSControlledDisplayCase displayCase1(
                    UAgoodsair=300,
                    Mgoods=200,
                    CPgoods=1000,
                    UAairwall=500,
                    Mwall=260,
                    CPwall=385,
                    UAwallrefmax=4000,
                    Mrefmax=1,
                    Mair=50,
                    CPair=1000,
                    Taufill=40,
                    Tairmax=5,
                    Tairmin=2,
                    Twallstart=0,
                    Tairstart=0,
                    Tgoodsstart=2,
                    Mrefstart=1)          annotation (extent=[-80,-40;-40,0]);
                  DEVSControl.DEVSCompressorRack2C compressorRack2C(
                    DB=0.2,
                    comp=50,
                    Kp=-75,
                    Taoi=50,
                    Nuvol=0.81,
                    Vsl=0.08)                       annotation (extent=[78,-58;98,-38]);
                  Modelica.Blocks.Sources.Step AirLoad(
                    height=-1200,
                    offset=3000,
                    startTime=7200) annotation (extent=[-66,70;-46,90]);
                  Modelica.Blocks.Sources.Step Mrefconst(
                    startTime=7200,
                    height=-0.2,
                    offset=0.2) annotation (extent=[34,70;54,90]);
                  Modelica.Blocks.Sources.Step Psucdesired(
                    startTime=7200,
                    height=0.2,
                    offset=1.4) annotation (extent=[54,-60;74,-40]);
                  Continuous2.ConstraintSat constraintSat(NumDisp=2,
                    MaxTair={5,5},
                    MinTair={2,2}) annotation (extent=[-16,-70;4,-50]);
                  Continuous2.NumberSwitches numberSwitches annotation (extent=[-16,-96;4,-76]);
                  Continuous2.PowerConsup powerConsup annotation (extent=[52,-94;72,-74]);
                  Modelica.Blocks.Sources.Step MaxPsuc(
                    startTime=7200,
                    height=0.2,
                    offset=1.7)     annotation (extent=[-42,-64;-30,-52]);
                  Modelica.Blocks.Math.Sum sum1(nin=2) annotation (extent=[-34,-98;-24,-86]);
                equation
                  connect(displayCase.m, suctionMainfold.m) annotation (points=[-42,24;-12,24;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(displayCase1.m, suctionMainfold.m) annotation (points=[-42,-36;-12,-36;-12,1.9;41.8,1.9], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, suctionMainfold.Vcomp) annotation (points=[88,-39.2;
                  88,5; 78.2,5],                                                                                  style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase.Qairload) annotation (points=[-45,80;-16,80;-16,48;-42,48], style(rgbcolor={0,0,127}));
                  connect(AirLoad.y, displayCase1.Qairload) annotation (points=[-45,80;-16,80;-16,-12;-42,-12], style(rgbcolor={0,0,127}));
                  connect(Mrefconst.y, suctionMainfold.Mrefconst) annotation (points=[55,80;60.26,80;60.26,32.9], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase.Psuc) annotation (points=[41.8,8.1;9.9,8.1;9.9,28.4;-42,28.4], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, displayCase1.Psuc) annotation (points=[41.8,8.1;10,8.1;10,-31.6;-42,-31.6], style(rgbcolor={0,0,127}));
                  connect(Psucdesired.y, compressorRack2C.SetPoint) annotation (points=[75,-50;78,-50;78,-48;78.6,-48], style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, compressorRack2C.Psuc) annotation (points=[41.8,8.1;
                  32,8.1; 32,-70; 88,-70; 88,-55],                                                                                style(rgbcolor={0,0,127}));

                  annotation (Diagram,    experiment(StopTime=14400),    experimentSetupOutput);
                  connect(suctionMainfold.Psuc, constraintSat.Psuc) annotation (points=[41.8,8.1;10,8.1;10,-48;-18,-48;-18,-52], style(rgbcolor={0,0,127}));

                  connect(MaxPsuc.y, constraintSat.MaxPsuc) annotation (points=[-29.4,-58;-18,-58], style(rgbcolor={0,0,127}));
                  connect(displayCase.Tair, constraintSat.Tair[1]) annotation (points=[-82,48;-90,48;-90,-69;-18,-69], style(rgbcolor={0,0,127}));
                  connect(displayCase1.Tair, constraintSat.Tair[2]) annotation (points=[-82,-12;-88,-12;-88,-67;-18,-67], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.numCompON, numberSwitches.NumComp) annotation (points=[97.8,-48;102,-48;102,-100;-42,-100;-42,-80;-18,-80], style(rgbcolor={0,0,127}));
                  connect(displayCase1.valveOpen, sum1.u[2]) annotation (points=[-60,-42;-60,-91.4;-35,-91.4], style(rgbcolor={0,0,127}));
                  connect(displayCase.valveOpen, sum1.u[1]) annotation (points=[-60,18;-96,18;-96,-92.6;-35,-92.6], style(rgbcolor={0,0,127}));
                  connect(sum1.y, numberSwitches.NumValve) annotation (points=[-23.5,-92;-18,-92], style(rgbcolor={0,0,127}));
                  connect(compressorRack2C.Vcomp, powerConsup.Fcomp) annotation (points=[88,-39.2;
                  88,-28; 44,-28; 44,-78; 50,-78],                                                                            style(rgbcolor={0,0,127}));
                  connect(suctionMainfold.Psuc, powerConsup.Psuc) annotation (points=[41.8,8.1;32,8.1;32,-90;50,-90], style(rgbcolor={0,0,127}));
                end atomicdevslib;
        end Experiments;
      end Supermarket;

      package Evolution

        package organism "Draft package to construct new atomic DEVS models"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<p>
This package contains the basic structures needed in atomic models.
It can be duplicated, renamed and modified to construct new atomic DEVS models.
</p>
<p>
The user can implement the behavior of a new DEVS model modifying the con, int, ext, out, and ta functions.<br>
The state can be defined using the st record, and initialized using the initst function.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model org "Draft model to construct new atomic DEVS models"
            Integer X;
            Integer Y;
            parameter Real Numgen = 2;
            parameter Real Nummod = 2;
            parameter Real StressThreshold = 0.8; // 80%
                     extends AtomicDEVS(numIn=8,numOut=8,
            redeclare record State = st);
            redeclare function Fcon = con(env=0);
            redeclare function Fext = ext;
            redeclare function Fint = int(env=0);
            redeclare function Fout = out;
            redeclare function Fta = ta;
            //redeclare function initState = initst(numgen=Numgen,nummod=Nummod,stt=StressThreshold);
            redeclare function initState = initst(numgen=2,nummod=2,stt=0.8);
            Interfaces.outPort out1     annotation (extent=[-116,90; -96,110]);
            Interfaces.inPort in1     annotation (extent=[-116,70; -96,90]);
            Interfaces.outPort out2     annotation (extent=[-24,90; -4,110]);
            Interfaces.inPort in2     annotation (extent=[-4,90; 16,110]);
            Interfaces.outPort out3     annotation (extent=[96,90; 116,110]);
            Interfaces.inPort in3     annotation (extent=[96,70; 116,90]);
            Interfaces.outPort out4     annotation (extent=[-116,-20; -96,0]);
            Interfaces.inPort in4     annotation (extent=[-116,0; -96,20]);
            Interfaces.outPort out5     annotation (extent=[96,0; 116,20]);
            Interfaces.inPort in5     annotation (extent=[96,-20; 116,0]);
            Interfaces.outPort out6     annotation (extent=[-116,-110; -96,-90]);
            Interfaces.inPort in6     annotation (extent=[-116,-90; -96,-70]);
            Interfaces.outPort out7     annotation (extent=[-4,-110; 16,-90]);
            Interfaces.inPort in7     annotation (extent=[-24,-110; -4,-90]);
            Interfaces.outPort out8     annotation (extent=[96,-108; 116,-88]);
            Interfaces.inPort in8     annotation (extent=[96,-90; 116,-70]);
            Modelica.Blocks.Interfaces.RealInput Env
              annotation (extent=[-60,-20; -20,20]);
            Modelica.Blocks.Interfaces.RealInput St annotation (extent=[0,-20; 40,20]);
          algorithm
            when sample(0.01,1) then
              if S.phase > 0 then
                Modelica.Utilities.Streams.print("["+String(X)+","+String(Y)+"] LIFE CICLE: phase "+String(S.phase)+" : age "+String(S.age)+" : GENREL["+String(S.genrel[1,1])+","+String(S.genrel[1,2])+";"+String(S.genrel[2,1])+","+String(S.genrel[2,2])+"] : GENVAL["+String(S.genval[1])+","+String(S.genval[2])+"] : MODREL["+String(S.modrel[1,1])+","+String(S.modrel[1,2])+";"+String(S.modrel[2,1])+","+String(S.modrel[2,2])+"] : eMOD["+String(S.eMOD[1])+","+String(S.eMOD[2])+"] : neighbors["+String(S.neighbors[1])+","+String(S.neighbors[2])+","+String(S.neighbors[3])+","+String(S.neighbors[4])+","+String(S.neighbors[5])+","+String(S.neighbors[6])+","+String(S.neighbors[7])+","+String(S.neighbors[8])+"]", "OUTPUT.txt");
              end if;
            end when;

          equation
            // INPUT PORTS
            iEvent[1] = in1.event;
            iQueue[1] = in1.queue;
            iEvent[2] = in2.event;
            iQueue[2] = in2.queue;
            iEvent[3] = in3.event;
            iQueue[3] = in3.queue;
            iEvent[4] = in4.event;
            iQueue[4] = in4.queue;
            iEvent[5] = in5.event;
            iQueue[5] = in5.queue;
            iEvent[6] = in6.event;
            iQueue[6] = in6.queue;
            iEvent[7] = in7.event;
            iQueue[7] = in7.queue;
            iEvent[8] = in8.event;
            iQueue[8] = in8.queue;
            // OUTPUT PORTS
            oEvent[1] = out1.event;
            oQueue[1] = out1.queue;
            oEvent[2] = out2.event;
            oQueue[2] = out2.queue;
            oEvent[3] = out3.event;
            oQueue[3] = out3.queue;
            oEvent[4] = out4.event;
            oQueue[4] = out4.queue;
            oEvent[5] = out5.event;
            oQueue[5] = out5.queue;
            oEvent[6] = out6.event;
            oQueue[6] = out6.queue;
            oEvent[7] = out7.event;
            oQueue[7] = out7.queue;
            oEvent[8] = out8.event;
            oQueue[8] = out8.queue;
            annotation (Diagram);
          end org;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            input Real env;
           // input Real str;
            output st sout;
          algorithm
            // first internal
            sout := ext(int(s,env=env),0,bag);
            //----------------
            // first external
            //soutput := ext(s,e,bag);
            //sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            input Real env;
            //input Real str;
            output st sout;
          protected
            Real probmut;
            Real valGEN[2]; // CHANGE TO CORRECT NUMGEN VALUE
            Real aux;
            Real r;
            Real str;
          algorithm
            sout := s;
            sout.updateport := zeros(8); // reset update ports
            if sout.action == 0 then // usual life cicle transition
              /****** UPDATE AGE *********/
              sout.age := s.age+1;
              if sout.age <= 1 then
                sout.phase := 1; //infant
              else
                sout.phase := 2; // adult
              end if;
              /****** MUTATION GENREL **********/
              if str > sout.stth then // stress greater than threshold
                Modelica.Utilities.Streams.print("MUTACION GENREL");
                for i in 1:sout.ngen loop // change GENREL
                  for j in 1:sout.ngen loop
                    if sout.pchangegen[i,j] >= RandomUniform(0) then
                      sout.genrel[i,j] := 1 - sout.genrel[i,j];
                    end if;
                  end for;
                end for;
                for i in 1:sout.ngen loop // change MODREL
                  for j in 1:sout.nmod loop
                    if sout.pchangemod[i,j] >= RandomUniform(0) then
                      sout.modrel[i,j] := 1 - sout.modrel[i,j];
                    end if;
                  end for;
                end for;
              end if;
              /****** MUTATION GENVAL **********/
              for i in 1:sout.nmod loop
                if str >= sout.stth then // stress greater than thershold
                  probmut :=0.4;
                else
                  probmut := 0.1;
                end if;
                if RandomUniform(0) <= probmut then // modify GENVAL depending on probmut
                  sout.genval[i] := RandomUniform(0)*5;
                   Modelica.Utilities.Streams.print("MUTACION GENVAL");
                end if;
              end for;
              /****** COMPROBACION VIABILIDAD ***/
              // calcular valGEN
              valGEN := zeros(sout.ngen);
              for i in 1:sout.ngen loop
                for j in 1:sout.ngen loop
                  valGEN[i] := valGEN[i] + sout.genval[j]*sout.genrel[i,j];
                end for;
              end for;
              // calcular expresionMOD (eMOD)
              sout.eMOD := zeros(sout.nmod);
              for i in 1:sout.nmod loop
                for j in 1:sout.ngen loop
                  sout.eMOD[i] := sout.eMOD[i] + valGEN[j]*sout.modrel[i,j];
                end for;
              end for;
              // FILTRO AMBIENTE
              for i in 1:sout.nmod loop
                if env > sout.eMOD[i] then
                  // DEAD by environment
                  Modelica.Utilities.Streams.print("DEAD by environment");
                  sout := initst(sout.ngen,sout.nmod,sout.stth); // reinit state
                end if;
              end for;
              // FILTRO RESTRICCION EMBRIOLOGICA
              for i in 1:sout.nmod loop
                if sout.eMOD[i] < 1 or sout.eMOD[i] > 15 then
                  // DEAD by embrio restriction
                  Modelica.Utilities.Streams.print(String(sout.eMOD[i])+" DEAD by embrio restriction");
                  sout := initst(sout.ngen,sout.nmod,sout.stth); // reinit state
                end if;
              end for;
              // FILTRO EDAD
              if sout.age >= 5 then
                // DEAD by age
                Modelica.Utilities.Streams.print("DEAD by age");
                sout := initst(sout.ngen,sout.nmod,sout.stth); // reinit state
              end if;
              if sout.phase == 0 then // DIED - send phase update to neigbors
                sout.action := 4; //send phase update to neigbors
                sout.updateport := sout.neighbors;
                sout.sigma := 0; // generate output.
              end if;
              /****** REPRODUCTION ******/
              // adulto => edad > 1 (1 cicle without reprod.)
              if sout.age > 1 then
                //Modelica.Utilities.Streams.print("Reproducing ...");
                sout.action := 1; // Reproduce using output function
                sout.sigma := 0; // inmediate internal transition to generate output
              end if;
            else // empty internal transition after output generation
              //Modelica.Utilities.Streams.print("EMPTY INTERNAL");
              sout.action := 0; // passive
              if sout.phase > 0 then
                sout.sigma := 1; // new life cicle
              else
                sout.sigma := Modelica.Constants.inf; // empty cell
              end if;
            end if;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            //input Real t;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := getEvent(bag);
              /************* RECEIVED NEIGHBOR STATE UPDATE */
              if x.Type == 1 then // neighbor changed its state
               // sout.action := 4; // REVISAR
               // sout.sigma := 0; // REVISAR
                sout.sigma := s.sigma-e; // continue with normal life cicle
                sout.neighbors[x.Port] := integer(x.Value);
              /************* CREATE NEW ORGANISM */
              elseif x.Type == 2 then  // new organism in cell GENREL VALUE
                //Modelica.Utilities.Streams.print("NEW ORG type 2");
                sout.phase := 1; // infant
                sout.sigma := 0; // inmediate internal transition to update neighbors state (in action 5)
                sout.action := 5;  //  askupdateneighbors, to initialize the state of neighbors
                sout.age := 0;
                for i in 1:s.ngen loop
                  for j in 1:s.ngen loop
                    sout.genrel[i,j] := integer(Objects.ObjReadPos(integer(x.Value),i,j));
                  end for;
                end for;
                Objects.ObjDelete(integer(x.Value));
                sout.ngen := s.ngen;  // number of genes
                sout.nmod := s.nmod; // number of modules
                sout.neighbors := zeros(8); // phase of neighbors
                sout.environment := 0;
                sout.stress := 0;
                sout.Tlasttran := 0;//t; // time of last transition
                sout.updateport := zeros(8);
              elseif x.Type == 3 then // new organism GENVAL VALUE
                //Modelica.Utilities.Streams.print("NEW ORG type 3");
                for i in 1:s.nmod loop
                  sout.genval[i] := Objects.ObjReadPos(integer(x.Value),i);
                  //Modelica.Utilities.Streams.print(String(i)+": read genval "+String(sout.genval[i]));
                end for;
                Objects.ObjDelete(integer(x.Value));
              elseif x.Type == 4 then // new organism MODREL VALUE
                //Modelica.Utilities.Streams.print("NEW ORG type 4");
                for i in 1:s.ngen loop
                  for j in 1:s.nmod loop
                    sout.modrel[i,j] := integer(Objects.ObjReadPos(integer(x.Value),i,j));
                  end for;
                end for;
                Objects.ObjDelete(integer(x.Value));
              /************* SEND NEIGHBOR STATE UPDATE */
              elseif x.Type == 5 then // updateneighbor message
                //Modelica.Utilities.Streams.print("received message type 5");
                sout.action := 4; // send phase to neighbor
                sout.sigma := 0; // inmediate internal transition to send phase update
                sout.updateport[x.Port] := 1;
                sout.neighbors[x.Port] := 1;  // neighbor that sent message is not empty
              /************* SEND NEIGHBOR STATE UPDATE */
              elseif x.Type == 6 then // INITIAL message
                //Modelica.Utilities.Streams.print("INITIAL");
                sout.phase := 1; // infant
                sout.sigma := 0; // inmediate internal transition to update neighbors state (in action 5)
                sout.action := 5;  //  askupdateneighbors, to initialize the state of neighbors
              end if;
            end for;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
            Integer kl[8];
            Integer k;
            Integer j;
            Boolean found;
            Real u;
          algorithm
            /**** REPRODUCTION *****/
            if s.action == 1 then // reproduction
              // find empty neighbor
              kl := zeros(8);
              j := 1;
              for i in 1:8 loop
                if s.neighbors[i] == 0 then
                  kl[j] := i;
                  j := j+1;
                  found := true;
                end if;
              end for;
              if found then // found empty neighbor
                // select random neighbor between those in kl
                u := RandomUniform(0);
                k := kl[integer(u*(j-1))+1];
                // generate and send GENREL object
                y.Type := 2;
                y.Value := Objects.ObjCreate(s.ngen,s.ngen); // generate genrel object
                for i in 1:s.ngen loop
                  for j in 1:s.ngen loop
                    Objects.ObjUpdatePos(integer(y.Value),s.genrel[i,j],i,j);
                  end for;
                end for;
                sendEvent(queue[k],y);
                // generate and send GENVAL object
                y.Type := 3;
                y.Value := Objects.ObjCreate(s.nmod); // generate genval object
                for i in 1:s.nmod loop
                  Objects.ObjUpdatePos(integer(y.Value),s.genval[i],i);
                  //Modelica.Utilities.Streams.print(String(i)+": written genval "+String(s.genval[i]));
                end for;
                sendEvent(queue[k],y);
                // generate and send MODREL
                y.Type := 4;
                y.Value := Objects.ObjCreate(s.ngen,s.nmod); // generate modrel object
                for i in 1:s.ngen loop
                  for j in 1:s.nmod loop
                    Objects.ObjUpdatePos(integer(y.Value),s.modrel[i,j],i,j);
                  end for;
                end for;
                sendEvent(queue[k],y);
              end if;
            /********* SEND PHASE UPDATE TO NEIGHBORS *****/
            elseif s.action == 4 then
              y.Type := 1;
              y.Value := s.phase;
              for i in 1:8 loop
                if (s.updateport[i] == 1) then
                  sendEvent(queue[i],y);
                end if;
              end for;
            /********* ASK NEIGHBORS TO UPDATE PHASE *****/
            elseif s.action == 5 then
              //Modelica.Utilities.Streams.print("output action 5");
              y.Type := 5;
              for i in 1:nports loop
                sendEvent(queue[i],y);
              end for;
            end if;

          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st "state of the model"
            Integer phase; // 0 empty, 1 infant, 2 adult
            Real sigma;
            Integer action; // -1 dead, 0 passive/living, 4 send phase to neigbor, 1 reproduction, 2 postmutation, 3 changestate, 5 ask neighbor to update phase
            Integer age;
            Integer ngen; // number of genes
            Integer nmod; // number of modules
            Integer genrel[2,2]; // nget x ngen matrix of inter-genetic relations
            Real genval[2]; // nmod vector of genetic values
            Integer modrel[2,2]; // ngen x nmod matrix for relation between genes and modules
            Real pchangegen[2,2]; // ngen x ngen matrix of probabilities for changes in GENREL
            Real pchangemod[2,2]; // ngen x nmod matrix of probabilities for changes in MODREL
            Real eMOD[2]; // nmod vector of values for modules
            Integer neighbors[8]; // phase of neighbors
            Real environment;
            Real stress;
            Real stth; // stress threshold for mutations;
            Real Tlasttran; // time of last transition
            Integer updateport[8]; // port of neighbors to update
          end st;

          function initst "state initalization function"
            input Real numgen;
            input Real nummod;
            input Real stt;
            output st out;
          algorithm
            out.phase := 0;
            out.sigma := Modelica.Constants.inf; // first internal event never happens
            out.action := 0; // passive
            out.age := 0;
            out.genrel:= [1,0; 1,1];
            out.genval := {2,2};
            out.modrel := [0,1; 1,1];
            out.pchangegen := [0.2,0.2; 0.2,0.2];
            out.pchangemod := [0.2,0.2; 0.2,0.2];
            out.eMOD := {0,0};
            out.ngen := integer(numgen);  // number of genes
            out.nmod := integer(nummod); // number of modules
            out.neighbors := zeros(8); // phase of neighbors
            out.environment := 0;
            out.stress := 0;
            out.stth := stt;
            out.Tlasttran := 0; // time of last transition
            out.updateport := zeros(8);
          end initst;
        end organism;

        model Test1
          organism.org org annotation (extent=[-18,18; 2,38]);
          annotation (Diagram);
          RandomLib.Sources.Continuous Environment
            annotation (extent=[-80,-60; -60,-40]);
          RandomLib.Sources.Continuous Stress
            annotation (extent=[40,-60; 60,-40]);
          AuxModels.Display display annotation (extent=[-48,56; -28,76]);
          AuxModels.Display display1 annotation (extent=[-22,56; -2,76]);
          AuxModels.Display display2 annotation (extent=[4,56; 24,76]);
          AuxModels.Display display3 annotation (extent=[32,20; 52,40]);
          AuxModels.Display display4 annotation (extent=[32,-10; 52,10]);
          AuxModels.Display display5 annotation (extent=[-18,-10; 2,10]);
          AuxModels.Display display6 annotation (extent=[-68,-18; -48,2]);
          AuxModels.Display display7 annotation (extent=[-70,22; -50,42]);
        equation
          connect(Environment.y, org.Env) annotation (points=[-61,-50; -42,-50;
                -42,28; -12,28], style(color=74, rgbcolor={0,0,127}));
          connect(Stress.y, org.St) annotation (points=[59,-50; 26,-50; 26,28;
                -6,28], style(color=74, rgbcolor={0,0,127}));
          connect(display.inPort1, org.out1) annotation (points=[-47,66; -42,66;
                -42,38; -18.6,38], style(color=0, rgbcolor={0,0,0}));
          connect(display5.inPort1, org.out7) annotation (points=[-17,0; -12,0;
                -12,18; -7.4,18], style(color=0, rgbcolor={0,0,0}));
          connect(display4.inPort1, org.out8) annotation (points=[33,0; 18,0;
                18,18.2; 2.6,18.2], style(color=0, rgbcolor={0,0,0}));
          connect(display3.inPort1, org.out5) annotation (points=[33,30; 17.8,
                30; 17.8,29; 2.6,29], style(color=0, rgbcolor={0,0,0}));
          connect(org.out3, display2.inPort1) annotation (points=[2.6,38; 2.6,
                52; 5,52; 5,66], style(color=0, rgbcolor={0,0,0}));
          connect(org.out2, display1.inPort1) annotation (points=[-9.4,38; -16,
                38; -16,66; -21,66], style(color=0, rgbcolor={0,0,0}));
          connect(org.out4, display7.inPort1) annotation (points=[-18.6,27;
                -44.3,27; -44.3,32; -69,32], style(color=0, rgbcolor={0,0,0}));
          connect(org.out6, display6.inPort1) annotation (points=[-18.6,18; -42,
                18; -42,-8; -67,-8], style(color=0, rgbcolor={0,0,0}));
        end Test1;

        model TestNxN "Draft model for constructing new coupled DEVS models"
          import DESLib.DEVSLib.SRC.*;
        annotation(preferedView="diagram",
          Documentation(info="<HTML>
<p>
This package contains the basic structures needed in coupled models.
It can be duplicated, renamed and modified to construct new coupled DEVS models.
</p>
<p>
This model includes one input and one output ports. Any other requiered port can be included.<br>
The components of the new coupled model can be included and interconnected using drag and drop.
</p>


</HTML>
"),         Coordsys(extent=[-100,-100; 100,100], scale=0.1),
            Diagram);
          parameter Integer size = 5;
          annotation (Diagram);
          organism.org M[size,size];
          DEVSLib.AuxModels.BreakLoop BLi[size,size,4];
          DEVSLib.AuxModels.BreakLoop BLo[size,size,4];

          Integer[size,size] state;
          Integer population(start = 0);
          Real AVGpop( start = 0);
          Integer popsum;

          RandomLib.Sources.Continuous Environment(distribution=1, p1=0)
                                                   annotation (extent=[-80,-20; -60,0]);
          RandomLib.Sources.Continuous Stress(distribution=1, p1=0)
                                              annotation (extent=[40,-20; 60,0]);
          SimpleModels.Generator.generator generator(
            firstGeneration=0,
            numGenerations=1,
            outValue=6) annotation (extent=[-80,40; -60,60]);
        equation

          for i in 0:size-1 loop
            for j in 0:size-1 loop
              M[i+1,j+1].X =  i+1;
              M[i+1,j+1].Y =  j+1;
              connect(M[i+1,j+1].in1, BLi[i+1,j+1,1].OUT);
              connect(BLi[i+1,j+1,1].IN, M[mod(i-1,size)+1,mod(j-1,size)+1].out8);
              connect(M[i+1,j+1].out1, BLo[i+1,j+1,1].IN);
              connect(BLo[i+1,j+1,1].OUT, M[mod(i-1,size)+1,mod(j-1,size)+1].in8);

              connect(M[i+1,j+1].in2, BLi[i+1,j+1,2].OUT);
              connect(BLi[i+1,j+1,2].IN, M[mod(i-1,size)+1,mod(j,size)+1].out7);
              connect(M[i+1,j+1].out2, BLo[i+1,j+1,2].IN);
              connect(BLo[i+1,j+1,2].OUT, M[mod(i-1,size)+1,mod(j,size)+1].in7);

              connect(M[i+1,j+1].in3, BLi[i+1,j+1,3].OUT);
              connect(BLi[i+1,j+1,3].IN, M[mod(i-1,size)+1,mod(j+1,size)+1].out6);
              connect(M[i+1,j+1].out3, BLo[i+1,j+1,3].IN);
              connect(BLo[i+1,j+1,3].OUT, M[mod(i-1,size)+1,mod(j+1,size)+1].in6);

              connect(M[i+1,j+1].in4, BLi[i+1,j+1,4].OUT);
              connect(BLi[i+1,j+1,4].IN, M[mod(i,size)+1,mod(j-1,size)+1].out5);
              connect(M[i+1,j+1].out4, BLo[i+1,j+1,4].IN);
              connect(BLo[i+1,j+1,4].OUT, M[mod(i,size)+1,mod(j-1,size)+1].in5);

              connect(Environment.y, M[i+1,j+1].Env);
              connect(Stress.y, M[i+1,j+1].St);
            end for;
          end for;

          connect(generator.outPort1,M[3,3].in1);

          for i in 1:size loop
            for j in 1:size loop
              state[i,j] = M[i,j].S.phase;
            end for;
          end for;

        algorithm

          when sample(0.01,1) then
            Modelica.Utilities.Streams.print("** TIME = "+String(time)+"*****************************************************","OUTPUT.txt");
            for i in 1:size loop
              //Modelica.Utilities.Streams.print(String(state[i,1])+","+String(state[i,2])+","+String(state[i,3]));
              Modelica.Utilities.Streams.print(String(state[i,1])+","+String(state[i,2])+","+String(state[i,3])+","+String(state[i,4])+","+String(state[i,5]),"OUTPUT.txt");
              //Modelica.Utilities.Streams.print(String(state[i,1])+","+String(state[i,2])+","+String(state[i,3])+","+String(state[i,4])+","+String(state[i,5])+","+String(state[i,6])+","+String(state[i,7])+","+String(state[i,8])+","+String(state[i,9])+","+String(state[i,10]),"OUTPUT.txt");
            end for;
            population := 0;
            for i in 1:size loop
              for j in 1:size loop
                if state[i,j] > 0 then
                  population := population +1;
                end if;
              end for;
            end for;
            popsum := popsum + population;
            if time > 1 then
              AVGpop := popsum / time;
            end if;
            Modelica.Utilities.Streams.print("POPULATION = "+String(population),"OUTPUT.txt");
            Modelica.Utilities.Streams.print("*******************************************************************************************************","OUTPUT.txt");
          end when;

        end TestNxN;

      package Objects
        function ObjCreate
          input Integer rows = 1;
          input Integer cols = 1;
          input Real foo = 1;
          output Integer out;
        external "C" out =  ObjCreate(rows,cols);
        annotation(Include="#include <objects.c>");
        end ObjCreate;

        function ObjDelete
          input Integer addr;
         //  output Integer out;
        external "C" ObjDelete(addr);
        annotation(Include="#include <objects.c>");
        end ObjDelete;

        function ObjUpdateSize
          input Integer addr;
          input Integer rows;
          input Integer cols;
          output Integer out;
          external "C" out =  ObjUpdateSize(addr,rows,cols);
        annotation(Include="#include <objects.c>");
        end ObjUpdateSize;

        function ObjUpdatePos
          input Integer addr;
          input Real value;
          input Integer row = 1;
          input Integer col = 1;
          output Integer out;
          external "C" out = ObjUpdatePos(addr,value,row,col);
        annotation(Include="#include <objects.c>");
        end ObjUpdatePos;

        function ObjUpdate
          input Integer P "pointer to object value";
          input Real[i1,i2] newvalue "new value for the object";
          input Integer i1 "index1";
          input Integer i2 "index2";
          output Integer out;
        algorithm
           for i in 1:i1 loop
             for j in 1:i2 loop
               out := ObjUpdatePos(P,newvalue[i,j],i,j);
             end for;
           end for;
        end ObjUpdate;

        function ObjReadPos
          input Integer addr;
          input Integer row = 1;
          input Integer col = 1;
          input Real foo=1;
          output Real out;
          external "C" out = ObjReadPos(addr,row,col);
        annotation(Include="#include <objects.c>");
        end ObjReadPos;

        function ObjReadI1
          input Integer addr;
          output Integer out;
          external "C" out = ObjReadI1(addr);
        annotation(Include="#include <objects.c>");
        end ObjReadI1;

        function ObjReadI2
          input Integer addr;
          output Integer out;
          external "C" out = ObjReadI2(addr);
        annotation(Include="#include <objects.c>");
        end ObjReadI2;

      end Objects;

      end Evolution;

      package GameOfLife
        package Cell "Draft package to construct new atomic DEVS models"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<p>
This package contains the basic structures needed in atomic models.
It can be duplicated, renamed and modified to construct new atomic DEVS models.
</p>
<p>
The user can implement the behavior of a new DEVS model modifying the con, int, ext, out, and ta functions.<br>
The state can be defined using the st record, and initialized using the initst function.
</p>


</HTML>
"));
          import DESLib.DEVSLib.SRC.*;
          model cell "Draft model to construct new atomic DEVS models"
                     extends AtomicDEVS(numIn=8,numOut=8, redeclare record
                State =                                                            st);
            redeclare function Fcon = con;
            redeclare function Fext = ext;
            redeclare function Fint = int;
            redeclare function Fout = out;
            redeclare function Fta = ta;
            //redeclare function initState = initst(numgen=Numgen,nummod=Nummod,stt=StressThreshold,firstcicle=FirstCicle);
            redeclare function initState = initst;
            Interfaces.outPort out1     annotation (extent=[-116,90; -96,110]);
            Interfaces.inPort in1     annotation (extent=[-116,70; -96,90]);
            Interfaces.outPort out2     annotation (extent=[-24,90; -4,110]);
            Interfaces.inPort in2     annotation (extent=[-4,90; 16,110]);
            Interfaces.outPort out3     annotation (extent=[96,90; 116,110]);
            Interfaces.inPort in3     annotation (extent=[96,70; 116,90]);
            Interfaces.outPort out4     annotation (extent=[-116,-20; -96,0]);
            Interfaces.inPort in4     annotation (extent=[-116,0; -96,20]);
            Interfaces.outPort out5     annotation (extent=[96,0; 116,20]);
            Interfaces.inPort in5     annotation (extent=[96,-20; 116,0]);
            Interfaces.outPort out6     annotation (extent=[-116,-110; -96,-90]);
            Interfaces.inPort in6     annotation (extent=[-116,-90; -96,-70]);
            Interfaces.outPort out7     annotation (extent=[-4,-110; 16,-90]);
            Interfaces.inPort in7     annotation (extent=[-24,-110; -4,-90]);
            Interfaces.outPort out8     annotation (extent=[96,-108; 116,-88]);
            Interfaces.inPort in8     annotation (extent=[96,-90; 116,-70]);
          equation
            // INPUT PORTS
            iEvent[1] = in1.event;
            iQueue[1] = in1.queue;
            iEvent[2] = in2.event;
            iQueue[2] = in2.queue;
            iEvent[3] = in3.event;
            iQueue[3] = in3.queue;
            iEvent[4] = in4.event;
            iQueue[4] = in4.queue;
            iEvent[5] = in5.event;
            iQueue[5] = in5.queue;
            iEvent[6] = in6.event;
            iQueue[6] = in6.queue;
            iEvent[7] = in7.event;
            iQueue[7] = in7.queue;
            iEvent[8] = in8.event;
            iQueue[8] = in8.queue;
            // OUTPUT PORTS
            oEvent[1] = out1.event;
            oQueue[1] = out1.queue;
            oEvent[2] = out2.event;
            oQueue[2] = out2.queue;
            oEvent[3] = out3.event;
            oQueue[3] = out3.queue;
            oEvent[4] = out4.event;
            oQueue[4] = out4.queue;
            oEvent[5] = out5.event;
            oQueue[5] = out5.queue;
            oEvent[6] = out6.event;
            oQueue[6] = out6.queue;
            oEvent[7] = out7.event;
            oQueue[7] = out7.queue;
            oEvent[8] = out8.event;
            oQueue[8] = out8.queue;
            annotation (Diagram);
          end cell;

          function con "Confluent Transtition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            st soutput;
          algorithm
            // first internal
            //sout := ext(int(s),0,bag);
            //----------------
            // first external
            soutput := ext(s,e,bag);
            sout := int(soutput);
          end con;

          function int "Internal Transition Function"
            input st s;
            output st sout;
          algorithm
            sout := s;
            if sout.action == 1 then
              sout.sigma := 0.5; // next cycle
              sout.action := 0; // deactivate update
            else
              if sout.phase == 0 then // dead, maybe borns
                if sum(sout.neighbors) == 3 then
                  //Modelica.Utilities.Streams.print("BORN "+String(sum(sout.neighbors)));
                  sout.phase := 1;
                  sout.sigma := 0.5;
                  sout.action := 1; // update
                else
                  sout.sigma := 1; // next cycle
                end if;
              else // alive, maybe die
                if (sum(sout.neighbors) < 2 or sum(sout.neighbors) > 3) then
                  //Modelica.Utilities.Streams.print("DIES "+String(sum(sout.neighbors)));
                  sout.phase := 0; // dies
                  sout.sigma := 0.5; // update
                  sout.action := 1; // update
                else
                  //Modelica.Utilities.Streams.print("CONTINUE");
                  sout.sigma := 1;
                end if;
              end if;
              Modelica.Utilities.Streams.print("- "+String(sout.phase)+" "+String(s.phase));
            end if;
          end int;

          function ext "External Transition Function"
            input st s;
            input Real e;
            input Integer bag;
            output st sout;
          protected
            Integer numreceived;
            stdEvent x;
          algorithm
            sout := s;
            numreceived := numEvents(bag);
            for i in 1:numreceived loop
              x := getEvent(bag);
              /************* RECEIVED NEIGHBOR STATE UPDATE */
              if x.Type == 1 then // neighbor changed its state
                sout.neighbors[x.Port] := integer(x.Value);
                sout.sigma := 0.5;
                //sout.sigma := sout.sigma - e;
              /************* CREATE NEW CELL */
              elseif x.Type == 2 then  // initial message (cell borns)
                //Modelica.Utilities.Streams.print("INIT BORN "+String(sum(sout.neighbors)));
                sout.sigma := 0.5; // send update to neighbors
                sout.action := 1; // update
                sout.phase := 1; // born
              end if;
            end for;
          end ext;

          function out "Output Function"
            input st s;
            input Integer queue[nports];
            input Integer nports;
          protected
            stdEvent y;
            Integer k;
          algorithm
            //Modelica.Utilities.Streams.print("UPDATE NEIGH");
            /********* SEND PHASE UPDATE TO NEIGHBORS *****/
            if s.action == 1 then
              y.Type := 1;
              y.Value := s.phase;
              for i in 1:8 loop
                sendEvent(queue[i],y);
              end for;
            end if;
          end out;

          function ta "Time Advance Function"
            input st s;
            output Real sigma;
          algorithm
            sigma := s.sigma;
          end ta;

          record st "state of the model"
            Integer phase; // 0 dead, 1 alive
            Real sigma;
            Integer action;
            Integer neighbors[8]; // phase of neighbors
          end st;

          function initst "state initalization function"
            output st out;
          algorithm
            out.phase := 0;
            out.action := 0; // 0 = pasive, 1 = update
            out.sigma := Modelica.Constants.inf;
            out.neighbors := zeros(8); // phase of neighbors
          end initst;
        end Cell;

        model test_torus "Draft model for constructing new coupled DEVS models"
          import DESLib.DEVSLib.SRC.*;
        annotation(preferedView="diagram",
          Documentation(info="<HTML>
<p>
This package contains the basic structures needed in coupled models.
It can be duplicated, renamed and modified to construct new coupled DEVS models.
</p>
<p>
This model includes one input and one output ports. Any other requiered port can be included.<br>
The components of the new coupled model can be included and interconnected using drag and drop.
</p>


</HTML>
"),         Coordsys(extent=[-100,-100; 100,100], scale=0.1),
            Diagram);
          parameter Integer size = 10;

          annotation (Diagram);
          Cell.cell M[size,size];
          DEVSLib.AuxModels.BreakLoop BLi[size,size,4];
          DEVSLib.AuxModels.BreakLoop BLo[size,size,4];

          AuxModels.Generator initiator(numGenerations=1, outValue=2,
            firstGeneration=0)
            annotation (extent=[-78,24; -58,44]);

          Real state[size,size];
          Real x[size,size] = zeros(size,size);
          Real y[size,size] = zeros(size,size);
          AuxModels.DUP3 dup  annotation (extent=[-58,4; -38,24]);
          AuxModels.DUP3 dup1 annotation (extent=[-38,-16; -18,4]);

        algorithm
          when sample(0.1,1) then
            Modelica.Utilities.Streams.print("** TIME = "+String(time)+"********");
            for i in 1:size loop
              //Modelica.Utilities.Streams.print(String(state[i,1])+","+String(state[i,2])+","+String(state[i,3])+","+String(state[i,4]));
              Modelica.Utilities.Streams.print(String(state[i,1])+","+String(state[i,2])+","+String(state[i,3])+","+String(state[i,4])+","+String(state[i,5])+","+String(state[i,6])+","+String(state[i,7])+","+String(state[i,8])+","+String(state[i,9])+","+String(state[i,10]));
            end for;
          end when;

        equation

          for i in 0:size-1 loop
            for j in 0:size-1 loop
              connect(M[i+1,j+1].in1, BLi[i+1,j+1,1].OUT);
              connect(BLi[i+1,j+1,1].IN, M[mod(i-1,size)+1,mod(j-1,size)+1].out8);
              connect(M[i+1,j+1].out1, BLo[i+1,j+1,1].IN);
              connect(BLo[i+1,j+1,1].OUT, M[mod(i-1,size)+1,mod(j-1,size)+1].in8);

              connect(M[i+1,j+1].in2, BLi[i+1,j+1,2].OUT);
              connect(BLi[i+1,j+1,2].IN, M[mod(i-1,size)+1,mod(j,size)+1].out7);
              connect(M[i+1,j+1].out2, BLo[i+1,j+1,2].IN);
              connect(BLo[i+1,j+1,2].OUT, M[mod(i-1,size)+1,mod(j,size)+1].in7);

              connect(M[i+1,j+1].in3, BLi[i+1,j+1,3].OUT);
              connect(BLi[i+1,j+1,3].IN, M[mod(i-1,size)+1,mod(j+1,size)+1].out6);
              connect(M[i+1,j+1].out3, BLo[i+1,j+1,3].IN);
              connect(BLo[i+1,j+1,3].OUT, M[mod(i-1,size)+1,mod(j+1,size)+1].in6);

              connect(M[i+1,j+1].in4, BLi[i+1,j+1,4].OUT);
              connect(BLi[i+1,j+1,4].IN, M[mod(i,size)+1,mod(j-1,size)+1].out5);
              connect(M[i+1,j+1].out4, BLo[i+1,j+1,4].IN);
              connect(BLo[i+1,j+1,4].OUT, M[mod(i,size)+1,mod(j-1,size)+1].in5);
            end for;
          end for;

          for i in 1:size loop
            for j in 1:size loop
              state[i,j] = M[i,j].S.phase;
            end for;
          end for;

          connect(initiator.outPort1, dup.IN)  annotation (points=[-59,34; -54,
                34; -54,14; -53,14],
                             style(color=0, rgbcolor={0,0,0}));
          connect(dup.out1, M[1, 2].in1);

          connect(dup.out2, M[2, 3].in1);
          connect(dup1.out1, M[3, 1].in1);
          connect(dup1.out2, M[3, 2].in1);
          connect(dup1.out3, M[3, 3].in1);

          connect(dup.out3, dup1.IN) annotation (points=[-46.6,10; -40,10; -40,
                -6; -33,-6],
                     style(color=0, rgbcolor={0,0,0}));
        end test_torus;

      package Objects
        function ObjCreate
          input Integer rows = 1;
          input Integer cols = 1;
          input Real foo = 1;
          output Integer out;
        external "C" out =  ObjCreate(rows,cols);
        annotation(Include="#include <objects.c>");
        end ObjCreate;

        function ObjDelete
          input Integer addr;
         //  output Integer out;
        external "C" ObjDelete(addr);
        annotation(Include="#include <objects.c>");
        end ObjDelete;

        function ObjUpdateSize
          input Integer addr;
          input Integer rows;
          input Integer cols;
          output Integer out;
          external "C" out =  ObjUpdateSize(addr,rows,cols);
        annotation(Include="#include <objects.c>");
        end ObjUpdateSize;

        function ObjUpdatePos
          input Integer addr;
          input Real value;
          input Integer row = 1;
          input Integer col = 1;
          output Integer out;
          external "C" out = ObjUpdatePos(addr,value,row,col);
        annotation(Include="#include <objects.c>");
        end ObjUpdatePos;

        function ObjUpdate
          input Integer P "pointer to object value";
          input Real[i1,i2] newvalue "new value for the object";
          input Integer i1 "index1";
          input Integer i2 "index2";
          output Integer out;
        algorithm
           for i in 1:i1 loop
             for j in 1:i2 loop
               out := ObjUpdatePos(P,newvalue[i,j],i,j);
             end for;
           end for;
        end ObjUpdate;

        function ObjReadPos
          input Integer addr;
          input Integer row = 1;
          input Integer col = 1;
          input Real foo=1;
          output Real out;
          external "C" out = ObjReadPos(addr,row,col);
        annotation(Include="#include <objects.c>");
        end ObjReadPos;

        function ObjReadI1
          input Integer addr;
          output Integer out;
          external "C" out = ObjReadI1(addr);
        annotation(Include="#include <objects.c>");
        end ObjReadI1;

        function ObjReadI2
          input Integer addr;
          output Integer out;
          external "C" out = ObjReadI2(addr);
        annotation(Include="#include <objects.c>");
        end ObjReadI2;

      end Objects;

      model LifeView "Virtual-lab view description"
                   extends Interactive.VLabModels.PartialView(fileName=
                "Life.java");

        annotation (Diagram, uses(VirtualLabBuilder(version="1")));
      parameter Integer size = 10;

        Interactive.ViewElements.Containers.MainWindow MF(
                                  Width=600, LayoutPolicy="VBox")
                                                           annotation (extent=[-54,42;
              -34,62]);
          Interactive.ViewElements.Containers.Canvas canvas(
          position={0,4,8},
          focalPoint={0,4,0},
          viewAngle=60)
            annotation (extent=[-28,42; -8,62]);
        Interactive.ViewElements.Drawables.PlanarGrid vista(
            dims={size,size,0},
            minRangeCellLUT=0,
            maxRangeCellLUT=1)
            annotation (extent=[0,40; 20,60]);
      equation

          connect(root.cLLeft, MF.pLLeft)        annotation (points=[-70.2,61;
              -70.2,54.8; -53,54.8],   style(color=0, rgbcolor={0,0,0}));
        annotation (
          Coordsys(extent=[-100, -100; 100, 100]),
          Documentation(info="<html><br><br>

This model has been composed by extending the <i>PartialView</i> class
and by instantiating and connecting the required components of the <i>VirtualLabBuilder</i> library.
The connection rules are described in the documentation of the <i>ViewElements</i> package.

    </html>"));
        connect(MF.cLRight, canvas.pLLeft) annotation (points=[-35,52; -30,52;
                -30,57; -27,57], style(color=0, rgbcolor={0,0,0}));
        connect(canvas.cRight, vista.pLeft) annotation (points=[-9,52; -6,52;
                -6,55; 1,55], style(color=0, rgbcolor={0,0,0}));
      end LifeView;

      model LifeInteractive "Interactive model"

        Interactive.VLabModels.VirtualLab interactive(
            redeclare model ModelI = test_torus,
            fileName="Program1.cpp",
            compilation=true,
          Tcom=0.01,
          redeclare model ViewI = LifeView)
          annotation (Placement(transformation(extent={{-50,-20},{42,46}},
                rotation=0)));

        parameter Integer size = 10;
        parameter Real lengthx = 2;
        parameter Real lengthy = 2;
        parameter Real lengthz = 0;
        protected
        Real s[size*size];
      equation
        for i in 1:size loop
          for j in 1:size loop
            s[i+(j-1)*size] = interactive.Model.state[i,j];
          end for;
        end for;

        interactive.View.vista.Position = {0, 0, 0};
        interactive.View.vista.Origin = {0, 0, 0};
        interactive.View.vista.Degrees = {0, 0, 0};
        interactive.View.vista.LengthX = lengthx;
        interactive.View.vista.LengthY = lengthy;
        interactive.View.vista.LengthZ = lengthz;
        interactive.View.vista.posGrid = {-lengthx/2,-lengthy/2,-0.03};

        interactive.View.vista.scalars = s;

        annotation (experiment(StopTime=1000, Interval=0.001), experimentSetupOutput(
              equdistant=false, events=false),
          uses(VirtualLabBuilder(version="1")),
          Documentation(
              Coordsys(extent=[-100, -100; 100, 100]),
      info="<html><br><br>

This model has been composed by instantiating the <i>VirtualLab</i> Model,
which is included in the <i>Interactive/i> library.
The <i>BBInteractive</i> model includes the equations required to link the model and the view variables.


    </html>"));
      end LifeInteractive;
      end GameOfLife;
    end Examples;

    package SRC "Source pacakges and models"
    annotation(preferedView="info",
      Documentation(info="<HTML>
<p>
This package contains the implementation of the internal components of the package.
For a detailed description of the implementation see the <a href=\"Modelica://DESLib.DEVSLib.SRC.DevelopersGuide\">Developers Guide</a>.
</p>



</HTML>
"),     uses(Modelica(version="2.2.1")));

    model DevelopersGuide "Developers Guide"

        annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Developers Guide of the DEVSLib Library</font></h3>

<p>
This guide includes a description of the implementation and internal behavior of the DEVSLib package components.
</p>


</html>"));
    class Atomic "AtomicDEVS Model"

    annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>AtomicDEVS Model Description</font></h3>

 <p>
This model represents the basic component for atomic Parallel DEVS models.<br>
It implements the behavior of an atomic model, independently of the actions performed by the transition, output and time advance functions.
Basically, it detects the occurrence of events and executes the functions that correspond to the detected events.
The actions performed by these functions have to be defined by the user<br>
The state of the model and its initialization also has to be defined by the user.
</p>


<p>
<h3><font color=\"#008000\">Interface Ports</font></h3>

The input ports have to be defined by the user.
In order to allow any number of input and output ports, four arrays (<i>iEvent</i>, <i>iQueue</i>, <i>oEvent</i> and <i>oQueue</i>) have been defined in the model
and internally represent the input (iEvent and iQueue) and output (oEvent and oQueue) ports.
The length of these arrays is defined by the parameters <i>numIn</i> and <i>numOut</i>.
The user must set these parameters to the correct number of input and output ports (the minimum value is 1).
An equation must be defined between the port and its corresponding position in each of the arrays.
The event variable of the port is connected to the Event array, and the queue variable is connected to the Queue array
(i.e. <i>inputPort1.event = iEvent[1]</i>, <i>inputPort1.queue = iQueue[1]</i> ,<i>outputPort1.event = oEvent[1]</i> and <i>outputPort1.queue = oQueue[1]</i>).<br>
<br>
</p>

<p>
<h3><font color=\"#008000\">Functions</font></h3>

Five replaceable functions are defined in this model and represent the transition, output and time advance functions (<i>Fcon, Fext, Fint, Fout and Fta</i>
respectively).
These functions do not perform any action and must be redeclared by the user.
Models without input or output ports does not need the external, confluent or internal transition functions, so they do not have to be redeclared.
The prototype of each function is:
<ul>
<li> sout := Fcon (s,e,bag), where <i>sout</i> is the state after the confluent transition,
<i>soutput</i> is the state that will be used by the output function to generate an output,
<i>s</i> is the current state, <i>e</i> is the elapsed time since the last transition and <i>bag</i> is the reference to the storage of input messages.
<li> sout := Fint (s), where <i>sout</i> is the state after the confluent transition, and <i>s</i> is the current state.
<li> sout := Fext (s,e,bag), where <i>sout</i> is the state after the confluent transition, and <i>s</i> is the current state,
<i>e</i> is the elapsed time since the last transition, and <i>bag</i> is the reference to the storage of input messages.
<li> sigma := Fta(s), where <i>sigma</i> is the time interval until the next internal transition, and <i>s</i> is the current state.
<li> Fout (s,queues[nports],nports), where <i>s</i> is the current state, <i>queues[nports]</i> is the array that contains references to the queues that will receive the messages sent by this function,
and <i>nports</i> is the number of ports connected to the model that executes the function.
</ul>

Alternatively to these parameters, any other parameter can be added to these functions.
New parameters can be added at the end of the function inputs.
The value of the parameter can be set in the redeclaration of the function, when constructing a new model.
Continuous or discrete input values can be passed to these functions using this method.
</p>

<p>
<h3><font color=\"#008000\">State</font></h3>

The state of the model is represented by the record <i>st</i>.
Any number of variables can be included in that record in order to represent the state of the model.
In Dymola 6.2, only simple types (Real, Integer and Boolean) can be included. This bug is fixed in newer versions.
The modified state can be redeclared in the construction of a new model.
The initialization of the state is performed by the <i>initState</i> function.
This function is called once before the beginning of the simulation.
</p>

<p>
<h3><font color=\"#008000\">Event Detection</font></h3>

The detection of the events is performed by the equations at the end of the model.<br>
The internal events are detected by the following equation:
<pre>
  internalEvent = if time >= pre(nextInternalTransition) then 1 else 0;
</pre>
Every time the time reaches the value of the nextInternalTransition variable, an internal transition is triggered.<br><br>

The external events are detected by the following equations:
<pre>
  for i in 1:numIn loop
     receivedEvents[i] = - integer(iEvent[i]);
     externalEventNum[i] = if (receivedEvents[i] &gt  pre(receivedEvents[i])) or (receivedEvents[i] &lt  pre(receivedEvents[i])) then 1 else 0;
  end for;
  externalEvent = if sum(externalEventNum) > 0 then 2 else 0;
</pre>
The for loop is neccesary to check the reception of new messages in all the input ports that could have been defined by the user.
 <br><br>
The confluent event checks the simultaneous occurrence of internal and external events.
<pre>
  confluentEvent = externalEvent + internalEvent;
</pre>

The transition to execute is decided by the following equations, that use the previously calculated variables:
<pre>
  confluentTransition = confluentEvent == 3 and time > 0;
  externalTransition = externalEvent == 2 and time > 0 and not confluentTransition;
  internalTransition = internalEvent == 1 and time > 0 and not confluentTransition;
</pre>

</p>
</html>"));

    end Atomic;

    class Communication "Model Communication"

    annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Communication between DEVSLib models</font></h3>

The interfaces for model communication are arranged in the package Interfaces.

<p>
<h3><font color=\"#008000\">Input Ports</font></h3>

The inPort model represents the interface for input messages.<br>
It contains two variables, the event and the queue.<br>
The event represents that a new message or messages have been received at the port.
It is a flow variable because multiple models can send messages to the same port, so the total number of messages must be summed up.<br>
The queue represents a reference to the storage space for the incoming messages.
</p>

<p>
<h3><font color=\"#008000\">Output Ports</font></h3>

The outPort model represents the interface for output messages.<br>
It contains two variables, the event and the queue.<br>
The event represents that a new message or messages have been sent through the port.
It is a flow variable because multiple models can send messages to the same port, so the total number of messages must be summed up.<br>
The queue represents a reference to the storage space for the outgoing messages.
</p>

<p>

<h3><font color=\"#008000\">Messages</font></h3>
<p>
The communication between models is performed as a message passing mechanism.
The output function of one model generates a message that is sent to the input port of a connected model.
The model that receives the message executes an external transition.
</p>
<p>
Due to the restrictions of the Modelica language, this message pasing mechanism has been implemented using external functions that store messages in dynamic memory.
Only references to the messages and the data structures that store the messages (queues, lists, etc) are stored in the Modelica variables.
The implementation of these external functions is included in the file <i>events.c</i>.
</p>


<h3><font color=\"#008000\">Message Management Functions</font></h3>

Several functions for message and queue management have been included in the library:
<ul>
<li>  sendEvent, sends a message to a queue (message is stored a the end of the queue).
<li>  sendEventLIFO, sends a message to a queue (message is stored at the beginning of the queue).
<li>  sendEventLVF, sends a message to a queue (message is stored by increasing 'value').
<li>  sendEventHVF, sends a message to a queue (message is stored by decreasing 'value').
<li>  readEvent, reads a message from the 'position' of the queue.
<li>  getEvent, reads a message from the 'position' of the queue.
<li>  numEvents, returns the number of messages in the queue.
<li>  eventOrder, returns the ordering value for the message in that 'position' of the queue.
<li>  CreateQueue, creates a new queue for messages.
</ul>

</html>"));

    end Communication;

    class HybridModels "Hybrid Models"

    annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>Hybrid System Modeling</font></h3>

 <p>
This package contains the implementation of the models used to translate continuous-time signals into series of messages and viceversa.
The same models have been implemented to work with Real values or Boolean values.
<ul>
<li> The DiCo/DiBo models translate messages into a picewise continuous-time or boolean signal. <br>
The value of the output signal is equaled to the value of the last message received, in the case of real signals.
In the case of boolean signals, the value is true if the message value is greater that 0, and false otherwise.
<li> The Quantizer model implements a first order quantization method, translating the real input into series of messages with value equal to the input signal.<br>
 A message is generated every time the input signal changes its value in a quantum <i>q</i>.
The <i>Threshold</i> parameter is used to define the accuracy in the detection of the signal variation.
The <i>EType</i> parameter defines the value for the type variable of the generated message.
<li> The CrossDown/BCrossDown translate Real and Boolean signals into messages when the signal crosses a predefined value downwards. <br>
The generated message has a value equal to the signal (or 0 in the boolean case) and a type equal to the <i>EType</i> parameter.
<li> The CrossUP/BCrossUP translate Real and Boolean signals into messages when the signal crosses a predefined value upwards. <br>
The generated message has a value equal to the signal (or 1 in the boolean case) and a type equal to the <i>EType</i> parameter.
</ul>
</p>


</html>"));

    end HybridModels;

    class QSSIntegrators "QSS Integrators"

    annotation (DocumentationClass=true, Documentation(info="<html>
<h3><font color=\"#008000\" size=5>QSS Integrators</font></h3>
 <p>
This package contains the implementation of the QSS1, QSS2, and QSS3 integration methods.
</p>
<p>
In order to correctly communicate the value and the derivatives of the variable, several messages are used.<br>
A message with type == 1 transports the value of the variable, a message with type == 2 transports the value of the first derivative, and a message with type == 3 transports the second derivative.
These messages have to be sent simultaneously.
</p>

</html>"));

    end QSSIntegrators;

    end DevelopersGuide;

      partial model AtomicDEVS "Partial atomic DEVS model"

      annotation(preferedView="info",
        Window(
          x=0.02,
          y=0.01,
          width=0.2,
          height=0.57,
          library=1,
          autolayout=1),
        version="2.2",
        versionDate="2005-04-15",
        Settings(NewStateSelection=true),
        Documentation(info="<HTML>


<p>
This model represents the basic component for atomic Parallel DEVS models.<br>
It implements the behavior of an atomic model, independently of the actions performed by the transition, output and time advance functions.
Basically, it detects the occurrence of events and executes the functions that correspond to the detected events.
The actions performed by these functions have to be defined by the user<br>
The state of the model and its initialization also has to be defined by the user.
</p>


<p>
<h3><font color=\"#008000\">Interface Ports</font></h3>

The input ports have to be defined by the user.
In order to allow any number of input and output ports, four arrays (<i>iEvent</i>, <i>iQueue</i>, <i>oEvent</i> and <i>oQueue</i>) have been defined in the model
and internally represent the input (iEvent and iQueue) and output (oEvent and oQueue) ports.
The length of these arrays is defined by the parameters <i>numIn</i> and <i>numOut</i>.
The user must set these parameters to the correct number of input and output ports (the minimum value is 1).
An equation must be defined between the port and its corresponding position in each of the arrays.
The event variable of the port is connected to the Event array, and the queue variable is connected to the Queue array
(i.e. <i>inputPort1.event = iEvent[1]</i>, <i>inputPort1.queue = iQueue[1]</i> ,<i>outputPort1.event = oEvent[1]</i> and <i>outputPort1.queue = oQueue[1]</i>).<br>
<br>
</p>

<p>
<h3><font color=\"#008000\">Functions</font></h3>

Five replaceable functions are defined in this model and represent the transition, output and time advance functions (<i>Fcon, Fext, Fint, Fout and Fta</i>
respectively).
These functions do not perform any action and must be redeclared by the user.
Models without input or output ports does not need the external, confluent or internal transition functions, so they do not have to be redeclared.
The prototype of each function is:
<ul>
<li> sout := Fcon (s,e,bag), where <i>sout</i> is the state after the confluent transition,
<i>soutput</i> is the state that will be used by the output function to generate an output,
<i>s</i> is the current state, <i>e</i> is the elapsed time since the last transition and <i>bag</i> is the reference to the storage of input messages.
<li> sout := Fint (s), where <i>sout</i> is the state after the confluent transition, and <i>s</i> is the current state.
<li> sout := Fext (s,e,bag), where <i>sout</i> is the state after the confluent transition, and <i>s</i> is the current state,
<i>e</i> is the elapsed time since the last transition, and <i>bag</i> is the reference to the storage of input messages.
<li> sigma := Fta(s), where <i>sigma</i> is the time interval until the next internal transition, and <i>s</i> is the current state.
<li> Fout (s,queues[nports],nports), where <i>s</i> is the current state, <i>queues[nports]</i> is the array that contains references to the queues that will receive the messages sent by this function,
and <i>nports</i> is the number of ports connected to the model that executes the function.
</ul>

Alternatively to these parameters, any other parameter can be added to these functions.
New parameters can be added at the end of the function inputs.
The value of the parameter can be set in the redeclaration of the function, when constructing a new model.
Continuous or discrete input values can be passed to these functions using this method.
</p>

<p>
<h3><font color=\"#008000\">State</font></h3>

The state of the model is represented by the record <i>st</i>.
Any number of variables can be included in that record in order to represent the state of the model.
In Dymola 6.2, only simple types (Real, Integer and Boolean) can be included. This bug is fixed in newer versions.
The modified state can be redeclared in the construction of a new model.
The initialization of the state is performed by the <i>initState</i> function.
This function is called once before the beginning of the simulation.
</p>

<p>
<h3><font color=\"#008000\">Event Detection</font></h3>

The detection of the events is performed by the equations at the end of the model.<br>
The internal events are detected by the following equation:
<pre>
  internalEvent = if time >= pre(nextInternalTransition) then 1 else 0;
</pre>
Every time the time reaches the value of the nextInternalTransition variable, an internal transition is triggered.<br><br>

The external events are detected by the following equations:
<pre>
  for i in 1:numIn loop
     receivedEvents[i] = - integer(iEvent[i]);
     externalEventNum[i] = if (receivedEvents[i] &gt  pre(receivedEvents[i])) or (receivedEvents[i] &lt  pre(receivedEvents[i])) then 1 else 0;
  end for;
  externalEvent = if sum(externalEventNum) > 0 then 2 else 0;
</pre>
The for loop is neccesary to check the reception of new messages in all the input ports that could have been defined by the user.
 <br><br>
The confluent event checks the simultaneous occurrence of internal and external events.
<pre>
  confluentEvent = externalEvent + internalEvent;
</pre>

The transition to execute is decided by the following equations, that use the previously calculated variables:
<pre>
  confluentTransition = confluentEvent == 3 and time > 0;
  externalTransition = externalEvent == 2 and time > 0 and not confluentTransition;
  internalTransition = internalEvent == 1 and time > 0 and not confluentTransition;
</pre>

</p>
</HTML>
"),       uses(Modelica(version="2.2.1")));

        parameter String name="AtomicDEVS";
        replaceable record State = stdState;
        inner State S "Current State";
        parameter Integer numIn = 1 "Num of input ports";
        parameter Integer numOut = 1 "Num of output ports";

      protected
        Real iEvent[numIn];
        Integer iQueue[numIn];
        Real oEvent[numOut];
        Integer oQueue[numOut];

        replaceable function Fout "Output Function"
          input State s;
          input Integer queue[nports];
          input Integer nports;
        end Fout;

        replaceable function Fcon "Confluent Transtition Function"
          input State s;
          input Real e;
          input Integer bag;
          output State sout;
        algorithm
          sout := s;
        end Fcon;

        replaceable function Fint "Internal Transition Function"
          input State s;
          output State sout;
        algorithm
          sout := s;
        end Fint;

        replaceable function Fext "External Transition Function"
          input State s;
          input Real e;
          input Integer bag;
          output State sout;
        algorithm
          sout := s;
        end Fext;

        replaceable function Fta "Time advance Function"
          input State s;
          output Real sigma;
        algorithm
          sigma := Modelica.Constants.inf;
        end Fta;

        replaceable partial function initState "Initial State Function"
          output State out;
        end initState;

      //public
      protected
        State Sout;
        Integer Bag;
        Integer receivedEvents[numIn];
        Integer externalEventNum[numIn];
        //Integer receivedEvents;
        Real nextInternalTransition(start = Modelica.Constants.inf);//(start=FirstInternalEvent);
        Real previousInternalTransition( start = 0);
        inner Real elapsedTime( start = 0);
        Integer internalEvent(start = 0);
        Integer externalEvent( start = 0);
        Integer confluentEvent(start = 0);
        Boolean init(start = true);
        stdEvent event;
        Integer numreceived;
        Boolean confluentTransition( start = false);
        Boolean internalTransition( start = false);
        Boolean externalTransition( start = false);
        //Real runTransition( start = Modelica.Constants.inf);
        //Integer e;
        Real timeadvance;

      /*****************************************************************************************************/
      algorithm
        // Initialization
        when initial() and not (time > 0) and pre(init) then
          S := initState();
          Sout := S;
          previousInternalTransition := 0;
          timeadvance := Fta(S);
          if timeadvance < 0 then
            timeadvance := 0;
          end if;
          nextInternalTransition := timeadvance;
          for i in 1:numIn loop
            iQueue[i] := CreateQueue();
          end for;
          Bag := CreateQueue();
          init := false;
        end when;

        // *************************************************************************************
        // *************************************************************************************
        when internalTransition then
      //  when pre(internalTransition) then
          //Modelica.Utilities.Streams.print(String(time)+" "+name+" INTERNAL TRANS "+String(pre(internalEvent))+", "+String(pre(externalEvent)));
          //Modelica.Utilities.Streams.print(String(time)+" "+name+" INTERNAL TRANS "+String(nextInternalTransition)+", "+String(time >= nextInternalTransition));
          //LogVariable(time);
          //LogVariable(nextInternalTransition);
          //while time >= nextInternalTransition loop
            //Modelica.Utilities.Streams.print(String(time)+" "+name+" INTERNAL TRANS ");
            // output generation
            Fout(S, pre(oQueue), numOut);
            for i in 1:numOut loop
              if numEvents(pre(oQueue[i])) > 0 then
                oEvent[i] := pre(oEvent[i]) +1;
              end if;
            end for;
            //internal transition
            S := Fint(S);
            timeadvance := Fta(S);
            if timeadvance < 0 then
              timeadvance := 0;
            end if;
            //LogVariable(timeadvance);
            nextInternalTransition := time + timeadvance;
            previousInternalTransition := time;
          //end while;
          while time >= nextInternalTransition loop // time advance was zero?
            //Modelica.Utilities.Streams.print(String(time)+" "+name+" INTERNAL TRANS ");
            // output generation
            Fout(S, pre(oQueue), numOut);
            for i in 1:numOut loop
              if numEvents(pre(oQueue[i])) > 0 then
                oEvent[i] := pre(oEvent[i]) +1;
              end if;
            end for;
            //internal transition
            S := Fint(S);
            timeadvance := Fta(S);
            if timeadvance < 0 then
              timeadvance := 0;
            end if;
            //LogVariable(timeadvance);
            nextInternalTransition := time + timeadvance;
            previousInternalTransition := time;
          end while;
        end when;

        // *************************************************************************************
        // *************************************************************************************
        // Check for external events
        when externalTransition then
        //when pre(externalTransition) then
      //    Modelica.Utilities.Streams.print(String(time)+" "+name+" EXTERNAL TRANS "+String(pre(internalEvent))+", "+String(pre(externalEvent)));
         //Modelica.Utilities.Streams.print(String(time)+" "+name+" EXTERNAL TRANS ");
          // fill bag
          for i in 1:numIn loop
            numreceived := numEvents(iQueue[i]);
            for j in 1:numreceived loop
              event := getEvent(iQueue[i]);
              event.Port := i;
              sendEvent(Bag,event);
            end for;
          end for;
          S := Fext(S,elapsedTime,Bag);
          timeadvance := Fta(S);
          if timeadvance < 0 then
            timeadvance := 0;
          end if;
          nextInternalTransition := time + timeadvance;
          previousInternalTransition := time;
          // Inmediate Internal transition?
      /*    if nextInternalTransition <= time then
      // output generation
      Fout(S,pre(oQueue), numOut);
      for i in 1:numOut loop
        if numEvents(pre(oQueue[i])) > 0 then
          oEvent[i] := pre(oEvent[i]) +1;
        end if;
      end for;
      // internal transition
      S := Fint(S);
      timeadvance := Fta(S);
      if timeadvance < 0 then
        timeadvance := 0;
      end if;
      nextInternalTransition := time + timeadvance;
      previousInternalTransition := time;
    else
      nextInternalTransition := nextInternalTransition;
      previousInternalTransition := previousInternalTransition;
    end if;*/
        end when;

        // *************************************************************************************
        // *************************************************************************************
        when confluentTransition then
      //  when ore(confluentTransition) then
      //    Modelica.Utilities.Streams.print(String(time)+" "+name+" CONFLUENT TRANS "+String(pre(internalEvent))+", "+String(pre(externalEvent)));
          //Modelica.Utilities.Streams.print(String(time)+" "+name+" CONFLUENT TRANS ");
          // fill bag
          for i in 1:numIn loop
            numreceived := numEvents(iQueue[i]);
            for j in 1:numreceived loop
              event := getEvent(iQueue[i]);
              event.Port := i;
              sendEvent(Bag,event);
            end for;
          end for;
          Fout(S, pre(oQueue), numOut);
          for i in 1:numOut loop
            if numEvents(pre(oQueue[i])) > 0 then
              oEvent[i] := pre(oEvent[i]) +1;
            end if;
          end for;
          S := Fcon(S,elapsedTime,Bag);

          timeadvance := Fta(S);
          if timeadvance < 0 then
            timeadvance := 0;
          end if;
          nextInternalTransition := time + timeadvance;
          //Modelica.Utilities.Streams.print(String(time)+" "+name+" CONFLUENTS TRANS : next int at: "+String(nextInternalTransition));
          previousInternalTransition := time;
        end when;

      equation
        // model equations
        elapsedTime = time - pre(previousInternalTransition);

        // INTERNAL EVENT DETECTION
        internalEvent = if time >= pre(nextInternalTransition) then 1 else 0;
        //LogVariable(internalTransition);

        // EXTERNAL EVENT DETECTION
        for i in 1:numIn loop
          receivedEvents[i] = - integer(iEvent[i]);
          externalEventNum[i] = if (receivedEvents[i] > pre(receivedEvents[i])) or (receivedEvents[i] < pre(receivedEvents[i])) then 1 else 0;
        end for;
        externalEvent = if sum(externalEventNum) > 0 then 2 else 0;

        // CONFLUENT EVENT DETECTION
        confluentEvent = externalEvent + internalEvent;

        // TRANSITION DETECTION
        confluentTransition = confluentEvent == 3;// and time > 0;
        externalTransition = externalEvent == 2 and not confluentTransition;// and time > 0 ;
        internalTransition = internalEvent == 1 and not confluentTransition;// and time > 0 ;

        annotation (Diagram, Icon);
      end AtomicDEVS;

      package Interfaces "Interfaces that represent DEVS ports"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package contains the interfaces used by the DEVS models to send and receive messages.
</p>



</HTML>
"),       uses(Modelica(version="2.2.1")));

        connector inPort "Input port for receiving messages"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<p>
This model represents the interface for input messages.<br>
It contains two variables, the event and the queue.<br>
The event represents that a new message or messages have been received at the port.
It is a flow variable because multiple models can send messages to the same port, so the total number of messages must be summed up.<br>
The queue represents a reference to the storage space for the incoming messages.
</p>

</HTML>
"),         uses(Modelica(version="2.2.1")));

          flow input Real event;
          Integer queue;
          annotation (Diagram, Icon(Rectangle(extent=[-60,60; 60,-60], style(
                        color=0,
                        rgbcolor={0,0,0},
                        fillColor=42,
                        rgbfillColor={150,0,0},
                        fillPattern=1)), Text(
                      extent=[-60,60; 60,-60],
                      string="%name",
                      style(
                        color=7,
                        rgbcolor={255,255,255},
                        fillColor=42,
                        rgbfillColor={150,0,0},
                        fillPattern=1))));
        end inPort;

        connector outPort "Output port for sending messages"

        annotation(preferedView="info",
          Documentation(info="<HTML>
<p>
This model represents the interface for output messages.<br>
It contains two variables, the event and the queue.<br>
The event represents that a new message or messages have been sent through the port.
It is a flow variable because multiple models can send messages to the same port, so the total number of messages must be summed up.<br>
The queue represents a reference to the storage space for the outgoing messages.
</p>

</HTML>
"),         uses(Modelica(version="2.2.1")));
          flow output Real event;
          Integer queue;
             annotation (Icon(Polygon(points=[-60,60; 60,0; 60,0; -60,-60; -60,60],
                                       style(
                          color=0,
                          rgbcolor={0,0,0},
                          fillColor=42,
                          rgbfillColor={150,0,0},
                          fillPattern=1)), Text(
                        extent=[-60,60; 40,-60],
                        string="%name",
                        style(
                          color=7,
                          rgbcolor={255,255,255},
                          fillColor=3,
                          rgbfillColor={0,0,255},
                          fillPattern=1))), Diagram);
        end outPort;

      end Interfaces;

      replaceable record stdEvent "standard event/message"
        Integer Port;
        Integer Type;
        Real Value;
      end stdEvent;

      replaceable record stdState "Standard model state"

      end stdState;

      function sendEvent
        "Sends a message to a queue (message is stored a the end of the queue)"
        input Integer queue;
        input stdEvent e;
        output Integer out;
      external "C" out = QAdd(queue,e.Port,e.Type,e.Value,0);
        annotation (Include="#include <events.c>");
      end sendEvent;

      function sendEventLIFO
        "sends a message to a queue (message is stored at the beginning of the queue)"
        input Integer queue;
        input stdEvent e;
        output Integer out;
      external "C" out = QAddFirst(queue,e.Port,e.Type,e.Value,0);
        annotation (Include="#include <events.c>");
      end sendEventLIFO;

    function sendEventLVF
        "sends a message to a queue (message is stored by increasing 'value')"
      input Integer queue;
      input stdEvent e;
      input Real value;
      output Integer out;
    external "C" out = QAddLVF(queue,e.Port,e.Type,e.Value,0,value);

      annotation (Include="#include <events.c>");
    end sendEventLVF;

    function sendEventHVF
        "sends a message to a queue (message is stored by decreasing 'value')"
      input Integer queue;
      input stdEvent e;
      input Real value;
      output Integer out;
    external "C" out = QAddLVF(queue,e.Port,e.Type,e.Value,0,value);

      annotation (Include="#include <events.c>");
    end sendEventHVF;

      function readEvent "reads a message from the 'position' of the queue"
        input Integer queue;
        input Integer position = 1;
        output stdEvent e;
        output Integer out;
      external "C" out = QRead(queue,position,e.Port,e.Type,e.Value);
        annotation (Include="#include <events.c>");
      end readEvent;

      function getEvent "reads a message from the 'position' of the queue"
        input Integer queue;
        input Integer position = 1;
        output stdEvent e;
        output Integer out;
      external "C" out = QGetPos(queue,position,e.Port,e.Type,e.Value);
        annotation (Include="#include <events.c>");
      end getEvent;

      function numEvents "Returns the number of messages in the queue"
        input Integer queue;
        output Integer out;
      algorithm
        if queue <> 0 then
          out := queueSize(queue);
        else
          out := 0;
        end if;
      end numEvents;

      function queueSize "Returns the number of messages in the queue"
        input Integer queue;
        output Integer out;
      external "C" out = QSize(queue);
        annotation (Include="#include <events.c>");
      end queueSize;

      function eventOrder
        "Returns the ordering value for the message in that 'position' of the queue"
        input Integer queue;
        input Integer position = 1;
        output Real out;
      external "C" out =  QEventOrder(queue,position);
        annotation (Include="#include <events.c>");
      end eventOrder;

      function CreateQueue "Creates a new queue for messages"
        output Integer q;
      external "C" q =  QCreate();
        annotation (Include="#include <events.c>");
      end CreateQueue;

      package DupEvent "Duplicates and forwards messages"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model duplicates and forwards each received message.
Each received message is stored in an internal queue and sent through the two output ports of the model.
</p>
<p>
This model is necessary since an output port can not be connected to several input ports,
because the output port can not store the references to the queues for incoming messages in the input ports.
</p>



</HTML>
"),       uses(Modelica(version="2.2.1")));

        model dupevent
                   extends AtomicDEVS(numOut=2,redeclare record State = st);
          redeclare function Fcon = con;
          redeclare function Fext = ext;
          redeclare function Fint = int;
          redeclare function Fout = out;
          redeclare function Fta = ta;
          redeclare function initState = initst;
          annotation (Diagram, Icon(Line(points=[0,20; -10,20; -10,-20; 0,-20],
                        style(
                        color=0,
                        rgbcolor={0,0,0},
                        thickness=4)), Line(points=[-20,0; -10,0], style(
                        color=0,
                        rgbcolor={0,0,0},
                        thickness=4))));

          Interfaces.outPort out1
                           annotation (extent=[-4,10;16,30]);
          Interfaces.inPort in1
                         annotation (extent=[-36,-10;-16,10]);
          Interfaces.outPort out2
                           annotation (extent=[-4,-30;16,-10]);
        equation
          iEvent[1] = in1.event;
          iQueue[1] = in1.queue;

          oEvent[1] = out1.event;
          oQueue[1] = out1.queue;
          oEvent[2] = out2.event;
          oQueue[2] = out2.queue;

        end dupevent;

        function con "Confluent Transtition Function"
          input st s;
          input Real e;
          input Integer bag;
          output st sout;
        algorithm
          sout := ext(int(s),0,bag);
        end con;

        function int "Internal Transition Function"
          input st s;
          output st sout;
        protected
          Integer n;
        algorithm
          sout := s;
          sout.sigma := Modelica.Constants.inf;
          sout.n := 0;
        end int;

        function ext "External Transition Function"
          input st s;
          input Real e;
          input Integer bag;
          output st sout;
        protected
          Integer numreceived;
          stdEvent x;
        algorithm
          sout := s;
          numreceived := numEvents(bag);
          sout.n := numreceived;
          for i in 1:numreceived loop
            x := DEVSLib.SRC.getEvent(bag);
            sout.e := integer(x.Value);
            sendEvent(sout.q,x);
          end for;
          sout.sigma := 0;
        end ext;

        function out "Output Function"
          input st s;
          input Integer queue[nports];
          input Integer nports;
        protected
          stdEvent y;
        algorithm
          for i in 1:s.n loop
            y := getEvent(s.q);
            sendEvent(queue[1], y);
            sendEvent(queue[2], y);
          end for;
        end out;

        function ta "Time Advance Function"
          input st s;
          output Real sigma;
        algorithm
          sigma := s.sigma;
        end ta;

        record st
          Integer phase; // 1 = passive, 2 = active
          Real sigma;
          Integer q;
          Integer n;
          Integer e;
        end st;

        function initst
          output st out;
        algorithm
          out.phase := 1; // passive
          out.sigma := Modelica.Constants.inf;
          out.q := CreateQueue();
          out.n := 0;
          out.e := 0;
        end initst;
      end DupEvent;

      model breakloop "Breaks algebraic loops in coupled models"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model can be used to break an algebraic loops in coupled models.
This model includes a \"pre\" in the detection of external events, that breaks the loop.
</p>

</HTML>
"));

        Integer e;
        Interfaces.inPort IN
                       annotation (extent=[-38,-10; -18,10]);
        Interfaces.outPort OUT
                         annotation (extent=[-4,-10;16,10]);
      protected
        Integer received;
        Integer receivedNum;
        Integer ext;
        Integer numreceived;
        Boolean extdetected;
        stdEvent x;
        Boolean init( start = true);
        //Real run;

      algorithm
        // Initialization
        when initial() and not (time > 0) and pre(init) then
          IN.queue := CreateQueue();
          init := false;
        end when;

      /*  when extdetected then
    run := time + 1e-6;
  end when;

  when time >= run then
    //Modelica.Utilities.Streams.print(String(time)+" BreakLoop: RUN EXTERNAL");
    numreceived := numEvents(IN.queue);
    for i in 1:numreceived loop
      x := DEVSLib.SRC.getEvent(IN.queue);
      e := integer(x.Value);
      DEVSLib.SRC.sendEvent(pre(OUT.queue),x);
      OUT.event := pre(OUT.event) + 1;
    end for;

  end when;
*/
        when extdetected then
          //Modelica.Utilities.Streams.print(String(time)+" BreakLoop: RUN EXTERNAL");
          numreceived := numEvents(IN.queue);
          for i in 1:numreceived loop
            x := DEVSLib.SRC.getEvent(IN.queue);
            e := integer(x.Value);
            DEVSLib.SRC.sendEvent(pre(OUT.queue),x);
            OUT.event := pre(OUT.event) + 1;
          end for;
        end when;

      equation
        received = - integer(IN.event);
        receivedNum = if (received > pre(received)) or (received < pre(received)) then 1 else 0;
        ext = if pre(receivedNum) > 0 then 2 else 0;
        extdetected = ext == 2;

       annotation (Icon(Line(points=[-8,20; -28,-20], style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=4)), Line(points=[6,20; -14,-20], style(
                      color=0,
                      rgbcolor={0,0,0},
                      thickness=4))));
      end breakloop;

      package Select "Forwards the message following the condition"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model forwards each received message through one of the output ports of the model, depending on the value of the condition parameter.
If the condition is true, the message is sent through the port 1, and otherwise it is sent through the port 2.
</p>



</HTML>
"),       uses(Modelica(version="2.2.1")));

        model select
          replaceable Boolean Condition = true
            "Condition for selecting output branch";
                   extends AtomicDEVS(numOut=2,redeclare record State = st);
          redeclare function Fcon = con(c=Condition);
          redeclare function Fext = ext(c=Condition);
          redeclare function Fint = int;
          redeclare function Fout = out;
          redeclare function Fta = ta;
          redeclare function initState = initst;
          annotation (Diagram, Icon(
              Line(points=[20,40; -10,40; -10,-40; 20,-40],
                        style(
                        color=3,
                        rgbcolor={0,0,255},
                        thickness=4)), Line(points=[-40,0; -10,0], style(
                        color=3,
                        rgbcolor={0,0,255},
                        thickness=4)),
                    Text(
                      extent=[0,60; 20,40],
                      style(
                        color=3,
                        rgbcolor={0,0,255},
                        thickness=4),
                      string="T"),
                    Text(
                      extent=[0,-40; 20,-60],
                      style(
                        color=3,
                        rgbcolor={0,0,255},
                        thickness=4),
                      string="F")));
          Interfaces.outPort out1
                           annotation (extent=[16,30;36,50]);
          Interfaces.inPort in1
                         annotation (extent=[-56,-10;-36,10]);
          Interfaces.outPort out2
                           annotation (extent=[16,-50;36,-30]);
        equation
          iEvent[1] = in1.event;
          iQueue[1] = in1.queue;

          oEvent[1] = out1.event;
          oQueue[1] = out1.queue;
          oEvent[2] = out2.event;
          oQueue[2] = out2.queue;

        end select;

        function con "Confluent Transtition Function"
          input st s;
          input Real e;
          input Integer bag;
          input Boolean c;
          output st sout;
        algorithm
          sout := ext(int(s),0,bag,c=c);
        end con;

        function int "Internal Transition Function"
          input st s;
          output st sout;
        protected
          Integer n;
        algorithm
          sout := s;
          sout.sigma := Modelica.Constants.inf;
          sout.n := 0;
          sout.port := 0;
        end int;

        function ext "External Transition Function"
          input st s;
          input Real e;
          input Integer bag;
          input Boolean c;
          output st sout;
        protected
          Integer numreceived;
          stdEvent x;
        algorithm
          sout := s;
          numreceived := numEvents(bag);
          sout.n := numreceived;
          for i in 1:numreceived loop
            x := DEVSLib.SRC.getEvent(bag);
            sendEvent(sout.q,x);
          end for;
          if c then
            sout.T := integer(x.Value);
            sout.port := 1; // if condition, send through port 1
          else
            sout.F := integer(x.Value);
            sout.port := 2; // otherwise, send through port 2
          end if;
          sout.sigma := 0;
        end ext;

        function out "Output Function"
          input st s;
          input Integer queue[nports];
          input Integer nports;
        protected
          stdEvent y;
        algorithm
          for i in 1:s.n loop
            y := getEvent(s.q);
            sendEvent(queue[s.port], y);
          end for;
        end out;

        function ta "Time Advance Function"
          input st s;
          output Real sigma;
        algorithm
          sigma := s.sigma;
        end ta;

        record st
          Integer phase; // 1 = passive, 2 = active
          Real sigma;
          Integer q;
          Integer n;
          Integer port;
          Integer T; // last entity sent throught true
          Integer F; // last entity sent throught false
        end st;

        function initst
          output st out;
        algorithm
          out.phase := 1; // passive
          out.sigma := Modelica.Constants.inf;
          out.q := CreateQueue();
          out.n := 0;
          out.port := 0;
          out.T := 0;
          out.F := 0;
        end initst;
      end Select;

      package Hybrid "Interface models for hybrid system modeling"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package contains the implementation of the models used to translate continuous-time signals into series of messages and viceversa.
The same models have been implemented to work with Real values or Boolean values.
<ul>
<li> The DiCo/DiBo models translate messages into a picewise continuous-time or boolean signal.
<li> The Quantizer model implements a first order quantization method, translating the real input into series of messages with value equal to the input signal.
<li> The CrossDown/BCrossDown translate Real and Boolean signals into messages when the signal crosses a predefined value downwards.
<li> The CrossUP/BCrossUP translate Real and Boolean signals into messages when the signal crosses a predefined value upwards.
</ul>
</p>
</HTML>
"),       uses(Modelica(version="2.2.1")));

        model DiCo

        annotation(preferedView="info",
          Documentation(info="<HTML>
<p>
This model translate messages into a piecewise continuous-time signal.
</p>
<p>
The value of the output signal is equaled to the value of the last message received.
</p>
</HTML>"),         uses(Modelica(version="2.2.1")));

          Interfaces.inPort inport
                        annotation (extent=[-116,-10;-96,10]);
          Modelica.Blocks.Interfaces.RealOutput y annotation (extent=[100,-10; 120,10]);
          annotation (Icon,Diagram);
          Modelica.Blocks.Interfaces.BooleanOutput change annotation (extent=[100,-50;120,-30]);
        protected
          Boolean externalEvent( start = false);
          Integer receivedEvents;
          Integer eventType;
          stdEvent e;
          Boolean init( start = true);
        algorithm
          when initial() and not (time > 0) and pre(init) then
            inport.queue := CreateQueue();
            init := false;
          end when;

          when pre(externalEvent) then
            e := DEVSLib.SRC.getEvent(inport.queue);
            //Modelica.Utilities.Streams.print("****************** RECEIVED NEW EVENT ON DICO! VALUE = "+String(e.Value));
            y := e.Value;
            change := not change;
            eventType := e.Type;
          end when;
        equation
          receivedEvents =  - integer(inport.event);
          externalEvent = (receivedEvents > pre(receivedEvents)) or (receivedEvents < pre(receivedEvents));
        end DiCo;

      model Quantizer "first order quantizer of a Real signal"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model implements a first order quantization method, translating the real input into series of messages with value equal to the input signal.
</p>
<p>
A message is generated every time the input signal changes its value in a quantum <i>q</i>.
The <i>Threshold</i> parameter is used to define the accuracy in the detection of the signal variation.
The <i>EType</i> parameter defines the value for the type variable of the generated message.
</p>


</HTML>
"),       uses(Modelica(version="2.2.1")));

        replaceable Real q = 1 "Quantum value";
        parameter Real Threshold = 1e-010 "Detection value threshold";
        parameter Integer EType = 1 "Type of the message to generate";

        Interfaces.outPort outport annotation (extent=[96,-10;116,10]);
        Modelica.Blocks.Interfaces.RealInput u annotation (extent=[-120,-10;-100,10]);
        annotation (Icon(
                    Rectangle(extent=[-100,100; 100,-100], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-62,62; 60,-60], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-60,60; -40,60; -40,40; -20,40; -20,20; 0,20; 0,0; 20,
                          0; 20,-20; 40,-20; 40,-40; 60,-40; 60,-60], style(color=10,
                          rgbcolor={135,135,135})),
                    Text(
                      extent=[0,14; 6,6],
                      style(color=10, rgbcolor={135,135,135}),
                      string="q")),
                                Diagram);

      //public
        protected
        stdEvent e;
        Real preEvent( start = u);

      algorithm
        // FIRST ORDER QUANTIFIER
        when ((u >= preEvent+q+Threshold) or (u <= preEvent-q-Threshold)) and (time > 0) then
          //Modelica.Utilities.Streams.print(String(time)+": U ="+String(u)+" PREEVENT = "+String(preEvent));
          e.Type := EType;
          e.Value := u;
          e.Port := 1;
          sendEvent(pre(outport.queue), e);
          outport.event := pre(outport.event) + 1;
          preEvent := u;
        end when;
      end Quantizer;

      model QuantizerUP "first order quantizer of a Real signal"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model implements a first order quantization method, translating the real input into series of messages with value equal to the input signal.
</p>
<p>
A message is generated every time the input signal changes its value in a quantum <i>q</i>.
The <i>Threshold</i> parameter is used to define the accuracy in the detection of the signal variation.
The <i>EType</i> parameter defines the value for the type variable of the generated message.
</p>


</HTML>
"),       uses(Modelica(version="2.2.1")),
          Icon(Line(
                  points=[60,80;80,80], style(
                  color=0, rgbcolor={0,0,255},
                  smooth=Smooth.None)),
                Line(
                  points=[80,80;80,60], style(
                  color=0,
                  rgbcolor={0,0,255},
                  smooth=Smooth.None)),
                Rectangle(extent=[-100,100;100,-100], style(color=3, rgbcolor={0,0,255})),
                Line(
                  points=[-70,-70; 80,80],style(
                  rgbcolor={0,0,255},
                  origin={9,9})),
                Line(
                  points=[-70,-70; -70,-50; -50,-50; -50,-30; -30,-30; -30,-10; -10,
                  -10; -10,10; 10,10; 10,30; 30,30; 30,50; 50,50],  style(
                  rgbcolor={135,135,135},
                  origin={0,0})),
                Text( extent=[-3,24; 3,16],
                      style(color=10, rgbcolor={135,135,135}),
                      string="q")));

        replaceable Real q = 1 "Quantum value";
        parameter Real Threshold = 1e-010 "Detection value threshold";
        parameter Integer EType = 1 "Type of the message to generate";

        Interfaces.outPort outport annotation (extent=[96,-10;116,10]);
        Modelica.Blocks.Interfaces.RealInput u annotation (extent=[-120,-10;-100,10]);

      //public
        protected
        stdEvent e;
        Real preEvent( start = u);

      algorithm
        // FIRST ORDER QUANTIFIER
        when (u >= preEvent+q+Threshold) and (time > 0) then
          //Modelica.Utilities.Streams.print(String(time)+": U ="+String(u)+" PREEVENT = "+String(preEvent));
          e.Type := EType;
          e.Value := u;
          e.Port := 1;
          sendEvent(pre(outport.queue), e);
          outport.event := pre(outport.event) + 1;
          preEvent := u;
        end when;
      end QuantizerUP;

      model QuantizerDOWN "first order quantizer of a Real signal"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model implements a first order quantization method, translating the real input into series of messages with value equal to the input signal.
</p>
<p>
A message is generated every time the input signal changes its value in a quantum <i>q</i>.
The <i>Threshold</i> parameter is used to define the accuracy in the detection of the signal variation.
The <i>EType</i> parameter defines the value for the type variable of the generated message.
</p>


</HTML>
"),       uses(Modelica(version="2.2.1")),
          Icon(Line(
                  points=[80,-80;80,-60], style(
                  rgbcolor={0,0,255},
                  smooth=Smooth.None)),
                Line(
                  points=[60,-80;80,-80], style(
                  rgbcolor={0,0,255},
                  smooth=Smooth.None)),
                Line(points=[-62,62;80,-80], style( rgbcolor={0,0,255})),
                Line(points=[-60,60;-40,60;-40,40;-20,40;-20,20;0,20;0,0;20,0;20,-20;40,-20;40,-40;60,-40;60,-60], style(
                    rgbcolor={135,135,135})),
                Text( extent=[7,14; 13,6],
                      style(color=10, rgbcolor={135,135,135}),
                      string="q"),
                Rectangle(extent=[-100,100;100,-100], style(lineColor={0,0,255}))));

        replaceable Real q = 1 "Quantum value";
        parameter Real Threshold = 1e-010 "Detection value threshold";
        parameter Integer EType = 1 "Type of the message to generate";

        Interfaces.outPort outport annotation (extent=[96,-10;116,10]);
        Modelica.Blocks.Interfaces.RealInput u annotation (extent=[-120,-10;-100,10]);

      //public
        protected
        stdEvent e;
        Real preEvent( start = u);

      algorithm
        // FIRST ORDER QUANTIFIER
        when (u <= preEvent-q-Threshold) and (time > 0) then
          //Modelica.Utilities.Streams.print(String(time)+": U ="+String(u)+" PREEVENT = "+String(preEvent));
          e.Type := EType;
          e.Value := u;
          e.Port := 1;
          sendEvent(pre(outport.queue), e);
          outport.event := pre(outport.event) + 1;
          preEvent := u;
        end when;
      end QuantizerDOWN;

      model CrossDOWN
          "generates a message of type EType, when crossing Value downwards"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model translates a real signal into messages when the signal crosses a predefined <i>Value</i> downwards.
The generated message has a value equal to the signal and a type equal to the <i>EType</i> parameter.
</p>

</HTML>
"),       uses(Modelica(version="2.2.1")));

        replaceable Real Value = 1 "Cross value";
        //parameter Real Threshold = 1e-010 "Detection value threshold";
        parameter Integer EType = 1 "Type of the event to generate";

        Interfaces.outPort outport annotation (extent=[96,-10;116,10]);
        Modelica.Blocks.Interfaces.RealInput u annotation (extent=[-120,-10;-100,10]);
        annotation (Icon(
                    Rectangle(extent=[-100,100; 100,-100],
                                                       style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-60,60; 60,-60], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[40,-50; 60,-60], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[50,-40; 60,-60], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-60,0; 60,0], style(color=10, rgbcolor={135,135,135})),
                    Text(
                      extent=[22,6; 60,0],
                      style(color=10, rgbcolor={135,135,135}),
                      string="Value")), Diagram);

      //public
        protected
        stdEvent e;
        Boolean above( start = u > Value);
      algorithm
        when (u < Value) and time > 0 and above then
         // Modelica.Utilities.Streams.print("CROSS DOWN: u "+String(u)+" value "+String(Value));
          above := false;
          e.Type := EType;
          e.Value := u;
          e.Port := 1;
          sendEvent(pre(outport.queue), e);
          outport.event := pre(outport.event) + 1;
        end when;
        when (u > Value) and time > 0 and not above then
          above := true;
        end when;

      end CrossDOWN;

      model CrossUP
          "generates a message of type EType, when crossing Value upwards"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model translates a real signal into messages when the signal crosses a predefined <i>Value</i> upwards.
The generated message has a value equal to the signal and a type equal to the <i>EType</i> parameter.
</p>

</HTML>
"),       uses(Modelica(version="2.2.1")));

        replaceable Real Value = 1 "Cross value";
        //parameter Real Threshold = 1e-010 "Detection value threshold";
        parameter Integer EType = 1 "Type of the event to generate";

        Interfaces.outPort outport annotation (Placement(transformation(extent=
                    {{96,-10},{116,10}}, rotation=0)));
        Modelica.Blocks.Interfaces.RealInput u annotation (Placement(
                transformation(extent={{-120,-10},{-100,10}}, rotation=0)));
        annotation (Icon(
                    Rectangle(extent=[-100,100; 100,-100],
                                                       style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-60,-60; 60,60], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[40,50; 60,60], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[50,40; 60,60], style(color=3, rgbcolor={0,0,255})),
                    Line(points=[-60,0; 60,0], style(color=10, rgbcolor={135,135,135})),
                    Text(
                      extent=[22,6; 60,0],
                      style(color=10, rgbcolor={135,135,135}),
                      string="Value")),
                                Diagram);

      //public
        protected
        stdEvent e;
        Boolean below( start = u < Value);
      algorithm
        when (u > Value) and time > 0 and pre(below) then
         // Modelica.Utilities.Streams.print(String(time)+": U= "+String(u)+" Value "+String(Value));
         // Modelica.Utilities.Streams.print("CROSS UP");
          below := false;
          e.Type := EType;
          e.Value := u;
          e.Port := 1;
          sendEvent(pre(outport.queue), e);
          outport.event := pre(outport.event) + 1;
        end when;
        when (u < Value) and time > 0 and not below then
          below := true;
        end when;
      end CrossUP;

        model DiBo

        annotation(preferedView="info",
          Documentation(info="<HTML>
<p>
This model translates messages into a boolean signal.
</p>
<p>
The value of the output signal is true if the value of the received message is greater than 0, or false otherwise.
</p>
</HTML>"),         uses(Modelica(version="2.2.1")));

          SRC.Interfaces.inPort inport annotation (extent=[-116,-10;-96,10]);
          annotation (Icon, Diagram);
          Modelica.Blocks.Interfaces.BooleanOutput change annotation (extent=[100,-10; 120,10]);
          parameter String name = "dibo";
        protected
          Boolean externalEvent( start = false);
          Integer receivedEvents;
          Integer numreceived;
          Integer eventType;
          SRC.stdEvent e;
          Boolean init( start = true);
          Real y;
        algorithm
          when initial() and not (time > 0) and pre(init) then
            inport.queue := SRC.CreateQueue();
            init := false;
          end when;

          when pre(externalEvent) then
              numreceived := numEvents(inport.queue);
            for i in 1:numreceived loop
              e := DEVSLib.SRC.getEvent(inport.queue);
              //Modelica.Utilities.Streams.print("****************** RECEIVED NEW EVENT ON DIBO "+name+" ! VALUE = "+String(e.Value));
              y := e.Value;
              if y > 0 then
                change := true;
              else
                change := false;
              end if;
              eventType := e.Type;
            end for;
          end when;
        equation
          receivedEvents =  - integer(inport.event);
          externalEvent = (receivedEvents > pre(receivedEvents)) or (receivedEvents < pre(receivedEvents));
        end DiBo;

      model BCrossDOWN
          "generates event of type EType, when crossing Value downwards"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model translates a boolean signal into messages when the signal changes from true to false.
The generated message has value 0 and a type equal to the <i>EType</i> parameter.
</p>

</HTML>
"),       uses(Modelica(version="2.2.1")),
            Icon(Text(
                extent=[-60,100; 60,60],
                style(color=3, rgbcolor={0,0,255}),
                string="BOOL")));

        //parameter Real Threshold = 1e-010 "Detection value threshold";
        parameter Integer EType = 1 "Type of the event to generate";

        Interfaces.outPort outport annotation (extent=[96,-10;116,10]);
        annotation (Icon(
                Rectangle(extent=[-100,100;100,-100], style( lineColor={0,0,255})),
                Line(points=[-60,60;60,-60], style( rgbcolor={0,0,255})),
                Line(points=[40,-50;60,-60], style( rgbcolor={0,0,255})),
                Line(points=[50,-40;60,-60], style( rgbcolor={0,0,255})),
                Line(points=[-60,0;60,0], style( rgbcolor={135,135,135})),
                Text(
                  extent=[22,6;60,0], style(
                  lineColor={135,135,135}),
                  textString="Value")),
                          Diagram);

      //public
        protected
        stdEvent e;
        public
        Modelica.Blocks.Interfaces.BooleanInput u
            annotation (extent=[-140,-20;-100,20]);
      algorithm
        when not u and pre(u) and time > 0 then //(not edge(u)) and time > 0 then
          Modelica.Utilities.Streams.print(String(time)+" CROSS DOWN: u "+String(not edge(u)));
          e.Type := EType;
          e.Value := 0;
          e.Port := 1;
          sendEvent(outport.queue, e);
          outport.event := pre(outport.event) + 1;
        end when;

      end BCrossDOWN;

      model BCrossUP

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model translates a boolean signal into messages when the signal changes from false to true.
The generated message has value 1 and a type equal to the <i>EType</i> parameter.
</p>

</HTML>
"),       uses(Modelica(version="2.2.1")));

        parameter Real Threshold = 1e-010 "Detection value threshold";
        parameter Integer EType = 1 "Type of the event to generate";

        Interfaces.outPort outport annotation (extent=[96,-10;116,10]);
        annotation (Icon(
                Rectangle(extent=[-100,100;100,-100], style(lineColor={0,0,255})),
                Line(points=[-60,-60;60,60], style(rgbcolor={0,0,255})),
                Line(points=[40,50;60,60], style(rgbcolor={0,0,255})),
                Line(points=[50,40;60,60], style(rgbcolor={0,0,255})),
                Line(points=[-60,0;60,0], style(rgbcolor={135,135,135})),
                Text(
                  extent=[22,6;60,0], style(
                  lineColor={135,135,135}),
                  textString="Value")),
                          Diagram);

      //public
        protected
        stdEvent e;
        public
        Modelica.Blocks.Interfaces.BooleanInput u
            annotation (extent=[-140,-20;-100,20]);
      algorithm
        when edge(u) and time > 0 then
          //Modelica.Utilities.Streams.print(String(time)+": U= "+String(u)+" Value "+String(Value));
          //Modelica.Utilities.Streams.print("CROSS UP");
          e.Type := EType;
          e.Value := 1;
          e.Port := 1;
          sendEvent(outport.queue, e);
          outport.event := pre(outport.event) + 1;
        end when;

      end BCrossUP;

      model CondGen "Conditional message generator"

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This model translates a real signal into messages when the signal crosses a predefined <i>Value</i> upwards.
The generated message has a value equal to the signal and a type equal to the <i>EType</i> parameter.
</p>

</HTML>
"),       uses(Modelica(version="2.2.1")),
            Icon(
                Text(
                  extent=[-80,100;80,20], style(
                  lineColor={0,0,255}),
                  textString="Cond"),
                Rectangle(extent=[-100,100;100,-100], style( lineColor={0,0,255})),
                Line(points=[-64,-60;76,-60], style(rgbcolor={0,0,255})),
                Line(points=[-44,-60;-44,0], style(rgbcolor={0,0,255})),
                Line(points=[-4,-60;-4,0], style(rgbcolor={0,0,255})),
                Line(points=[36,-60;36,0], style(rgbcolor={0,0,255})),
                Polygon(
                  points=[76,-60;60,-66;60,-54;76,-60], style(
                  lineColor={0,0,255},
                  fillColor={0,0,255},
                  fillPattern=FillPattern.Solid))));

        replaceable Boolean Condition = false "Cross value";
        //parameter Real Threshold = 1e-010 "Detection value threshold";
        parameter Integer Type = 1 "Type of the event to generate";
        parameter Integer Value = 1 "Type of the event to generate";

        Interfaces.outPort outport annotation (extent=[96,-10;116,10]);
        annotation (Icon(
              Rectangle(extent=[-100,100; 100,-100],
                                                 style(color=3, rgbcolor={0,0,255})),
              Line(points=[-60,0; 60,0], style(color=10, rgbcolor={135,135,135})),
              Text(
                extent=[22,6; 60,0],
                style(color=10, rgbcolor={135,135,135}),
                string="Value")),
                          Diagram(graphics));

      //public
        protected
        stdEvent e;
      algorithm
        when Condition then
          e.Type := Type;
          e.Value := Value;
          e.Port := 1;
          sendEvent(pre(outport.queue), e);
          outport.event := pre(outport.event) + 1;
        end when;
      end CondGen;
      end Hybrid;

      package QSS "QSS integration methods"

          function pow
          input Real a;
          input Real b;
          output Real y;
          algorithm
          y := a^b;
          end pow;

       function minposroot
         input Real[4] coeff;
         input Integer order;
         output Real mpr;
        protected
         Real disc;
         Real sd;
         Real r1;
         Real q;
         Real r;
         Real s;
         Real t;
         Real rho;
         Real th;
         Real rho13;
         Real costh3;
         Real sinth3;
         Real spt;
         Real smti32;
       algorithm
         if (order == 0) then
           mpr := Modelica.Constants.inf;
         elseif (order == 1) then
           if coeff[2] == 0 then
             mpr := Modelica.Constants.inf;
           else
             mpr :=-coeff[1]/coeff[2];
           end if;
           if mpr < 0 then
             mpr := Modelica.Constants.inf;
           end if;
         elseif (order == 2) then
           if (coeff[3]==0) then
             if (coeff[2]==0) then
               mpr := Modelica.Constants.inf;
             else
               mpr := -coeff[1]/coeff[2];
             end if;
             if (mpr<0) then
               mpr := Modelica.Constants.inf;
             end if;
           else
             disc := coeff[2]*coeff[2]-4*coeff[3]*coeff[1];
             if (disc < 0) then
               mpr := Modelica.Constants.inf;
             else
               sd := sqrt(disc);
               r1 := (-coeff[2]+sd)/2/coeff[3];
               if (r1>0) then
                 mpr:=r1;
               else
                 mpr:= Modelica.Constants.inf;
               end if;
               r1 := (-coeff[2]-sd)/2/coeff[3];
               if ((r1 > 0) and (r1 < mpr)) then
                  mpr := r1;
               end if;
             end if;
           end if;
         elseif (order == 3) then
           if ((coeff[4]==0) or (1000*abs(coeff[4]) < abs(coeff[3]))) then
             mpr := minposroot(coeff,2);
           else
             q := (3*coeff[4]*coeff[2]-coeff[3]*coeff[3])/9/coeff[4]/coeff[4];
             r := (9*coeff[4]*coeff[3]*coeff[2]-27*coeff[4]*coeff[4]*coeff[1]-2*coeff[3]*coeff[3]*coeff[3])/54/coeff[4]/coeff[4]/coeff[4];
             disc := q*q*q + r*r;
             mpr := Modelica.Constants.inf;
             if (disc >= 0) then
               //only one real root
               sd := sqrt(disc);
               if (r+sd>0) then
                 s := pow(r+sd,1.0/3);
               else
                 s := -pow(abs(r+sd),1.0/3);
               end if;
               if (r-sd>0) then
                 t:=pow(r-sd,1.0/3);
               else
                 t:=-pow(abs(r-sd),1.0/3);
               end if;
               r1:=s + t - coeff[3]/3/coeff[4];
               if (r1>0) then
                 mpr:=r1;
               end if;
             else
               rho:=sqrt(-q*q*q);
               th:=Modelica.Math.acos(r/rho);
               rho13:=pow(rho,1.0/3);
               costh3:=cos(th/3);
               sinth3:=sin(th/3);
               spt:=rho13*2*costh3;
               smti32:=-rho13*sinth3*sqrt(3);
               r1:=spt-coeff[3]/3/coeff[4];
               if (r1>0) then
                 mpr:=r1;
               end if;
               r1:=-spt/2 - coeff[3]/3/coeff[4] + smti32;
               if ((r1>0) and (r1<mpr)) then
                 mpr:=r1;
               end if;
               r1:=r1-2*smti32;
               if ((r1>0) and (r1<mpr)) then
                 mpr:=r1;
               end if;
             end if;
           end if;
         end if;
       end minposroot;

      annotation(preferedView="info",
        Documentation(info="<HTML>
<p>
This package contains the implementation of the QSS1, QSS2, and QSS3 integration methods.
</p>
<p>
In order to correctly communicate the value and the derivatives of the variable, several messages are used.<br>
A message with type == 1 transports the value of the variable, a message with type == 2 transports the value of the first derivative, and a message with type == 3 transports the second derivative.
These messages have to be sent simultaneously.
</p>



</HTML>
"),       uses(Modelica(version="2.2.1")));

      package IntegratorQSS1

          import DESLib.DEVSLib.SRC.*;
            model integratorQSS1
              parameter Real quantum=1;
              parameter Real startX=10;
                       extends AtomicDEVS(numIn=1,
              redeclare record State = st);
              redeclare function Fcon = con;
              redeclare function Fext = ext;
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(q=quantum,x=startX);
              annotation (Diagram,Icon);
              Interfaces.outPort outPort1     annotation (extent=[80,-10;100,10]);
              Interfaces.inPort inPort1     annotation (extent=[-100,-10;-80,10]);

            Real Y( start = startX);
            Real preTime( start = 0);
            equation
              when externalTransition then
                preTime = time;
              end when;
              Y = S.X + S.u*(time-preTime);

              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;

            end integratorQSS1;

            function con "Confluent Transtition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            algorithm
              sout.phase := 1;
              sout := ext(int(s),0,bag);
            end con;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;

              if s.u == 0 then
                sout.y := s.X;
              else
            //  y := s.X + s.dQ * s.u / abs(s.u);
                sout.y := s.X + s.sigma * s.u;
              end if;

              sout.X := sout.X + sout.sigma * sout.u;
              sout.q := sout.X;
              if (sout.u == 0) then
                sout.sigma := Modelica.Constants.inf;
              else
                sout.sigma := abs(sout.dQ/sout.u);  // --------------------
              end if;
            end int;

            function ext "External Transition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
          protected
              Integer numreceived;
              stdEvent x;
            algorithm
              sout := s;
              numreceived := numEvents(bag);
              sout.X := sout.X + sout.u * e;
              for i in 1:numreceived loop
                x := getEvent(bag);
                if x.Type == 1 then
                  sout.u := x.Value;
                end if;
              end for;
              if sout.sigma > 0 then
                if sout.u == 0 then
                  sout.sigma := Modelica.Constants.inf;
                else
                  if sout.u > 0 then
                    sout.sigma := (sout.q + sout.dQ - sout.X)/sout.u; // ----
                  else
                    sout.sigma := (sout.q - sout.dQ - sout.X)/sout.u; // ----
                  end if;
                  if (abs(sout.X - sout.q) > sout.dQ) then
                    sout.sigma := 0;
                  end if;
                end if;
              end if;
            end ext;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent y;
            algorithm
              y.Type := 1;
              if s.u == 0 then
                y.Value := s.X;
              else
            //          y.Value := s.X + s.dQ * s.u / abs(s.u);
                y.Value := s.X + s.sigma * s.u;
              end if;
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase;
              Real sigma;
              Real dQ;
              Real X;
              Real q;
              Real u;
              Real S;
              Real y;
            end st;

            function initst
              input Real q;
              input Real x;
              output st out;
            algorithm
              out.phase := 2;
              out.sigma := 0;
              out.dQ := q;
              out.X := x;
              out.q := x;
            //        out.q := floor(x/q)*q;
              out.u := 0;
              out.S := 0;
              out.y := 0;
            end initst;
      end IntegratorQSS1;

          package IntegratorQSS2

            import DESLib.DEVSLib.SRC.*;
            model integratorQSS2
              parameter Real quantum=1;
              parameter Real startX=10;
                       extends AtomicDEVS(numIn=1,
              redeclare record State = st);
              redeclare function Fcon = con;
              redeclare function Fext = ext;
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(q=quantum,x=startX);
              annotation (Diagram,Icon);
              Interfaces.outPort outPort1     annotation (extent=[80,-10;100,10]);
              Interfaces.inPort inPort1     annotation (extent=[-100,-10;-80,10]);

              Real Y( start = startX);
              Real preTime( start = 0);
            equation
              when externalTransition then
                preTime = time;
              end when;
              Y = S.X + S.u*(time-preTime) + S.mu*(time-preTime)*(time-preTime);

              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
            end integratorQSS2;

            function con "Confluent Transtition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            algorithm
              sout := ext(int(s),0,bag);
            end con;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;

              sout.y0 := s.X + s.u*s.sigma + s.mu*s.sigma*s.sigma;
              sout.y1 := s.u + 2*s.mu*s.sigma;

              //    sout.X := sout.X + sout.sigma*sout.u + sout.mu/2*sout.sigma*sout.sigma;
              sout.X := sout.X + sout.u*sout.sigma + sout.mu*sout.sigma*sout.sigma;
              sout.u := sout.u + 2*sout.mu*sout.sigma;
              //sout.S0 := s.X + s.u*s.sigma + s.mu*s.sigma*s.sigma/2;

              sout.q := sout.X;
              //sout.u := sout.u + sout.sigma*sout.mu;
              sout.uq := sout.u;
              if (sout.mu == 0) then
                sout.sigma := Modelica.Constants.inf;
              else
                //sout.sigma := sqrt(2*sout.dQ/abs(sout.mu));
                sout.sigma := sqrt(abs(sout.dQ/sout.mu));
              end if;
            end int;

            function ext "External Transition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
          protected
              Integer numreceived;
              stdEvent x;
              Real diffxq[4];
              Real dt1;
            algorithm
              sout := s;
              numreceived := numEvents(bag);
              //sout.X := sout.X + e*sout.u + sout.mu*e*e/2;
              sout.X := sout.X + sout.u*e + sout.mu*e*e;
              for i in 1:numreceived loop
                x := getEvent(bag);
                if x.Type == 1 then
                  sout.u := x.Value;
                elseif x.Type == 2 then
                  //sout.mu := x.Value;
                  sout.mu := x.Value/2;
                end if;
              end for;
              if sout.sigma <> 0 then
                sout.q := sout.q + sout.uq*e;
                diffxq[1] := sout.q - sout.X - sout.dQ;
                diffxq[2] := sout.uq - sout.u;
                diffxq[3] := -sout.mu;
                diffxq[4] := 0;
                sout.sigma := minposroot(diffxq,2);
                diffxq[1] := sout.q - sout.X + sout.dQ;
                dt1 := minposroot(diffxq,2);
                if (dt1 < sout.sigma) then
                  sout.sigma := dt1;
                end if;
                if (abs(sout.X - sout.q) > sout.dQ) then
                  sout.sigma := 0;
                end if;
                /*sout.sigma := getSigma2(
          sout.dQ,
          sout.sigma,
          sout.q,
          sout.uq,
          sout.u,
          sout.mu,
          sout.X,
          e);*/
              end if;
              //Modelica.Utilities.Streams.print(" INTEG SIGMA= "+String(sout.sigma));
            end ext;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
              output Integer port[nports];
          protected
              stdEvent y;
            algorithm
              port[1] := 1;
              y.Type := 1;
              //y.Value := s.X + s.u*s.sigma + s.mu*s.sigma*s.sigma/2;
              y.Value := s.X + s.u*s.sigma + s.mu*s.sigma*s.sigma;
              //Modelica.Utilities.Streams.print(String(y.Value));
              sendEvent(queue[1],y);
              y.Type := 2;
              //y.Value := s.u + s.mu*s.sigma;
              y.Value := s.u + 2*s.mu*s.sigma;
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase;
              Real sigma;
              Real dQ;
              Real X;
              Real q;
              Real u;
              Real mu;
              Real uq;
              Real y0;
              Real y1;
            end st;

            function initst
              input Real q;
              input Real x;
              output st out;
            algorithm
              out.phase := 0;
              out.sigma := 0; // first internal event never happens
              out.dQ := q;
              out.X := x;
              out.u := 0;
              out.mu := 0;
              out.q := x;
              out.uq := 0;
              out.y0 := 0;
              out.y1 := 0;
            end initst;

          function getSigma1
            input Real quantum;
            input Real sigma;
            input Real q;
            input Real mq;
            input Real u;
            input Real mu;
            input Real X;
            input Real e;
            output Real sigma_o;
          protected
            Real a;
            Real b;
            Real c;
            Real s;

          algorithm
            if (sigma<>0) then
              a:= mu/2;
              b:= u - mq;
              c:= X - q + quantum;
              sigma_o:=Modelica.Constants.inf;
              if (a==0) then
                if (b<>0) then
                   s:=-c/b;
                   if (s>0) then
                     sigma_o:=s;
                   end if;
                   c:=X - q - quantum;
                   s:=-c/b;
                   if s>0 and s<sigma_o then
                     sigma_o:=s;
                   end if;
                end if;
              else
                if b*b-4*a*c > 0 then
                  s:=(-b + sqrt(b*b - 4*a*c))/2/a;
                  if (s>0) then
                    sigma_o:=s;
                  end if;
                  s:=(-b - sqrt(b*b - 4*a*c))/2/a;
                  if s>0 and s<sigma_o then
                    sigma_o:=s;
                  end if;
                end if;
                c:= X - q - quantum;
                if b*b-4*a*c > 0 then
                  s:=(-b + sqrt(b*b - 4*a*c))/2/a;
                  if s>0 and s<sigma_o then
                    sigma_o:=s;
                  end if;
                  s:=(-b - sqrt(b*b - 4*a*c))/2/a;
                  if s>0 and s<sigma_o then
                    sigma_o:=s;
                  end if;
                end if;
              end if;
              if (X-q)>quantum or (q-X)>quantum then
                sigma_o:=0;
              end if;
            end if;
          end getSigma1;

          function getSigma2
            input Real quantum;
            input Real sigma;
            input Real q;
            input Real mq;
            input Real u;
            input Real mu;
            input Real X;
            input Real e;
            output Real sigma_o;
          protected
            Real a;
            Real b;
            Real c;
            Real s;
            Real s2;
            Real D;
            Real denom;
            Real tol2 = quantum;
          algorithm
            a := mu/2;
            b := u - mq;
            c := X - q;
            if (b < 0) then
              a := -a;
              b := -b;
              c := -c;
            end if;
            D := b*b - 4*a*(c - tol2);
            if (D >= 0) then
              denom := -b-sqrt(D);
              if (denom >= 0) then
                s := Modelica.Constants.inf;
              else
                s :=(2*(c - tol2))/denom;
              end if;
            else
              s :=(-b - sqrt(b*b - 4*a*(c + tol2)))/(2*a);
            end if;
            sigma_o:=s;
          end getSigma2;
          end IntegratorQSS2;

          package IntegratorQSS3

            import DESLib.DEVSLib.SRC.*;
            model integratorQSS3
              parameter Real quantum=1;
              parameter Real startX=10;
                       extends AtomicDEVS(numIn=1,
              redeclare record State = st);
              redeclare function Fcon = con;
              redeclare function Fext = ext;
              redeclare function Fint = int;
              redeclare function Fout = out;
              redeclare function Fta = ta;
              redeclare function initState = initst(q=quantum,x=startX);
              annotation (Diagram,Icon);
              Interfaces.outPort outPort1     annotation (extent=[80,-10;100,10]);
              Interfaces.inPort inPort1     annotation (extent=[-100,-10;-80,10]);
              Real Y( start = startX);
              Real preTime( start = 0);
            equation
              when externalTransition then
                preTime = time;
              end when;
              Y = S.X + S.u*(time-preTime) + S.mu*(time-preTime)*(time-preTime) + S.pu*(time-preTime)*(time-preTime)*(time-preTime);

              iEvent[1] = inPort1.event;
              iQueue[1] = inPort1.queue;

              oEvent[1] = outPort1.event;
              oQueue[1] = outPort1.queue;
            end integratorQSS3;

            function con "Confluent Transtition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
            algorithm
              sout := ext(int(s),0,bag);
            end con;

            function int "Internal Transition Function"
              input st s;
              output st sout;
            algorithm
              sout := s;
              // outputs
              sout.y0 := s.X + s.u*s.sigma + s.mu*s.sigma*s.sigma + s.pu*s.sigma*s.sigma*s.sigma;
              sout.y1 := s.u + s.mu*s.sigma + s.pu*s.sigma*s.sigma;
              sout.y2 := s.mu + s.pu*s.sigma;
              // --

              sout.X := sout.X + sout.u*sout.sigma + sout.mu*sout.sigma*sout.sigma + sout.pu*sout.sigma*sout.sigma*sout.sigma;
              sout.u := sout.u + sout.mu*sout.sigma + sout.pu*sout.sigma*sout.sigma;
              sout.mu := sout.mu + sout.pu*sout.sigma;
              sout.q := sout.X;
              sout.uq := sout.u;
              sout.mq := sout.mu;
              if (sout.pu == 0) then
                sout.sigma := Modelica.Constants.inf;
              else
                sout.sigma := pow(abs(sout.dQ/sout.pu),1.0/3);
              end if;
            end int;

            function ext "External Transition Function"
              input st s;
              input Real e;
              input Integer bag;
              output st sout;
          protected
              Integer numreceived;
              stdEvent x;
              Real diffxq[4];
              Real dt1;
            algorithm
              sout := s;
              numreceived := numEvents(bag);
              sout.X := sout.X + sout.u*e + sout.mu*e*e + sout.pu*e*e*e;
              for i in 1:numreceived loop
                x := getEvent(bag);
                if x.Type == 1 then
                  sout.u := x.Value;
                elseif x.Type == 2 then
                  sout.mu := x.Value/2;
                elseif x.Type == 3 then
                  sout.pu := x.Value/3;
                end if;
              end for;
              if sout.sigma > 0 then
                sout.q := sout.q + sout.uq*e + sout.mq*e*e;
                sout.uq := sout.uq + sout.mq*e;
                diffxq[1] := sout.q - sout.X - sout.dQ;
                diffxq[2] := sout.uq  - sout.u;
                diffxq[3] := sout.mq - sout.mu;
                diffxq[4] := -sout.pu;
                sout.sigma := minposroot(diffxq,3);
                diffxq[1] := sout.q - sout.X + sout.dQ;
                dt1 := minposroot(diffxq,3);
                if (dt1 < sout.sigma) then
                  sout.sigma := dt1;
                end if;
                if (abs(sout.X - sout.q) > sout.dQ) then
                  sout.sigma := 0;
                end if;
              end if;
            end ext;

            function out "Output Function"
              input st s;
              input Integer queue[nports];
              input Integer nports;
          protected
              stdEvent y;
            algorithm
              y.Type := 1;
              y.Value := s.X + s.u*s.sigma + s.mu*s.sigma*s.sigma + s.pu*s.sigma*s.sigma*s.sigma;
              sendEvent(queue[1],y);
              y.Type := 2;
              y.Value := s.u + s.mu*s.sigma + s.pu*s.sigma*s.sigma;
              sendEvent(queue[1],y);
              y.Type := 3;
              y.Value := s.mu + s.pu*s.sigma;
              sendEvent(queue[1],y);
            end out;

            function ta "Time Advance Function"
              input st s;
              output Real sigma;
            algorithm
              sigma := s.sigma;
            end ta;

            record st
              Integer phase;
              Real sigma;
              Real dQ;
              Real X;
              Real u;
              Real mu;
              Real pu;
              Real q;
              Real uq;
              Real mq;
              Real y0;
              Real y1;
              Real y2;
            end st;

            function initst
              input Real q;
              input Real x;
              output st out;
            algorithm
              out.phase := 0;
              out.sigma := 0; // first internal event never happens
              out.dQ := q;
              out.X := x;
              out.q := x;
              out.u := 0;
              out.mu := 0;
              out.mq := 0;
              out.pu := 0;
              out.y0 := 0;
              out.y1 := 0;
              out.y2 := 0;
            end initst;

          function getSigmaQSS3_2
            input Real dQ;
            input Real sigma_i;
            input Real q;
            input Real mq;
            input Real pq;
            input Real u;
            input Real mu;
            input Real pu;
            input Real X;
           // input Real e;
            output Real sigma;
          protected
            Real a;
            Real b;
            Real c;
            Real s;
            Real v;
            Real w;
            Real A;
            Real B;
            Real i1;
            Real i2;
            Real x1;
            Real x2;
            Real y1;
            Real y2;
            Real y3;
            Real arg;
            Real pidiv3;

          algorithm
            //